#region Using declarations
using System;
using System.Windows;
using System.Windows.Media;
using System.Linq;
using NinjaTrader.Core.FloatingPoint;
using NinjaTrader.Gui;
#endregion

//This namespace holds Indicators in this folder and is required. Do not change it. 
namespace NinjaTrader.NinjaScript.AddOns
{
    /// <summary>
    /// A utility class to calculate delta. For use in any indicator. 
    /// To use in your own indicator see Sim22_DeltaV3 for an example.
    /// Sim22 NT8b12. Aug 2016.
    /// 
    /// </summary>
    public class Sim22_DeltaUtilities
    {
        public enum DeltaCalculationTypeEnum
        {
            BidAsk,
            UpDownTick,
            UpDownTickWithContinuation,
            UpDownOneTickWithContinuation,
            Hybrid,
        }

        public enum DeltaTradeSizeFilterTypeEnum
        {
            GreaterOrEqualTo,
            LessThanOrEqualTo,
            None
        }

        public enum DeltaCotType
        {
            None,
            HighCot,
            LowCot,
            CombinedCot
        }

        public enum DeltaEmaType
        {
            None,
            Continuous,
            SessionRestart
        }

        private int lastDirection = 0;
        private double p1 = 0d;
        private bool startLookingForReversal = false;

        public double GetCurrentDelta(DeltaCalculationTypeEnum deltaCalculationType, DeltaTradeSizeFilterTypeEnum deltaFilterType, double ask, double bid, double price, double volume, bool invertDelta, long filterSize)
        {
            double p = price; double a = ask; double b = bid; double vol = volume; double delta = 0;

            int direction = lastDirection;

            if ((deltaFilterType == DeltaTradeSizeFilterTypeEnum.GreaterOrEqualTo) && (vol < filterSize))
                return 0.0;

            if ((deltaFilterType == DeltaTradeSizeFilterTypeEnum.LessThanOrEqualTo) && (vol > filterSize))
                return 0.0;

            switch (deltaCalculationType)
            {
                case DeltaCalculationTypeEnum.BidAsk:
                    //Delta value is the trade +-volume when price hits either the bid or ask.
                    if (a > b)
                    {
                        if (p >= a)
                        {
                            delta = vol;
                        }

                        if (p <= b)
                        {
                            delta = -vol;
                        }

                        if (p < a && p > b)
                        {
                            if (a - p < p - b)
                            {
                                delta = vol;
                            }
                            if (a - p > p - b)
                            {
                                delta = -vol;
                            }
                        }
                    }
                    else if (a < b)
                    {
                        if (p >= b)
                        {
                            delta = vol;

                        }
                        if (p <= a)
                        {
                            delta = -vol;

                        }
                        if (p < b && p > a)
                        {
                            if (b - p < p - a)
                            {
                                delta = vol;

                            }
                            if (b - p > p - a)
                            {
                                delta = -vol;

                            }
                        }
                    }

                    else if (a.ApproxCompare(b) == 0)
                    {
                        if (p > a)
                        {
                            delta = vol;
                        }
                        if (p < b)
                        {
                            delta = -vol;
                        }
                    }
                    break;

                case DeltaCalculationTypeEnum.UpDownTick:
                    //Delta value is the trade +-volume of the first tick when price moves
                    if (p1.ApproxCompare(0) == 0)
                        delta = 0.0;

                    if (p > p1)
                    {
                        delta = vol;
                    }
                    else if (p < p1)
                    {
                        delta = -vol;
                    }
                    else
                    {
                        delta = 0.0;
                    }
                    break;

                case DeltaCalculationTypeEnum.UpDownTickWithContinuation:
                case DeltaCalculationTypeEnum.UpDownOneTickWithContinuation:
                    //Delta value is the trade +-volume when price moves in a trending direction.
                    if (p > p1)
                    {
                        direction = 1;
                    }
                    else if (p < p1)
                    {
                        direction = -1;
                    }

                    if (deltaCalculationType == DeltaCalculationTypeEnum.UpDownOneTickWithContinuation)
                        delta = direction;
                    else
                        delta = direction * vol;
                    break;

                case DeltaCalculationTypeEnum.Hybrid:
                    //Delta value is the trade +-volume when a possible reversal changes the direction.
                    if (p > p1)
                    {
                        direction = 1;
                        startLookingForReversal = false;
                    }
                    else if (p < p1)
                    {
                        direction = -1;
                        startLookingForReversal = false;
                    }

                    if (!startLookingForReversal)
                        if (direction == 1)
                            startLookingForReversal = (p <= b);
                        else
                            startLookingForReversal = (p >= a);

                    if (p.ApproxCompare(p1) == 0)
                    {
                        if ((direction == 1) && startLookingForReversal && ((p.ApproxCompare(a) == 0) || (p < a && p > b)))
                            direction = -1;

                        else if ((direction == -1) && startLookingForReversal && ((p.ApproxCompare(b) == 0) || (p < a && p > b)))
                            direction = 1;
                    }

                    delta = direction * vol;
                    break;
            }

            p1 = p;
            lastDirection = direction;

            if (invertDelta)
                return -delta;
             
            return delta;
        }
    }
}
