// 
// Copyright (C) 2015, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//
#region Using declarations
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Input;
using System.Windows.Media;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Gui;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui.SuperDom;
using NinjaTrader.Data;
using NinjaTrader.NinjaScript;
using NinjaTrader.Core.FloatingPoint;
using NinjaTrader.NinjaScript.DrawingTools;
#endregion

// This namespace holds indicators in this folder and is required. Do not change it.
namespace NinjaTrader.NinjaScript.Indicators.Sim22.Gapless
{
    /// <summary>
    /// Exponential Moving Average. The Exponential Moving Average is an indicator that
    /// shows the average value of a security's price over a period of time. When calculating
    /// a moving average. The Sim22_EmaGaplessV2 applies more weight to recent prices than the SMA. 
    /// @"Intraday EMA that ignores price gaps at the start of the session. Useful only for trend trading from the first bars of the session. Sim22 Dec 2015 NT8b6".
    /// Thanks to Fat Tails for this original NT7 code. Original concept sourced from the TradeStation forums.
    /// Updated Aug 2016 NT8b12. Tested for tick replay. Now works on EOD charts. Can be used with any input.
	/// Updated Nov 2016 NT8b13. Added user selection of Ema plot type.
    /// </summary>
    public class Sim22_EmaGaplessV2 : Indicator
	{
		private double constant1;
		private double constant2;
        private string displayName;
        private double prevInput;
        private int startBar = 0;

		protected override void OnStateChange()
		{
			if (State == State.SetDefaults)
			{
				Description					= @"Intraday EMA that ignores price gaps at the start of the session. Sim22 Aug 2016 NT8b12";
				Name						= "Sim22_EMA_GaplessV2";
				IsOverlay					= true;
				IsSuspendedWhileInactive	= true;
				Period						= 14;
				GaplessEmaType                 = GaplessEmaTypeEnum.SessionStartGapless;
				AddPlot(Brushes.Orange, "EmaGapless");
			}
			else if (State == State.Configure)
			{
				constant1 = 2.0 / (1 + Period);
				constant2 = 1 - (2.0 / (1 + Period));
                displayName = "EMAv2Gl " + (GaplessEmaType != GaplessEmaTypeEnum.Normal ? GaplessEmaType.ToString() : "") + " (" + Period + ")";
            }
		}

		public override string DisplayName
		{
		    get { return displayName ?? Name; }
		}
		
		protected override void OnBarUpdate()
		{
		    try
		    {
		        if (GaplessEmaType == GaplessEmaTypeEnum.None)
		            return;
                if (CurrentBar < 0)
                    return;

                //for tickreplay. Make sure only updates on each price change
                if (Bars.IsTickReplay && State == State.Historical && !IsFirstTickOfBar && prevInput.CompareTo(Input[0]) == 0)
                    return;

                prevInput = Input[0];

                double gap = 0d;

                
                if (CurrentBar == 0)
                    Value[0] = Input[0];
                else
                {
                    //Added EOD ability
                    if (CurrentBar > 0 && (!Bars.BarsType.IsIntraday || (Bars.IsFirstBarOfSession && Bars.IsResetOnNewTradingDay)))
                    {  
                        if (GaplessEmaType == GaplessEmaTypeEnum.SessionStartGapless)
                        {
                            //Instrument prices only
                            if (Input is PriceSeries)
                            {
                                gap = Open[0] - Close[1];

                                if (Bars.BarsType.IsIntraday)
                                    if (Plots[0].PlotStyle == PlotStyle.Line || Plots[0].PlotStyle == PlotStyle.Square)
                                        PlotBrushes[0][0] = Brushes.Transparent;
                            }
                            else
                            {
                                gap = Input[0] - Input[1];
                            }
                        }
                        else if (GaplessEmaType == GaplessEmaTypeEnum.SessionRestart)
                        {
                            startBar = CurrentBar - 1;

                            Value[0] = Input[0];

                            if (Bars.BarsType.IsIntraday)
                                if (Plots[0].PlotStyle == PlotStyle.Line || Plots[0].PlotStyle == PlotStyle.Square)
                                    PlotBrushes[0][0] = Brushes.Transparent;
                          
                        }
                    }
                    
                    if (GaplessEmaType == GaplessEmaTypeEnum.SessionRestart)
                    {
                        constant1 = 2.0 / (1 + Math.Min(Period, CurrentBar - startBar));
                        constant2 = 1 - (2.0 / (1 + Math.Min(Period, CurrentBar - startBar)));
                    }
                    // Valid for all settings
                    Value[0] = (Input[0] * constant1 + constant2 * (Value[1] + gap));

                }
		    }
		    catch (Exception ex)
		    {
		        Print(Name + " : " + ex);
		    }
		}

		#region Properties
		[Range(1, int.MaxValue), NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Period", GroupName = "NinjaScriptParameters", Order = 0)]
		public int Period
		{ get; set; }
		
		[NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "What type of plot?", GroupName = "NinjaScriptParameters", Order = 1)]
		public GaplessEmaTypeEnum GaplessEmaType
		{ get; set; }

       

        #endregion
    }
}

public enum GaplessEmaTypeEnum
{
    None,
    Normal,
    SessionStartGapless,
    SessionRestart
}

#region NinjaScript generated code. Neither change nor remove.

namespace NinjaTrader.NinjaScript.Indicators
{
	public partial class Indicator : NinjaTrader.Gui.NinjaScript.IndicatorRenderBase
	{
		private Sim22.Gapless.Sim22_EmaGaplessV2[] cacheSim22_EmaGaplessV2;
		public Sim22.Gapless.Sim22_EmaGaplessV2 Sim22_EmaGaplessV2(int period, GaplessEmaTypeEnum gaplessEmaType)
		{
			return Sim22_EmaGaplessV2(Input, period, gaplessEmaType);
		}

		public Sim22.Gapless.Sim22_EmaGaplessV2 Sim22_EmaGaplessV2(ISeries<double> input, int period, GaplessEmaTypeEnum gaplessEmaType)
		{
			if (cacheSim22_EmaGaplessV2 != null)
				for (int idx = 0; idx < cacheSim22_EmaGaplessV2.Length; idx++)
					if (cacheSim22_EmaGaplessV2[idx] != null && cacheSim22_EmaGaplessV2[idx].Period == period && cacheSim22_EmaGaplessV2[idx].GaplessEmaType == gaplessEmaType && cacheSim22_EmaGaplessV2[idx].EqualsInput(input))
						return cacheSim22_EmaGaplessV2[idx];
			return CacheIndicator<Sim22.Gapless.Sim22_EmaGaplessV2>(new Sim22.Gapless.Sim22_EmaGaplessV2(){ Period = period, GaplessEmaType = gaplessEmaType }, input, ref cacheSim22_EmaGaplessV2);
		}
	}
}

namespace NinjaTrader.NinjaScript.MarketAnalyzerColumns
{
	public partial class MarketAnalyzerColumn : MarketAnalyzerColumnBase
	{
		public Indicators.Sim22.Gapless.Sim22_EmaGaplessV2 Sim22_EmaGaplessV2(int period, GaplessEmaTypeEnum gaplessEmaType)
		{
			return indicator.Sim22_EmaGaplessV2(Input, period, gaplessEmaType);
		}

		public Indicators.Sim22.Gapless.Sim22_EmaGaplessV2 Sim22_EmaGaplessV2(ISeries<double> input , int period, GaplessEmaTypeEnum gaplessEmaType)
		{
			return indicator.Sim22_EmaGaplessV2(input, period, gaplessEmaType);
		}
	}
}

namespace NinjaTrader.NinjaScript.Strategies
{
	public partial class Strategy : NinjaTrader.Gui.NinjaScript.StrategyRenderBase
	{
		public Indicators.Sim22.Gapless.Sim22_EmaGaplessV2 Sim22_EmaGaplessV2(int period, GaplessEmaTypeEnum gaplessEmaType)
		{
			return indicator.Sim22_EmaGaplessV2(Input, period, gaplessEmaType);
		}

		public Indicators.Sim22.Gapless.Sim22_EmaGaplessV2 Sim22_EmaGaplessV2(ISeries<double> input , int period, GaplessEmaTypeEnum gaplessEmaType)
		{
			return indicator.Sim22_EmaGaplessV2(input, period, gaplessEmaType);
		}
	}
}

#endregion
