#region Using declarations
using System;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.Windows.Media;
using System.Xml.Serialization;
using NinjaTrader.Gui;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Data;

#endregion

public enum DeltaV3BarWidthTypeEnum
{
	Auto,
	UsePlotWidth	
}

//This namespace holds Indicators in this folder and is required. Do not change it. 
namespace NinjaTrader.NinjaScript.Indicators.Sim22
{
    /// <summary>
    /// Based on Gomi's GOM delta.
    /// Zondor 'unGom' Bid-Ask conversion to NT8 with Tick Replay.
    /// Sim22 addition of various other GOM formulae, commitment of traders calculations, heiken ashi, Ema, code tidying and extra features. NT8b12 Aug 2016.
	/// Added plot width ability Dec 2016
    /// </summary>
    public class Sim22_DeltaV3 : Indicator
    {
        private Sim22.Gapless.Sim22_EmaGaplessV2 emaGapless;

        private Series<int> deltaColorSeries;

        private NinjaTrader.NinjaScript.AddOns.Sim22_DeltaUtilities deltaUtilities = new NinjaTrader.NinjaScript.AddOns.Sim22_DeltaUtilities();
		private Series<double>	deltaOpenSeries;
		private Series<double>	deltaCloseSeries;
		private Series<double>	deltaHighSeries;
		private Series<double>	deltaLowSeries;
        
        private double      currentDeltaD 	= 0d;
		private double 		barDeltaD       = 0d;
        private double      highCotD        = 0d;
        private double      lowCotD         = 0d;

        private string filterStr = string.Empty;
        private string cotStr = string.Empty;
        private string haStr = string.Empty;

        private double deltaOpen;
        private double deltaHigh;
        private double deltaLow;
        private double deltaClose;

        //Experimental HA
        private NinjaTrader.NinjaScript.AddOns.Sim22_HaUtilities heikenAshiUtilities = new NinjaTrader.NinjaScript.AddOns.Sim22_HaUtilities();

	    private NinjaTrader.NinjaScript.AddOns.Sim22_HaUtilities.HeikenAshiValues heikenAshiValues;
	    private NinjaTrader.NinjaScript.AddOns.Sim22_HaUtilities.OhlcValues ohlcValues;
        private Series<NinjaTrader.NinjaScript.AddOns.Sim22_HaUtilities.HeikenAshiValues> heikenAshiSeries;
        private Series<NinjaTrader.NinjaScript.AddOns.Sim22_HaUtilities.OhlcValues> ohlcSeries;

        protected override void OnStateChange()
		{   
			if (State == State.SetDefaults)
			{
				Description					= @"Delta. Gomi, Zondor, Sim22";
				Name						= "Sim22_DeltaV3";
				Calculate					= Calculate.OnEachTick;
				MaximumBarsLookBack 		= MaximumBarsLookBack.Infinite;
				IsOverlay					= false;
				DisplayInDataBox			= true;
				DrawOnPricePanel			= true;
				DrawHorizontalGridLines		= false;
				DrawVerticalGridLines		= false;
				PaintPriceMarkers			= true;
				ScaleJustification			= ScaleJustification.Right;
				IsSuspendedWhileInactive	= false;

                BarUpBrush                  = new System.Windows.Media.SolidColorBrush(Color.FromArgb(255, 42, 147, 42));
                OutlineUpBrush              = new System.Windows.Media.SolidColorBrush(Color.FromArgb(255, 90, 231, 90));

                BarDnBrush                  = new System.Windows.Media.SolidColorBrush(Color.FromArgb(255, 167, 81, 80));
                OutlineDnBrush              = new System.Windows.Media.SolidColorBrush(Color.FromArgb(255, 184, 124, 124));

                PlotCumulativeDelta			= true;
				ResetDelta			        = false;
				FilterSize			        = 1;
				InvertDelta			        = false;
			    ShowPaintBars               = true;
			    ShowPaintBarsOutline        = true;
			    ShowAboveBelowEmaColors     = false;
                PeriodEma                   = 20;
			    //PeriodSmooth              = 10;
                DeltaType                   = NinjaTrader.NinjaScript.AddOns.Sim22_DeltaUtilities.DeltaCalculationTypeEnum.BidAsk;
                FilterType                  = NinjaTrader.NinjaScript.AddOns.Sim22_DeltaUtilities.DeltaTradeSizeFilterTypeEnum.None;
			    DeltaCotType                = NinjaTrader.NinjaScript.AddOns.Sim22_DeltaUtilities.DeltaCotType.None;
			    HeikenAshiCalculationType   = NinjaTrader.NinjaScript.AddOns.Sim22_HaUtilities.HeikenAshiCalculationType.None;
                // Nov 2016
                DeltaGaplessEmaType         = GaplessEmaTypeEnum.Normal;
				DeltaV3BarWidthType			= DeltaV3BarWidthTypeEnum.Auto;

                AddPlot(new Stroke(Brushes.Orange, DashStyleHelper.Solid, 2), PlotStyle.Line, "Ema"); //Placed first to keep price marker below Delta price marker
                AddPlot(new Stroke(Brushes.RoyalBlue, DashStyleHelper.Solid, 6), PlotStyle.Bar, "Delta");
                AddLine(new Stroke(Brushes.Gray, DashStyleHelper.Dot, 1), 0, "ZeroLine");

            }
            else if (State == State.Configure)
            {
                emaGapless = Sim22_EmaGaplessV2(Values[1], PeriodEma, DeltaGaplessEmaType);
                
                Calculate = Calculate.OnEachTick;

                BarUpBrush.Freeze();							
				BarDnBrush.Freeze();
				OutlineUpBrush.Freeze();
				OutlineDnBrush.Freeze();

                deltaColorSeries = new Series<int>(this, MaximumBarsLookBack.Infinite);

                deltaOpenSeries = new Series<double>(this, MaximumBarsLookBack.Infinite);
				deltaCloseSeries = new Series<double>(this, MaximumBarsLookBack.Infinite);
				deltaHighSeries = new Series<double>(this, MaximumBarsLookBack.Infinite);
				deltaLowSeries = new Series<double>(this, MaximumBarsLookBack.Infinite);

                //Experimental HA
                heikenAshiSeries = new Series<NinjaTrader.NinjaScript.AddOns.Sim22_HaUtilities.HeikenAshiValues>(this, MaximumBarsLookBack.Infinite);
                ohlcSeries = new Series<NinjaTrader.NinjaScript.AddOns.Sim22_HaUtilities.OhlcValues>(this, MaximumBarsLookBack.Infinite);

                switch (FilterType)
				{
				    case AddOns.Sim22_DeltaUtilities.DeltaTradeSizeFilterTypeEnum.GreaterOrEqualTo:
				        filterStr	= "Filter >= " + FilterSize;
				        break;
				    case AddOns.Sim22_DeltaUtilities.DeltaTradeSizeFilterTypeEnum.LessThanOrEqualTo:
				        filterStr	= "Filter <= " + FilterSize;
				        break;
				}

			    switch (DeltaCotType)
			    {
			       
			        case NinjaTrader.NinjaScript.AddOns.Sim22_DeltaUtilities.DeltaCotType.HighCot:
			            cotStr = " (High COT)";
			            break;
			        case NinjaTrader.NinjaScript.AddOns.Sim22_DeltaUtilities.DeltaCotType.LowCot:
                        cotStr = " (Low COT)";
                        break;
			        case NinjaTrader.NinjaScript.AddOns.Sim22_DeltaUtilities.DeltaCotType.CombinedCot:
                        cotStr = " (Total COT)";
                        break;
			        
			    }
			    switch (HeikenAshiCalculationType)
			    {
			        case NinjaTrader.NinjaScript.AddOns.Sim22_HaUtilities.HeikenAshiCalculationType.None:
			            haStr = "";
			            break;
			        case NinjaTrader.NinjaScript.AddOns.Sim22_HaUtilities.HeikenAshiCalculationType.DanValcu:
			            haStr = " (HA)";
			            break;
			        case NinjaTrader.NinjaScript.AddOns.Sim22_HaUtilities.HeikenAshiCalculationType.SylvainVervoort:
                        haStr = " (HAsv)";
                        break;
			    }
                //For future development:
			    //switch (HeikenAshiSmoothingType)
			    //{
			    //    case Sim22_HAUtilities.HeikenAshiSmoothingType.None:
			    //        break;
			    //    case Sim22_HAUtilities.HeikenAshiSmoothingType.EMA:
       //                 OpenMA = EMA(Open, PeriodSmooth);
       //                 break;
			    //    case Sim22_HAUtilities.HeikenAshiSmoothingType.HMA:
			    //        break;
			    //    case Sim22_HAUtilities.HeikenAshiSmoothingType.SMA:
			    //        break;
			    //}
			}
            else if (State == State.Historical)
            {
            }
		}

	    public override string FormatPriceMarker(double price)
	    {
	        return price.ToString("N0");
	    }

	    public override string DisplayName
	    {
	        get { return "DeltaV3 " + DeltaType + cotStr + haStr + (InvertDelta ? " (Inverted)" : "") + (ResetDelta ? Environment.NewLine + "EOD reset " : "") + (!string.IsNullOrEmpty(filterStr) ? Environment.NewLine + filterStr : ""); }
	    }

	    protected override void OnMarketData(MarketDataEventArgs e)
	    {
	        try
	        {
	            if (CurrentBar <= -1)
	                return;
	            if (e.MarketDataType != MarketDataType.Last)
	                return;
	            if (CurrentBars[0] <= -1)
	                return;

	            if (BarsInProgress == 0)
	            {
	                if (IsFirstTickOfBar)
	                {
	                    if (CurrentBar == 0)
	                        barDeltaD = 0.0;

	                    if (CurrentBar > 0)
	                        barDeltaD = deltaCloseSeries[1];

	                    if (!PlotCumulativeDelta || (Bars.IsResetOnNewTradingDay && Bars.IsFirstBarOfSession && ResetDelta))
	                    {
	                        barDeltaD = 0.0;
	                    }

	                    deltaLow = deltaHigh = deltaOpen = barDeltaD;

	                    highCotD = 0d;
	                    lowCotD = 0d;
	                }
                    // Calculate delta via Delta Utilities AddOn
	                currentDeltaD = deltaUtilities.GetCurrentDelta(DeltaType, FilterType, e.Ask, e.Bid, e.Price, e.Volume, InvertDelta, FilterSize);

	                deltaClose = 0d;
                    // If a COT is selected
	                if (DeltaCotType != NinjaTrader.NinjaScript.AddOns.Sim22_DeltaUtilities.DeltaCotType.None)
	                {
	                    if (IsFirstTickOfBar)
	                    {
	                        highCotD = 0d;
	                        lowCotD = 0d;
	                    }

	                    if (e.Price >= High[0])
	                    {
	                        highCotD = 0d;
	                    }
	                    else
	                    {
	                        highCotD += currentDeltaD;
	                    }

	                    if (e.Price <= Low[0])
	                    {
	                        lowCotD = 0d;
	                    }
	                    else
	                    {
	                        lowCotD += currentDeltaD;
	                    }

	                    switch (DeltaCotType)
	                    {
	                        case NinjaTrader.NinjaScript.AddOns.Sim22_DeltaUtilities.DeltaCotType.HighCot:
	                            deltaClose = highCotD + barDeltaD;
	                            break;
	                        case NinjaTrader.NinjaScript.AddOns.Sim22_DeltaUtilities.DeltaCotType.LowCot:
	                            deltaClose = lowCotD + barDeltaD;
	                            break;

	                        case NinjaTrader.NinjaScript.AddOns.Sim22_DeltaUtilities.DeltaCotType.CombinedCot:
	                            deltaClose = (highCotD + lowCotD) + barDeltaD;
	                            break;
	                    }
	                }
	                else
	                    deltaClose = (barDeltaD += currentDeltaD);

	                deltaHigh = Math.Max(deltaHigh, deltaClose);
	                deltaLow = Math.Min(deltaLow, deltaClose);
                }
	        }
            catch (Exception ex)
            {
                Print("DeltaV3 exception at line 405:" + ex);
            }
        }

	    protected override void OnBarUpdate()
	    {
            deltaCloseSeries[0] = deltaClose;

            if (HeikenAshiCalculationType != NinjaTrader.NinjaScript.AddOns.Sim22_HaUtilities.HeikenAshiCalculationType.None && PlotCumulativeDelta)
            {
                ohlcValues.Open = deltaOpen;
                ohlcValues.High = deltaHigh;
                ohlcValues.Low = deltaLow;
                ohlcValues.Close = deltaClose;

                ohlcSeries[0] = ohlcValues;

                /*********Original HA code
                // Please Note to avoid confusion: deltaCloseSeries[0] is the real delta value, Values[1][0] (the plot) is the HA close value.

                //                    Values[1][0] = (deltaClose + deltaOpen + deltaHigh + deltaLow) * 0.25; // Calculate the HA close
                //                    deltaOpenSeries[0] = (deltaOpenSeries[1] + Values[1][1]) * 0.5; // Calculate the HA open
                //                    deltaHighSeries[0] = Math.Max(deltaHigh, deltaOpenSeries[0]); // Calculate the HA high
                //                    deltaLowSeries[0] = Math.Min(deltaLow, deltaOpenSeries[0]); // Calculate the HA low
                **********/

                if (CurrentBar <= 0 || (Bars.IsFirstBarOfSession && ResetDelta))
                {
                    heikenAshiValues.HaOpen = deltaOpen;
                    heikenAshiValues.HaHigh = deltaHigh;
                    heikenAshiValues.HaLow = deltaLow;
                    heikenAshiValues.HaClose = deltaClose;

                    heikenAshiSeries[0] = heikenAshiValues;
                }
                else
                {
                    heikenAshiSeries[0] = heikenAshiUtilities.GetHeikenAshi(HeikenAshiCalculationType, heikenAshiSeries, ohlcSeries);
                }

                deltaOpenSeries[0] = heikenAshiSeries[0].HaOpen;
                deltaHighSeries[0] = heikenAshiSeries[0].HaHigh;
                deltaLowSeries[0] = heikenAshiSeries[0].HaLow;
                // Note: deltaCloseSeries[0] is the real delta value, Values[1][0] (the plot) is the HA close value
                Values[1][0] = heikenAshiSeries[0].HaClose;
            }
            else
            {
                Values[1][0] = deltaClose;
                deltaOpenSeries[0] = deltaOpen;
                deltaHighSeries[0] = deltaHigh;
                deltaLowSeries[0] = deltaLow;
                deltaCloseSeries[0] = deltaClose;
            }

            try
            {
                // for ema values
                if (DeltaGaplessEmaType != GaplessEmaTypeEnum.None)
                {
                    if (CurrentBar < 0)
                        return;

                    Values[0][0] = emaGapless[0];
                }
                // Determine colors
                if (ShowAboveBelowEmaColors)
                {
                    if (Values[1][0] > Values[0][0])
                        deltaColorSeries[0] = 1;
                    if (Values[1][0] < Values[0][0])
                        deltaColorSeries[0] = -1;
                }
                else
                {
                    if (Values[1][0] > deltaOpenSeries[0])
                        deltaColorSeries[0] = 1;
                    if (Values[1][0] < deltaOpenSeries[0])
                        deltaColorSeries[0] = -1;
                }
                // Set colors for plots
                if (deltaColorSeries[0] == 1)
                {
                    PlotBrushes[1][0] = BarUpBrush;
                    if (ShowPaintBars)
                        BarBrush = BarUpBrush;
                    if (ShowPaintBarsOutline)
                        CandleOutlineBrush = OutlineUpBrush;
                }    
                else if (deltaColorSeries[0] == -1)
                {
                    PlotBrushes[1][0] = BarDnBrush;
                    if (ShowPaintBars)
                        BarBrush = BarDnBrush;
                    if (ShowPaintBarsOutline)
                        CandleOutlineBrush = OutlineDnBrush;
                }
                else if (CurrentBar != 0)
                {
                    // Show colors of prior bar
                    PlotBrushes[1][0] = PlotBrushes[0][1];
                    if (ShowPaintBars)
                        BarBrush = BarBrushes[1];
                    if (ShowPaintBarsOutline)
                        CandleOutlineBrush = CandleOutlineBrushes[1];
                }            
                else
                {
                    // Just in case
                    PlotBrushes[1][0] = Brushes.Gray;
                    if (ShowPaintBars)
                        BarBrush = Brushes.Gray;
                    if (ShowPaintBarsOutline)
                        CandleOutlineBrush = Brushes.Gray;
                }

            }
            catch (Exception ex)
            {
                Print("DeltaV3 exception at line 399, EMA calculation:" + ex);
            }

        }

        public override void OnCalculateMinMax()
	    {
            try
            {

                if (Bars == null)
                    return;

                int lastBar = ChartBars.ToIndex;
                int firstBar = ChartBars.FromIndex;
       
                double min = double.MaxValue;
                double max = double.MinValue;

                for (int index = firstBar; index <= lastBar; index++)
                {
                    min = Math.Min(min, deltaLowSeries.GetValueAt(index));
                    max = Math.Max(max, deltaHighSeries.GetValueAt(index));
                }

                if ((max - min) < 1)
                {
                    min -= 1;
                    max += 1;
                }

                MinValue = min;
                MaxValue = max;

            }
            catch (Exception ex)
            {
                Print("DeltaV3 exception at OnCalculateMinMax() at line 460:" + ex);
            }
        }

	    protected override void OnRender(ChartControl chartControl, ChartScale chartScale)
	    {
	        if (Bars == null || ChartControl == null || Bars.Instrument == null || !IsVisible)
	        {
	            return;
	        }

	        try
	        {
	            int cbti = ChartBars.ToIndex;
	            int cbfi = ChartBars.FromIndex;


	            SharpDX.Direct2D1.Brush brushBarDX = Brushes.Gray.ToDxBrush(RenderTarget);
	            SharpDX.Direct2D1.Brush brushOutlineDX = Brushes.Black.ToDxBrush(RenderTarget);

	            SharpDX.Direct2D1.Brush brushZeroDX = Lines[0].BrushDX;
	            SharpDX.Direct2D1.StrokeStyle lzSS = Lines[0].StrokeStyle;
	            float zeroLineY = chartScale.GetYByValue(0);
	            RenderTarget.DrawLine(new SharpDX.Vector2(ChartPanel.X, zeroLineY), new SharpDX.Vector2(ChartPanel.W, zeroLineY), brushZeroDX, 1f, lzSS);

				// Changed Dec 2016
	            float width = DeltaV3BarWidthType == DeltaV3BarWidthTypeEnum.Auto ? (float) Math.Max(2, ChartBars.Properties.ChartStyle.BarWidth*2) : Plots[1].Width;

	            for (int idx = cbfi; idx <= cbti; idx++)
	            {
	                double opValue = deltaOpenSeries.GetValueAt(idx);
	                double hiValue = deltaHighSeries.GetValueAt(idx);
	                double loValue = deltaLowSeries.GetValueAt(idx);
	                double clValue = Values[1].GetValueAt(idx);

	                double clValue1 = idx > 0 ? Values[1].GetValueAt(idx - 1) : clValue;
	                double opValue1 = idx > 0 ? deltaOpenSeries.GetValueAt(idx - 1) : opValue;

	                float yOpen = chartScale.GetYByValue(opValue);
	                float yHi = chartScale.GetYByValue(hiValue);
	                float yLo = chartScale.GetYByValue(loValue);
	                float yClose = chartScale.GetYByValue(clValue);

	                var height = (int) Math.Abs(yOpen - yClose);
	                var top = (opValue > clValue ? yOpen : yClose);
	                var bottom = (opValue < clValue ? yOpen : yClose);
	                float xCenter = chartControl.GetXByBarIndex(ChartBars, idx);
	                float xPosition = xCenter - (float) (width * 0.5f);


	                SharpDX.RectangleF rect = new SharpDX.RectangleF(xPosition, top, width, height);

                    if (deltaColorSeries.GetValueAt(idx) == 1)
	                {
	                    brushBarDX = BarUpBrush.ToDxBrush(RenderTarget);
	                    brushOutlineDX = OutlineUpBrush.ToDxBrush(RenderTarget);
	                }
	                else if (deltaColorSeries.GetValueAt(idx) == -1)
                    {
	                    brushBarDX = BarDnBrush.ToDxBrush(RenderTarget);
	                    brushOutlineDX = OutlineDnBrush.ToDxBrush(RenderTarget);
	                }
	                else
	                {
                        brushBarDX = Brushes.Gray.ToDxBrush(RenderTarget);
                        brushOutlineDX = Brushes.Gray.ToDxBrush(RenderTarget);
                    }
	              
	                RenderTarget.FillRectangle(rect, brushBarDX);
	                RenderTarget.DrawRectangle(rect, brushOutlineDX);
	                RenderTarget.DrawLine(new SharpDX.Vector2(xCenter, yHi), new SharpDX.Vector2(xCenter, top), brushOutlineDX, 1f);
	                RenderTarget.DrawLine(new SharpDX.Vector2(xCenter, yLo), new SharpDX.Vector2(xCenter, bottom), brushOutlineDX, 1f);
	            }

	            brushBarDX.Dispose();
	            brushOutlineDX.Dispose();
	            brushZeroDX.Dispose();
	            lzSS.Dispose();

	            if (DeltaGaplessEmaType != GaplessEmaTypeEnum.None)
	            {
	                //Avg path
	                SharpDX.Direct2D1.PathGeometry lineGeometryAvg = new SharpDX.Direct2D1.PathGeometry(Core.Globals.D2DFactory);

	                SharpDX.Direct2D1.GeometrySink sinkAvg = lineGeometryAvg.Open();

	                sinkAvg.BeginFigure(new SharpDX.Vector2(chartControl.GetXByBarIndex(ChartBars, cbfi), chartScale.GetYByValue(Values[0].GetValueAt(cbfi))), SharpDX.Direct2D1.FigureBegin.Filled);

	                for (int idx = cbfi; idx <= cbti; idx++)
	                {
	                    double indyValue = Values[0].GetValueAt(idx);

	                    float avgY = chartScale.GetYByValue(indyValue);
	                    float x = chartControl.GetXByBarIndex(ChartBars, idx);

	                    sinkAvg.AddLine(new SharpDX.Vector2(x, avgY));
	                }

	                SharpDX.Direct2D1.Brush avgBrush = Plots[0].BrushDX;
	                SharpDX.Direct2D1.StrokeStyle avgStrokeStyle = Plots[0].StrokeStyle;
	                float strokeWidthAvg = Plots[0].Width;

	                //Avg path plot
	                sinkAvg.EndFigure(SharpDX.Direct2D1.FigureEnd.Open);
	                sinkAvg.Close();
	                RenderTarget.AntialiasMode = SharpDX.Direct2D1.AntialiasMode.PerPrimitive;
	                RenderTarget.DrawGeometry(lineGeometryAvg, avgBrush, strokeWidthAvg, avgStrokeStyle);
	                RenderTarget.AntialiasMode = SharpDX.Direct2D1.AntialiasMode.Aliased; //
	                lineGeometryAvg.Dispose();
	                avgBrush.Dispose();

	                avgStrokeStyle.Dispose();
	            }
	        }
            catch (Exception ex)
            {
                Print("DeltaV3 exception at OnRender() at line 586:" + ex);
            }
        }

	    #region Properties

	    [Browsable(false)]
	    [XmlIgnore()]
	    public Series<double> DeltaEma
	    {
	        get { return Values[0]; }
	    }

	    [Browsable(false)]
	    [XmlIgnore()]
	    public Series<double> DeltaClose
	    {
	        get { return Values[1]; }
	    }

	    [Browsable(false)]
	    [XmlIgnore()]
	    public Series<double> DeltaOpen
	    {
	        get { return deltaOpenSeries; }
	    }

	    [Browsable(false)]
	    [XmlIgnore()]
	    public Series<double> DeltaHigh
	    {
	        get { return deltaHighSeries; }
	    }

	    [Browsable(false)]
	    [XmlIgnore()]
	    public Series<double> DeltaLow
	    {
	        get { return deltaLowSeries; }
	    }

		[NinjaScriptProperty]
        [Display(Name = "Bar plot width", Description = "Choose between auto or fixed width. For fixed width use the plot width to adjust.", Order = 0, GroupName = "0. Bar width type")]
        public DeltaV3BarWidthTypeEnum DeltaV3BarWidthType { get; set; }
		
		[NinjaScriptProperty]
	    [Display(Name = "Delta type", Description = "", Order = 0, GroupName = "1. Delta")]
	    public AddOns.Sim22_DeltaUtilities.DeltaCalculationTypeEnum DeltaType { get; set; }

	    [NinjaScriptProperty]
	    [Display(Name = "COT type", Description = "Set to plot commitment of traders or not.", GroupName = "1. Delta", Order = 1)]
	    public AddOns.Sim22_DeltaUtilities.DeltaCotType DeltaCotType { get; set; }

	    [NinjaScriptProperty]
	    [Display(Name = "Cumulative delta?", GroupName = "1. Delta", Order = 2)]
	    public bool PlotCumulativeDelta { get; set; }

	    [NinjaScriptProperty]
	    [Display(Name = "EOD reset?", GroupName = "1. Delta", Order = 3)]
	    public bool ResetDelta { get; set; }

	    [NinjaScriptProperty]
	    [Display(Name = "Invert delta?", GroupName = "1. Delta", Order = 4)]
	    public bool InvertDelta { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "EMA plot type", Description = "Plot EMA as normal, reset the EMA at the session start or ignore the session delta gap.", Order = 5, GroupName = "1. Delta")]
        public GaplessEmaTypeEnum DeltaGaplessEmaType { get; set; }

        [Range(1, 300), NinjaScriptProperty]
	    [Display(Name = "EMA period", GroupName = "1. Delta", Order = 6)]
	    public int PeriodEma { get; set; }
		
		 [NinjaScriptProperty]
	    [Display(Name = "Filter type", Description = "", Order = 0, GroupName = "2. Filter")]
	    public AddOns.Sim22_DeltaUtilities.DeltaTradeSizeFilterTypeEnum FilterType { get; set; }

	    [Range(1, int.MaxValue), NinjaScriptProperty]
	    [Display(Name = "Filter size", GroupName = "2. Filter", Order = 1)]
	    public int FilterSize { get; set; }
		
		 //HeikenAshiCalculationType
	    [NinjaScriptProperty]
	    [Display(Name = "HA type", Description = "Use Dan Valcu's for normal HA or Sylvain Vervoort's model", Order = 0, GroupName = "3. Heiken Ashi")]
	    public AddOns.Sim22_HaUtilities.HeikenAshiCalculationType HeikenAshiCalculationType { get; set; }

	    //[NinjaScriptProperty]
	    //[Display(Name = "HA smoothing type", Description = "Use various smoothing calculations.", Order = 1, GroupName = "3. Heiken Ashi")]
	    //public AddOns.Sim22_HAUtilities.HeikenAshiSmoothingType HeikenAshiSmoothingType { get; set; }

     //   [Range(1, 300), NinjaScriptProperty]
     //   [Display(Name = "HA smoothing period", GroupName = "3. Heiken Ashi", Order = 2)]
     //   public int PeriodSmooth { get; set; }

       
        [NinjaScriptProperty]
        [Display(Name = "Show EMA above/below colors?", Order = 1, GroupName = "4. Colors")]
        public bool ShowAboveBelowEmaColors { get; set; }


        [XmlIgnore]
	    [Display(ResourceType = typeof(Custom.Resource), Name = "Up color", Order = 6, GroupName = "4. Colors")]
	    public Brush BarUpBrush { get; set; }

	    [Browsable(false)]
	    public string BarUpBrushSerialize
	    {
	        get { return Serialize.BrushToString(BarUpBrush); }
	        set { BarUpBrush = Serialize.StringToBrush(value); }
	    }

	    [XmlIgnore]
	    [Display(ResourceType = typeof(Custom.Resource), Name = "Down color", Order = 8, GroupName = "4. Colors")]
	    public Brush BarDnBrush { get; set; }

	    [Browsable(false)]
	    public string BarDnBrushSerialize
	    {
	        get { return Serialize.BrushToString(BarDnBrush); }
	        set { BarDnBrush = Serialize.StringToBrush(value); }
	    }

	    [XmlIgnore]
	    [Display(ResourceType = typeof(Custom.Resource), Name = "Up color outline", Order = 7, GroupName = "4. Colors")]
	    public Brush OutlineUpBrush { get; set; }

	    [Browsable(false)]
	    public string OutlineUpBrushSerialize
	    {
	        get { return Serialize.BrushToString(OutlineUpBrush); }
	        set { OutlineUpBrush = Serialize.StringToBrush(value); }
	    }

	    [XmlIgnore]
	    [Display(ResourceType = typeof(Custom.Resource), Name = "Dn color outline", Order = 9, GroupName = "4. Colors")]
	    public Brush OutlineDnBrush { get; set; }

        [Browsable(false)]
	    public string OutlineDnBrushSerialize
	    {
	        get { return Serialize.BrushToString(OutlineDnBrush); }
	        set { OutlineDnBrush = Serialize.StringToBrush(value); }
	    }

        [NinjaScriptProperty]
        [Display(Name = "Show paint bars?", Order = 1, GroupName = "5. Paint bars")]
        public bool ShowPaintBars { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Show paint bars outline?", Order = 2, GroupName = "5. Paint bars")]
        public bool ShowPaintBarsOutline { get; set; }

        

	   

	    #endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.

namespace NinjaTrader.NinjaScript.Indicators
{
	public partial class Indicator : NinjaTrader.Gui.NinjaScript.IndicatorRenderBase
	{
		private Sim22.Sim22_DeltaV3[] cacheSim22_DeltaV3;
		public Sim22.Sim22_DeltaV3 Sim22_DeltaV3(DeltaV3BarWidthTypeEnum deltaV3BarWidthType, AddOns.Sim22_DeltaUtilities.DeltaCalculationTypeEnum deltaType, AddOns.Sim22_DeltaUtilities.DeltaCotType deltaCotType, bool plotCumulativeDelta, bool resetDelta, bool invertDelta, GaplessEmaTypeEnum deltaGaplessEmaType, int periodEma, AddOns.Sim22_DeltaUtilities.DeltaTradeSizeFilterTypeEnum filterType, int filterSize, AddOns.Sim22_HaUtilities.HeikenAshiCalculationType heikenAshiCalculationType, bool showAboveBelowEmaColors, bool showPaintBars, bool showPaintBarsOutline)
		{
			return Sim22_DeltaV3(Input, deltaV3BarWidthType, deltaType, deltaCotType, plotCumulativeDelta, resetDelta, invertDelta, deltaGaplessEmaType, periodEma, filterType, filterSize, heikenAshiCalculationType, showAboveBelowEmaColors, showPaintBars, showPaintBarsOutline);
		}

		public Sim22.Sim22_DeltaV3 Sim22_DeltaV3(ISeries<double> input, DeltaV3BarWidthTypeEnum deltaV3BarWidthType, AddOns.Sim22_DeltaUtilities.DeltaCalculationTypeEnum deltaType, AddOns.Sim22_DeltaUtilities.DeltaCotType deltaCotType, bool plotCumulativeDelta, bool resetDelta, bool invertDelta, GaplessEmaTypeEnum deltaGaplessEmaType, int periodEma, AddOns.Sim22_DeltaUtilities.DeltaTradeSizeFilterTypeEnum filterType, int filterSize, AddOns.Sim22_HaUtilities.HeikenAshiCalculationType heikenAshiCalculationType, bool showAboveBelowEmaColors, bool showPaintBars, bool showPaintBarsOutline)
		{
			if (cacheSim22_DeltaV3 != null)
				for (int idx = 0; idx < cacheSim22_DeltaV3.Length; idx++)
					if (cacheSim22_DeltaV3[idx] != null && cacheSim22_DeltaV3[idx].DeltaV3BarWidthType == deltaV3BarWidthType && cacheSim22_DeltaV3[idx].DeltaType == deltaType && cacheSim22_DeltaV3[idx].DeltaCotType == deltaCotType && cacheSim22_DeltaV3[idx].PlotCumulativeDelta == plotCumulativeDelta && cacheSim22_DeltaV3[idx].ResetDelta == resetDelta && cacheSim22_DeltaV3[idx].InvertDelta == invertDelta && cacheSim22_DeltaV3[idx].DeltaGaplessEmaType == deltaGaplessEmaType && cacheSim22_DeltaV3[idx].PeriodEma == periodEma && cacheSim22_DeltaV3[idx].FilterType == filterType && cacheSim22_DeltaV3[idx].FilterSize == filterSize && cacheSim22_DeltaV3[idx].HeikenAshiCalculationType == heikenAshiCalculationType && cacheSim22_DeltaV3[idx].ShowAboveBelowEmaColors == showAboveBelowEmaColors && cacheSim22_DeltaV3[idx].ShowPaintBars == showPaintBars && cacheSim22_DeltaV3[idx].ShowPaintBarsOutline == showPaintBarsOutline && cacheSim22_DeltaV3[idx].EqualsInput(input))
						return cacheSim22_DeltaV3[idx];
			return CacheIndicator<Sim22.Sim22_DeltaV3>(new Sim22.Sim22_DeltaV3(){ DeltaV3BarWidthType = deltaV3BarWidthType, DeltaType = deltaType, DeltaCotType = deltaCotType, PlotCumulativeDelta = plotCumulativeDelta, ResetDelta = resetDelta, InvertDelta = invertDelta, DeltaGaplessEmaType = deltaGaplessEmaType, PeriodEma = periodEma, FilterType = filterType, FilterSize = filterSize, HeikenAshiCalculationType = heikenAshiCalculationType, ShowAboveBelowEmaColors = showAboveBelowEmaColors, ShowPaintBars = showPaintBars, ShowPaintBarsOutline = showPaintBarsOutline }, input, ref cacheSim22_DeltaV3);
		}
	}
}

namespace NinjaTrader.NinjaScript.MarketAnalyzerColumns
{
	public partial class MarketAnalyzerColumn : MarketAnalyzerColumnBase
	{
		public Indicators.Sim22.Sim22_DeltaV3 Sim22_DeltaV3(DeltaV3BarWidthTypeEnum deltaV3BarWidthType, AddOns.Sim22_DeltaUtilities.DeltaCalculationTypeEnum deltaType, AddOns.Sim22_DeltaUtilities.DeltaCotType deltaCotType, bool plotCumulativeDelta, bool resetDelta, bool invertDelta, GaplessEmaTypeEnum deltaGaplessEmaType, int periodEma, AddOns.Sim22_DeltaUtilities.DeltaTradeSizeFilterTypeEnum filterType, int filterSize, AddOns.Sim22_HaUtilities.HeikenAshiCalculationType heikenAshiCalculationType, bool showAboveBelowEmaColors, bool showPaintBars, bool showPaintBarsOutline)
		{
			return indicator.Sim22_DeltaV3(Input, deltaV3BarWidthType, deltaType, deltaCotType, plotCumulativeDelta, resetDelta, invertDelta, deltaGaplessEmaType, periodEma, filterType, filterSize, heikenAshiCalculationType, showAboveBelowEmaColors, showPaintBars, showPaintBarsOutline);
		}

		public Indicators.Sim22.Sim22_DeltaV3 Sim22_DeltaV3(ISeries<double> input , DeltaV3BarWidthTypeEnum deltaV3BarWidthType, AddOns.Sim22_DeltaUtilities.DeltaCalculationTypeEnum deltaType, AddOns.Sim22_DeltaUtilities.DeltaCotType deltaCotType, bool plotCumulativeDelta, bool resetDelta, bool invertDelta, GaplessEmaTypeEnum deltaGaplessEmaType, int periodEma, AddOns.Sim22_DeltaUtilities.DeltaTradeSizeFilterTypeEnum filterType, int filterSize, AddOns.Sim22_HaUtilities.HeikenAshiCalculationType heikenAshiCalculationType, bool showAboveBelowEmaColors, bool showPaintBars, bool showPaintBarsOutline)
		{
			return indicator.Sim22_DeltaV3(input, deltaV3BarWidthType, deltaType, deltaCotType, plotCumulativeDelta, resetDelta, invertDelta, deltaGaplessEmaType, periodEma, filterType, filterSize, heikenAshiCalculationType, showAboveBelowEmaColors, showPaintBars, showPaintBarsOutline);
		}
	}
}

namespace NinjaTrader.NinjaScript.Strategies
{
	public partial class Strategy : NinjaTrader.Gui.NinjaScript.StrategyRenderBase
	{
		public Indicators.Sim22.Sim22_DeltaV3 Sim22_DeltaV3(DeltaV3BarWidthTypeEnum deltaV3BarWidthType, AddOns.Sim22_DeltaUtilities.DeltaCalculationTypeEnum deltaType, AddOns.Sim22_DeltaUtilities.DeltaCotType deltaCotType, bool plotCumulativeDelta, bool resetDelta, bool invertDelta, GaplessEmaTypeEnum deltaGaplessEmaType, int periodEma, AddOns.Sim22_DeltaUtilities.DeltaTradeSizeFilterTypeEnum filterType, int filterSize, AddOns.Sim22_HaUtilities.HeikenAshiCalculationType heikenAshiCalculationType, bool showAboveBelowEmaColors, bool showPaintBars, bool showPaintBarsOutline)
		{
			return indicator.Sim22_DeltaV3(Input, deltaV3BarWidthType, deltaType, deltaCotType, plotCumulativeDelta, resetDelta, invertDelta, deltaGaplessEmaType, periodEma, filterType, filterSize, heikenAshiCalculationType, showAboveBelowEmaColors, showPaintBars, showPaintBarsOutline);
		}

		public Indicators.Sim22.Sim22_DeltaV3 Sim22_DeltaV3(ISeries<double> input , DeltaV3BarWidthTypeEnum deltaV3BarWidthType, AddOns.Sim22_DeltaUtilities.DeltaCalculationTypeEnum deltaType, AddOns.Sim22_DeltaUtilities.DeltaCotType deltaCotType, bool plotCumulativeDelta, bool resetDelta, bool invertDelta, GaplessEmaTypeEnum deltaGaplessEmaType, int periodEma, AddOns.Sim22_DeltaUtilities.DeltaTradeSizeFilterTypeEnum filterType, int filterSize, AddOns.Sim22_HaUtilities.HeikenAshiCalculationType heikenAshiCalculationType, bool showAboveBelowEmaColors, bool showPaintBars, bool showPaintBarsOutline)
		{
			return indicator.Sim22_DeltaV3(input, deltaV3BarWidthType, deltaType, deltaCotType, plotCumulativeDelta, resetDelta, invertDelta, deltaGaplessEmaType, periodEma, filterType, filterSize, heikenAshiCalculationType, showAboveBelowEmaColors, showPaintBars, showPaintBarsOutline);
		}
	}
}

#endregion
