#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Enter the description of your new custom indicator here
    /// </summary>
    [Description("Enter the description of your new custom indicator here")]
    public class DailyLevels : Indicator
    {
        #region Variables
        // Wizard generated variables
        // User defined variables (add any user defined variables below)
			private CustomDashStyle dashStyleType	= CustomDashStyle.Solid; //Default dashStyle setting. 
			private Color  			highLine 		= Color.Red; //Default Color setting for High line
			private Color  			lowLine  		= Color.Red; //Default Color setting for low line
			private int 			width 			= 2; // Default setting for lineWidth
			private int    			barsSinceSessionStart;
		    private double 			highSinceSession;
			private double 			lowSinceSession;
			private int				CBofDay;

			
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
            Overlay				= true;
			CalculateOnBarClose = false;
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {	
			if (Bars.FirstBarOfSession)  //Captures Current Bar at start of day to allow for unique per session lines. 
				CBofDay = CurrentBar;
			
			if (Bars.Period.Id == PeriodType.Day || Bars.Period.Id == PeriodType.Week || Bars.Period.Id == PeriodType.Month || Bars.Period.Id == PeriodType.Year)
			{
				Print("DailyLevels indicator is designed for intraday charts. Please choose a lower time frame such as 15 minute.");
				DrawTextFixed("Message", "Daily levels indicator is designed for intraday charts. Please choose a lower time frame such as 15 minute.", TextPosition.TopRight);				
			}
			
			else
				RemoveDrawObject("Message");
			
			barsSinceSessionStart = Bars.BarsSinceSession;
			highSinceSession = MAX(High, barsSinceSessionStart + 1)[0];
			lowSinceSession	 = MIN(Low, barsSinceSessionStart + 1)[0];
			
			switch (dashStyleType)
			{
				
				case CustomDashStyle.Solid:
				{
					DrawLine("High " + CBofDay, true, 0, highSinceSession, barsSinceSessionStart, highSinceSession, HighLine, DashStyle.Solid, Width);
					DrawLine("Low " + CBofDay, true, 0, lowSinceSession, barsSinceSessionStart, lowSinceSession, LowLine, DashStyle.Solid, Width);
					break;
				}
				
				case CustomDashStyle.Dash:
				{
					DrawLine("High " + CBofDay, true, 0, highSinceSession, barsSinceSessionStart, highSinceSession, HighLine, DashStyle.Dash, Width);
					DrawLine("Low " + CBofDay, true, 0, lowSinceSession, barsSinceSessionStart, lowSinceSession, LowLine, DashStyle.Dash, Width);
					break;
				}
				
				case CustomDashStyle.DashDot:
				{
					DrawLine("High " + CBofDay, true, 0, highSinceSession, barsSinceSessionStart, highSinceSession, HighLine, DashStyle.DashDot, Width);
					DrawLine("Low " + CBofDay, true, 0, lowSinceSession, barsSinceSessionStart, lowSinceSession, LowLine, DashStyle.DashDot, Width);
					break;
				}
				
				case CustomDashStyle.DashDotDot:
				{
					DrawLine("High " + CBofDay, true, 0, highSinceSession, barsSinceSessionStart, highSinceSession, HighLine, DashStyle.DashDotDot, Width);
					DrawLine("Low " + CBofDay, true, 0, lowSinceSession, barsSinceSessionStart, lowSinceSession, LowLine, DashStyle.DashDotDot, Width);
					break;
				}
				
				case CustomDashStyle.Dot:
				{
					DrawLine("High " + CBofDay, true, 0, highSinceSession, barsSinceSessionStart, highSinceSession, HighLine, DashStyle.Dot, Width);
					DrawLine("Low " + CBofDay, true, 0, lowSinceSession, barsSinceSessionStart, lowSinceSession, LowLine, DashStyle.Dot, Width);
					break;
				}
			
			}
			
			
        }
		

        #region Properties
        [Description("")]
        [GridCategory("Parameters")]
        public int Width
        {
            get { return width; }
            set { width = Math.Max(1, value); }
        }
		
		[XmlIgnore()]
		[Description("Color for High Line")]
		[Category("Parameters")]
		public Color HighLine
		{
			get { return highLine; }
			set { highLine = value; }
		}
		
		[Browsable(false)]
		public string HighLineSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(highLine); }
			set { highLine = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		[XmlIgnore()]
		[Description("Color for Low Line")]
		[Category("Parameters")]
		public Color LowLine
		{
			get { return lowLine; }
			set { lowLine = value; }
		}
		
		[Browsable(false)]
		public string LowLineSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(lowLine); }
			set { lowLine = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		[Description("DashStyle")]
		[Category("Parameters")]
		public CustomDashStyle DashStyleType
		{
			get { return dashStyleType; }
			set { dashStyleType = value; }
		}
		
        #endregion
    }
}

public enum CustomDashStyle
{
	Solid,
	Dash,
	DashDot,
	DashDotDot,
	Dot,
	
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private DailyLevels[] cacheDailyLevels = null;

        private static DailyLevels checkDailyLevels = new DailyLevels();

        /// <summary>
        /// Enter the description of your new custom indicator here
        /// </summary>
        /// <returns></returns>
        public DailyLevels DailyLevels(CustomDashStyle dashStyleType, Color highLine, Color lowLine, int width)
        {
            return DailyLevels(Input, dashStyleType, highLine, lowLine, width);
        }

        /// <summary>
        /// Enter the description of your new custom indicator here
        /// </summary>
        /// <returns></returns>
        public DailyLevels DailyLevels(Data.IDataSeries input, CustomDashStyle dashStyleType, Color highLine, Color lowLine, int width)
        {
            if (cacheDailyLevels != null)
                for (int idx = 0; idx < cacheDailyLevels.Length; idx++)
                    if (cacheDailyLevels[idx].DashStyleType == dashStyleType && cacheDailyLevels[idx].HighLine == highLine && cacheDailyLevels[idx].LowLine == lowLine && cacheDailyLevels[idx].Width == width && cacheDailyLevels[idx].EqualsInput(input))
                        return cacheDailyLevels[idx];

            lock (checkDailyLevels)
            {
                checkDailyLevels.DashStyleType = dashStyleType;
                dashStyleType = checkDailyLevels.DashStyleType;
                checkDailyLevels.HighLine = highLine;
                highLine = checkDailyLevels.HighLine;
                checkDailyLevels.LowLine = lowLine;
                lowLine = checkDailyLevels.LowLine;
                checkDailyLevels.Width = width;
                width = checkDailyLevels.Width;

                if (cacheDailyLevels != null)
                    for (int idx = 0; idx < cacheDailyLevels.Length; idx++)
                        if (cacheDailyLevels[idx].DashStyleType == dashStyleType && cacheDailyLevels[idx].HighLine == highLine && cacheDailyLevels[idx].LowLine == lowLine && cacheDailyLevels[idx].Width == width && cacheDailyLevels[idx].EqualsInput(input))
                            return cacheDailyLevels[idx];

                DailyLevels indicator = new DailyLevels();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.DashStyleType = dashStyleType;
                indicator.HighLine = highLine;
                indicator.LowLine = lowLine;
                indicator.Width = width;
                Indicators.Add(indicator);
                indicator.SetUp();

                DailyLevels[] tmp = new DailyLevels[cacheDailyLevels == null ? 1 : cacheDailyLevels.Length + 1];
                if (cacheDailyLevels != null)
                    cacheDailyLevels.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheDailyLevels = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Enter the description of your new custom indicator here
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.DailyLevels DailyLevels(CustomDashStyle dashStyleType, Color highLine, Color lowLine, int width)
        {
            return _indicator.DailyLevels(Input, dashStyleType, highLine, lowLine, width);
        }

        /// <summary>
        /// Enter the description of your new custom indicator here
        /// </summary>
        /// <returns></returns>
        public Indicator.DailyLevels DailyLevels(Data.IDataSeries input, CustomDashStyle dashStyleType, Color highLine, Color lowLine, int width)
        {
            return _indicator.DailyLevels(input, dashStyleType, highLine, lowLine, width);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Enter the description of your new custom indicator here
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.DailyLevels DailyLevels(CustomDashStyle dashStyleType, Color highLine, Color lowLine, int width)
        {
            return _indicator.DailyLevels(Input, dashStyleType, highLine, lowLine, width);
        }

        /// <summary>
        /// Enter the description of your new custom indicator here
        /// </summary>
        /// <returns></returns>
        public Indicator.DailyLevels DailyLevels(Data.IDataSeries input, CustomDashStyle dashStyleType, Color highLine, Color lowLine, int width)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.DailyLevels(input, dashStyleType, highLine, lowLine, width);
        }
    }
}
#endregion
