// 
// Copyright (C) 2008, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Detects common candlestick patterns and marks them on the chart.
    /// </summary>
    [Description("Detects common candlestick patterns and marks them on the chart.")]
    public class CandleStickPatternAll : Indicator
    {
        #region Variables
		private Color 			downColor;
		private bool			downTrend;
		private BoolSeries 		patternFound;
		
		private Font 			textFont 							= new Font("Courier", 11, FontStyle.Bold);
		private int				trendStrength						= 4;
		private Color 			txtColor;
		private Color 			upColor;
		private bool			upTrend;
		
		private int				lineCountOver  = 10;
		private int				lineCountUnder = -10;
		private bool			bearFound;
		private bool			bullFound;
		private bool			starFound;
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
			AllowRemovalOfDrawObjects = true;
			patternFound = new BoolSeries(this);
            CalculateOnBarClose	= true;
            Overlay				= true;
            PriceTypeSupported	= false;
        }
		
		
        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
			
			
			if (CurrentBar == 0 && ChartControl != null)
			{
				downColor = ChartControl.GetAxisBrush(ChartControl.BackColor).Color;
				txtColor = downColor;
				if (downColor == Color.Black)
					upColor = Color.Transparent;
				else
					upColor = Color.Black;
			}
			
			// Calculates trend lines and prevailing trend for patterns that require a trend
			if (TrendStrength > 0 && CurrentBar >= TrendStrength)
				CalculateTrendLines();
			lineCountOver  = 10;
			lineCountUnder = -10;
			bullFound = false;
			bearFound = false;
			starFound = false;
			
//				BearishBeltHold:			

					#region Bearish Belt Hold
					if (CurrentBar < 5)			//I set this to 5 as that is the number of bars needed for the "Rising Three Methods"
						return;
													
					if (Close[1] > Open[1] && Open[0] > Close[1] + 5 * TickSize && Open[0] == High[0] && Close[0] < Open[0]&& !(TrendStrength > 0 && !upTrend) ) 
					{
						if (ChartControl != null)
						{
							BarColorSeries.Set(CurrentBar - 1, upColor);
							CandleOutlineColorSeries.Set(CurrentBar - 1, downColor);
							BarColor									= downColor;
						}
						
						IText  aText = DrawText("BearBelt Hold" + CurrentBar, false, "BearBelt Hold", 0, High[0], lineCountOver, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						bearFound = true;
						lineCountOver  += 15;
						
						patternFound.Set(true);
					}
					#endregion
				
				if (bearFound == false)
					{
//				BearishEngulfing:
					
						#region Bearish Engulfing
						
						if (Close[1] > Open[1] && Close[0] < Open[0] && Open[0] > Close[1] && Close[0] < Open[1])
						{
							BarColor = downColor;
							IText  aText = DrawText("BearEngulfing" + CurrentBar, false, "BearEngulfing", 0, High[0], lineCountOver, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
							aText.Locked = false;
							bearFound = true;
							lineCountOver  += 15;
							patternFound.Set(true);
						}
						#endregion
	
					if (bearFound == false)
						{	
//				BearishHaramiCross:
		
							#region Bearish Harami Cross
							
							if (Open[0] <= Close[1] && Close[0] >= Open[1] && ((Close[0] >= Open[0] && Close[0] <= Open[0] + TickSize) || (Close[0] <= Open[0] && Close[0] >= Open[0] - TickSize))&& !(TrendStrength > 0 && !upTrend))
							{
								BarColor = downColor;
								IText  aText = DrawText("BearHarami Cross" + CurrentBar, false, "BearHarami Cross", 0, High[0], lineCountOver, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
								aText.Locked = false;
								starFound = true;
								bearFound = true;
								lineCountOver  += 15;
								patternFound.Set(true);
							}
							#endregion
							
						if (starFound == false)
							{	
//				BearishHarami:
			
								#region Bearish Harami
						
						if (Close[0] < Open[0] && Open[0] <= Close[1] && Close[0] >= Open[1] && !(TrendStrength > 0 && !upTrend))
						{
							BarColor = downColor;
							IText  aText = DrawText("BearHarami" + CurrentBar, false, "BearHarami", 0, High[0], lineCountOver, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
							aText.Locked = false;
							bearFound = true;
							lineCountOver  += 15;
							patternFound.Set(true);
						}
						#endregion
							}
						}
					}	

				if (bearFound == false)
				{
//				BullishBeltHold:

					#region Bullish Belt Hold
					
					if (Close[1] < Open[1] && Open[0] < Close[1] - 5 * TickSize && Open[0] == Low[0] && Close[0] > Open[0] && !(TrendStrength > 0 && !downTrend))
					{
						if (ChartControl != null)
						{
							BarColorSeries.Set(CurrentBar - 1, downColor);
							BarColor								= upColor;
							CandleOutlineColorSeries.Set(CurrentBar, downColor);
						}
						
						IText  aText = DrawText("BullBelt Hold" + CurrentBar, false, "BullBelt Hold", 0, Low[0], lineCountUnder, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						bullFound = true;
						lineCountUnder  -= 15;
						patternFound.Set(true);
					}
					#endregion

					if (bullFound == false)
					{
//				BullishEngulfing:

						#region Bullish Engulfing
					
					if (Close[1] < Open[1] && Close[0] > Open[0] && Close[0] > Open[1] && Open[0] < Close[1] && !(TrendStrength > 0 && !downTrend))
					{
						if (ChartControl != null)
						{
							BarColor								= upColor;
							CandleOutlineColorSeries.Set(CurrentBar, downColor);
						}
						
						IText  aText = DrawText("BullEngulfing" + CurrentBar, false, "BullEngulfing", 0, Low[0], lineCountUnder, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						bullFound = true;
						lineCountUnder  -= 15;
						patternFound.Set(true);
					}
					#endregion

						if (bullFound == false)
						{
//				BullishHaramiCross:

							#region Bullish Harami Cross
					
					if (Open[0] >= Close[1] && Close[0] <= Open[1] && ((Close[0] >= Open[0] && Close[0] <= Open[0] + TickSize) || (Close[0] <= Open[0] && Close[0] >= Open[0] - TickSize)) && !(TrendStrength > 0 && !downTrend))
					{
						if (ChartControl != null)
						{
							BarColor								= upColor;
							CandleOutlineColorSeries.Set(CurrentBar, downColor);
						}
						
						IText  aText = DrawText("BullHarami Cross" + CurrentBar, false, "BullHarami Cross", 0, Low[0], lineCountUnder, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						bullFound = true;
						starFound = true;
						lineCountUnder  -= 15;
						patternFound.Set(true);
					}
					#endregion

							if (starFound == false)
							{
//				BullishHarami:

								#region Bullish Harami

					
					if (Close[0] > Open[0] && Open[0] >= Close[1] && Close[0] <= Open[1] && !(TrendStrength > 0 && !downTrend))
					{
						if (ChartControl != null)
						{
							BarColor								= upColor;
							CandleOutlineColorSeries.Set(CurrentBar, downColor);
						}
							
						IText  aText = DrawText("BullHarami" + CurrentBar, false, "BullHarami", 0, Low[0], lineCountUnder, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						bullFound = true;
						starFound = true;
						lineCountUnder  -= 15;
						patternFound.Set(true);
					}
					#endregion
							}
						}
					}
				}	
//				DarkCloudCover:

					#region Dark Cloud Cover
					
					if (Open[0] > High[1] && Close[1] > Open[1] && Close[0] < Open[0] && Close[0] <= Close[1] - (Close[1] - Open[1]) / 2 && Close[0] >= Open[1]&& !(TrendStrength > 0 && !upTrend))	
					{
						if (ChartControl != null)
						{
							CandleOutlineColorSeries.Set(CurrentBar - 1, downColor);
							BarColorSeries.Set(CurrentBar - 1, upColor);
							BarColor									= downColor;
						}
						
						IText  aText = DrawText("Dark Cloud Cover" + CurrentBar, false, "Dark Cloud Cover", 1, High[0], lineCountOver, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						
						lineCountOver  += 15;
						patternFound.Set(true);
					}
					#endregion

					
//				Doji:

					#region Doji
					if (Close[0] == Open[0] || (Close[0] >= Open[0] && Close[0] <= Open[0] + TickSize) || (Close[0] <= Open[0] && Close[0] >= Open[0] - TickSize))
					{
						if (ChartControl != null)
						{
							BarColor								= upColor;
							CandleOutlineColorSeries.Set(CurrentBar, downColor);
						}
						
						int yOffset = Close[0] > Close[Math.Min(1, CurrentBar)] ? 10 : -10;
						if (yOffset > 0)
						{
						IText  aText = DrawText("Doji Text" + CurrentBar, false, "Doji", 0, Low[0], lineCountUnder,  txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						lineCountUnder  =- 10;
						}
							else
							{
							IText  aText = DrawText("Doji Text" + CurrentBar, false, "Doji", 0, High[0], lineCountOver,  txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
							aText.Locked = false;
							lineCountOver  += 15;
							}
							patternFound.Set(true);
					}
					#endregion

					
//				DownsideTasukiGap:

					#region Downside Tasuki Gap
					
					if (Close[2] < Open[2] && Close[1] < Open[1] && Close[0] > Open[0]
						&& High[1] < Low[2]
						&& Open[0] > Close[1] && Open[0] < Open[1]
						&& Close[0] > Open[1] && Close[0] < Close[2])
					{
						if (ChartControl != null)
						{
							BarColor								= upColor;
							CandleOutlineColorSeries.Set(CurrentBar, downColor);
							BarColorSeries.Set(CurrentBar - 1, downColor);
							BarColorSeries.Set(CurrentBar - 2, downColor);
						}
						
						IText  aText = DrawText("Downside Tasuki Gap" + CurrentBar, false, "Downside Tasuki Gap", 1, High[0], lineCountOver, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						
						patternFound.Set(true);
						lineCountOver  += 15;
					}
					#endregion

					
//				EveningStar:

					#region Evening Star
					
					if (Close[2] > Open[2] && Close[1] > Close[2] && Open[0] < (Math.Abs((Close[1] - Open[1])/2) + Open[1]) && Close[0] < Open[0])
					{
						if (ChartControl != null)
						{
							if (Close[0] > Open[0])
							{
								BarColor								= upColor;
								CandleOutlineColorSeries.Set(CurrentBar, downColor);
							}
							else
								BarColor								= downColor;
							
							if (Close[1] > Open[1])
							{
								BarColorSeries.Set(CurrentBar - 1, upColor);
								CandleOutlineColorSeries.Set(CurrentBar - 1, downColor);
							}
							else
								BarColorSeries.Set(CurrentBar - 1, downColor);
							
							if (Close[2] > Open[2])
							{
								BarColorSeries.Set(CurrentBar - 2, upColor);
								CandleOutlineColorSeries.Set(CurrentBar - 2, downColor);
							}
							else
								BarColorSeries.Set(CurrentBar - 2, downColor);
						}

						IText  aText = DrawText("Evening Star Text" + CurrentBar, false, "Evening Star", 1, High[0], lineCountOver, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;	
						
						patternFound.Set(true);
						lineCountOver  += 15;
					}
					#endregion

					
//				FallingThreeMethods:

					#region Falling Three Methods
					
					if (Close[4] < Open[4] && Close[0] < Open[0] && Close[0] < Low[4]
						&& High[3] < High[4] && Low[3] > Low[4]
						&& High[2] < High[4] && Low[2] > Low[4]
						&& High[1] < High[4] && Low[1] > Low[4])
					{
						if (ChartControl != null)
						{
							BarColor						= downColor;
							BarColorSeries.Set(CurrentBar - 4, downColor);
						
							int x = 1;
							while (x < 4)
							{
								if (Close[x] > Open[x])
								{
									BarColorSeries.Set(CurrentBar - x, upColor);
									CandleOutlineColorSeries.Set(CurrentBar - x, downColor);
								}
								else
									BarColorSeries.Set(CurrentBar - x, downColor);
								x++;
							}
						}
						
						IText  aText = DrawText("Falling 3 Methods" + CurrentBar, false, "Falling 3 Methods", 2, High[0], lineCountOver, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						
						patternFound.Set(true);
						lineCountOver  += 15;
					}
					#endregion

					
//				Hammer:

					#region Hammer
					
					if (Low[0] < Open[0] - 5 * TickSize && Math.Abs(Open[0] - Close[0]) < (0.10 * (High[0] - Low[0])) && (High[0] - Close[0]) < (0.25 * (High[0] - Low[0])) && !(TrendStrength > 0 && (!downTrend || MIN(Low, TrendStrength)[0] != Low[0])))
					{
						if (ChartControl != null)
						{
							if (Close[0] > Open[0])
							{
								BarColor								= upColor;
								CandleOutlineColorSeries.Set(CurrentBar, downColor);
							}
							else
								BarColor								= downColor;
						}
						
						IText  aText = DrawText("Hammer" + CurrentBar, false, "Hammer", 0, Low[0], lineCountUnder, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						
						patternFound.Set(true);
						lineCountUnder  -= 15;
					}
					#endregion

					
//				HangingMan:

					#region Hanging Man
					
					if (Low[0] < Open[0] - 5 * TickSize && Math.Abs(Open[0] - Close[0]) < (0.10 * (High[0] - Low[0])) && (High[0] - Close[0]) < (0.25 * (High[0] - Low[0])) && !(TrendStrength > 0 && (!upTrend || MAX(High, TrendStrength)[0] != High[0])))
					{
						if (ChartControl != null)
						{
							if (Close[0] > Open[0])
							{
								BarColor								= upColor;
								CandleOutlineColorSeries.Set(CurrentBar, downColor);
							}
							else
								BarColor								= downColor;
						}
						
						IText  aText = DrawText("Hanging Man" + CurrentBar, false, "Hanging Man", 0,High[0], lineCountOver, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						
						patternFound.Set(true);
						lineCountOver  += 15;
					}
					#endregion

					
//				InvertedHammer:

					#region Inverted Hammer
					
					if (High[0] > Open[0] + 5 * TickSize && Math.Abs(Open[0] - Close[0]) < (0.10 * (High[0] - Low[0])) && (Close[0] - Low[0]) < (0.25 * (High[0] - Low[0])) && !(TrendStrength > 0 && (!upTrend || MAX(High, TrendStrength)[0] != High[0])))
					{
						if (ChartControl != null)
						{
							if (Close[0] > Open[0])
							{
								BarColor								= upColor;
								CandleOutlineColorSeries.Set(CurrentBar, downColor);
							}
							else
								BarColor								= downColor;
						}
							
						IText  aText = DrawText("Inverted Hammer" + CurrentBar, false, "InvertedHammer", 0, High[0], lineCountOver,txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						
						patternFound.Set(true);
						lineCountOver  -= 15;
					}
					#endregion

					
//				MorningStar:

					#region Morning Star
					
					if (Close[2] < Open[2] && Close[1] < Close[2] && Open[0] > (Math.Abs((Close[1] - Open[1])/2) + Open[1]) && Close[0] > Open[0])
					{
						if (ChartControl != null)
						{
							if (Close[0] > Open[0])
							{
								BarColor								= upColor;
								CandleOutlineColorSeries.Set(CurrentBar, downColor);
							}
							else
								BarColor								= downColor;
							
							if (Close[1] > Open[1])
							{
								BarColorSeries.Set(CurrentBar - 1, upColor);
								CandleOutlineColorSeries.Set(CurrentBar - 1, downColor);
							}
							else
								BarColorSeries.Set(CurrentBar - 1, downColor);
							
							if (Close[2] > Open[2])
							{
								BarColorSeries.Set(CurrentBar - 2, upColor);
								CandleOutlineColorSeries.Set(CurrentBar - 2, downColor);
							}
							else
								BarColorSeries.Set(CurrentBar - 2, downColor);
						}
							
						IText  aText = DrawText("Morning Star Text" + CurrentBar, false, "Morning Star", 1, Low[0], lineCountUnder, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;		
						
						patternFound.Set(true);
						lineCountUnder  -= 15;
					}
					#endregion

					
//				PiercingLine:

					#region Piercing Line
					
					if (Open[0] < Low[1] && Close[1] < Open[1] && Close[0] > Open[0] && Close[0] >= Close[1] + (Open[1] - Close[1]) / 2 && Close[0] <= Open[1] && !(TrendStrength > 0 && !downTrend))	
					{
						if (ChartControl != null)
						{
							CandleOutlineColorSeries.Set(CurrentBar - 1, downColor);
							BarColorSeries.Set(CurrentBar - 1, upColor);
							BarColor									= downColor;
						}
						
						IText  aText = DrawText("Piercing Line" + CurrentBar, false, "Piercing Line", 1, Low[0], lineCountUnder, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						
						patternFound.Set(true);
						lineCountUnder  -= 15;
					}
					
					#endregion

					
//				RisingThreeMethods:

					#region Rising Three Methods
					
					if (Close[4] > Open[4] && Close[0] > Open[0] && Close[0] > High[4]
						&& High[3] < High[4] && Low[3] > Low[4]
						&& High[2] < High[4] && Low[2] > Low[4]
						&& High[1] < High[4] && Low[1] > Low[4])
					{
						if (ChartControl != null)
						{
							BarColor									= upColor;
							CandleOutlineColorSeries.Set(CurrentBar, downColor);
							BarColorSeries.Set(CurrentBar - 4, upColor);
							CandleOutlineColorSeries.Set(CurrentBar - 4, downColor);
						
							int x = 1;
							while (x < 4)
							{
								if (Close[x] > Open[x])
								{
									BarColorSeries.Set(CurrentBar - x, upColor);
									CandleOutlineColorSeries.Set(CurrentBar - x, downColor);
								}
								else
									BarColorSeries.Set(CurrentBar - x, downColor);
								x++;
							}
						}
						
						IText  aText = DrawText("Rising 3 Methods" + CurrentBar, false, "Rising 3 Methods", 2, Low[0], lineCountUnder, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						
						patternFound.Set(true);
					}
					#endregion

					
//				ShootingStar:

					#region Shooting Star
					
					if (High[0] > Open[0] && (High[0] - Open[0]) >= 2 * (Open[0] - Close[0]) && Close[0] < Open[0] && (Close[0] - Low[0]) <= 2 * TickSize && !(TrendStrength > 0 && !upTrend))
					{
						if (ChartControl != null)
							BarColor = downColor;
						
						IText  aText = DrawText("Shooting Star" + CurrentBar, false, "Shooting Star", 0, High[0], lineCountOver, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						
						patternFound.Set(true);
						lineCountOver  += 15;
					}
					#endregion

					
//				StickSandwich:

					#region Stick Sandwich
					
					if (Close[2] == Close[0] && Close[2] < Open[2] && Close[1] > Open[1] && Close[0] < Open[0])
					{
						if (ChartControl != null)
						{
							BarColor									= downColor;
							BarColorSeries.Set(CurrentBar - 1, upColor);
							CandleOutlineColorSeries.Set(CurrentBar - 1, downColor);
							BarColorSeries.Set(CurrentBar - 2, downColor);
						}
						
						IText  aText = DrawText("Stick Sandwich" + CurrentBar, false, "Stick Sandwich", 1, Low[0], lineCountUnder, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						
						patternFound.Set(true);
					}
					#endregion

					
//				ThreeBlackCrows:

					#region Three Black Crows
					
					if (patternFound[1] == false && patternFound[2] == false 
						&& Close[0] < Open[0] && Close[1] < Open[1] && Close[2] < Open[2]
						&& Close[0] < Close[1] && Close[1] < Close[2]
						&& Open[0] < Open[1] && Open[0] > Close[1]
						&& Open[1] < Open[2] && Open[1] > Close[2]
						&& !(TrendStrength > 0 && !upTrend))
					{
						if (ChartControl != null)
						{
							BarColor						= downColor;
							BarColorSeries.Set(CurrentBar - 1, downColor);
							BarColorSeries.Set(CurrentBar - 2, downColor);
						}
						
						IText  aText = DrawText("3 Black Crows" + CurrentBar, false, "3 Black Crows", 1, High[0], lineCountOver, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						
						patternFound.Set(true);
						lineCountOver  += 15;
					}
					#endregion

					
//				ThreeWhiteSoldiers:

					#region Three White Soldiers
					
					if (patternFound[1] == false && patternFound[2] == false 
						&& Close[0] > Open[0] && Close[1] > Open[1] && Close[2] > Open[2]
						&& Close[0] > Close[1] && Close[1] > Close[2]
						&& Open[0] < Close[1] && Open[0] > Open[1]
						&& Open[1] < Close[2] && Open[1] > Open[2]
						&& !(TrendStrength > 0 && !downTrend))
					{
						if (ChartControl != null)
						{
							BarColor									= upColor;
							CandleOutlineColorSeries.Set(CurrentBar, downColor);
							BarColorSeries.Set(CurrentBar - 1, upColor);
							CandleOutlineColorSeries.Set(CurrentBar - 1, downColor);
							BarColorSeries.Set(CurrentBar - 2, upColor);
							CandleOutlineColorSeries.Set(CurrentBar - 2, downColor);
						}
						
						IText  aText = DrawText("3 White Soldiers" + CurrentBar, false, "3 White Soldiers", 1, Low[2], lineCountUnder, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						
						patternFound.Set(true);
						lineCountUnder  -= 15;
					}
					#endregion

					
//				UpsideGapTwoCrows:

					#region Upside Gap Two Crows
					
					if (Close[2] > Open[2] && Close[1] < Open[1] && Close[0] < Open[0]
						&& Low[1] > High[2]
						&& Close[0] > High[2]
						&& Close[0] < Close[1] && Open[0] > Open[1]
						&& !(TrendStrength > 0 && !upTrend))
					{
						if (ChartControl != null)
						{
							BarColor									= downColor;
							BarColorSeries.Set(CurrentBar - 1, downColor);
							BarColorSeries.Set(CurrentBar - 2, upColor);
							CandleOutlineColorSeries.Set(CurrentBar - 2, downColor);
						}
						
						IText  aText = DrawText("Upside Gap Two Crows" + CurrentBar, false, "Upside Gap Two Crows", 1,High[0], lineCountOver, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						
						patternFound.Set(true);
						lineCountOver  += 15;
					}
					#endregion

					
//				UpsideTasukiGap:

					#region Upside Tasuki Gap
					if (Close[2] > Open[2] && Close[1] > Open[1] && Close[0] < Open[0]
						&& Low[1] > High[2]
						&& Open[0] < Close[1] && Open[0] > Open[1]
						&& Close[0] < Open[1] && Close[0] > Close[2])
					{
						if (ChartControl != null)
						{
							BarColor									= downColor;
							BarColorSeries.Set(CurrentBar - 1, upColor);
							CandleOutlineColorSeries.Set(CurrentBar - 1, downColor);
							BarColorSeries.Set(CurrentBar - 2, upColor);
							CandleOutlineColorSeries.Set(CurrentBar - 2, downColor);
						}
						
						IText  aText = DrawText("Upside Tasuki Gap" + CurrentBar, false, "Upside Tasuki Gap", 1, High[0], lineCountOver, txtColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						aText.Locked = false;
						
						patternFound.Set(true);
						lineCountOver  += 15;
					}
					#endregion


					}

        #region Properties
		[Browsable(false)]
        [XmlIgnore()]
        public BoolSeries PatternFound
        {
            get { return patternFound; }
        }
		
		[Description("Number of bars required to define a trend when a pattern requires a prevailing trend. A value of zero will disable trend requirement.")]
		[Category("Parameters")]
		[Gui.Design.DisplayName("Trend strength")]
        public int TrendStrength
        {
            get { return trendStrength; }
			set { trendStrength = Math.Max(0, value); }
        }
		


        #endregion
		
		#region Misc
		public override string ToString()
		{
			return Name + "(" + "" + ")";
		}
		
		// Calculate trend lines and prevailing trend
		private void CalculateTrendLines()
		{
			// Calculate up trend line
			int upTrendStartBarsAgo		= 0;
			int upTrendEndBarsAgo 		= 0;
			int upTrendOccurence 		= 1;
			
			while (Low[upTrendEndBarsAgo] <= Low[upTrendStartBarsAgo])
			{
				upTrendStartBarsAgo 	= Swing(TrendStrength).SwingLowBar(0, upTrendOccurence + 1, CurrentBar);
				upTrendEndBarsAgo 		= Swing(TrendStrength).SwingLowBar(0, upTrendOccurence, CurrentBar);
					
				if (upTrendStartBarsAgo < 0 || upTrendEndBarsAgo < 0)
					break;

				upTrendOccurence++;
			}
			
			
			// Calculate down trend line	
			int downTrendStartBarsAgo	= 0;
			int downTrendEndBarsAgo 	= 0;
			int downTrendOccurence 		= 1;
			
			while (High[downTrendEndBarsAgo] >= High[downTrendStartBarsAgo])
			{
				downTrendStartBarsAgo 		= Swing(TrendStrength).SwingHighBar(0, downTrendOccurence + 1, CurrentBar);
				downTrendEndBarsAgo 		= Swing(TrendStrength).SwingHighBar(0, downTrendOccurence, CurrentBar);
					
				if (downTrendStartBarsAgo < 0 || downTrendEndBarsAgo < 0)
					break;
					
				downTrendOccurence++;
			}
			
			if (upTrendStartBarsAgo > 0 && upTrendEndBarsAgo > 0 && upTrendStartBarsAgo < downTrendStartBarsAgo)
			{
				upTrend 	= true;
				downTrend 	= false;	
			}
			else if (downTrendStartBarsAgo > 0 && downTrendEndBarsAgo > 0  && upTrendStartBarsAgo > downTrendStartBarsAgo)
			{
				upTrend 	= false;
				downTrend 	= true;
			}
			else
			{
				upTrend 	= false;
				downTrend 	= false;
			}			
		}
		
		#endregion
    }
}



#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private CandleStickPatternAll[] cacheCandleStickPatternAll = null;

        private static CandleStickPatternAll checkCandleStickPatternAll = new CandleStickPatternAll();

        /// <summary>
        /// Detects common candlestick patterns and marks them on the chart.
        /// </summary>
        /// <returns></returns>
        public CandleStickPatternAll CandleStickPatternAll(int trendStrength)
        {
            return CandleStickPatternAll(Input, trendStrength);
        }

        /// <summary>
        /// Detects common candlestick patterns and marks them on the chart.
        /// </summary>
        /// <returns></returns>
        public CandleStickPatternAll CandleStickPatternAll(Data.IDataSeries input, int trendStrength)
        {
            if (cacheCandleStickPatternAll != null)
                for (int idx = 0; idx < cacheCandleStickPatternAll.Length; idx++)
                    if (cacheCandleStickPatternAll[idx].TrendStrength == trendStrength && cacheCandleStickPatternAll[idx].EqualsInput(input))
                        return cacheCandleStickPatternAll[idx];

            lock (checkCandleStickPatternAll)
            {
                checkCandleStickPatternAll.TrendStrength = trendStrength;
                trendStrength = checkCandleStickPatternAll.TrendStrength;

                if (cacheCandleStickPatternAll != null)
                    for (int idx = 0; idx < cacheCandleStickPatternAll.Length; idx++)
                        if (cacheCandleStickPatternAll[idx].TrendStrength == trendStrength && cacheCandleStickPatternAll[idx].EqualsInput(input))
                            return cacheCandleStickPatternAll[idx];

                CandleStickPatternAll indicator = new CandleStickPatternAll();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.TrendStrength = trendStrength;
                Indicators.Add(indicator);
                indicator.SetUp();

                CandleStickPatternAll[] tmp = new CandleStickPatternAll[cacheCandleStickPatternAll == null ? 1 : cacheCandleStickPatternAll.Length + 1];
                if (cacheCandleStickPatternAll != null)
                    cacheCandleStickPatternAll.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheCandleStickPatternAll = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Detects common candlestick patterns and marks them on the chart.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.CandleStickPatternAll CandleStickPatternAll(int trendStrength)
        {
            return _indicator.CandleStickPatternAll(Input, trendStrength);
        }

        /// <summary>
        /// Detects common candlestick patterns and marks them on the chart.
        /// </summary>
        /// <returns></returns>
        public Indicator.CandleStickPatternAll CandleStickPatternAll(Data.IDataSeries input, int trendStrength)
        {
            return _indicator.CandleStickPatternAll(input, trendStrength);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Detects common candlestick patterns and marks them on the chart.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.CandleStickPatternAll CandleStickPatternAll(int trendStrength)
        {
            return _indicator.CandleStickPatternAll(Input, trendStrength);
        }

        /// <summary>
        /// Detects common candlestick patterns and marks them on the chart.
        /// </summary>
        /// <returns></returns>
        public Indicator.CandleStickPatternAll CandleStickPatternAll(Data.IDataSeries input, int trendStrength)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.CandleStickPatternAll(input, trendStrength);
        }
    }
}
#endregion
