//Indicator: VWAP
//Author: cvax
//Version: 1.0 Release Candidate 1
//
//Changelog:
//			v1.0rc1	-	Optimized VWAP and SD calculation functions
//						Removed forced CalculateOnBarClose setting for newer SD calculations
//			v1.0b10 -	Added ability for user defined multiplier on SD bands
//						Added ability to disable accuracy warning
//						Fixed bug in Standard Deviation Population calculations
//						Fixed bug with VWAP input series selection
//						Fixed bug when using in a NinjaScript Strategy
//						Fixed bug with real-time VWAP calculations on tick/volume/range charts							
//			v1.0b9 -	Added Start and End Times
//						Replaced SD multiplier with native support for up to 3 SD bands
//			v1.0b8 -	Added support for more accurate SD calculations
//						Fixed several plotting bugs
//						Optimized code
//			v1.0b7 -	Added compatibility with NinjaTrader 6.5
//			v1.0b6 -	Fixed bug loading VWAP when there is no data connection
//			v1.0b5 -	Added checks to ensure proper VWAP usage
//			v1.0b4 -	Replaced Standard Deviation plot with a DataSeries
//			v1.0b3 -	Added coloring between VWAP bands
//			v1.0b2 -	Added VWAP bands
//						Added on/off switches for plots
//			v1.0b1 -	Added VWAP standard deviation plot

#region Using declarations
using System;
using System.Collections;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    [Description("VWAP is the ratio of the value traded to total volume traded over a particular time horizon (usually one day). It is a measure of the average price a stock traded at over the trading horizon.")]
    [Gui.Design.DisplayName("VWAP (Volume-Weighted Average Price)")]
    public class VWAP : Indicator
    {
        #region Variables
        //VWAP Variables
		private int			numStdDev		= 3;
		public double		SD				= 0;
		private double		sd1Multi		= 1;
		private double		sd2Multi		= 2;
		private double		sd3Multi		= 3;
		
		private BandType	bandType		= BandType.VWAP;
		private bool		showvwapstddev	= true;
		private bool		showWarning		= true;
		private bool		estvwap			= true;
		private Color		fontColor;
		private double		previousvolume	= 0;
		private double		currentvolume	= 0;
		private double		tickvolume		= 0;
		private double		vwapavg			= 0;
		private double		vwapsum			= 0;
		private double		volwap			= 0;
		private ArrayList	volumearray		= new ArrayList();
		private ArrayList	pricearray		= new ArrayList();
		private ArrayList	vwaparray		= new ArrayList();
		
		private ArrayList	priceRangeArray = new ArrayList();
		private ArrayList	priceRangeVolArray = new ArrayList();
		
		private DataSeries	vwapsd;
		private DateTime	loadtime;
		private DateTime	sessionend;
		private int			previousBar		= 0;
		private int			timecheck		= 1;
		
		private TimeSpan	startTime;
		private DateTime	startTimeDate;
		private TimeSpan	endTime;
		private DateTime	endTimeDate;
		private bool 		useSessionBegin = false;
		
		//Plot Coloring
		private Color bandAreaColor = Color.Blue;
		private int bandAreaColorOpacity = 1;
		
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
			Add(new Plot(Color.Orange, "VWAP"));
			Add(new Plot(Color.Blue, "SD1 Upper"));
			Add(new Plot(Color.Blue, "SD1 Lower"));
			Add(new Plot(Color.Blue, "SD2 Upper"));
			Add(new Plot(Color.Blue, "SD2 Lower"));
			Add(new Plot(Color.Blue, "SD3 Upper"));
			Add(new Plot(Color.Blue, "SD3 Lower"));			
			
			vwapsd = new DataSeries(this);
			AutoScale			= false;
			BarsRequired		= 1;
            CalculateOnBarClose	= false;
            Overlay				= true;
            PriceTypeSupported	= true;
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
			if (Bars == null)
				return;
			//Only allowed on Intraday charts.
			if (Bars.Period.Id == PeriodType.Day || Bars.Period.Id == PeriodType.Week || Bars.Period.Id == PeriodType.Month || Bars.Period.Id == PeriodType.Year)
				return;
			//Remembers the DateTime that the indicator was started on
			if (CurrentBar == 0)
			{
				if(Bars.MarketDataSubscribed)
					loadtime = Now;
				else
					loadtime = DateTime.Now;
				
				sessionend = new DateTime(Bars.Session.NextBeginTime.Year, Bars.Session.NextBeginTime.Month, Bars.Session.NextBeginTime.Day, Bars.Session.NextEndTime.Hour, Bars.Session.NextEndTime.Minute, Bars.Session.NextEndTime.Second);
				startTimeDate = new DateTime(Bars.Session.NextBeginTime.Year, Bars.Session.NextBeginTime.Month, Bars.Session.NextBeginTime.Day, startTime.Hours, startTime.Minutes, startTime.Seconds);
				endTimeDate = new DateTime(Bars.Session.NextBeginTime.Year, Bars.Session.NextBeginTime.Month, Bars.Session.NextBeginTime.Day, endTime.Hours, endTime.Minutes, endTime.Seconds);
				
				// Checks to ensure End Time is not before Start Time: if it is set End Time to SessionEnd Time
				if (DateTime.Compare(endTimeDate, startTimeDate) < 0)
					endTimeDate = sessionend;
				
				// Determines whether to use SessionBegin Time or Start Time: If Start Time undefined/less than SessionBegin, use SessionBegin
				if (startTime.TotalMinutes == 0 || ToTime(startTimeDate) < ToTime(Bars.Session.NextBeginTime))
					useSessionBegin = true;
			}

			// If Time is outside range of Start Time and End Time do not calculate further
			if (ToTime(Time[0]) < ToTime(startTimeDate) || (ToTime(endTimeDate) != 0 && ToTime(Time[0]) > ToTime(endTimeDate)))
				return;

			//Recalculates VWAP on start of every new session			
			if ((useSessionBegin && Bars.FirstBarOfSession && FirstTickOfBar) ||
				(useSessionBegin == false && FirstTickOfBar && ToTime(Time[1]) < ToTime(startTimeDate) && ToTime(Time[0]) >= ToTime(startTimeDate)))
			{
				estvwap = false;
				if (bandType == BandType.AvgVWAP)	CalculateOnBarClose = false;
				timecheck = 1;
				vwapsum = 0;
				vwapavg = 0;
				volwap = 0;
				volumearray.Clear();
				pricearray.Clear();
				vwaparray.Clear();
				priceRangeArray.Clear();
				priceRangeVolArray.Clear();
			}
			//Checks to see if load time was before or after market start
			else if(loadtime > Time[0] && timecheck == 1)
			{
				estvwap = true;
				timecheck = 0;
			}
			else if (timecheck == 1)
			{
				estvwap = false;
				if (bandType == BandType.AvgVWAP)	CalculateOnBarClose = false;
				timecheck = 0;
			}
			
			// Calculate current volume
			if(FirstTickOfBar)
			{
				tickvolume = Volume[0];
				previousvolume = Volume[0];
				currentvolume = Volume[0];
				
				previousBar = CurrentBar;
			}
			else
			{
				/* Check to see if this is a new bar. For some reason, FirstTickOfBar isn't reliable when using
				Market Replay to determine if it is a new bar. */
				if (previousBar == CurrentBar)
					previousvolume = currentvolume;
				else
					previousvolume = 0;
				
				previousBar = CurrentBar;
				currentvolume = Volume[0];
				tickvolume = currentvolume - previousvolume;
			}
			
			// Calculate VWAP. If using Avg VWAP for SD calcs, need to prevent heavy weighting on incoming ticks vs historical bars.
			if (estvwap && bandType == BandType.AvgVWAP)
			{
				CalculateOnBarClose = true;
				volumearray.Add(tickvolume);
				pricearray.Add(Input[0]);
				volwap = vwap(volumearray, pricearray);
				VWAPLine.Set(volwap);
			}
			else
			{
				volumearray.Add(tickvolume);
				pricearray.Add(Input[0]);
				volwap = vwap(volumearray, pricearray);
				VWAPLine.Set(volwap);
			}
			
			if(showvwapstddev)
			{
				switch (bandType)
				{
					case BandType.AvgVWAP:
					{
						vwaparray.Add(volwap);
						vwapsum = vwapsum + volwap;
						vwapavg = vwapsum / vwaparray.Count;
						SD = StdDeviation(vwaparray, vwapavg);
						vwapsd.Set(SD);
						PlotSD(SD);
						break;
					}
					
					case BandType.VWAP:
					{
						// Create array for price ranges (1 entry per traded price)
						if (priceRangeArray.Contains(Input[0]))
						{
							int index = priceRangeArray.IndexOf(Input[0]);
							double newVol = (double)priceRangeVolArray[index] + tickvolume;
							priceRangeVolArray[index] = newVol;
						}
						else
						{
							priceRangeArray.Add(Input[0]);
							priceRangeVolArray.Add(tickvolume);
						}
						
						SD = StdDeviationProb(priceRangeArray, volwap);
						vwapsd.Set(SD);
						PlotSD(SD);						
						break;
					}
				}
			}
			
			// Plotting
			if (estvwap && showWarning)
			{
				if (ChartControl != null)
					fontColor = ChartControl.GetAxisBrush(ChartControl.BackColor).Color;
				else
					fontColor = Color.Black;
				DrawTextFixed("VWAPError", "VWAP is most accurate when loaded before market start. \nThe current VWAP values displayed are estimates.", TextPosition.BottomRight, fontColor, new Font("Arial", 8), Color.Transparent, Color.Transparent, 0);
			}
			else
				RemoveDrawObject("VWAPError");
		}

        #region Properties
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries VWAPLine
        {
            get { return Values[0]; }
        }
		
		[Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries SD1Upper
        {
            get { return Values[1]; }
        }
		
		[Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries SD1Lower
        {
            get { return Values[2]; }
        }
		
		[Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries SD2Upper
        {
            get { return Values[3]; }
        }
		
		[Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries SD2Lower
        {
            get { return Values[4]; }
        }
		
		[Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries SD3Upper
        {
            get { return Values[5]; }
        }
		
		[Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries SD3Lower
        {
            get { return Values[6]; }
        }
		
		[Description("Std Dev 1 Multiplier")]
		[Category("Plots")]
		[Gui.Design.DisplayName("\t\t\tSD1 Multiplier")]
		public double SD1Multi
		{
			get { return sd1Multi; }
			set { sd1Multi = value; }
		}
		
		[Description("Std Dev 2 Multiplier")]
		[Category("Plots")]
		[Gui.Design.DisplayName("\t\tSD2 Multiplier")]
		public double SD2Multi
		{
			get { return sd2Multi; }
			set { sd2Multi = value; }
		}
		
		[Description("Std Dev 3 Multiplier")]
		[Category("Plots")]
		[Gui.Design.DisplayName("\tSD3 Multiplier")]
		public double SD3Multi
		{
			get { return sd3Multi; }
			set { sd3Multi = value; }
		}
		
		[Description("Choose what to calculate SD on. AvgVWAP calculates SD on the average VWAP. VWAP calculates SD on input price with VWAP as the mean.")]
		[Category("Settings")]
		[Gui.Design.DisplayName("\t\tStd Dev Calc Mode")]
		public BandType BType
		{
			get { return bandType; }
			set { bandType = value; }
		}
		
		[Description("Show VWAP standard deviation.")]
		[Category("Settings")]
		[Gui.Design.DisplayName("\t\t\tShow Std Dev")]
		public bool ShowVWAPStdDev
		{
			get { return showvwapstddev; }
			set { showvwapstddev = value; }
		}
		
		[Description("# of Standard Deviation bands for VWAP. (Max 3)")]
		[Category("Settings")]
		[Gui.Design.DisplayName("\t# of Std Dev")]
		public int NumStdDev
		{
			get { return numStdDev; }
			set { numStdDev = Math.Max(1, Math.Min(value, 3)); }
		}
		
		[Description("VWAP Start Time")]
		[Category("Settings")]
		[Gui.Design.DisplayName("\t\t\t\t\t\tStart Time")]
		public TimeSpan StartTime
		{
			get { return startTime; }
			set { startTime = value; }
		}
		
		[Description("VWAP End Time")]
		[Category("Settings")]
		[Gui.Design.DisplayName("\t\t\t\t\tEnd Time")]
		public TimeSpan EndTime
		{
			get { return endTime; }
			set { endTime = value; }
		}
		
		[Description("Show accuracy warning when potential inaccuracies exist.")]
		[Category("Settings")]
		[Gui.Design.DisplayName("\t\t\t\tAccuracy Warning")]
		public bool ShowWarning
		{
			get { return showWarning; }
			set { showWarning = value; }
		}
		
		[Description("Band Color")]
		[Category("Visual")]
		[Gui.Design.DisplayName("Band Area Color")]
		public Color BandAreaColor
		{
			get { return bandAreaColor; }
			set { bandAreaColor = value; /*RecalculateColors();*/ }
		}
		
		[Browsable(false)]
		public string BandAreaColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(bandAreaColor); }
			set { bandAreaColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}

		[Description("Band Color Opacity (least-most: 0-10)")]
		[Category("Visual")]
		[Gui.Design.DisplayName("Color Opacity")]
		public int BandAreaColorOpacity
		{
			get { return bandAreaColorOpacity; }
			set { bandAreaColorOpacity = Math.Max(value, 0); }
		}
		#endregion
		
		#region Miscellaneous
		private double vwap(ArrayList volume, ArrayList price)
		{
			int x = 0;
			double numerator = 0;
			double denominator = 0;
			while (x < volume.Count)
			{
				
				numerator = numerator + ((double)price[x] * (double)volume[x]);
				denominator = denominator + (double)volume[x];
				x++;
			}
			
			if (denominator > 0)
				return numerator / denominator;
			else
				return 0;
		}
		
		private double StdDeviation(ArrayList array, double avg)
		{
			int x = 0;
			double sd = 0;
			while (x < array.Count)
			{
				sd += (((double)array[x] - avg) * ((double)array[x] - avg));
				x++;
			}
			if (sd > 0)
				sd = System.Math.Sqrt(sd / (array.Count));
			return sd;
		}
		
		private double StdDeviationProb(ArrayList array, double avg)
		{
			double totalVolume = 0;
			int x = 0;
			double sd = 0;
			
			for (int n = 0; n <= Bars.BarsSinceSession; n++ )
				totalVolume += Volume[n];
			
			while (x < priceRangeArray.Count)
			{
				int index = priceRangeArray.IndexOf(array[x]);
				
				if (totalVolume > 0)
					sd += (((double)array[x] - avg) * ((double)array[x] - avg) * ((double)priceRangeVolArray[index] / totalVolume));

				x++;
			}
			if (sd > 0)
				sd = System.Math.Sqrt(sd);
			return sd;
		}
		
		private void PlotSD(double SD)
		{
			if (numStdDev == 3)
			{
				SD1Upper.Set(VWAPLine[0] + sd1Multi * SD);
				SD1Lower.Set(VWAPLine[0] - sd1Multi * SD);
				SD2Upper.Set(VWAPLine[0] + sd2Multi * SD);
				SD2Lower.Set(VWAPLine[0] - sd2Multi * SD);
				SD3Upper.Set(VWAPLine[0] + sd3Multi * SD);
				SD3Lower.Set(VWAPLine[0] - sd3Multi * SD);
				DrawRegion("ColorVWAP", CurrentBar, 0, SD1Upper, SD1Lower, Color.Transparent, bandAreaColor, bandAreaColorOpacity);
				DrawRegion("ColorVWAP2", CurrentBar, 0, SD2Upper, SD2Lower, Color.Transparent, bandAreaColor, bandAreaColorOpacity);
				DrawRegion("ColorVWAP3", CurrentBar, 0, SD3Upper, SD3Lower, Color.Transparent, bandAreaColor, bandAreaColorOpacity);
			}
			else if (numStdDev == 2)
			{
				SD1Upper.Set(VWAPLine[0] + sd1Multi * SD);
				SD1Lower.Set(VWAPLine[0] - sd1Multi * SD);
				SD2Upper.Set(VWAPLine[0] + sd2Multi * SD);
				SD2Lower.Set(VWAPLine[0] - sd2Multi * SD);
				DrawRegion("ColorVWAP", CurrentBar, 0, SD1Upper, SD1Lower, Color.Transparent, bandAreaColor, bandAreaColorOpacity);
				DrawRegion("ColorVWAP2", CurrentBar, 0, SD2Upper, SD2Lower, Color.Transparent, bandAreaColor, bandAreaColorOpacity);
			}
			else
			{
				SD1Upper.Set(VWAPLine[0] + sd1Multi * SD);
				SD1Lower.Set(VWAPLine[0] - sd1Multi * SD);
				DrawRegion("ColorVWAP", CurrentBar, 0, SD1Upper, SD1Lower, Color.Transparent, bandAreaColor, bandAreaColorOpacity);
			}
		}
		
		private DateTime Now
		{
			get { return (Bars.MarketData.Connection.Options.Provider == Cbi.Provider.Replay ? Bars.MarketData.Connection.Now : DateTime.Now); }
		}
		#endregion
    }
}

public enum BandType
{
	AvgVWAP,
	VWAP,
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private VWAP[] cacheVWAP = null;

        private static VWAP checkVWAP = new VWAP();

        /// <summary>
        /// VWAP is the ratio of the value traded to total volume traded over a particular time horizon (usually one day). It is a measure of the average price a stock traded at over the trading horizon.
        /// </summary>
        /// <returns></returns>
        public VWAP VWAP()
        {
            return VWAP(Input);
        }

        /// <summary>
        /// VWAP is the ratio of the value traded to total volume traded over a particular time horizon (usually one day). It is a measure of the average price a stock traded at over the trading horizon.
        /// </summary>
        /// <returns></returns>
        public VWAP VWAP(Data.IDataSeries input)
        {
            if (cacheVWAP != null)
                for (int idx = 0; idx < cacheVWAP.Length; idx++)
                    if (cacheVWAP[idx].EqualsInput(input))
                        return cacheVWAP[idx];

            lock (checkVWAP)
            {
                if (cacheVWAP != null)
                    for (int idx = 0; idx < cacheVWAP.Length; idx++)
                        if (cacheVWAP[idx].EqualsInput(input))
                            return cacheVWAP[idx];

                VWAP indicator = new VWAP();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                Indicators.Add(indicator);
                indicator.SetUp();

                VWAP[] tmp = new VWAP[cacheVWAP == null ? 1 : cacheVWAP.Length + 1];
                if (cacheVWAP != null)
                    cacheVWAP.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheVWAP = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// VWAP is the ratio of the value traded to total volume traded over a particular time horizon (usually one day). It is a measure of the average price a stock traded at over the trading horizon.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.VWAP VWAP()
        {
            return _indicator.VWAP(Input);
        }

        /// <summary>
        /// VWAP is the ratio of the value traded to total volume traded over a particular time horizon (usually one day). It is a measure of the average price a stock traded at over the trading horizon.
        /// </summary>
        /// <returns></returns>
        public Indicator.VWAP VWAP(Data.IDataSeries input)
        {
            return _indicator.VWAP(input);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// VWAP is the ratio of the value traded to total volume traded over a particular time horizon (usually one day). It is a measure of the average price a stock traded at over the trading horizon.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.VWAP VWAP()
        {
            return _indicator.VWAP(Input);
        }

        /// <summary>
        /// VWAP is the ratio of the value traded to total volume traded over a particular time horizon (usually one day). It is a measure of the average price a stock traded at over the trading horizon.
        /// </summary>
        /// <returns></returns>
        public Indicator.VWAP VWAP(Data.IDataSeries input)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.VWAP(input);
        }
    }
}
#endregion
