//
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// HeikenAshi2 technique discussed in the article 'Using Heiken-Ashi Technique' in February 2004 issue of TASC magazine.
    /// </summary>
    [Description("HeikenAshi2 technique discussed in the article 'Using Heiken-Ashi Technique' in February 2004 issue of TASC magazine.")]
    public class HeikenAshi2 : Indicator
    {
        #region Variables

        private Color           barColorDown         = Color.Red;
        private Color           barColorUp           = Color.Lime;
        private SolidBrush      brushDown            = null;
        private SolidBrush      brushUp              = null;
        private Color           savePenColor         = Color.Transparent;
        private Color           savePen2Color        = Color.Transparent;
        private Color           saveDownColor        = Color.Empty;
        private Color           saveUpColor          = Color.Empty;
        private Color           shadowColorUp        = Color.Lime;
        private Color           shadowColorDown      = Color.Red;
        private Color           shadowColorDoji      = Color.Gold;
      	private Pen             shadowPenUp          = null;
        private Pen				shadowPenDown		 = null;
        private Pen				shadowPenDoji		 = null;
		private int             shadowWidth          = 1;

        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
            Add(new Plot(Color.Gray, PlotStyle.Line, "HAOpen"));
            Add(new Plot(Color.Gray, PlotStyle.Line, "HAHigh"));
            Add(new Plot(Color.Gray, PlotStyle.Line, "HALow"));
            Add(new Plot(Color.Gray, PlotStyle.Line, "HAClose"));
            PaintPriceMarkers   = false;
            CalculateOnBarClose = false;
            PlotsConfigurable   = false;
            Overlay             = true;
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {

            if (CurrentBar == 0)
            {
                HAOpen.Set(Open[0]);
                HAHigh.Set(High[0]);
                HALow.Set(Low[0]);
                HAClose.Set(Close[0]);
                return;
            }

            HAClose.Set((Open[0] + High[0] + Low[0] + Close[0]) * 0.25); // Calculate the close
            HAOpen.Set((HAOpen[1] + HAClose[1]) * 0.5); // Calculate the open
            HAHigh.Set(Math.Max(High[0], HAOpen[0])); // Calculate the high
            HALow.Set(Math.Min(Low[0], HAOpen[0])); // Calculate the low
        }

        #region Properties

        [Browsable(false)]
        [XmlIgnore]
        public DataSeries HAOpen
        {
            get { return Values[0]; }
        }

        [Browsable(false)]
        [XmlIgnore]
        public DataSeries HAHigh
        {
            get { return Values[1]; }
        }

        [Browsable(false)]
        [XmlIgnore]
        public DataSeries HALow
        {
            get { return Values[2]; }
        }

        [Browsable(false)]
        [XmlIgnore]
        public DataSeries HAClose
        {
            get { return Values[3]; }
        }

        [XmlIgnore]
        [Description("Color of down bars.")]
        [Category("Visual")]
        [Gui.Design.DisplayNameAttribute("Candle Color Down")]
        public Color BarColorDown
        {
            get { return barColorDown; }
            set { barColorDown = value; }
        }

        /// <summary>
        /// </summary>
        [Browsable(false)]
        public string BarColorDownSerialize
        {
            get { return Gui.Design.SerializableColor.ToString(barColorDown); }
            set { barColorDown = Gui.Design.SerializableColor.FromString(value); }
        }

        /// <summary>
        /// </summary>
        [XmlIgnore]
        [Description("Color of up bars.")]
        [Category("Visual")]
        [Gui.Design.DisplayNameAttribute("Candle Color Up")]
        public Color BarColorUp
        {
            get { return barColorUp; }
            set { barColorUp = value; }
        }

        /// <summary>
        /// </summary>
        [Browsable(false)]
        public string BarColorUpSerialize
        {
            get { return Gui.Design.SerializableColor.ToString(barColorUp); }
            set { barColorUp = Gui.Design.SerializableColor.FromString(value); }
        }

        /// <summary>
        /// </summary>
        [XmlIgnore]
        [Description("Color of Shadow Line for UpClose.")]
        [Category("Visual")]
        [Gui.Design.DisplayNameAttribute("Shadow Color Up")]
        public Color ShadowColorUp
        {
            get { return shadowColorUp; }
            set { shadowColorUp = value; }
        }

        /// <summary>
        /// </summary>
        [Browsable(false)]
        public string ShadowColorUpSerialize
        {
            get { return Gui.Design.SerializableColor.ToString(shadowColorUp); }
            set { shadowColorUp = Gui.Design.SerializableColor.FromString(value); }
        }

        /// <summary>
        /// </summary>
        [XmlIgnore]
        [Description("Color of Shadow Line for DownClose.")]
        [Category("Visual")]
        [Gui.Design.DisplayNameAttribute("Shadow Color Down")]
        public Color ShadowColorDown
        {
            get { return shadowColorDown; }
            set { shadowColorDown = value; }
        }

        /// <summary>
        /// </summary>
        [Browsable(false)]
        public string ShadowColorDownSerialize
        {
            get { return Gui.Design.SerializableColor.ToString(shadowColorDown); }
            set { shadowColorDown = Gui.Design.SerializableColor.FromString(value); }
        }

         /// <summary>
        /// </summary>
        [XmlIgnore]
        [Description("Color of Doji Shadow Line.")]
        [Category("Visual")]
        [Gui.Design.DisplayNameAttribute("Shadow Color Doji")]
        public Color ShadowColorDoji
        {
            get { return shadowColorDoji; }
            set { shadowColorDoji = value; }
        }

        /// <summary>
        /// </summary>
        [Browsable(false)]
        public string ShadowColorDojiSerialize
        {
            get { return Gui.Design.SerializableColor.ToString(shadowColorDoji); }
            set { shadowColorDoji = Gui.Design.SerializableColor.FromString(value); }
        }

       /// <summary>
        /// </summary>
        [Description("Width of shadow line.")]
        [Category("Visual")]
        [Gui.Design.DisplayNameAttribute("Shadow width")]
        public int ShadowWidth
        {
            get { return shadowWidth; }
            set { shadowWidth = Math.Max(value, 1); }
        }

        #endregion

        #region Miscellaneous

        protected override void OnStartUp()
        {
            if (ChartControl == null || Bars == null)
                return;
            savePenColor                        = Bars.BarsData.ChartStyle.Pen.Color;
            savePen2Color                       = Bars.BarsData.ChartStyle.Pen2.Color;
            saveUpColor                         = Bars.BarsData.ChartStyle.UpColor;
            saveDownColor                       = Bars.BarsData.ChartStyle.DownColor;
           	Bars.BarsData.ChartStyle.Pen.Color  = Color.Empty;
            Bars.BarsData.ChartStyle.Pen2.Color = Color.Empty;
            Bars.BarsData.ChartStyle.DownColor  = Color.Empty;
            Bars.BarsData.ChartStyle.UpColor    = Color.Empty;
            brushUp                             = new SolidBrush(barColorUp);
            brushDown                           = new SolidBrush(barColorDown);
           	shadowPenUp                         = new Pen(shadowColorUp, shadowWidth);
            shadowPenDown                       = new Pen(shadowColorDown, shadowWidth);
            shadowPenDoji                       = new Pen(shadowColorDoji, shadowWidth);
       }

        protected override void OnTermination()
        {
            if (ChartControl != null && Bars != null)
            {
                Bars.BarsData.ChartStyle.Pen.Color = savePenColor;
                Bars.BarsData.ChartStyle.Pen2.Color = savePen2Color;
                Bars.BarsData.ChartStyle.UpColor = saveUpColor;
                Bars.BarsData.ChartStyle.DownColor = saveDownColor;
            }

            if (brushUp != null) brushUp.Dispose();
            if (brushDown != null) brushDown.Dispose();
           	if (shadowPenUp != null) shadowPenUp.Dispose();
            if (shadowPenDown != null) shadowPenDown.Dispose();
            if (shadowPenDoji != null) shadowPenDoji.Dispose();
       }

        public override void GetMinMaxValues(ChartControl chartControl, ref double min, ref double max)
        {
            if (Bars == null || ChartControl == null)
                return;

            for (int idx = FirstBarIndexPainted; idx <= LastBarIndexPainted; idx++)
            {
                double tmpHigh = HAHigh.Get(idx);
                double tmpLow = HALow.Get(idx);

                if (tmpHigh != 0 && tmpHigh > max)
                    max = tmpHigh;
                if (tmpLow != 0 && tmpLow < min)
                    min = tmpLow;
            }
        }

        public override void Plot(Graphics graphics, Rectangle bounds, double min, double max)
        {
            if (Bars == null || ChartControl == null)
                return;

            int barPaintWidth = Math.Max(3, 1 + 2 * ((int)Bars.BarsData.ChartStyle.BarWidth - 1) + 2 * shadowWidth);

            for (int idx = FirstBarIndexPainted; idx <= LastBarIndexPainted; idx++)
            {
                if (idx - Displacement < 0 || idx - Displacement >= BarsArray[0].Count || (!ChartControl.ShowBarsRequired && idx - Displacement < BarsRequired))
                    continue;
                double valH = HAHigh.Get(idx);
                double valL = HALow.Get(idx);
                double valC = HAClose.Get(idx);
                double valO = HAOpen.Get(idx);
                int x = ChartControl.GetXByBarIdx(BarsArray[0], idx);
                int y1 = ChartControl.GetYByValue(this, valO);
                int y2 = ChartControl.GetYByValue(this, valH);
                int y3 = ChartControl.GetYByValue(this, valL);
                int y4 = ChartControl.GetYByValue(this, valC);

                
                if (y4 == y1)
				{
               		graphics.DrawLine(shadowPenDoji, x, y2, x, y3);
                    graphics.DrawLine(shadowPenDoji, x - barPaintWidth / 2, y1, x + barPaintWidth / 2, y1);
				}
                else
                {
                    if (y4 > y1)
					{
                		graphics.DrawLine(shadowPenDown, x, y2, x, y3);
						graphics.FillRectangle(brushDown, x - barPaintWidth / 2, y1, barPaintWidth, y4 - y1);
                   		graphics.DrawRectangle(shadowPenDown, (x - barPaintWidth / 2) + (shadowPenUp.Width / 2), Math.Min(y4, y1), barPaintWidth - shadowPenUp.Width, Math.Abs(y4 - y1));
                    }
					else
					{
                		graphics.DrawLine(shadowPenUp, x, y2, x, y3);
                        graphics.FillRectangle(brushUp, x - barPaintWidth / 2, y4, barPaintWidth, y1 - y4);
                   		graphics.DrawRectangle(shadowPenUp, (x - barPaintWidth / 2) + (shadowPenDown.Width / 2), Math.Min(y4, y1), barPaintWidth - shadowPenDown.Width, Math.Abs(y4 - y1));
					}
				}
            }
        }
        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private HeikenAshi2[] cacheHeikenAshi2 = null;

        private static HeikenAshi2 checkHeikenAshi2 = new HeikenAshi2();

        /// <summary>
        /// HeikenAshi2 technique discussed in the article 'Using Heiken-Ashi Technique' in February 2004 issue of TASC magazine.
        /// </summary>
        /// <returns></returns>
        public HeikenAshi2 HeikenAshi2()
        {
            return HeikenAshi2(Input);
        }

        /// <summary>
        /// HeikenAshi2 technique discussed in the article 'Using Heiken-Ashi Technique' in February 2004 issue of TASC magazine.
        /// </summary>
        /// <returns></returns>
        public HeikenAshi2 HeikenAshi2(Data.IDataSeries input)
        {
            if (cacheHeikenAshi2 != null)
                for (int idx = 0; idx < cacheHeikenAshi2.Length; idx++)
                    if (cacheHeikenAshi2[idx].EqualsInput(input))
                        return cacheHeikenAshi2[idx];

            lock (checkHeikenAshi2)
            {
                if (cacheHeikenAshi2 != null)
                    for (int idx = 0; idx < cacheHeikenAshi2.Length; idx++)
                        if (cacheHeikenAshi2[idx].EqualsInput(input))
                            return cacheHeikenAshi2[idx];

                HeikenAshi2 indicator = new HeikenAshi2();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                Indicators.Add(indicator);
                indicator.SetUp();

                HeikenAshi2[] tmp = new HeikenAshi2[cacheHeikenAshi2 == null ? 1 : cacheHeikenAshi2.Length + 1];
                if (cacheHeikenAshi2 != null)
                    cacheHeikenAshi2.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheHeikenAshi2 = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// HeikenAshi2 technique discussed in the article 'Using Heiken-Ashi Technique' in February 2004 issue of TASC magazine.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.HeikenAshi2 HeikenAshi2()
        {
            return _indicator.HeikenAshi2(Input);
        }

        /// <summary>
        /// HeikenAshi2 technique discussed in the article 'Using Heiken-Ashi Technique' in February 2004 issue of TASC magazine.
        /// </summary>
        /// <returns></returns>
        public Indicator.HeikenAshi2 HeikenAshi2(Data.IDataSeries input)
        {
            return _indicator.HeikenAshi2(input);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// HeikenAshi2 technique discussed in the article 'Using Heiken-Ashi Technique' in February 2004 issue of TASC magazine.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.HeikenAshi2 HeikenAshi2()
        {
            return _indicator.HeikenAshi2(Input);
        }

        /// <summary>
        /// HeikenAshi2 technique discussed in the article 'Using Heiken-Ashi Technique' in February 2004 issue of TASC magazine.
        /// </summary>
        /// <returns></returns>
        public Indicator.HeikenAshi2 HeikenAshi2(Data.IDataSeries input)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.HeikenAshi2(input);
        }
    }
}
#endregion
