#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// This is a NinjaTrader version of Dr. Brett Steenbarger's adjusted cumulative TICK.  It should be used on a one minute TICK data series.
    /// </summary>
    [Description("This is a NinjaTrader version of Dr. Brett Steenbarger's adjusted cumulative TICK.  It should be used on a one minute TICK data series.")]
    public class AdjustedCumulativeTICK : Indicator
    {
        #region Variables
        // Wizard generated variables
            private int historicalAdjustmentPeriod = 20; // Default setting for HistoricalAdjustmentPeriod
        // User defined variables (add any user defined variables below)
		private int _adjustmentBarCount;
		private double _totalTICK = 0.0f;
		private double _dailyTICK = 0.0f;
		private DateTime _lastTradeDate = DateTime.MinValue;
		private System.Collections.Generic.Queue<double> _tickValues = null;
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
            Add(new Plot(Color.FromKnownColor(KnownColor.Black), PlotStyle.Line, "CumulativeTICK"));
            Add(new Line(Color.FromKnownColor(KnownColor.Black), 0, "Zero"));
            Overlay				= false;
			
			this._adjustmentBarCount = this.historicalAdjustmentPeriod * 390;
			this._tickValues = new System.Collections.Generic.Queue<double>(this.historicalAdjustmentPeriod);
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
            // Use this method for calculating your indicator values. Assign a value to each
            // plot below by replacing 'Close[0]' with your own formula.
            DateTime tradeDate = Time[0].Date;
			bool plot = true;
			if(this._lastTradeDate != tradeDate)
			{
				this._dailyTICK = 0.0f;
				plot = false;
			}
			
			this._lastTradeDate = tradeDate;
			
			double avgTICK = ((High[0]+Low[0]+Close[0]) / 3.0f);
						
			if(CurrentBar > this._adjustmentBarCount)
			{
				double oldTICK = this._tickValues.Dequeue();
				double cumulativeAverage = this._totalTICK / (double)this._adjustmentBarCount;
				this._totalTICK -= oldTICK;
								
				avgTICK -= cumulativeAverage;
				this._dailyTICK += avgTICK;
				
				if(plot)
				{
					CumulativeTICK.Set(this._dailyTICK);
				}
			}
			
			if(this._tickValues.Count > this._adjustmentBarCount)
			{
				Print(String.Format("An error occurred... we are not using the right average count: {0}", this._tickValues.Count));
			}
			
			this._totalTICK += avgTICK;
			this._tickValues.Enqueue(avgTICK);
        }

        #region Properties
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries CumulativeTICK
        {
            get { return Values[0]; }
        }

        [Description("Number of days to adjust each average TICK by (Dr. Steenbarger used 20).")]
        [GridCategory("Parameters")]
        public int HistoricalAdjustmentPeriod
        {
            get { return historicalAdjustmentPeriod; }
            set { historicalAdjustmentPeriod = Math.Max(1, value); }
        }
        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private AdjustedCumulativeTICK[] cacheAdjustedCumulativeTICK = null;

        private static AdjustedCumulativeTICK checkAdjustedCumulativeTICK = new AdjustedCumulativeTICK();

        /// <summary>
        /// This is a NinjaTrader version of Dr. Brett Steenbarger's adjusted cumulative TICK.  It should be used on a one minute TICK data series.
        /// </summary>
        /// <returns></returns>
        public AdjustedCumulativeTICK AdjustedCumulativeTICK(int historicalAdjustmentPeriod)
        {
            return AdjustedCumulativeTICK(Input, historicalAdjustmentPeriod);
        }

        /// <summary>
        /// This is a NinjaTrader version of Dr. Brett Steenbarger's adjusted cumulative TICK.  It should be used on a one minute TICK data series.
        /// </summary>
        /// <returns></returns>
        public AdjustedCumulativeTICK AdjustedCumulativeTICK(Data.IDataSeries input, int historicalAdjustmentPeriod)
        {
            if (cacheAdjustedCumulativeTICK != null)
                for (int idx = 0; idx < cacheAdjustedCumulativeTICK.Length; idx++)
                    if (cacheAdjustedCumulativeTICK[idx].HistoricalAdjustmentPeriod == historicalAdjustmentPeriod && cacheAdjustedCumulativeTICK[idx].EqualsInput(input))
                        return cacheAdjustedCumulativeTICK[idx];

            lock (checkAdjustedCumulativeTICK)
            {
                checkAdjustedCumulativeTICK.HistoricalAdjustmentPeriod = historicalAdjustmentPeriod;
                historicalAdjustmentPeriod = checkAdjustedCumulativeTICK.HistoricalAdjustmentPeriod;

                if (cacheAdjustedCumulativeTICK != null)
                    for (int idx = 0; idx < cacheAdjustedCumulativeTICK.Length; idx++)
                        if (cacheAdjustedCumulativeTICK[idx].HistoricalAdjustmentPeriod == historicalAdjustmentPeriod && cacheAdjustedCumulativeTICK[idx].EqualsInput(input))
                            return cacheAdjustedCumulativeTICK[idx];

                AdjustedCumulativeTICK indicator = new AdjustedCumulativeTICK();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.HistoricalAdjustmentPeriod = historicalAdjustmentPeriod;
                Indicators.Add(indicator);
                indicator.SetUp();

                AdjustedCumulativeTICK[] tmp = new AdjustedCumulativeTICK[cacheAdjustedCumulativeTICK == null ? 1 : cacheAdjustedCumulativeTICK.Length + 1];
                if (cacheAdjustedCumulativeTICK != null)
                    cacheAdjustedCumulativeTICK.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheAdjustedCumulativeTICK = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// This is a NinjaTrader version of Dr. Brett Steenbarger's adjusted cumulative TICK.  It should be used on a one minute TICK data series.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.AdjustedCumulativeTICK AdjustedCumulativeTICK(int historicalAdjustmentPeriod)
        {
            return _indicator.AdjustedCumulativeTICK(Input, historicalAdjustmentPeriod);
        }

        /// <summary>
        /// This is a NinjaTrader version of Dr. Brett Steenbarger's adjusted cumulative TICK.  It should be used on a one minute TICK data series.
        /// </summary>
        /// <returns></returns>
        public Indicator.AdjustedCumulativeTICK AdjustedCumulativeTICK(Data.IDataSeries input, int historicalAdjustmentPeriod)
        {
            return _indicator.AdjustedCumulativeTICK(input, historicalAdjustmentPeriod);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// This is a NinjaTrader version of Dr. Brett Steenbarger's adjusted cumulative TICK.  It should be used on a one minute TICK data series.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.AdjustedCumulativeTICK AdjustedCumulativeTICK(int historicalAdjustmentPeriod)
        {
            return _indicator.AdjustedCumulativeTICK(Input, historicalAdjustmentPeriod);
        }

        /// <summary>
        /// This is a NinjaTrader version of Dr. Brett Steenbarger's adjusted cumulative TICK.  It should be used on a one minute TICK data series.
        /// </summary>
        /// <returns></returns>
        public Indicator.AdjustedCumulativeTICK AdjustedCumulativeTICK(Data.IDataSeries input, int historicalAdjustmentPeriod)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.AdjustedCumulativeTICK(input, historicalAdjustmentPeriod);
        }
    }
}
#endregion
