// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// Bollinger_TripleState_v4NT7 Bands are plotted at standard deviation levels above and below a moving average. Since standard deviation is a measure of volatility, the bands are self-adjusting: widening during volatile markets and contracting during calmer periods.
	/// </summary>
	[Description("Bollinger_TripleState_v4NT7 Bands are plotted at standard deviation levels above and below a moving average. Since standard deviation is a measure of volatility, the bands are self-adjusting: widening during volatile markets and contracting during calmer periods.")]
	public class BollingerTripleState_v4NT7 : Indicator
	{
		#region Variables
		private	double		numStdDev	= 2;
		private int			period		= 14;
		private double		wide_Tick_Threashold = 8;
		private double		narrow_Tick_Threashold = 4;
		private Color		wideColor = Color.Red;
		private Color		narrowColor = Color.Yellow;
		private Color 		neutralColor = Color.Navy;
		private Color		sMALineColor = Color.Black;
		private bool 		myLines = true;
		private bool		shading = true;
		
		private DataSeries  value1;
		private DataSeries  value2;
		
		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Plot(Color.Navy, PlotStyle.Line, "Upper"));
			Add(new Plot(Color.Navy,PlotStyle.Line, "Lower"));
			Add(new Plot(Color.Black,PlotStyle.Line, "Middle"));
			
			Plots[0].Pen.Width = 3;
			Plots[1].Pen.Width = 3;
			Plots[2].Pen.Width = 1;
			
			value1	= new DataSeries(this);
			value2	= new DataSeries(this);
			
			Overlay				= true;
			PriceTypeSupported	= true;
			
		}

		/// <summary>
		/// Called on each bar update event (incoming tick)
		/// </summary>
		protected override void OnBarUpdate()
		{
			if(CurrentBar < Period) return;
			
			double line_upper = SMA(Period)[0] + NumStdDev * StdDev(Period)[0];//get upper line value
			double line_lower = SMA(Period)[0] - NumStdDev * StdDev(Period)[0];//get lower line value
			
			Upper.Set(line_upper);//set the ds values to lines above
			Lower.Set(line_lower);
		
			double diff = Upper[0] - Lower[0];//calc diff btw line
		
			double wtt = wide_Tick_Threashold * TickSize;
			double ntt = narrow_Tick_Threashold * TickSize;
			
			if(!myLines)
			{
				WideColor = Color.Transparent;
				NarrowColor = Color.Transparent;
				NeutralColor = Color.Transparent;
				SMALineColor = Color.Transparent;
			}
			
			if( diff >= wtt)
			{
					PlotColors[0][0] = WideColor;
					PlotColors[1][0] = WideColor;
				
				if(shading)
				{
					DrawRegion("reg" + CurrentBar, 1, 0, Upper , Lower, Color.Empty, Color.Red, 1);
				}
			}
			else if(diff < wtt && diff > ntt)
			{
					PlotColors[0][0] = NeutralColor;
					PlotColors[1][0] = NeutralColor;	
				
				if(shading)
				{	
					DrawRegion("reg" + CurrentBar, 1, 0, Upper , Lower, Color.Empty, Color.Navy, 1);
				}
			}
			else if(diff <= ntt)
			{
					PlotColors[0][0] = NarrowColor;
					PlotColors[1][0] = NarrowColor;
				
				if(shading)
				{	
					DrawRegion("reg" + CurrentBar, 1, 0, Upper , Lower, Color.Empty, Color.Yellow, 1);
				}
			}
			
			Middle.Set(SMA(Period)[0]);
			PlotColors[2][0] = SMALineColor;
	
			DrawTextFixed("Threashold Determiner", "Actual Upper/Lower Band Spread: " + Math.Round(diff,4) + "\n\nUser Defined Threasholds:\n  Narrow: " + ntt + "\n  Wide: " + wtt , TextPosition.TopLeft);
		}

		#region Properties
				[Description("Do you want shading")]
		[Category("Parameters")]
		public bool Shading
		{
			get { return shading; }
			set { shading= value; }
		}
		
		[Description("Do you want to draw the lines")]
		[Category("Parameters")]
		public bool MyLines
		{
			get { return myLines; }
			set { myLines= value; }
		}
				[Description("threashold to draw narrow color")]
		[Category("Parameters")]
		public double Wide_Threashold
		{
			get { return wide_Tick_Threashold; }
			set { wide_Tick_Threashold = Math.Max(0.001, value); }
		}
		
				[Description("threashold to draw narrow color")]
		[Category("Parameters")]
		public double Narrow_Tick_Threashold
		{
			get { return narrow_Tick_Threashold; }
			set { narrow_Tick_Threashold = Math.Max(0.001, value); }
		}
		[XmlIgnore()]
        [Description("Wide Color")]
        [Category("Colors")]
        public Color WideColor
        {
            get { return wideColor; }
            set { wideColor = value; }
        }
				[Browsable(false)]
		public string WideColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(WideColor); }
			set { WideColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
				[XmlIgnore()]
        [Description("Narrow Color")]
        [Category("Colors")]
        public Color NarrowColor
        {
            get { return narrowColor; }
            set { narrowColor = value; }
        }
						[Browsable(false)]
		public string NarrowColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(NarrowColor); }
			set { NarrowColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
				[XmlIgnore()]
        [Description("Neutral Color")]
        [Category("Colors")]
        public Color NeutralColor
        {
            get { return neutralColor; }
            set { neutralColor = value; }
        }
						[Browsable(false)]
		public string NeutralColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(NeutralColor); }
			set { NeutralColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
						[XmlIgnore()]
        [Description("SMA Line Color")]
        [Category("Colors")]
        public Color SMALineColor
        {
            get { return sMALineColor; }
            set { sMALineColor = value; }
        }
						[Browsable(false)]
		public string SMALineColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(SMALineColor); }
			set { SMALineColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		/// <summary>
		/// Gets the lower value.
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Lower
		{
			get { return Values[0]; }
		}
		
		/// <summary>
		/// Get the upper value.
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Upper
		{
			get { return Values[1]; }
		}
		/// <summary>
		/// Get the middle value.
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Middle
		{
			get { return Values[2]; }
		}

		/// <summary>
		/// </summary>
		[Description("Number of standard deviations")]
		[Category("Parameters")]
		[Gui.Design.DisplayNameAttribute("# of std. dev.")]
		public double NumStdDev
		{
			get { return numStdDev; }
			set { numStdDev = Math.Max(0, value); }
		}

		/// <summary>
		/// </summary>
		[Description("Numbers of bars used for calculations")]
		[Category("Parameters")]
		public int Period
		{
			get { return period; }
			set { period = Math.Max(1, value); }
		}
		//added ben


		//added by ben V
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries UpperWide
		{
			get { return Values[1]; }
		}
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries UpperNarrow
		{
			get { return Values[2]; }
		}
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries LowerWide
		{
			get { return Values[5]; }
		}
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries LowerNarrow
		{
			get { return Values[6]; }
		}
		
		
		
		
		
		
		
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private BollingerTripleState_v4NT7[] cacheBollingerTripleState_v4NT7 = null;

        private static BollingerTripleState_v4NT7 checkBollingerTripleState_v4NT7 = new BollingerTripleState_v4NT7();

        /// <summary>
        /// Bollinger_TripleState_v4NT7 Bands are plotted at standard deviation levels above and below a moving average. Since standard deviation is a measure of volatility, the bands are self-adjusting: widening during volatile markets and contracting during calmer periods.
        /// </summary>
        /// <returns></returns>
        public BollingerTripleState_v4NT7 BollingerTripleState_v4NT7(bool myLines, double narrow_Tick_Threashold, double numStdDev, int period, bool shading, double wide_Threashold)
        {
            return BollingerTripleState_v4NT7(Input, myLines, narrow_Tick_Threashold, numStdDev, period, shading, wide_Threashold);
        }

        /// <summary>
        /// Bollinger_TripleState_v4NT7 Bands are plotted at standard deviation levels above and below a moving average. Since standard deviation is a measure of volatility, the bands are self-adjusting: widening during volatile markets and contracting during calmer periods.
        /// </summary>
        /// <returns></returns>
        public BollingerTripleState_v4NT7 BollingerTripleState_v4NT7(Data.IDataSeries input, bool myLines, double narrow_Tick_Threashold, double numStdDev, int period, bool shading, double wide_Threashold)
        {
            if (cacheBollingerTripleState_v4NT7 != null)
                for (int idx = 0; idx < cacheBollingerTripleState_v4NT7.Length; idx++)
                    if (cacheBollingerTripleState_v4NT7[idx].MyLines == myLines && Math.Abs(cacheBollingerTripleState_v4NT7[idx].Narrow_Tick_Threashold - narrow_Tick_Threashold) <= double.Epsilon && Math.Abs(cacheBollingerTripleState_v4NT7[idx].NumStdDev - numStdDev) <= double.Epsilon && cacheBollingerTripleState_v4NT7[idx].Period == period && cacheBollingerTripleState_v4NT7[idx].Shading == shading && Math.Abs(cacheBollingerTripleState_v4NT7[idx].Wide_Threashold - wide_Threashold) <= double.Epsilon && cacheBollingerTripleState_v4NT7[idx].EqualsInput(input))
                        return cacheBollingerTripleState_v4NT7[idx];

            lock (checkBollingerTripleState_v4NT7)
            {
                checkBollingerTripleState_v4NT7.MyLines = myLines;
                myLines = checkBollingerTripleState_v4NT7.MyLines;
                checkBollingerTripleState_v4NT7.Narrow_Tick_Threashold = narrow_Tick_Threashold;
                narrow_Tick_Threashold = checkBollingerTripleState_v4NT7.Narrow_Tick_Threashold;
                checkBollingerTripleState_v4NT7.NumStdDev = numStdDev;
                numStdDev = checkBollingerTripleState_v4NT7.NumStdDev;
                checkBollingerTripleState_v4NT7.Period = period;
                period = checkBollingerTripleState_v4NT7.Period;
                checkBollingerTripleState_v4NT7.Shading = shading;
                shading = checkBollingerTripleState_v4NT7.Shading;
                checkBollingerTripleState_v4NT7.Wide_Threashold = wide_Threashold;
                wide_Threashold = checkBollingerTripleState_v4NT7.Wide_Threashold;

                if (cacheBollingerTripleState_v4NT7 != null)
                    for (int idx = 0; idx < cacheBollingerTripleState_v4NT7.Length; idx++)
                        if (cacheBollingerTripleState_v4NT7[idx].MyLines == myLines && Math.Abs(cacheBollingerTripleState_v4NT7[idx].Narrow_Tick_Threashold - narrow_Tick_Threashold) <= double.Epsilon && Math.Abs(cacheBollingerTripleState_v4NT7[idx].NumStdDev - numStdDev) <= double.Epsilon && cacheBollingerTripleState_v4NT7[idx].Period == period && cacheBollingerTripleState_v4NT7[idx].Shading == shading && Math.Abs(cacheBollingerTripleState_v4NT7[idx].Wide_Threashold - wide_Threashold) <= double.Epsilon && cacheBollingerTripleState_v4NT7[idx].EqualsInput(input))
                            return cacheBollingerTripleState_v4NT7[idx];

                BollingerTripleState_v4NT7 indicator = new BollingerTripleState_v4NT7();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.MyLines = myLines;
                indicator.Narrow_Tick_Threashold = narrow_Tick_Threashold;
                indicator.NumStdDev = numStdDev;
                indicator.Period = period;
                indicator.Shading = shading;
                indicator.Wide_Threashold = wide_Threashold;
                Indicators.Add(indicator);
                indicator.SetUp();

                BollingerTripleState_v4NT7[] tmp = new BollingerTripleState_v4NT7[cacheBollingerTripleState_v4NT7 == null ? 1 : cacheBollingerTripleState_v4NT7.Length + 1];
                if (cacheBollingerTripleState_v4NT7 != null)
                    cacheBollingerTripleState_v4NT7.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheBollingerTripleState_v4NT7 = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Bollinger_TripleState_v4NT7 Bands are plotted at standard deviation levels above and below a moving average. Since standard deviation is a measure of volatility, the bands are self-adjusting: widening during volatile markets and contracting during calmer periods.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.BollingerTripleState_v4NT7 BollingerTripleState_v4NT7(bool myLines, double narrow_Tick_Threashold, double numStdDev, int period, bool shading, double wide_Threashold)
        {
            return _indicator.BollingerTripleState_v4NT7(Input, myLines, narrow_Tick_Threashold, numStdDev, period, shading, wide_Threashold);
        }

        /// <summary>
        /// Bollinger_TripleState_v4NT7 Bands are plotted at standard deviation levels above and below a moving average. Since standard deviation is a measure of volatility, the bands are self-adjusting: widening during volatile markets and contracting during calmer periods.
        /// </summary>
        /// <returns></returns>
        public Indicator.BollingerTripleState_v4NT7 BollingerTripleState_v4NT7(Data.IDataSeries input, bool myLines, double narrow_Tick_Threashold, double numStdDev, int period, bool shading, double wide_Threashold)
        {
            return _indicator.BollingerTripleState_v4NT7(input, myLines, narrow_Tick_Threashold, numStdDev, period, shading, wide_Threashold);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Bollinger_TripleState_v4NT7 Bands are plotted at standard deviation levels above and below a moving average. Since standard deviation is a measure of volatility, the bands are self-adjusting: widening during volatile markets and contracting during calmer periods.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.BollingerTripleState_v4NT7 BollingerTripleState_v4NT7(bool myLines, double narrow_Tick_Threashold, double numStdDev, int period, bool shading, double wide_Threashold)
        {
            return _indicator.BollingerTripleState_v4NT7(Input, myLines, narrow_Tick_Threashold, numStdDev, period, shading, wide_Threashold);
        }

        /// <summary>
        /// Bollinger_TripleState_v4NT7 Bands are plotted at standard deviation levels above and below a moving average. Since standard deviation is a measure of volatility, the bands are self-adjusting: widening during volatile markets and contracting during calmer periods.
        /// </summary>
        /// <returns></returns>
        public Indicator.BollingerTripleState_v4NT7 BollingerTripleState_v4NT7(Data.IDataSeries input, bool myLines, double narrow_Tick_Threashold, double numStdDev, int period, bool shading, double wide_Threashold)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.BollingerTripleState_v4NT7(input, myLines, narrow_Tick_Threashold, numStdDev, period, shading, wide_Threashold);
        }
    }
}
#endregion
