#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    [Description("Customization of Ichimoku Cloud system.")]
//	[Gui.Design.DisplayName("a Ichimoku Ver 1.1")]	 // name as shown in available indicators list.
    public class AaIchimokuNT7 : Indicator
	/*
		Version 1.0 - March 3/3/2010 Implemented version control of program changes.
		Version 1.1 - March 3/4/2010 Added ADX backlighting and changed the Strength display to only show abreviations ["T K X = Tenkan Sen / Kijun Sen Cross"].
	*/
    {
        #region Variables
            private int periodFast 			= 9;	// Default setting for PeriodFast
            private int periodMedium 		= 26;	// Default setting for PeriodMedium
            private int periodSlow 			= 52;	// Default setting for PeriodSlow
			private int kumoOpacity 		= 2;	// Opacity of the kumo.
		
			private int tenkanDirection0 	= 0;	// Variables to determine when a crossing has occured between the old (direction__1) and the current (direction__0).
			private int tenkanDirection1 	= 0;
			private int kijunDirection0		= 0;
			private int kijunDirection1		= 0;
			private int kumoDirection0		= 0;
			private int kumoDirection1		= 0;
			private int senkouDirection0	= 0;
			private int senkouDirection1	= 0;
			private int chikouDirection0	= 0;
			private int chikouDirection1	= 0;
			private int ichimokuDirection0	= 0;
			private int ichimokuDirection1	= 0;

			private int tenkanBarCount		= 0;	// Counters that indicate how long the bars will display before 'graying' out.
			private int kijunBarCount		= 0;
			private int kumoBarCount		= 0;
			private int senkouBarCount		= 0;
			private int chikouBarCount		= 0;
			private int ichimokuBarCount	= 0;

			private int tenkanSignal		= 0;	// Relative signal strength based on Bullish (+) or Bearish (-) indications. 
			private int kijunSignal			= 0;	
			private int kumoSignal			= 0;	
			private int senkouSignal		= 0;	
			private int chikouSignal		= 0;	
			private int ichimokuSignal		= 0;	
		
			private double mediumTerm	= 0;
			private double shortTerm	= 0;
		
			private bool displayTenkan		= true;		// Display the respective crossing on the panel.
			private bool displayKijun		= false;	
			private bool displayKumo		= false;	
			private bool displaySenkou		= false;	
			private bool displayChikou		= false;	
			private bool displayIchimoku	= false;	
			private bool displayStrength	= true;		// Display the stength indicators in the upper left corner.
			private bool displayADXcolor	= true;		// Display a backlight for no trading as defined byt the ADX.
		
			private bool alertTenkan	= false;	// Turn alert on and off.
			private bool alertKijun		= false;	// Turn alert on and off.
			private bool alertKumo		= false;	// Turn alert on and off.
			private bool alertSenkou	= false;	// Turn alert on and off.
			private bool alertChikou	= false;	// Turn alert on and off.
			private bool alertIchimoku	= false;	// Turn alert on and off.
			private bool modifySignal	= false;	// Allows modification to signal strength immediately following a Tenkan crossing.
		
			private bool spanAonTop		= true;		// True if SenkouSpanA is greater than SenkouSpanB.
		
			private Color kumoColorUp 	= Color.Blue;
			private Color kumoColorDown	= Color.Red;
		
			private SolidBrush 		tenkanUpColor		= new SolidBrush(Color.PaleGreen); 
			private SolidBrush 		tenkanDownColor		= new SolidBrush(Color.Pink); 
			private SolidBrush 		kijunUpColor		= new SolidBrush(Color.PaleGreen); 
			private SolidBrush 		kijunDownColor		= new SolidBrush(Color.Pink); 
			private SolidBrush 		kumoUpColor			= new SolidBrush(Color.PaleGreen); 
			private SolidBrush 		kumoDownColor		= new SolidBrush(Color.Pink); 
			private SolidBrush 		senkouUpColor		= new SolidBrush(Color.PaleGreen); 
			private SolidBrush 		senkouDownColor		= new SolidBrush(Color.Pink); 
			private SolidBrush 		chikouUpColor		= new SolidBrush(Color.PaleGreen); 
			private SolidBrush 		chikouDownColor		= new SolidBrush(Color.Pink); 
			private SolidBrush 		ichimokuUpColor		= new SolidBrush(Color.PaleGreen); 
			private SolidBrush 		ichimokuDownColor	= new SolidBrush(Color.Pink); 
		
			// Indicator variables
			int x1 = 20;
			int xGap = 4;
			int yGap = 4;
			int yGapHeading = 8;
			int y1 = 20;
			int width = 17;
			int height = 6;
			
			string str = "";
			string textStr = "";
		
			// ADX Backlighting coloring system
            private int adxPeriods 	= 14; 	// Number of periods for calculating the ADX.
			private int adxCutOff	= 20;	// Number below which the no zone coloring scheme works.


		
        #endregion

        protected override void Initialize()
        {
            #region Setup
				Add(new Plot( new Pen(Color.Red, 2),	    PlotStyle.Line, "TenkanSen"));	// Short term momentum  (short term average of highest-high & lowest-low)
				Add(new Plot( new Pen(Color.Blue, 2),	    PlotStyle.Line, "KijunSen"));	// Medium term momentum (medium term average of highest-high & lowest-low)
				Add(new Plot( new Pen(Color.DarkViolet, 7), PlotStyle.Line, "ChikouSpan"));	// Lagging price (for visual comparison with current price)
				
				Add(new Plot( new Pen(Color.Black, 1),  "SenkouSpanA" ));		// Predicts levels of future support/resistance  (Values[3])
				Add(new Plot( new Pen(Color.Gray, 1),   "SenkouSpanB" ));		// Predicts levels of future support/resistance	 (Values[4])
				
				CalculateOnBarClose	= true;						// Updated once per bar (as opposed to on every tick.)
				Overlay				= true;						// Display on top of the price in panel # 1.
				PriceTypeSupported	= false;
				DisplayInDataBox	= false;
				PaintPriceMarkers	= false;
			#endregion
        }
        protected override void OnBarUpdate()
        {
			#region Initial
				// Wait until we have enough bars.
				if ((CurrentBar < PeriodMedium) || (CurrentBar < PeriodFast))
					return;
				if (displayADXcolor)
				{
					int x = (int)ADX(AdxPeriods)[0];
					if(x < AdxCutOff)
					{
						BackColor = Color.FromArgb(255 - x * 12,139,69,19);
					}
				}
			#endregion
				
			#region Set Calculations
				shortTerm = ( MAX(High,PeriodFast)[0]   + MIN(Low,PeriodFast)[0]   ) / 2;
				mediumTerm = ( MAX(High,PeriodMedium)[0] + MIN(Low,PeriodMedium)[0] ) / 2;
			
				KijunSen.Set(mediumTerm);	// Medium term momentum. Kijun means "trend" in Japanese.
				TenkanSen.Set(shortTerm);	// Short term momentum. The Tenkan line is a signal line that works in conjunction with the trend line (kijun).
				ChikouSpan.Set( PeriodMedium, Close[0] );		// The lagging line is a plot of the current price in the past, for easy visual comparison.
	
				// The leading lines, create a Cloud-like formation ("Kumo"). This is an area of support or resistance.
				if ((CurrentBar < PeriodFast+PeriodMedium) || (CurrentBar < PeriodMedium+PeriodMedium) || (CurrentBar < PeriodSlow+PeriodMedium)) return;
				// Average of the TenkanSen & Kijunsen, but shifted PeriodMedium bars forward in time.
				SenkouSpanA.Set( ( MAX(High,PeriodFast)[PeriodMedium] + MIN(Low,PeriodFast)[PeriodMedium]
								+ MAX(High,PeriodMedium)[PeriodMedium] + MIN(Low,PeriodMedium)[PeriodMedium] ) / 4 );  
				// The long-term (slow) average of the lowest-low & highest-high, shifted forward in time.
				SenkouSpanB.Set( ( MAX(High,PeriodSlow)[PeriodMedium] + MIN(Low,PeriodSlow)[PeriodMedium] ) / 2 );  
				/* In each of the following indicators, a strength value is calculated for 
				*/
			#endregion
				
			#region Tenkan Crossing // Signal when the tenkan sen crosses over the kijun sen.
				if (mediumTerm <= shortTerm)
					tenkanDirection0 = 1;
				else
					tenkanDirection0 = -1;
				if (tenkanDirection0 != tenkanDirection1)
				{
					modifySignal = true;  // Logic to allow signal modification because of crossing.
					tenkanDirection1=tenkanDirection0;
					if (tenkanDirection0 == 1)
					{
						tenkanSignal = 1;  // Reset the signal strength when the cross occurs.
						if (DisplayTenkan)
							BackColor = TenkanUpColor; // Long signal
						if (AlertTenkan)
							Alert("TenkanCrossUp",NinjaTrader.Cbi.Priority.High, "Tenken Sen/ Kijun Sen cross UP", "tenkanUp.wav", 10, Color.Black, Color.Yellow);
					}
					else
					{
						tenkanSignal = -1;  // Reset the signal strength when the cross occurs.
						if (DisplayTenkan)
							BackColor = TenkanDownColor; // Short signal
						if (AlertTenkan)
							Alert("TenkanCrossDown",NinjaTrader.Cbi.Priority.High, "Tenken Sen/ Kijun Sen cross DOWN", "tenkanDown.wav", 10, Color.Black, Color.Yellow);
					}
				}
				// Signal strength modifications
				if (modifySignal)
				{
					modifySignal = false;	
					// Strength of crossing in relation to the kumo.
					if (KijunSen[0] > SenkouSpanA[0] && KijunSen[0] > SenkouSpanB[0] &&	TenkanSen[0] > SenkouSpanA[0] && TenkanSen[0] > SenkouSpanB[0]) // Crossed above the cloud.
					{
						if (tenkanSignal == 1)
							tenkanSignal = tenkanSignal + 2; // Crossing occured above the cloud. Strong signal.
					}
					else if (KijunSen[0] < SenkouSpanA[0] && KijunSen[0] < SenkouSpanB[0] && TenkanSen[0] < SenkouSpanA[0] && TenkanSen[0] < SenkouSpanB[0]) // Crossed below the cloud.
					{
						if (tenkanSignal == -1)
							tenkanSignal = tenkanSignal - 2; // Crossing occured below the cloud.  Strong signal.
					}
					else // Crossed in the cloud.
					{
						if (tenkanSignal == 1)
							tenkanSignal = tenkanSignal + 1;
						else
							tenkanSignal = tenkanSignal - 1;
					}
					// Further modifications due to price action.
					if (Low[0] > SenkouSpanA[0] && Low[0] > SenkouSpanB[0] && tenkanSignal > 0) // Price action above the cloud.
						tenkanSignal = tenkanSignal + 1; 
					if (High[0] < SenkouSpanA[0] && High[0] < SenkouSpanB[0] && tenkanSignal < 0) // Price action below the cloud.
						tenkanSignal = tenkanSignal - 1; 
					
					// Further modifications due to Chikou location.
					if (ChikouSpan[PeriodMedium] > SenkouSpanA[PeriodMedium] && ChikouSpan[PeriodMedium] > SenkouSpanB[PeriodMedium] && tenkanSignal > 0) // Chikou above the cloud.
						tenkanSignal = tenkanSignal + 1; 
					if (ChikouSpan[PeriodMedium] < SenkouSpanA[PeriodMedium] && ChikouSpan[PeriodMedium] < SenkouSpanB[PeriodMedium] && tenkanSignal < 0) // Chikou below the cloud.
						tenkanSignal = tenkanSignal - 1; 
					tenkanBarCount = 0;
				}
				tenkanBarCount = tenkanBarCount + 1;
			#endregion
			
			#region Kijun Crossing // signal when price action crosses over the kijun sen.
				if (mediumTerm < Low[0])
					kijunDirection0 = 1;
				else if (mediumTerm > High[0])
					kijunDirection0 = -1;
				if (kijunDirection0 != kijunDirection1)
				{
					modifySignal = true;  // Logic to allow signal modification because of crossing.
					kijunDirection1 = kijunDirection0;
					if (kijunDirection0 == 1)
					{
						kijunSignal = 1;  // Reset the signal strength when the cross occurs.
						if (DisplayKijun)
							BackColor = KijunUpColor;  // Long signal
						if (AlertKijun)
							Alert("KijunCrossUp",NinjaTrader.Cbi.Priority.High, "Kijun Sen cross UP", "kijunUp.wav", 10, Color.Black, Color.Yellow);
					}
					else
					{
						kijunSignal = -1;  // Reset the signal strength when the cross occurs.
						if (DisplayKijun)
							BackColor = KijunDownColor; // Short signal
						if (AlertKijun)
							Alert("KijunCrossDown",NinjaTrader.Cbi.Priority.High, "Kijun Sen cross DOWN", "kijunDown.wav", 10, Color.Black, Color.Yellow);
					}
				}
				// Signal strength modifications
				if (modifySignal)
				{
					modifySignal = false;	
					// Strength of crossing in relation to the kumo.
					if (KijunSen[0] > SenkouSpanA[0] && KijunSen[0] > SenkouSpanB[0]) // Crossed above the cloud.
					{
						if (kijunSignal == 1)
							kijunSignal = kijunSignal + 2; // Crossing occured above the cloud. Strong signal.
					}
					else if (KijunSen[0] < SenkouSpanA[0] && KijunSen[0] < SenkouSpanB[0] ) // Crossed below the cloud.
					{
						if (kijunSignal == -1)
							kijunSignal = kijunSignal - 2; // Crossing occured below the cloud.  Strong signal.
					}
					else // Crossed in the cloud.
					{
						if (kijunSignal == 1)
							kijunSignal = kijunSignal + 1;
						else
							kijunSignal = kijunSignal - 1;
					}
					// Further modifications due to price action.
					if (Low[0] > SenkouSpanA[0] && Low[0] > SenkouSpanB[0] && kijunSignal > 0) // Price action above the cloud.
						kijunSignal = kijunSignal + 1; 
					if (High[0] < SenkouSpanA[1] && High[0] < SenkouSpanB[0] && kijunSignal < 0) // Price action below the cloud.
						kijunSignal = kijunSignal - 1; 
					
					// Further modifications due to Chikou location.
					if (ChikouSpan[PeriodMedium] > SenkouSpanA[PeriodMedium] && ChikouSpan[PeriodMedium] > SenkouSpanB[PeriodMedium] && kijunSignal > 0) // Chikou above the cloud.
						kijunSignal = kijunSignal + 1; 
					if (ChikouSpan[PeriodMedium] < SenkouSpanA[PeriodMedium] && ChikouSpan[PeriodMedium] < SenkouSpanB[PeriodMedium] && kijunSignal < 0) // Chikou below the cloud.
						kijunSignal = kijunSignal - 1; 
					kijunBarCount = 0;
				}
				kijunBarCount = kijunBarCount + 1;
			#endregion
			
			#region Kumo Breakout // Signal when price action completely exits the kumo.
				if (SenkouSpanA[0] > SenkouSpanB[0])
					spanAonTop = true;
				else
					spanAonTop = false;
				if (spanAonTop)
				{
					if (Low[0] > SenkouSpanA[0])
						kumoDirection0 = 1;
					if (High[0] < SenkouSpanB[0])
						kumoDirection0 = -1;
				}
				else
				{
					if (Low[0] > SenkouSpanB[0])
						kumoDirection0 = 1;
					if (High[0] < SenkouSpanA[0])
						kumoDirection0 = -1;
				}
				if (kumoDirection0 != kumoDirection1)
				{
					modifySignal = true;  // Logic to allow signal modification because of crossing.
					// Strength of crossing in relation to the kumo.
					kumoDirection1 = kumoDirection0;
					if (kumoDirection0 == 1)
					{
						kumoSignal = 1;  // Reset the signal strength when the cross occurs.
						if (DisplayKumo)
							BackColor = KumoUpColor;  // Long signal
						if (AlertKumo)
							Alert("KumoCrossUp",NinjaTrader.Cbi.Priority.High, "Kumo cross UP", "kumoUp.wav", 10, Color.Black, Color.Yellow);
					}
					else
					{
						kumoSignal = -1;  // Reset the signal strength when the cross occurs.
						if (DisplayKumo)
							BackColor = KumoDownColor; // Short signal
						if (AlertKumo)
							Alert("KumoCrossDown",NinjaTrader.Cbi.Priority.High, "Kumo cross DOWN", "kumoDown.wav", 10, Color.Black, Color.Yellow);
					}
					// Signal strength modifications
					if (modifySignal)
					{
						modifySignal = false;	
/*						// Further modifications due to cloud color.
						if (spanAonTop && kumoSignal > 0) // Bullish kumo
							kumoSignal = kumoSignal + 1;
						if (spanAonTop && kumoSignal < 0) // Bearish kumo
							kumoSignal = kumoSignal - 1;
*/														
						// Further modifications due to Chikou location.
						if (ChikouSpan[PeriodMedium] > SenkouSpanA[PeriodMedium] && ChikouSpan[PeriodMedium] > SenkouSpanB[PeriodMedium] && kumoSignal > 0) // Chikou above the cloud.
							kumoSignal = kumoSignal + 1; 
						if (ChikouSpan[PeriodMedium] < SenkouSpanA[PeriodMedium] && ChikouSpan[PeriodMedium] < SenkouSpanB[PeriodMedium] && kumoSignal < 0) // Chikou below the cloud.
							kumoSignal = kumoSignal - 1; 
					}
					kumoBarCount = 0;
				}
				kumoBarCount = kumoBarCount + 1;
			#endregion
			
			#region Senkou Crossing // Signal when Senko Span A line crosses over the Senko Span B line of the kumo.
				if (spanAonTop)
					senkouDirection0 = 1;
				else
					senkouDirection0 = -1;
						
				if (senkouDirection0 != senkouDirection1)
				{
					modifySignal = true;  // Logic to allow signal modification because of crossing.
					senkouDirection1 = senkouDirection0;
					if (senkouDirection0 == 1)
					{
						senkouSignal = 1;  // Reset the signal strength when the cross occurs.
						if (DisplaySenkou)
							BackColor = SenkouUpColor;  // Long signal
						if (AlertSenkou)
							Alert("SenkouCrossUp",NinjaTrader.Cbi.Priority.High, "Senkou cross UP", "Up.wav", 10, Color.Black, Color.Yellow);
					}
					else
					{
						senkouSignal = -1;  // Reset the signal strength when the cross occurs.
						if (DisplaySenkou)
							BackColor = SenkouDownColor; // Short signal
						if (AlertSenkou)
							Alert("SenkouCrossDown",NinjaTrader.Cbi.Priority.High, "Senkou cross DOWN", "Down.wav", 10, Color.Black, Color.Yellow);
					}
					// Signal strength modifications
					if (modifySignal)
					{
						modifySignal = false;	
						// Strength of crossing in relation to the kumo.
						if (Low[0] > SenkouSpanA[0] && Low[0] > SenkouSpanB[0]) // Crossed above the cloud.
						{
							if (kijunSignal == 1)
								kijunSignal = kijunSignal + 2; // Crossing occured above the cloud. Strong signal.
						}
						else if (High[0] < SenkouSpanA[0] && High[0] < SenkouSpanB[0] ) // Crossed below the cloud.
						{
							if (kijunSignal == -1)
								kijunSignal = kijunSignal - 2; // Crossing occured below the cloud.  Strong signal.
						}
						else // Crossed in the cloud.
						{
							if (kijunSignal == 1)
								kijunSignal = kijunSignal + 1;
							else
								kijunSignal = kijunSignal - 1;
						}
						// Further modifications due to Chikou location.
						if (ChikouSpan[PeriodMedium] > SenkouSpanA[PeriodMedium] && ChikouSpan[PeriodMedium] > SenkouSpanB[PeriodMedium] && kumoSignal > 0) // Chikou above the cloud.
							kumoSignal = kumoSignal + 1; 
						if (ChikouSpan[PeriodMedium] < SenkouSpanA[PeriodMedium] && ChikouSpan[PeriodMedium] < SenkouSpanB[PeriodMedium] && kumoSignal < 0) // Chikou below the cloud.
							kumoSignal = kumoSignal - 1; 
					}
					senkouBarCount = 0;
				}
				senkouBarCount = senkouBarCount + 1;
			#endregion
			
			#region Chikou Crossing // Signal when the chikou span crosses over price action.
				if (ChikouSpan[PeriodMedium] > High[PeriodMedium])
					chikouDirection0 = 1;
				else if (ChikouSpan[PeriodMedium] < Low[PeriodMedium])
					chikouDirection0 = -1;

				if (chikouDirection0 != chikouDirection1)
				{
					modifySignal = true;  // Logic to allow signal modification because of crossing.
					chikouDirection1 = chikouDirection0;
					if (chikouDirection0 == 1)
					{
						chikouSignal = 1;  // Reset the signal strength when the cross occurs.
						if (DisplayChikou)
							BackColor = ChikouUpColor;  // Long signal
						if (AlertChikou)
							Alert("ChikouCrossUp",NinjaTrader.Cbi.Priority.High, "Chikou cross UP", "Up.wav", 10, Color.Black, Color.Yellow);
					}
					else
					{
						chikouSignal = -1;  // Reset the signal strength when the cross occurs.
						if (DisplayChikou)
							BackColor = ChikouDownColor; // Short signal
						if (AlertChikou)
							Alert("ChikouCrossDown",NinjaTrader.Cbi.Priority.High, "Chikou cross DOWN", "Down.wav", 10, Color.Black, Color.Yellow);
					}
				}
				// Signal strength modifications
				if (modifySignal)
				{
					modifySignal = false;	
					// Strength of crossing in relation to the kumo.
					if (ChikouSpan[PeriodMedium] > SenkouSpanA[PeriodMedium] && ChikouSpan[PeriodMedium] > SenkouSpanB[PeriodMedium]) // Crossed above the cloud.
					{
						if (chikouSignal == 1)
							chikouSignal = chikouSignal + 2; // Crossing occured above the cloud. Strong signal.
					}
					else if (ChikouSpan[PeriodMedium] < SenkouSpanA[PeriodMedium] && ChikouSpan[PeriodMedium] < SenkouSpanB[PeriodMedium]) // Crossed below the cloud.
					{
						if (chikouSignal == -1)
							chikouSignal = chikouSignal - 2; // Crossing occured below the cloud.  Strong signal.
					}
					else // Crossed in the cloud.
					{
						if (chikouSignal == 1)
							chikouSignal = chikouSignal + 1;
						else
							chikouSignal = chikouSignal - 1;
					}
					// Further modifications due to price action.
					if (Low[PeriodMedium] > SenkouSpanA[PeriodMedium] && Low[PeriodMedium] > SenkouSpanB[PeriodMedium]) // Price action above the cloud.
						chikouSignal = chikouSignal + 1;
					if (High[PeriodMedium] < SenkouSpanA[PeriodMedium] && High[PeriodMedium] < SenkouSpanB[PeriodMedium]) // Price action below the cloud.
						chikouSignal = chikouSignal - 1;
					chikouBarCount = 0;
				}
				chikouBarCount = chikouBarCount + 1;
			#endregion
			
			#region Ichimoku Signal // Signal when all indicators indicate a bullish or bearish market.
				if (tenkanDirection0 == 1 && kijunDirection0 == 1 && kumoDirection0 == 1 && senkouDirection0 == 1 && chikouDirection0 == 1)
					ichimokuDirection0 = 1;
				if (tenkanDirection0 == -1 && kijunDirection0 == -1 && kumoDirection0 == -1 && senkouDirection0 == -1 && chikouDirection0 == -1)
					ichimokuDirection0 = -1;
				
				if (ichimokuDirection0 != ichimokuDirection1)
				{
					modifySignal = true;  // Logic to allow signal modification because of crossing.
					ichimokuDirection1 = ichimokuDirection0;
					if (ichimokuDirection0 == 1)
					{
						ichimokuSignal = (5 * (tenkanSignal + kijunSignal + kumoSignal + senkouSignal + chikouSignal)/22);  // Reset the signal strength when the cross occurs.
						if (DisplayIchimoku)
							BackColor = IchimokuUpColor;  // Long signal
						if (AlertIchimoku)
							Alert("IchimokuCrossUp",NinjaTrader.Cbi.Priority.High, "Ichimoku cross UP", "ichimokuUp.wav", 10, Color.Black, Color.Yellow);
					}
					else
					{
						ichimokuSignal = (5 * (tenkanSignal + kijunSignal + kumoSignal + senkouSignal + chikouSignal)/22);  // Reset the signal strength when the cross occurs.
						if (DisplayIchimoku)
							BackColor = IchimokuDownColor; // Short signal
						if (AlertIchimoku)
							Alert("IchimokuCrossDown",NinjaTrader.Cbi.Priority.High, "Ichimoku cross DOWN", "ichimokuDown.wav", 10, Color.Black, Color.Yellow);
					}
					ichimokuBarCount = 0;
				}
				ichimokuBarCount = ichimokuBarCount + 1;
			#endregion	
			

        }
		
        #region Properties
			
			#region Value Input
				[Browsable(false)]
				[XmlIgnore()]
				public DataSeries TenkanSen
				{
					get { return Values[0]; }
				}
		
				[Browsable(false)]
				[XmlIgnore()]
				public DataSeries KijunSen
				{
					get { return Values[1]; }
				}
		
				[Browsable(false)]
				[XmlIgnore()]
				public DataSeries ChikouSpan
				{
					get { return Values[2]; }
				}
		
				[Browsable(false)]
				[XmlIgnore()]
				public DataSeries SenkouSpanA
				{
					get { return Values[3]; }
				}
				
				[Browsable(false)]
				[XmlIgnore()]
				public DataSeries SenkouSpanB
				{
					get { return Values[4]; }
				}
			#endregion
			
			#region Run Parameters
				[Description("The number 9 represents a week and a half")]
				[Category("Parameters")]
				public int PeriodFast
				{
					get { return periodFast; }
					set { periodFast = Math.Max(2, value); }
				}
		
				[Description("The number 26 represents standard Japanese work month")]
				[Category("Parameters")]
				public int PeriodMedium
				{
					get { return periodMedium; }
					set { periodMedium = Math.Max(4, value); }
				}
		
				[Description("The number 52 represents two months")]
				[Category("Parameters")]
				public int PeriodSlow
				{
					get { return periodSlow; }
					set { periodSlow = Math.Max(6, value); }
				}
				
				[Description("Number of Periods in the ADX calculation.")]
				[Category("Parameters")]
				public int AdxPeriods
				{
					get { return adxPeriods; }
					set { adxPeriods = Math.Max(1, value); }
				}
		
				[Description("ADX values below this number will display on the screen.")]
				[Category("Parameters")]
				public int AdxCutOff
				{
					get { return adxCutOff; }
					set { adxCutOff = Math.Max(1, value); }
				}
			#endregion
			
			#region Displays
				[Description("Display the ADX backlighting.")]
				[Category("Displays")]
				public bool DisplayADXcolor
				{
					get { return displayADXcolor; }
					set { displayADXcolor = value; }
				}
		
				[Description("Display the signal indicator panel.")]
				[Category("Displays")]
				public bool DisplayStrength
				{
					get { return displayStrength; }
					set { displayStrength = value; }
				}
		
				[Description("Display the Tenkan Sen/Kijun Sen crossing.")]
				[Category("Displays")]
				public bool DisplayTenkan
				{
					get { return displayTenkan; }
					set { displayTenkan= value; }
				}
		
				[Description("Display the Kijun Sen crossing.")]
				[Category("Displays")]
				public bool DisplayKijun
				{
					get { return displayKijun; }
					set { displayKijun = value; }
				}
		
				[Description("Display the Kumo breakout.")]
				[Category("Displays")]
				public bool DisplayKumo
				{
					get { return displayKumo; }
					set { displayKumo = value; }
				}
		
				[Description("Display the Senkou Span crossing.")]
				[Category("Displays")]
				public bool DisplaySenkou
				{
					get { return displaySenkou; }
					set { displaySenkou = value; }
				}
				
				[Description("Display the Chikou Span crossing.")]
				[Category("Displays")]
				public bool DisplayChikou
				{
					get { return displayChikou; }
					set { displayChikou = value; }
				}
				
				[Description("Display all indicators satisfied.")]
				[Category("Displays")]
				public bool DisplayIchimoku
				{
					get { return displayIchimoku; }
					set { displayIchimoku = value; }
				}
			#endregion
			
			#region Alerts
				[Description("Enable the Tenkan Sen/Kijun Sen crossing alert.")]
				[Category("Alerts")]
				public bool AlertTenkan
				{
					get { return alertTenkan; }
					set { alertTenkan = value; }
				}
				
				[Description("Enable the Kijun Sen crossing alert.")]
				[Category("Alerts")]
				public bool AlertKijun
				{
					get { return alertKijun; }
					set { alertKijun = value; }
				}
				
				[Description("Enable the Kumo breakout alert.")]
				[Category("Alerts")]
				public bool AlertKumo
				{
					get { return alertKumo; }
					set { alertKumo = value; }
				}
				
				[Description("Enable the Senkou Span crossing alert.")]
				[Category("Alerts")]
				public bool AlertSenkou
				{
					get { return alertSenkou; }
					set { alertSenkou = value; }
				}
				
				[Description("Enable the Chikou Span crossing alert.")]
				[Category("Alerts")]
				public bool AlertChikou
				{
					get { return alertChikou; }
					set { alertChikou = value; }
				}
				
				[Description("Enable the complete Ichimoku alert.")]
				[Category("Alerts")]
				public bool AlertIchimoku
				{
					get { return alertIchimoku; }
					set { alertIchimoku = value; }
				}
			#endregion
			
			#region Back Colors
	/* 
					get { return barBrush.Color; }
					set
					{
						Color barAlphaColor = Color.FromArgb((int)((double)alphaPercent/100*(double)255),value);
						barBrush = new SolidBrush(barAlphaColor);
					}
	*/
				
				[Browsable(true)]
				[XmlIgnore()]
				[Description("Color of the upward Tenkan Sen/Kijun Sen crossing.")]
				[Category("Colors")]
				public Color TenkanUpColor
				{
					get { return tenkanUpColor.Color; }
					set	
						{ 
						Color junkTenkan = Color.PaleGreen;
						tenkanUpColor = new SolidBrush(junkTenkan);
						}
				}
// There has got to be a way to get the color selectable....sigh				
				[Browsable(true)]
				[XmlIgnore()]
				[Description("Color of the downward Tenkan Sen/Kijun Sen crossing.")]
				[Category("Colors")]
				public Color TenkanDownColor
				{
					get { return tenkanDownColor.Color; }
					set	{ Color tenkanDownColor = Color.Pink; }
				}
				
				[Browsable(true)]
				[XmlIgnore()]
				[Description("Color of the upward Kijun Sen crossing.")]
				[Category("Colors")]
				public Color KijunUpColor
				{
					get { return kijunUpColor.Color; }
					set	{ Color kijunUpColor = Color.PaleGreen; }
				}
				
				[Browsable(true)]
				[XmlIgnore()]
				[Description("Color of the downward Kijun Sen crossing.")]
				[Category("Colors")]
				public Color KijunDownColor
				{
					get { return kijunDownColor.Color; }
					set	{ Color kijunDownColor = Color.Pink; }
				}
				
				[Browsable(true)]
				[XmlIgnore()]
				[Description("Color of the upward Kumo breakout.")]
				[Category("Colors")]
				public Color KumoUpColor
				{
					get { return kumoUpColor.Color; }
					set	{ Color kumoUpColor = Color.PaleGreen; }
				}
				
				[Browsable(true)]
				[XmlIgnore()]
				[Description("Color of the downward Kumo breakout.")]
				[Category("Colors")]
				public Color KumoDownColor
				{
					get { return kumoDownColor.Color; }
					set	{ Color kumoDownColor = Color.Pink; }
				}
				
				[Browsable(true)]
				[XmlIgnore()]
				[Description("Color of the upward Kumo breakout.")]
				[Category("Colors")]
				public Color SenkouUpColor
				{
					get { return senkouUpColor.Color; }
					set	{ Color senkouUpColor = Color.PaleGreen; }
				}
				
				[Browsable(true)]
				[XmlIgnore()]
				[Description("Color of the downward Kumo breakout.")]
				[Category("Colors")]
				public Color SenkouDownColor
				{
					get { return senkouDownColor.Color; }
					set	{ Color senkouDownColor = Color.Pink; }
				}
				
				[Browsable(true)]
				[XmlIgnore()]
				[Description("Color of the upward Kumo breakout.")]
				[Category("Colors")]
				public Color ChikouUpColor
				{
					get { return chikouUpColor.Color; }
					set	{ Color chikouUpColor = Color.PaleGreen; }
				}
				
				[Browsable(true)]
				[XmlIgnore()]
				[Description("Color of the downward Kumo breakout.")]
				[Category("Colors")]
				public Color ChikouDownColor
				{
					get { return chikouDownColor.Color; }
					set	{ Color chikouDownColor = Color.Pink; }
				}
				
				[Browsable(true)]
				[XmlIgnore()]
				[Description("Color of the upward Kumo breakout.")]
				[Category("Colors")]
				public Color IchimokuUpColor
				{
					get { return ichimokuUpColor.Color; }
					set	{ Color ichimokuUpColor = Color.PaleGreen; }
				}
				
				[Browsable(true)]
				[XmlIgnore()]
				[Description("Color of the downward Kumo breakout.")]
				[Category("Colors")]
				public Color IchimokuDownColor
				{
					get { return ichimokuDownColor.Color; }
					set	{ Color ichimokuDownColor = Color.Pink; }
				}
				#endregion
			
        #endregion
		
		public override void Plot(Graphics graphics, Rectangle bounds, double min, double max)
		{
			#region Plot Kumo
				//  This Plot() method has been tweaked so it will color in two different colors
				//  in-between the two plots of Values[3] and Values[4].
				//
				base.Plot(graphics, bounds, min, max);
	
				if (Bars == null || ChartControl == null)
					return;
	
				SolidBrush brush;								// Set current brush color here.
	
				SolidBrush brushUP	 = new SolidBrush(Color.FromArgb(kumoOpacity*20,kumoColorUp));
				SolidBrush brushDOWN = new SolidBrush(Color.FromArgb(kumoOpacity*20,kumoColorDown));
	
				int	barWidth = ChartControl.ChartStyle.GetBarPaintWidth(ChartControl.BarWidth);
				SmoothingMode oldSmoothingMode = graphics.SmoothingMode;
				GraphicsPath path = new GraphicsPath();
	
				DataSeries series0 = (DataSeries) Values[3];	// Color in between these two plots.
				DataSeries series1 = (DataSeries) Values[4];
	
				brush = brushUP;								// Start with the upwards color.
				int barcount = 0;								// Start with leftmost bar.
				bool firstbar = true;							// Plotting the first bar.
				
				while (barcount < ChartControl.BarsPainted)		// Continue until all bars have been painted.
				{
					int count = 0;								// Counter for innner loop.
					for (int seriesCount = 3; seriesCount < 5; seriesCount++)
					{
						int	lastX = -1;
						int	lastY = -1;
						DataSeries	series = (DataSeries) Values[seriesCount];
						Gui.Chart.Plot plot = Plots[seriesCount];
		
						for (count = barcount; count < ChartControl.BarsPainted; count++)
						{
							int idx = ChartControl.LastBarPainted - ChartControl.BarsPainted + 1 + count;
							if (idx < 0 || idx >= Input.Count || (!ChartControl.ShowBarsRequired && idx < BarsRequired))
								continue;
	
							double val	= series.Get(idx);				// Get next y-value to be plotted.
							if (val == 0)								// If nothing to plot...
								continue;								// ...ignore the enrtry.
		
							int	x = (int) (ChartControl.CanvasRight - ChartControl.BarMarginRight - barWidth / 2
								+ (count - ChartControl.BarsPainted + 1) * ChartControl.BarSpace) + 1;
	
							int	y = (int) ((bounds.Y + bounds.Height) - ((val - min ) / Gui.Chart.ChartControl.MaxMinusMin(max, min)) * bounds.Height);
	
							double val0 = series0.Get(idx);
							double val1 = series1.Get(idx);
							if (((val0 > val1) && (brush != brushUP))	// Now going in wrong direction?
							|| ((val0 < val1) && (brush != brushDOWN)))
							{											// Yes.  Done with this loop.
								if (lastX >= 0)							// Was there a last point?
								{										// Yes.  Connect it to the position half-way to this one.
									path.AddLine( lastX - plot.Pen.Width / 2, lastY, (x +lastX - plot.Pen.Width) / 2, (lastY + y)/2);
																		// Plot vertex of cross-over of the lines (1/2 way point).
								}
								break;									// Done, exit inner loop to change color.
							}
	
							if (firstbar == false)						// Is this the first plotted bar of the chart?
							{											// No.  Plot all bars after the first one.
								if (count == barcount)					// First bar after direction change (and color swap)?
								{										// Yes.  Add line segment for cross-over, 1/2 bar back.
									double valm1 = series.Get(idx-1);	// Get prior y-value to be plotted.
									lastX = x - ChartControl.BarSpace/2;// Back up 1/2 a bar for x-value.
									lastY = (y + (int) ((bounds.Y + bounds.Height) - ((valm1 - min ) / Gui.Chart.ChartControl.MaxMinusMin(max, min)) * bounds.Height))/2;
								}
	
								path.AddLine(lastX - plot.Pen.Width / 2, lastY, x - plot.Pen.Width / 2, y);	// Connect last point to this one.
							}
							firstbar = false;							// No longer the first bar.
							lastX = x;									// Save current position for next time, so we can connect the dots.
							lastY = y;
						}
						path.Reverse();									// Go back the other direction.
					}
					graphics.SmoothingMode = SmoothingMode.AntiAlias;
					graphics.FillPath(brush, path);
					path.Reset();										// Eliminate points already colored.
	
					barcount = count;									// Get ready to process next segment.
					brush = (brush == brushUP) ? brushDOWN : brushUP;	// Switch colors for next segment.
				}
				graphics.SmoothingMode = oldSmoothingMode;				// Restore smoothing mode before exiting.
			#endregion
	
			#region Draw Indicator
				if (DisplayStrength)
				{
					Brush strBrush = new SolidBrush(Color.DodgerBlue);
					Font font = new Font( "Arial", 7);
					Rectangle rect;
					int curX = x1;
					int curY = y1;
		
					str = "  1     2      3     4      5";
					graphics.DrawString(str, font, strBrush, curX, curY );
					for( int i = 1; i < 7; i++ ) // Rows
					{
						int k = 0;
						bool paintRect = true;
						switch (i)
						{
							case 1:
								textStr = "T K X";
								if (tenkanBarCount < 5)
									k = tenkanSignal;
								break;
							case 2:
								textStr = "K X";
								if (kijunBarCount < 5)
									k = kijunSignal;
								break;
							case 3:
								textStr = "K B O";
								if (kumoBarCount < 5)
									k = kumoSignal;
								break;
							case 4:
								textStr = "S X";
								if (senkouBarCount < 5)
									k = senkouSignal;
								break;
							case 5:
								textStr = "C X";
								if (chikouBarCount < 5)
									k = chikouSignal;
								break;
							case 6:
								textStr = "Ichimoku";
								if (ichimokuBarCount < 5)
									k = ichimokuSignal;
								break;
						}
						curX = x1;
						curY = y1 + i * ( height + yGap ) + yGapHeading - yGap; // Increase the row count.
						for(int j = 1; j < 6; j++) // Columns
						{
							brush.Color = Color.LightGray;
							if (paintRect)
							{
								if (j == k)
									paintRect = false;
								if (k >= j)
									brush.Color = Color.Green;
								if (k < j && k != 0)
								{
									brush.Color = Color.Red;
									k = k + 1;
								}
							}
							rect = new Rectangle( curX, curY, width, height );
							graphics.FillRectangle(brush, rect );
							curX = x1 + j * ( width + xGap );
						}
						graphics.DrawString(textStr, font, strBrush, curX, curY - yGap);
					}
				}
			#endregion
		}
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private AaIchimokuNT7[] cacheAaIchimokuNT7 = null;

        private static AaIchimokuNT7 checkAaIchimokuNT7 = new AaIchimokuNT7();

        /// <summary>
        /// Customization of Ichimoku Cloud system.
        /// </summary>
        /// <returns></returns>
        public AaIchimokuNT7 AaIchimokuNT7(int adxCutOff, int adxPeriods, int periodFast, int periodMedium, int periodSlow)
        {
            return AaIchimokuNT7(Input, adxCutOff, adxPeriods, periodFast, periodMedium, periodSlow);
        }

        /// <summary>
        /// Customization of Ichimoku Cloud system.
        /// </summary>
        /// <returns></returns>
        public AaIchimokuNT7 AaIchimokuNT7(Data.IDataSeries input, int adxCutOff, int adxPeriods, int periodFast, int periodMedium, int periodSlow)
        {
            if (cacheAaIchimokuNT7 != null)
                for (int idx = 0; idx < cacheAaIchimokuNT7.Length; idx++)
                    if (cacheAaIchimokuNT7[idx].AdxCutOff == adxCutOff && cacheAaIchimokuNT7[idx].AdxPeriods == adxPeriods && cacheAaIchimokuNT7[idx].PeriodFast == periodFast && cacheAaIchimokuNT7[idx].PeriodMedium == periodMedium && cacheAaIchimokuNT7[idx].PeriodSlow == periodSlow && cacheAaIchimokuNT7[idx].EqualsInput(input))
                        return cacheAaIchimokuNT7[idx];

            lock (checkAaIchimokuNT7)
            {
                checkAaIchimokuNT7.AdxCutOff = adxCutOff;
                adxCutOff = checkAaIchimokuNT7.AdxCutOff;
                checkAaIchimokuNT7.AdxPeriods = adxPeriods;
                adxPeriods = checkAaIchimokuNT7.AdxPeriods;
                checkAaIchimokuNT7.PeriodFast = periodFast;
                periodFast = checkAaIchimokuNT7.PeriodFast;
                checkAaIchimokuNT7.PeriodMedium = periodMedium;
                periodMedium = checkAaIchimokuNT7.PeriodMedium;
                checkAaIchimokuNT7.PeriodSlow = periodSlow;
                periodSlow = checkAaIchimokuNT7.PeriodSlow;

                if (cacheAaIchimokuNT7 != null)
                    for (int idx = 0; idx < cacheAaIchimokuNT7.Length; idx++)
                        if (cacheAaIchimokuNT7[idx].AdxCutOff == adxCutOff && cacheAaIchimokuNT7[idx].AdxPeriods == adxPeriods && cacheAaIchimokuNT7[idx].PeriodFast == periodFast && cacheAaIchimokuNT7[idx].PeriodMedium == periodMedium && cacheAaIchimokuNT7[idx].PeriodSlow == periodSlow && cacheAaIchimokuNT7[idx].EqualsInput(input))
                            return cacheAaIchimokuNT7[idx];

                AaIchimokuNT7 indicator = new AaIchimokuNT7();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.AdxCutOff = adxCutOff;
                indicator.AdxPeriods = adxPeriods;
                indicator.PeriodFast = periodFast;
                indicator.PeriodMedium = periodMedium;
                indicator.PeriodSlow = periodSlow;
                Indicators.Add(indicator);
                indicator.SetUp();

                AaIchimokuNT7[] tmp = new AaIchimokuNT7[cacheAaIchimokuNT7 == null ? 1 : cacheAaIchimokuNT7.Length + 1];
                if (cacheAaIchimokuNT7 != null)
                    cacheAaIchimokuNT7.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheAaIchimokuNT7 = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Customization of Ichimoku Cloud system.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.AaIchimokuNT7 AaIchimokuNT7(int adxCutOff, int adxPeriods, int periodFast, int periodMedium, int periodSlow)
        {
            return _indicator.AaIchimokuNT7(Input, adxCutOff, adxPeriods, periodFast, periodMedium, periodSlow);
        }

        /// <summary>
        /// Customization of Ichimoku Cloud system.
        /// </summary>
        /// <returns></returns>
        public Indicator.AaIchimokuNT7 AaIchimokuNT7(Data.IDataSeries input, int adxCutOff, int adxPeriods, int periodFast, int periodMedium, int periodSlow)
        {
            return _indicator.AaIchimokuNT7(input, adxCutOff, adxPeriods, periodFast, periodMedium, periodSlow);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Customization of Ichimoku Cloud system.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.AaIchimokuNT7 AaIchimokuNT7(int adxCutOff, int adxPeriods, int periodFast, int periodMedium, int periodSlow)
        {
            return _indicator.AaIchimokuNT7(Input, adxCutOff, adxPeriods, periodFast, periodMedium, periodSlow);
        }

        /// <summary>
        /// Customization of Ichimoku Cloud system.
        /// </summary>
        /// <returns></returns>
        public Indicator.AaIchimokuNT7 AaIchimokuNT7(Data.IDataSeries input, int adxCutOff, int adxPeriods, int periodFast, int periodMedium, int periodSlow)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.AaIchimokuNT7(input, adxCutOff, adxPeriods, periodFast, periodMedium, periodSlow);
        }
    }
}
#endregion
