#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

public enum IndicatorMethod
{
	RSI,
	MACD,
	CCI,		
	StochasticsD,
	StochasticsK,
	FastStochasticsD,
	FastStochasticsK,
	StochasticsRSI,
	MFI,
	ROC,
	RVI,
	SMI,		// Stochastic Momentum Index
	DO,			// Derivative Oscillator
	Momentum,	// NinjaTraders Momentum Indicator
	LaguerreECO	// Laguerre-ized ergodic candle oscillator
}

public enum D3PriceType
{
	High_Low,
	Open_Close,
	SMA1,
	SMA2,
	EMA
}
	
namespace NinjaTrader.Indicator
{
	
    [Description("Divergence spotter incorporated with many common indicators. Ver 2.0 - David Anderson 5/2009")]
	public class D3SpotterV2 : Indicator
	{
		#region Variables
		
		private IndicatorMethod method = IndicatorMethod.DO;
		private D3PriceType pType = D3PriceType.High_Low;
		
		
//		 RSI variables
		private DataSeries		avgUp;
		private DataSeries		avgDown;
		private DataSeries		down;
		private int				period	= 14;
		private int				smooth	= 3;
		private DataSeries		up;
		
//		MACD variables
		private int				fast	= 12;
		private int				slow	= 26;
//		private int				smooth	= 9;
		private	DataSeries		fastEma;
		private	DataSeries		slowEma;
		
//		 Stochastics and Fast Stochastics
		private int				periodD	= 7;	// SlowDperiod
		private int				periodK	= 14;	// Kperiod
//		private int				smooth	= 3;	// SlowKperiod
		private DataSeries		den;
		private DataSeries		nom;
		
//		MFI
		private	DataSeries		negative;
		private	DataSeries		positive;
//		
//		RVI
		private double			dnAvgH	= 0;
		private double			dnAvgL	= 0;
		private double			upAvgH	= 0;
		private double			upAvgL	= 0;
//		
//---------------------------------------------------------------
		private IDataSeries Indicator;
		private int QueueLength = 3;

		private int [] HighBarsAgo;
		private int [] LowBarsAgo;
		
		private int ThisHigh, ThisLow;
		private int QHLength = 0, QLLength = 0;
		
		private int scanWidth = 30;
		
		private int A = 1;
		
		private int BarsAgo;
		private double priceDiffLimit 		= 0.0;
		private double indicatorDiffLimit 	= 0.0;
		
		private Color divergenceColor 			= Color.DarkMagenta;
		private DashStyle divergenceDashStyle 	= DashStyle.Dot;
		private int divergenceLineWidth 		= 3;
		private int markerDistanceFactor 		= 1;
		
		private string myAlert1 = "Alert1.wav";
		private string myAlert2 = "Alert2.wav";
		
//		private int period = 21;
//		private int smooth = 0;
		private bool showHiLo = false;
		private double wt;
		private DataSeries H_upavg, L_upavg, H_dnavg, L_dnavg;
// 
// variables for SMI
//
		private int			range		= 10;
		private int			emaperiod1	= 25;
		private int			emaperiod2	= 3;
		private int 		smiemaperiod = 3;
		private double  	offset 		= 1.0;
		
		private DataSeries	sms;
		private DataSeries	hls;
		private DataSeries 	smis;
		
		private bool 	activatealertsound = true;
		private bool 	showarrows=true;
		private bool 	showpaintbars=false;
		
//		For Alerts - for SMI
		private double	alertlevel 			= 40;
		private string	longwavfilename 	= "long.wav";
		private string 	shortwavfilename	= "short.wav";
		private bool 	writealerttext 		= true;
		private int 	BarAtLastCross 		= 0;
		
		private int smooth1 = 5; // Default setting for Smooth1
        private int smooth2 = 3; // Default setting for Smooth2
        private int smooth3 = 9; // Default setting for Smooth3 
		
//		For Laguerre-ized ergodic candle oscillator
		
		// Wizard generated variables
        private double gamma = 0.300; // Default setting for Gamma
        // User defined variables (add any user defined variables below)
	    private WMA weightma;
	    private EMA expma;
	    private int smoothing = 8;
	    private int trigger = 5;
	    private double[] ol,pol,hl,phl,ll,pll,cl,pcl;
	    private DataSeries unsmoothed;
	    private int lastSeenBar = -1;
		
		#endregion

		protected override void Initialize()
		{	
			switch (Method) {
				case IndicatorMethod.RSI:
					InitRSI();
					Indicator = RSI(Period,1);
					break;
				
				case IndicatorMethod.MACD:
					InitMACD();
					Indicator = MACD(Fast, Slow, Smooth);
					break;
				
				case IndicatorMethod.CCI:
					InitCCI();
					Indicator = CCI(Period);
					break;
				
				case IndicatorMethod.StochasticsD:
					InitStochastics();
					Indicator = Stochastics(PeriodD, PeriodK, Smooth).D;
					break;
				
				case IndicatorMethod.StochasticsK:
					InitStochastics();
					Indicator = Stochastics(PeriodD, PeriodK, Smooth).K;
					break;

				case IndicatorMethod.FastStochasticsD:
					InitStochastics();
					Indicator = StochasticsFast(PeriodD, PeriodK).D;
					break;
				
				case IndicatorMethod.FastStochasticsK:
					InitStochastics();
					Indicator = StochasticsFast(PeriodD, PeriodK).K;
					break;
				
				case IndicatorMethod.StochasticsRSI:
					InitStochasticsRSI();
					Indicator = StochRSI(Period);
					break;
					
				case IndicatorMethod.MFI:
					InitMFI();
					Indicator = MFI(Period);
					break;

				case IndicatorMethod.ROC:
					InitROC();
					Indicator = ROC(Period);
					break;
					
//				case IndicatorMethod.ERG:
//					InitERG();
//					Indicator = ERG(Period);
//					break;
					
				case IndicatorMethod.RVI:
					InitRVI();
					Indicator = RVI(Period);
					break;
					
				case IndicatorMethod.SMI:
					InitSMI();
					Indicator = SMI(EMAPeriod1, EMAPeriod2, Range, SMIEMAPeriod);
					break;
					
				case IndicatorMethod.DO:
					InitDO();
					Indicator = DerivativeOscillator(Period, Smooth1, Smooth2, Smooth3);
					break;
					
				case IndicatorMethod.Momentum:
					InitMomentum();
					Indicator = Momentum(Close, Period);
					break;
					
				case IndicatorMethod.LaguerreECO:
					InitLaguerreECO();
					Indicator = Z20090712LaguerreECO(Gamma, Smoothing, Trigger); // LaguerreECO();
					break;
					
			}
			
			HighBarsAgo	= new int[QueueLength];
			LowBarsAgo	= new int[QueueLength];
			
			for (int i=0; i<QueueLength; i++) {
				HighBarsAgo[i] 	= 0;
				LowBarsAgo[i] 	= 0;
			}
//			
//			PriceTypeSupported	= true;
			
		}

		protected override void OnBarUpdate()
		{		
			double PriceDiff, IndicatorDiff;

			if (CurrentBar < 10) return;
			
			switch (Method) {
				case IndicatorMethod.RSI:	PlotRSI(); break;
					
				case IndicatorMethod.MACD:	PlotMACD(); break;
				
				case IndicatorMethod.CCI:	PlotCCI(); break;
				
				case IndicatorMethod.StochasticsK: 
				case IndicatorMethod.StochasticsD: PlotStochastics(); break;

				case IndicatorMethod.FastStochasticsK: 
				case IndicatorMethod.FastStochasticsD: PlotFastStochastics(); break;
				
				case IndicatorMethod.StochasticsRSI: PlotStochasticsRSI(); break;
					
				case IndicatorMethod.MFI: PlotMFI(); break;
					
				case IndicatorMethod.ROC: PlotROC(); break;
					
//				case IndicatorMethod.ERG: PlotERG(); break;
					
				case IndicatorMethod.RVI: PlotRVI(); break;
					
				case IndicatorMethod.SMI: PlotSMI(); break;
					
				case IndicatorMethod.DO: PlotDO(); break;
					
				case IndicatorMethod.Momentum: PlotMomentum(); break;
					
				case IndicatorMethod.LaguerreECO: PlotLaguerreECO(); break;
					
			}
			
//--------------------------------------------------------------------
//--------------------------------------------------------------------
//Print("-- CurrentBar - A= " + (CurrentBar - A).ToString());
			
			switch (PType) 
			{
				case D3PriceType.High_Low:	
					ThisHigh = HighestBar(High, ScanWidth);
					break;
					
				case D3PriceType.Open_Close: 
					ThisHigh = HighestBar(Close, ScanWidth);
					break;
				
				case D3PriceType.SMA1:
					ThisHigh = HighestBar(SMA(High,1), ScanWidth);
					break;
			}

				
			if (ThisHigh == A) {

				Alert("MyAlrt1" + CurrentBar.ToString(), NinjaTrader.Cbi.Priority.High, "High Found", MyAlert1, 10, Color.Black, Color.Yellow);
				
				for (int i = QueueLength-1; i >= 1; i--) {
					HighBarsAgo[i] = HighBarsAgo[i-1];
				}
			
				HighBarsAgo[0] = CurrentBar - A;

//				Print("-- " + Time[A].ToLongTimeString() + " -> " + High[A].ToString());
				
//		for (int i = 0; i < QueueLength; i++) {
//			Print("HighBarsAgo[" + i.ToString("0") + "]=" + HighBarsAgo[i].ToString());
//		}
				
				DrawDot("Hdot" + CurrentBar.ToString(), true, A, High[A] + (TickSize*MDF), Color.Yellow);
				DrawOnPricePanel = false;
				
			Print("Indicator[A]: " + Indicator[A].ToString());
				
				DrawDot("IHdot" + CurrentBar.ToString(), true, A, Indicator[A], Color.Yellow);
				DrawOnPricePanel = true;
				
				if (++QHLength >= 2) {
					
					for(int i = 0; i < Math.Min(QHLength, QueueLength); i++) {
		
//						Print("i " + i.ToString());
						
						BarsAgo = CurrentBar - HighBarsAgo[i];
						
						IndicatorDiff	= Indicator[A] - Indicator[BarsAgo];
						
						switch (PType) 
						{
							case D3PriceType.High_Low:	
								PriceDiff	= High[A] - High[BarsAgo];
								break;
								
							case D3PriceType.Open_Close: 
								PriceDiff	= Close[A] - Close[BarsAgo];
								break;
							
							case D3PriceType.SMA1:
								PriceDiff	= SMA(High, 1)[A] - SMA(High, 1)[BarsAgo];
								break;
								
							default :
								PriceDiff	= High[A] - High[BarsAgo];
								break;
						}
						
						if (((IndicatorDiff < IndicatorDiffLimit) && (PriceDiff >= PriceDiffLimit)) || ((IndicatorDiff > IndicatorDiffLimit) && (PriceDiff <= PriceDiffLimit))) {						
								
//							Print("IndicatorDiff " + IndicatorDiff.ToString());
//							Print("PriceDiff " + PriceDiff.ToString());

							if ((BarsAgo - A) < ScanWidth) {

								Alert("MyAlrt2" + CurrentBar.ToString(), NinjaTrader.Cbi.Priority.High, "Divergence Found", MyAlert2, 10, Color.Black, Color.Yellow);
								
//								Print("HighestBar(High, QueueLength) = " +HighestBar(High, QueueLength).ToString());
									
//								Print("+++++ DrawLine> " + Time[BarsAgo].ToShortTimeString() + " to " + Time[A].ToShortTimeString());
								
								DrawLine("high"+CurrentBar.ToString() + BarsAgo.ToString(), true, BarsAgo, High[BarsAgo] + (TickSize*MDF), A, High[A] + (TickSize*MDF), DLC, DSD, DLW);								
								DrawTriangleDown(CurrentBar.ToString(), true, 0, High[0] + (TickSize*MDF), Color.Red);

//								DrawArrowDown("I" + CurrentBar.ToString(), true, 0, Indicator[0] + TickSize, Color.Red);
								DrawOnPricePanel = false;	
								DrawLine("IH"+CurrentBar.ToString() + BarsAgo.ToString(), true, BarsAgo, Indicator[BarsAgo], A, Indicator[A], DLC, DSD, DLW);
								DrawOnPricePanel = true;
							}
						}
					}
				}
			}
			
			switch (PType) 
			{
				case D3PriceType.High_Low:	
					ThisLow = LowestBar(Low, ScanWidth);
					break;
					
				case D3PriceType.Open_Close: 
					ThisLow = LowestBar(Close, ScanWidth);
					break;
					
				case D3PriceType.SMA1:
					ThisLow = LowestBar(SMA(Low,1), ScanWidth);
					break;
			}
				
			if (ThisLow == A) {
								
				for (int i = QueueLength-1; i >= 1; i--) {
					LowBarsAgo[i] = LowBarsAgo[i-1];
				}
			
				LowBarsAgo[0] = CurrentBar - A;
				
				DrawDot("Ldot" + CurrentBar.ToString(), true, A, Low[A] - (TickSize*MDF), Color.Yellow);
				DrawOnPricePanel = false;
				DrawDot("ILdot" + CurrentBar.ToString(), true, A, Indicator[A], Color.Yellow);
				DrawOnPricePanel = true;

				Alert("MyAlrt1" + CurrentBar.ToString(), NinjaTrader.Cbi.Priority.High, "Low Found", MyAlert1, 10, Color.Black, Color.Yellow);
				
				if (++QLLength >= 2) {
					
					for(int i = 0; i < Math.Min(QLLength, QueueLength); i++) {
		
						BarsAgo = CurrentBar - LowBarsAgo[i];
						
						IndicatorDiff 	= Indicator[A] - Indicator[BarsAgo];
						switch (PType) 
						{
							case D3PriceType.High_Low:	
								PriceDiff 		= Low[A] - Low[BarsAgo];	
								break;
								
							case D3PriceType.Open_Close: 
								PriceDiff 		= Close[A] - Close[BarsAgo];	
								break;
							
							case D3PriceType.SMA1:
								PriceDiff 		= SMA(Close,1)[A] - SMA(Close,1)[BarsAgo];	
								break;
								
							default:
								PriceDiff 		= Low[A] - Low[BarsAgo];	
								break;
						}	
						
						if (((IndicatorDiff > IndicatorDiffLimit) && (PriceDiff <= PriceDiffLimit)) || ((IndicatorDiff < IndicatorDiffLimit) && (PriceDiff >= PriceDiffLimit))) {	

//							Print("IndicatorDiff " + IndicatorDiff.ToString());
//							Print("PriceDiff " + PriceDiff.ToString());
								
							if ((BarsAgo - A) < ScanWidth) {

								DrawLine("low"+CurrentBar.ToString() + BarsAgo.ToString(), true, BarsAgo, Low[BarsAgo] - (TickSize*MDF), A, Low[A] - (TickSize*MDF), DLC, DSD, DLW);
									
								DrawTriangleUp(CurrentBar.ToString(), true, 0, Low[0] - (TickSize*MDF), Color.Green);
									
								DrawOnPricePanel = false;
								DrawLine("Ilow"+CurrentBar.ToString() + BarsAgo.ToString(), true, BarsAgo, Indicator[BarsAgo], A, Indicator[A], DLC, DSD, DLW);
								DrawOnPricePanel = true;
								
								Alert("MyAlrt2" + CurrentBar.ToString(), NinjaTrader.Cbi.Priority.High, "Divergence Found", MyAlert2, 10, Color.Black, Color.Yellow);
							}
						}
					}
				}
			}
			

		}
	
		#region Configured Indicators
	
//
// Relative Strength Index (RSI) a price following oscillator that ranges between 0 and 100
//
		private void InitRSI()
		{
			Add(new Plot(Color.Green, "RSI"));
			Add(new Plot(Color.Orange, "Avg"));

			Add(new Line(System.Drawing.Color.DarkViolet, 30, "Lower"));
			Add(new Line(System.Drawing.Color.YellowGreen, 70, "Upper"));

			avgUp		= new DataSeries(this);
			avgDown		= new DataSeries(this);
			down		= new DataSeries(this);
			up			= new DataSeries(this);
			

			DrawTextFixed("tag", "text", TextPosition.Center, Color.Black, new Font("Arial", 20), Color.Black, Color.Black, 5);
			//DrawTextFixed("tag1", "Text to draw", TextPosition.TopRight);
		}

		private void PlotRSI()
		{
			DrawOnPricePanel = false;
			DrawTextFixed("RSI", "RSI("+Period.ToString()+","+Smooth.ToString()+")", TextPosition.BottomLeft); //, Color.Black, new Font("Arial", 10), Color.Black, Color.Black, 5);
			DrawOnPricePanel = true;
			
			if (CurrentBar == 0)
			{
				down.Set(0);
				up.Set(0);
				return;
			}

			down.Set(Math.Max(Input[1] - Input[0], 0));
			up.Set(Math.Max(Input[0] - Input[1], 0));

			if ((CurrentBar + 1) < Period) 
			{
				if ((CurrentBar + 1) == (Period - 1))
					Avg.Set(50);
				return;
			}

			if ((CurrentBar + 1) == Period) 
			{
				// First averages 
				avgDown.Set(SMA(down, Period)[0]);
				avgUp.Set(SMA(up, Period)[0]);
			}  
			else 
			{
				// Rest of averages are smoothed
				avgDown.Set((avgDown[1] * (Period - 1) + down[0]) / Period);
				avgUp.Set((avgUp[1] * (Period - 1) + up[0]) / Period);
			}

			double rs	  = avgUp[0] / (avgDown[0] == 0 ? 1 : avgDown[0]);
			double rsi	  = 100 - (100 / (1 + rs));
			double rsiAvg = (2.0 / (1 + Smooth)) * rsi + (1 - (2.0 / (1 + Smooth))) * Avg[1];

			Avg.Set(rsiAvg);
			Value.Set(rsi);
		}
//
// The RVI (Relative Volatility Index) was developed by Donald Dorsey as a compliment to and a confirmation of momentum based indicators. When used to confirm other signals, only buy when the RVI is over 50 and only sell when the RVI is under 50.
//
		private void InitRVI()
		{
			Add(new Plot(Color.DarkOrange, "RVI"));
			Add(new Line(Color.LightGray, 50, "SignalLine"));

			Overlay				= false;
			PriceTypeSupported	= false;
		}
		
		private void PlotRVI()
		{
			DrawOnPricePanel = false;
			DrawTextFixed("RVI", "RVI("+Period.ToString()+")", TextPosition.BottomLeft); //, Color.Black, new Font("Arial", 10), Color.Black, Color.Black, 5);
			DrawOnPricePanel = true;
			
			if (CurrentBar == 0)
			{
				Value.Set(50);
				return;
			}

			double up;
			double dn;
			
			// RVI(High)
			up = 0;
			dn = 0;

			if (High[0] > High[1])
				up = StdDev(High, 10)[0];

			if (High[0] < High[1])
				dn = StdDev(High, 10)[0];

			upAvgH = (upAvgH * (Period - 1) + up) / Period;
			dnAvgH = (dnAvgH * (Period - 1) + dn) / Period;

			double rviH = 100 * (upAvgH / (upAvgH + dnAvgH));

			// RVI(Low)
			up = 0;
			dn = 0;

			if (Low[0] > Low[1])
				up = StdDev(Low, 10)[0];

			if (Low[0] < Low[1])
				dn = StdDev(Low, 10)[0];

			upAvgL = (upAvgL * (Period - 1) + up) / Period;
			dnAvgL = (dnAvgL * (Period - 1) + dn) / Period;

			double rviL = 100 * (upAvgL / (upAvgL + dnAvgL));

			double rvi = (rviH + rviL) / 2;

			Value.Set(rvi);
		}
//
// The MACD (Moving Average Convergence/Divergence) is a trend following momentum indicator that shows the relationship between two moving averages of prices.
//
		private void InitMACD()
		{
			Add(new Plot(Color.Green, "Macd"));
			Add(new Plot(Color.DarkViolet, "Avg"));
			Add(new Plot(new Pen(Color.Navy, 2), PlotStyle.Bar, "Diff"));

			Add(new Line(Color.DarkGray, 0, "Zero line"));

			fastEma	= new DataSeries(this);
			slowEma	= new DataSeries(this);
		}
		
		private void PlotMACD()
		{
			DrawOnPricePanel = false;
			DrawTextFixed("MACD", "MACD("+Fast.ToString()+","+Slow.ToString()+","+Smooth.ToString()+")", TextPosition.BottomLeft); //, Color.Black, new Font("Arial", 10), Color.Black, Color.Black, 5);
			DrawOnPricePanel = true;
			
			if (CurrentBar == 0)
			{
				fastEma.Set(Input[0]);
				slowEma.Set(Input[0]);
				Value.Set(0);
				Avg.Set(0);
				Diff.Set(0);
			}
			else
			{
				fastEma.Set((2.0 / (1 + Fast)) * Input[0] + (1 - (2.0 / (1 + Fast))) * fastEma[1]);
				slowEma.Set((2.0 / (1 + Slow)) * Input[0] + (1 - (2.0 / (1 + Slow))) * slowEma[1]);

				double macd		= fastEma[0] - slowEma[0];
				double macdAvg	= (2.0 / (1 + Smooth)) * macd + (1 - (2.0 / (1 + Smooth))) * Avg[1];
				
				Value.Set(macd);
				Avg.Set(macdAvg);
				Diff.Set(macd - macdAvg);
			}
		}
		
//
// The Commodity Channel Index (CCI) measures the variation of a security's price from its statistical mean. High values show that prices are unusually high compared to average prices whereas low values indicate that prices are unusually low.
//
		private void InitCCI()
		{
			Add(new Plot(Color.Orange, "CCI"));
			Add(new Line(Color.DarkGray, 200, "Level 2"));
			Add(new Line(Color.DarkGray, 100, "Level 1"));
			Add(new Line(Color.DarkGray, 0, "Zero line"));
			Add(new Line(Color.DarkGray, -100, "Level -1"));
			Add(new Line(Color.DarkGray, -200, "Level -2"));
		}
		
		private void PlotCCI()
		{
			DrawOnPricePanel = false;
			DrawTextFixed("CCI", "CCI("+Period.ToString()+")", TextPosition.BottomLeft); //, Color.Black, new Font("Arial", 10), Color.Black, Color.Black, 5);
			DrawOnPricePanel = true;
			
			if (CurrentBar == 0)
				Value.Set(0);
			else
			{
				double mean = 0;
				for (int idx = Math.Min(CurrentBar, Period - 1); idx >= 0; idx--)
					mean += Math.Abs(Typical[idx] - SMA(Typical, Period)[0]);
				Value.Set((Typical[0] - SMA(Typical, Period)[0]) / (mean == 0 ? 1 : (0.015 * (mean / Math.Min(Period, CurrentBar + 1)))));
			}
		}
		
		private void InitStochastics()
		{
			Add(new Plot(Color.Green, "D"));
			Add(new Plot(Color.Orange, "K"));

			Add(new Line(Color.DarkViolet, 20, "Lower"));
			Add(new Line(Color.YellowGreen, 80, "Upper"));

			den			= new DataSeries(this);
			nom			= new DataSeries(this);
		}
		
		private void PlotStochastics()
		{
			DrawOnPricePanel = false;
			DrawTextFixed("Stochastics", "Stochastics("+PeriodD.ToString()+","+PeriodK.ToString()+","+Smooth.ToString()+")", TextPosition.BottomLeft); //, Color.Black, new Font("Arial", 10), Color.Black, Color.Black, 5);
			DrawOnPricePanel = true;
				
			nom.Set(Close[0] - MIN(Low, PeriodK)[0]);
			den.Set(MAX(High, PeriodK)[0] - MIN(Low, PeriodK)[0]);										// PeriodK = Kperiod
			
			K.Set(100 * SUM(nom, Smooth)[0] / (SUM(den, Smooth)[0] == 0 ? 1.0 : SUM(den, Smooth)[0]));	// Smooth  = SlowKperiod
			D.Set(SMA(K, PeriodD)[0]);	
		}
		
		private void PlotFastStochastics()
		{
			DrawOnPricePanel = false;
			DrawTextFixed("StochasticsFast", "StochasticsFast("+PeriodD.ToString()+","+PeriodK.ToString()+")", TextPosition.BottomLeft); //, Color.Black, new Font("Arial", 10), Color.Black, Color.Black, 5);
			DrawOnPricePanel = true;
			
			nom.Set(Close[0] - MIN(Low, PeriodK)[0]);
			den.Set(MAX(High, PeriodK)[0] - MIN(Low, PeriodK)[0]);

			K.Set(100 * nom[0] / (den[0] == 0 ? 1.0 : den[0])); 
			D.Set(100 * SUM(nom, PeriodD)[0] / (SUM(den, PeriodD)[0] == 0 ? 1.0 : SUM(den, PeriodD)[0]));
		}
		
		private void InitMFI()
		{
			Add(new Plot(Color.Orange, "MFI"));
			Add(new Line(Color.DarkViolet, 20, "Lower"));
			Add(new Line(Color.YellowGreen, 80, "Upper"));
			
			negative			= new DataSeries(this);
			positive			= new DataSeries(this);
		}
		
		private void PlotMFI()
		{
			DrawOnPricePanel = false;
			DrawTextFixed("MFI", "MFI("+Period.ToString()+")", TextPosition.BottomLeft); //, Color.Black, new Font("Arial", 10), Color.Black, Color.Black, 5);
			DrawOnPricePanel = true;
			
			if (CurrentBar == 0)
				Value.Set(50);
			else
			{
				negative.Set(Typical[0] < Typical[1] ? Typical[0] * Volume[0] : 0);
				positive.Set(Typical[0] > Typical[1] ? Typical[0] * Volume[0] : 0);

				Value.Set(SUM(negative, Period)[0] == 0 ? 50 : 100.0 - (100.0 / (1 + SUM(positive, Period)[0] / SUM(negative, Period)[0])));
			}		
		}
		
		private void InitROC()
		{
			Add(new Line(Color.DarkGray, 0, "Zero line"));
			Add(new Plot(Color.Blue, "ROC"));
			PriceTypeSupported	= true;
		}
		
		private void PlotROC()
		{
			DrawOnPricePanel = false;
			DrawTextFixed("ROC", "ROC("+Period.ToString()+")", TextPosition.BottomLeft); //, Color.Black, new Font("Arial", 10), Color.Black, Color.Black, 5);
			DrawOnPricePanel = true;
			
            int barsAgo = Math.Min(CurrentBar, Period);
			Value.Set(((Input[0] - Input[barsAgo]) / Input[barsAgo]) * 100);		
		}
		
		
		private void InitStochasticsRSI()
		{
			Add(new Plot(Color.Green, "StochRSI"));

			Add(new Line(Color.Red,  0.8, "Overbought"));
			Add(new Line(Color.Blue, 0.5, "Neutral"));
			Add(new Line(Color.Red,  0.2, "Oversold"));

			Overlay				= false;
			PriceTypeSupported	= true;
		}
		
		private void PlotStochasticsRSI()
		{
			DrawOnPricePanel = false;
			DrawTextFixed("StochRSI", "StochRSI("+Period.ToString()+")", TextPosition.BottomLeft); //, Color.Black, new Font("Arial", 10), Color.Black, Color.Black, 5);
			DrawOnPricePanel = true;
			
			double rsi  = RSI(Input, Period, 1)[0];
			double rsiL = MIN(RSI(Input, Period, 1), Period)[0];
			double rsiH = MAX(RSI(Input, Period, 1), Period)[0];

			if (rsi != rsiL && rsiH != rsiL)
				Value.Set((rsi - rsiL) / (rsiH - rsiL));
			else
				Value.Set(0);
		}
		
		private void InitSMI()
		{
			Print("AlertLevel: " + alertlevel.ToString());
			Add(new Line(Color.Maroon, alertlevel, "Alert line"));
			Add(new Line(Color.DarkGray, 0, "Zero line"));
			Add(new Line(Color.DarkGreen, -alertlevel, "Alert line"));
			
			Lines[0].Pen.DashStyle = DashStyle.Dash;
			Lines[1].Pen.DashStyle = DashStyle.Solid;
			Lines[2].Pen.DashStyle = DashStyle.Dash;
			
			Add(new Plot(Color.Green, PlotStyle.Line, "SMI"));
			Add(new Plot(Color.Orange, "SMIEMA"));
	
			Add(new Plot(Color.FromKnownColor(KnownColor.Red), PlotStyle.Line, "SMIdn"));
			Add(new Plot(Color.FromKnownColor(KnownColor.Blue), PlotStyle.Line, "SMIup")); 
	
		    Plots[0].Pen.Width = 2;
			Plots[2].Pen.Width = 4;    // dn
			Plots[3].Pen.Width = 4;    // up

			//stochastic momentums
			sms			= new DataSeries(this);
			//high low diffs
			hls			= new DataSeries(this);
			//stochastic momentum indexes
			smis		= new DataSeries(this);
			
			//CalculateOnBarClose = false;
		}
		
		private void PlotSMI()
		{
			if (( CurrentBar < emaperiod2) | ( CurrentBar < EMAPeriod1)) {
				return;
			}
			
			//Stochastic Momentum = SM {distance of close - midpoint}
		 	sms.Set(Close[0] - 0.5 * ((MAX(High, range)[0] + MIN(Low, range)[0])));
			
			//High low diffs
			hls.Set(MAX(High, range)[0] - MIN(Low, range)[0]);

			//Stochastic Momentum Index = SMI
			double denom = 0.5*EMA(EMA(hls,EMAPeriod1),emaperiod2)[0];
 			smis.Set(100*(EMA(EMA(sms,EMAPeriod1),emaperiod2))[0] / (denom ==0 ? 1 : denom  ));
			
			//Set the current SMI line value
			//SMI.Set(smis[0]);
			
			if (Falling(smis))
			{
				SMIdn.Set(1, smis[1]);
				SMIdn.Set(smis[0]);
				// Sets the bar color to red
				if (showpaintbars)
					BarColor = Color.FromKnownColor(KnownColor.Red);
			}	
			if (Rising(smis))
			{
				SMIup.Set(1, smis[1]);
				SMIup.Set(smis[0]);
				// Sets the bar color to blue
				if (showpaintbars)
					BarColor = Color.FromKnownColor(KnownColor.Blue);
			}
			
			//Set the line value for the SMIEMA by taking the EMA of the SMI
			SMIEMA.Set(EMA(smis, smiemaperiod)[0]);

			if(CrossBelow(smis, SMIEMA,1))
			{
				if (BarAtLastCross!=CurrentBar && BarAtLastCross!=-CurrentBar)
					if (activatealertsound) PlaySound(shortwavfilename);
				BarAtLastCross=CurrentBar;
				if (showarrows) DrawArrowDown("SMIcrossd"+CurrentBar,false,0,High[0]+offset,Color.FromKnownColor(KnownColor.Red));
			}
			else if(CrossAbove(smis, SMIEMA,1))
			{
				if (BarAtLastCross!=CurrentBar && BarAtLastCross!=-CurrentBar)
						if (activatealertsound) PlaySound(longwavfilename);
				BarAtLastCross=-CurrentBar;
				if (showarrows) DrawArrowUp("SMIcrossu"+CurrentBar,false,0,Low[0]-offset,Color.FromKnownColor(KnownColor.Blue));
			}
			if(writealerttext)
			{
				if(BarAtLastCross < 0)
					DrawTextFixed("SMIalert","SMI crossed above signal line  " +(CurrentBar+BarAtLastCross)+" bars ago",TextPosition.TopLeft); //+(-alertlevel).ToString("0")+"   "
				else if(BarAtLastCross > 0)
					DrawTextFixed("SMIalert","SMI crossed below signal line  "+(CurrentBar-BarAtLastCross)+" bars ago",TextPosition.TopLeft); //+alertlevel.ToString("0")+"   "
			}
		}
//
// "Constance Brown's Derivative Oscillator as pusblished in 'Technical Analysis for the Trading Professional' p. 293")
//
		private void InitDO()
		{
			Add(new Plot(Color.FromKnownColor(KnownColor.Blue), PlotStyle.Line, "DerivativeOscUp"));
			Add(new Plot(Color.FromKnownColor(KnownColor.Red), PlotStyle.Line, "DerivativeOscDn"));
			Add(new Plot(Color.FromKnownColor(KnownColor.Black), PlotStyle.Line, "DerivativeOsc"));
			Add(new Line(Color.FromKnownColor(KnownColor.DarkOliveGreen), 0, "Zero"));
			
            Overlay				= false;
            PriceTypeSupported	= true;
			PaintPriceMarkers   = true;
			VerticalGridLines   = false;
			
			Plots[0].Pen.Width = 2;
			Plots[1].Pen.Width = 2;
			Plots[2].Pen.Width = 2;			
			
			Plots[0].PlotStyle = PlotStyle.Bar;
			Plots[1].PlotStyle = PlotStyle.Bar;
			
			Plots[0].Min = 0;

			Plots[1].Max = 0;
		}
			
		private void PlotDO()
		{
			double smooth1 = EMA(EMA(RSI(Input, Period, 1), Smooth1), Smooth2)[0];
			double smooth2 = SMA(EMA(EMA(RSI(Input, Period, 1), Smooth1), Smooth2), Smooth3)[0];
			
			DerivativeOsc.Set(smooth1 - smooth2);
            DerivativeOscUp.Set(smooth1 - smooth2);
			DerivativeOscDn.Set(smooth1 - smooth2);
		}
		
//
// NinjaTraders Momentum indicator
//
		private void InitMomentum()
		{
			Add(new Plot(Color.Green, "Momentum"));
			Add(new Line(Color.DarkViolet, 0, "Zero line"));

			PriceTypeSupported	= true;
		}
			
		private void PlotMomentum()
		{
			Value.Set(CurrentBar == 0 ? 0 : Input[0] - Input[Math.Min(CurrentBar, Period)]);
		}
//
// Laguerre-ized ergodic candle oscillator
//		
		private void InitLaguerreECO()
		{
			Add(new Plot(Color.FromKnownColor(KnownColor.Gold), PlotStyle.Line, "Trigger"));
            Add(new Plot(Color.FromKnownColor(KnownColor.Red), PlotStyle.Line, "ECO"));
            Add(new Plot(Color.FromKnownColor(KnownColor.Cyan), PlotStyle.Bar, "Histogram"));
            Add(new Line(Color.FromKnownColor(KnownColor.Blue), 0, "Zero"));
            Overlay				= false;
            PriceTypeSupported	= false;
			Plots[0].Pen.Width = 2;
			Plots[1].Pen.Width = 2;
			Plots[2].Pen.Width = 2;
			weightma = null;
			expma = null;
			ol = new double[4];
			pol = new double[4];
			hl = new double[4];
			phl = new double[4];
			ll = new double[4];
			pll = new double[4];
			cl = new double[4];
			pcl = new double[4];
			unsmoothed = new DataSeries(this);
			lastSeenBar = -1;
		}
		
		private void PlotLaguerreECO()
		{
			if(CurrentBar == 1) 
			{
				for(int i = 0; i < 4; ++i)
				{
					pol[i] = Open[0];
					phl[i] = High[0];
					pll[i] = Low[0];
					pcl[i] = Close[0];
					lastSeenBar = CurrentBar;
				}
			}

			if(CurrentBar != lastSeenBar) 
			{
				for(int i = 0; i < 4; ++i)
				{
					pol[i] = ol[i]; // remember previous bar value...
					phl[i] = hl[i]; 
					pll[i] = ll[i]; 
					pcl[i] = cl[i]; 
				}
				lastSeenBar = CurrentBar;
			}

			
			// update all the Laguerre numbers....
			ol[0] = (1-gamma)*Open[0] + gamma*pol[0];
			hl[0] = (1-gamma)*High[0] + gamma*phl[0];
			ll[0] = (1-gamma)*Low[0] + gamma*pll[0];
			cl[0] = (1-gamma)*Close[0] + gamma*pcl[0];
			
			for(int i = 1; i < 4; ++i)
			{
				ol[i] = -gamma*ol[i-1] + pol[i-1] + gamma*pol[i];
				hl[i] = -gamma*hl[i-1] + phl[i-1] + gamma*phl[i];
				ll[i] = -gamma*ll[i-1] + pll[i-1] + gamma*pll[i];
				cl[i] = -gamma*cl[i-1] + pcl[i-1] + gamma*pcl[i];
			}
						
			double value1 = 0.0;
			double value2 = 0.0;
			
			for(int i = 0; i < 4; ++i)
			{
				value1 += (cl[i] - ol[i]);
				value2 += (hl[i] - ll[i]);
			}
			
			if(value2 != 0.0) 
			{
				unsmoothed.Set(100.0*value1/value2);	
			} 
			else 
			{
				if((CurrentBar > 1) && unsmoothed.ContainsValue(1)) 
				{
					unsmoothed.Set(unsmoothed[1]);
			  	} 
				else 
				{
			    	unsmoothed.Set(0);	
			  	}
			}
			
			if(weightma == null) 
			{
				weightma = WMA(unsmoothed,smoothing);
				expma = EMA(weightma.Value,trigger);
			}
			
			ECO.Set(weightma[0]);
			TriggerLine.Set(expma[0]);
			Histogram.Set(weightma[0] - expma[0]);
		}
		
		#endregion
		
		#region Properties
		/// <summary>
		/// </summary>
		/// 

		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Avg
		{
			get { return Values[1]; }
		}

		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Default
		{
			get { return Values[0]; }
		}

		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("High/Low Scan Width")]
		[Category("Parameters")]
		public int ScanWidth
		{
			get { return scanWidth; }
			set { scanWidth = Math.Max(10, value); }

		}
		
		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Indicator Method")]
		[Category("Parameters")]
		public IndicatorMethod Method
		{
			get { return method; }
			set { method = value; }
		}
		
		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Price Type")]
		[Category("Parameters")]
		public D3PriceType PType
		{
			get { return pType; }
			set { pType = value; }
		}
		
		// RSI Parameters
		
		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Numbers of bars used for calculations")]
		[Category("Parameters")]
		public int Period
		{
			get { return period; }
			set { period = Math.Max(2, value); }
		}

		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Number of bars for smoothing")]
		[Category("Parameters")]
		public int Smooth
		{
			get { return smooth; }
			set { smooth = Math.Max(1, value); }
		}
		
		//MACD Parameters
		
        /// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Diff
		{
			get { return Values[2]; }
		}

		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Number of bars for fast EMA")]
		[Category("Parameters")]
		public int Fast
		{
			get { return fast; }
			set { fast = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Number of bars for slow EMA")]
		[Category("Parameters")]
		public int Slow
		{
			get { return slow; }
			set { slow = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Numbers of bars used for moving average over D values")]
		[Category("Parameters")]
		public int PeriodD
		{
			get { return periodD; }
			set { periodD = Math.Max(1, value); }
		}

		// Stochastics
		/// <summary>
		/// Gets the slow D value.
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries D
		{
			get { return Values[0]; }
		}

		/// <summary>
		/// Gets the slow K value.
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries K
		{
			get { return Values[1]; }
		}
		
		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Numbers of bars used for calculating the K values")]
		[Category("Parameters")]
		public int PeriodK
		{
			get { return periodK; }
			set { periodK = Math.Max(1, value); }
		}
		
		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Price Difference Limit for divergence")]
//		[Category("Parameters")]
		public double PriceDiffLimit
		{
			get { return priceDiffLimit; }
			set { priceDiffLimit = Math.Min(0, value); }
		}
		
		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Indicator Difference Limit for divergence")]
//		[Category("Parameters")]
		public double IndicatorDiffLimit
		{
			get { return indicatorDiffLimit; }
			set { indicatorDiffLimit = Math.Max(0, value); }
		}
		
		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Divergence Line Color")]
		[Category("Parameters")]
		public Color DLC
		{
			get { return divergenceColor; }
			set { divergenceColor = value;; }
		}
		
		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Divergence Dash Style")]
		[Category("Parameters")]
		public DashStyle DSD
		{
			get { return divergenceDashStyle; }
			set { divergenceDashStyle = value; }
		}
		
		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Divergence Line Width")]
		[Category("Parameters")]
		public int DLW
		{
			get { return divergenceLineWidth; }
			set { divergenceLineWidth = Math.Max(1,value);}
		}
		
		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Marker Distance Factor")]
		[Category("Parameters")]
		public int MDF
		{
			get { return markerDistanceFactor; }
			set { markerDistanceFactor = Math.Max(1,value);}
		}
		
		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Alert for local high/low")]
		[Category("Parameters")]
		public string MyAlert1
		{
			get { return myAlert1; }
			set { myAlert1 = value; }
		}
		
		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Alert for arrow")]
		[Category("Parameters")]
		public string MyAlert2
		{
			get { return myAlert2; }
			set { myAlert2 = value; }
		}
		
				/// <summary>
		/// Gets the fast D value.
		/// </summary>
//		[Browsable(false)]
//		[XmlIgnore()]
//		public DataSeries SMI
//		{
//			get { return Values[0]; }
//		}
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries SMIdn
		{
			get { return Values[2]; }
		}
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries SMIup
		{
			get { return Values[3]; }
		}
		/// <summary>
		/// Gets the fast K value.
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries SMIEMA
		{
			get { return Values[1]; }
		}

		/// <summary>
		/// </summary>
		[Description("Range for momentum calculation. ( Q )")]
		[Category("Parameters")]
		public int Range
		{
			get { return range; }
			set { range = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Description("1st ema smothing period. ( R )")]
		[Category("Parameters")]
		public int EMAPeriod1
		{
			get { return emaperiod1; }
			set { emaperiod1 = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Description("2nd ema smoothing period. ( S )")]
		[Category("Parameters")]
		public int EMAPeriod2
		{
			get { return emaperiod2; }
			set { emaperiod2 = Math.Max(1, value); }
		}
		
		/// <summary>
		/// </summary>
		[Description("SMI EMA smoothing period.")]
		[Category("Parameters")]
		public int SMIEMAPeriod
		{
			get { return smiemaperiod; }
			set { smiemaperiod = Math.Max(1, value); }
		}
		[Description("Activate Alert Sounds?")]
		[Category("Parameters")]
		public bool ActivateAlertSound
		{
			get { return activatealertsound; }
			set { activatealertsound = value; }
		}
		[Description("Sound file to play for long alert.")]
		[Category("Parameters")]
		public string LONGWAVFileName
		{
			get { return longwavfilename; }
			set { longwavfilename = value; }
		}
		[Description("Sound file to play for short alert.")]
		[Category("Parameters")]
		public string SHORTWAVFileName
		{
			get { return shortwavfilename; }
			set { shortwavfilename = value; }
		}
		[Description("Write alert text message to screen?")]
		[Category("Parameters")]
		public bool WriteAlertText
		{
			get { return writealerttext; }
			set { writealerttext = value; }
		}
		[Description("Show Paintbar on Candles?")]
		[Category("Parameters")]
		public bool ShowPaintbars
		{
			get { return showpaintbars; }
			set { showpaintbars = value; }
		}
		[Description("Show Arrow Signals?")]
		[Category("Parameters")]
		public bool ShowArrows
		{
			get { return showarrows; }
			set { showarrows = value; }
		}
		[Description("Arrow Signal Offset (in points)")]
		[Category("Parameters")]
		public double Offset
		{
			get { return offset; }
			set { offset = value; }
		}
		
		[Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries DerivativeOscUp
        {
            get { return Values[0]; }
        }
		
		[Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries DerivativeOscDn
        {
            get { return Values[1]; }
        }
		
		[Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries DerivativeOsc
        {
            get { return Values[2]; }

        }
		
		[Description("Length of smoothing EMA 1")]
        [Category("Parameters")]
        public int Smooth1
        {
            get { return smooth1; }
            set { smooth1 = Math.Max(1, value); }
        }

        [Description("Length of smoothing EMA 2")]
        [Category("Parameters")]
        public int Smooth2
        {
            get { return smooth2; }
            set { smooth2 = Math.Max(1, value); }
        }

        [Description("Length of final smoothing SMA")]
        [Category("Parameters")]
        public int Smooth3
        {
            get { return smooth3; }
            set { smooth3 = Math.Max(1, value); }

        }
//
// Laguerre-ized ergodic candle oscillator
//
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries ECO
        {
            get { return Values[1]; }
        }

        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries TriggerLine
        {
            get { return Values[0]; }
        }
		
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries Histogram
        {
            get { return Values[2]; }
        }

        [Description("gamma")]
        [Category("Parameters")]
        public double Gamma
        {
            get { return gamma; }
            set { gamma = Math.Max(0.000, value); }
        }
        [Description("smoothing")]
        [Category("Parameters")]
        public int Smoothing
        {
            get { return smoothing; }
            set { smoothing = Math.Max(1, value); }
        }
        [Description("trigger")]
        [Category("Parameters")]
        public int Trigger
        {
            get { return trigger; }
            set { trigger = Math.Max(1, value); }
        }
		
		#endregion
		
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private D3SpotterV2[] cacheD3SpotterV2 = null;

        private static D3SpotterV2 checkD3SpotterV2 = new D3SpotterV2();

        /// <summary>
        /// Divergence spotter incorporated with many common indicators. Ver 2.0 - David Anderson 5/2009
        /// </summary>
        /// <returns></returns>
        public D3SpotterV2 D3SpotterV2(bool activateAlertSound, Color dLC, int dLW, DashStyle dSD, int eMAPeriod1, int eMAPeriod2, int fast, double gamma, string lONGWAVFileName, int mDF, IndicatorMethod method, string myAlert1, string myAlert2, double offset, int period, int periodD, int periodK, D3PriceType pType, int range, int scanWidth, string sHORTWAVFileName, bool showArrows, bool showPaintbars, int slow, int sMIEMAPeriod, int smooth, int smooth1, int smooth2, int smooth3, int smoothing, int trigger, bool writeAlertText)
        {
            return D3SpotterV2(Input, activateAlertSound, dLC, dLW, dSD, eMAPeriod1, eMAPeriod2, fast, gamma, lONGWAVFileName, mDF, method, myAlert1, myAlert2, offset, period, periodD, periodK, pType, range, scanWidth, sHORTWAVFileName, showArrows, showPaintbars, slow, sMIEMAPeriod, smooth, smooth1, smooth2, smooth3, smoothing, trigger, writeAlertText);
        }

        /// <summary>
        /// Divergence spotter incorporated with many common indicators. Ver 2.0 - David Anderson 5/2009
        /// </summary>
        /// <returns></returns>
        public D3SpotterV2 D3SpotterV2(Data.IDataSeries input, bool activateAlertSound, Color dLC, int dLW, DashStyle dSD, int eMAPeriod1, int eMAPeriod2, int fast, double gamma, string lONGWAVFileName, int mDF, IndicatorMethod method, string myAlert1, string myAlert2, double offset, int period, int periodD, int periodK, D3PriceType pType, int range, int scanWidth, string sHORTWAVFileName, bool showArrows, bool showPaintbars, int slow, int sMIEMAPeriod, int smooth, int smooth1, int smooth2, int smooth3, int smoothing, int trigger, bool writeAlertText)
        {
            if (cacheD3SpotterV2 != null)
                for (int idx = 0; idx < cacheD3SpotterV2.Length; idx++)
                    if (cacheD3SpotterV2[idx].ActivateAlertSound == activateAlertSound && cacheD3SpotterV2[idx].DLC == dLC && cacheD3SpotterV2[idx].DLW == dLW && cacheD3SpotterV2[idx].DSD == dSD && cacheD3SpotterV2[idx].EMAPeriod1 == eMAPeriod1 && cacheD3SpotterV2[idx].EMAPeriod2 == eMAPeriod2 && cacheD3SpotterV2[idx].Fast == fast && Math.Abs(cacheD3SpotterV2[idx].Gamma - gamma) <= double.Epsilon && cacheD3SpotterV2[idx].LONGWAVFileName == lONGWAVFileName && cacheD3SpotterV2[idx].MDF == mDF && cacheD3SpotterV2[idx].Method == method && cacheD3SpotterV2[idx].MyAlert1 == myAlert1 && cacheD3SpotterV2[idx].MyAlert2 == myAlert2 && Math.Abs(cacheD3SpotterV2[idx].Offset - offset) <= double.Epsilon && cacheD3SpotterV2[idx].Period == period && cacheD3SpotterV2[idx].PeriodD == periodD && cacheD3SpotterV2[idx].PeriodK == periodK && cacheD3SpotterV2[idx].PType == pType && cacheD3SpotterV2[idx].Range == range && cacheD3SpotterV2[idx].ScanWidth == scanWidth && cacheD3SpotterV2[idx].SHORTWAVFileName == sHORTWAVFileName && cacheD3SpotterV2[idx].ShowArrows == showArrows && cacheD3SpotterV2[idx].ShowPaintbars == showPaintbars && cacheD3SpotterV2[idx].Slow == slow && cacheD3SpotterV2[idx].SMIEMAPeriod == sMIEMAPeriod && cacheD3SpotterV2[idx].Smooth == smooth && cacheD3SpotterV2[idx].Smooth1 == smooth1 && cacheD3SpotterV2[idx].Smooth2 == smooth2 && cacheD3SpotterV2[idx].Smooth3 == smooth3 && cacheD3SpotterV2[idx].Smoothing == smoothing && cacheD3SpotterV2[idx].Trigger == trigger && cacheD3SpotterV2[idx].WriteAlertText == writeAlertText && cacheD3SpotterV2[idx].EqualsInput(input))
                        return cacheD3SpotterV2[idx];

            lock (checkD3SpotterV2)
            {
                checkD3SpotterV2.ActivateAlertSound = activateAlertSound;
                activateAlertSound = checkD3SpotterV2.ActivateAlertSound;
                checkD3SpotterV2.DLC = dLC;
                dLC = checkD3SpotterV2.DLC;
                checkD3SpotterV2.DLW = dLW;
                dLW = checkD3SpotterV2.DLW;
                checkD3SpotterV2.DSD = dSD;
                dSD = checkD3SpotterV2.DSD;
                checkD3SpotterV2.EMAPeriod1 = eMAPeriod1;
                eMAPeriod1 = checkD3SpotterV2.EMAPeriod1;
                checkD3SpotterV2.EMAPeriod2 = eMAPeriod2;
                eMAPeriod2 = checkD3SpotterV2.EMAPeriod2;
                checkD3SpotterV2.Fast = fast;
                fast = checkD3SpotterV2.Fast;
                checkD3SpotterV2.Gamma = gamma;
                gamma = checkD3SpotterV2.Gamma;
                checkD3SpotterV2.LONGWAVFileName = lONGWAVFileName;
                lONGWAVFileName = checkD3SpotterV2.LONGWAVFileName;
                checkD3SpotterV2.MDF = mDF;
                mDF = checkD3SpotterV2.MDF;
                checkD3SpotterV2.Method = method;
                method = checkD3SpotterV2.Method;
                checkD3SpotterV2.MyAlert1 = myAlert1;
                myAlert1 = checkD3SpotterV2.MyAlert1;
                checkD3SpotterV2.MyAlert2 = myAlert2;
                myAlert2 = checkD3SpotterV2.MyAlert2;
                checkD3SpotterV2.Offset = offset;
                offset = checkD3SpotterV2.Offset;
                checkD3SpotterV2.Period = period;
                period = checkD3SpotterV2.Period;
                checkD3SpotterV2.PeriodD = periodD;
                periodD = checkD3SpotterV2.PeriodD;
                checkD3SpotterV2.PeriodK = periodK;
                periodK = checkD3SpotterV2.PeriodK;
                checkD3SpotterV2.PType = pType;
                pType = checkD3SpotterV2.PType;
                checkD3SpotterV2.Range = range;
                range = checkD3SpotterV2.Range;
                checkD3SpotterV2.ScanWidth = scanWidth;
                scanWidth = checkD3SpotterV2.ScanWidth;
                checkD3SpotterV2.SHORTWAVFileName = sHORTWAVFileName;
                sHORTWAVFileName = checkD3SpotterV2.SHORTWAVFileName;
                checkD3SpotterV2.ShowArrows = showArrows;
                showArrows = checkD3SpotterV2.ShowArrows;
                checkD3SpotterV2.ShowPaintbars = showPaintbars;
                showPaintbars = checkD3SpotterV2.ShowPaintbars;
                checkD3SpotterV2.Slow = slow;
                slow = checkD3SpotterV2.Slow;
                checkD3SpotterV2.SMIEMAPeriod = sMIEMAPeriod;
                sMIEMAPeriod = checkD3SpotterV2.SMIEMAPeriod;
                checkD3SpotterV2.Smooth = smooth;
                smooth = checkD3SpotterV2.Smooth;
                checkD3SpotterV2.Smooth1 = smooth1;
                smooth1 = checkD3SpotterV2.Smooth1;
                checkD3SpotterV2.Smooth2 = smooth2;
                smooth2 = checkD3SpotterV2.Smooth2;
                checkD3SpotterV2.Smooth3 = smooth3;
                smooth3 = checkD3SpotterV2.Smooth3;
                checkD3SpotterV2.Smoothing = smoothing;
                smoothing = checkD3SpotterV2.Smoothing;
                checkD3SpotterV2.Trigger = trigger;
                trigger = checkD3SpotterV2.Trigger;
                checkD3SpotterV2.WriteAlertText = writeAlertText;
                writeAlertText = checkD3SpotterV2.WriteAlertText;

                if (cacheD3SpotterV2 != null)
                    for (int idx = 0; idx < cacheD3SpotterV2.Length; idx++)
                        if (cacheD3SpotterV2[idx].ActivateAlertSound == activateAlertSound && cacheD3SpotterV2[idx].DLC == dLC && cacheD3SpotterV2[idx].DLW == dLW && cacheD3SpotterV2[idx].DSD == dSD && cacheD3SpotterV2[idx].EMAPeriod1 == eMAPeriod1 && cacheD3SpotterV2[idx].EMAPeriod2 == eMAPeriod2 && cacheD3SpotterV2[idx].Fast == fast && Math.Abs(cacheD3SpotterV2[idx].Gamma - gamma) <= double.Epsilon && cacheD3SpotterV2[idx].LONGWAVFileName == lONGWAVFileName && cacheD3SpotterV2[idx].MDF == mDF && cacheD3SpotterV2[idx].Method == method && cacheD3SpotterV2[idx].MyAlert1 == myAlert1 && cacheD3SpotterV2[idx].MyAlert2 == myAlert2 && Math.Abs(cacheD3SpotterV2[idx].Offset - offset) <= double.Epsilon && cacheD3SpotterV2[idx].Period == period && cacheD3SpotterV2[idx].PeriodD == periodD && cacheD3SpotterV2[idx].PeriodK == periodK && cacheD3SpotterV2[idx].PType == pType && cacheD3SpotterV2[idx].Range == range && cacheD3SpotterV2[idx].ScanWidth == scanWidth && cacheD3SpotterV2[idx].SHORTWAVFileName == sHORTWAVFileName && cacheD3SpotterV2[idx].ShowArrows == showArrows && cacheD3SpotterV2[idx].ShowPaintbars == showPaintbars && cacheD3SpotterV2[idx].Slow == slow && cacheD3SpotterV2[idx].SMIEMAPeriod == sMIEMAPeriod && cacheD3SpotterV2[idx].Smooth == smooth && cacheD3SpotterV2[idx].Smooth1 == smooth1 && cacheD3SpotterV2[idx].Smooth2 == smooth2 && cacheD3SpotterV2[idx].Smooth3 == smooth3 && cacheD3SpotterV2[idx].Smoothing == smoothing && cacheD3SpotterV2[idx].Trigger == trigger && cacheD3SpotterV2[idx].WriteAlertText == writeAlertText && cacheD3SpotterV2[idx].EqualsInput(input))
                            return cacheD3SpotterV2[idx];

                D3SpotterV2 indicator = new D3SpotterV2();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.ActivateAlertSound = activateAlertSound;
                indicator.DLC = dLC;
                indicator.DLW = dLW;
                indicator.DSD = dSD;
                indicator.EMAPeriod1 = eMAPeriod1;
                indicator.EMAPeriod2 = eMAPeriod2;
                indicator.Fast = fast;
                indicator.Gamma = gamma;
                indicator.LONGWAVFileName = lONGWAVFileName;
                indicator.MDF = mDF;
                indicator.Method = method;
                indicator.MyAlert1 = myAlert1;
                indicator.MyAlert2 = myAlert2;
                indicator.Offset = offset;
                indicator.Period = period;
                indicator.PeriodD = periodD;
                indicator.PeriodK = periodK;
                indicator.PType = pType;
                indicator.Range = range;
                indicator.ScanWidth = scanWidth;
                indicator.SHORTWAVFileName = sHORTWAVFileName;
                indicator.ShowArrows = showArrows;
                indicator.ShowPaintbars = showPaintbars;
                indicator.Slow = slow;
                indicator.SMIEMAPeriod = sMIEMAPeriod;
                indicator.Smooth = smooth;
                indicator.Smooth1 = smooth1;
                indicator.Smooth2 = smooth2;
                indicator.Smooth3 = smooth3;
                indicator.Smoothing = smoothing;
                indicator.Trigger = trigger;
                indicator.WriteAlertText = writeAlertText;
                Indicators.Add(indicator);
                indicator.SetUp();

                D3SpotterV2[] tmp = new D3SpotterV2[cacheD3SpotterV2 == null ? 1 : cacheD3SpotterV2.Length + 1];
                if (cacheD3SpotterV2 != null)
                    cacheD3SpotterV2.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheD3SpotterV2 = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Divergence spotter incorporated with many common indicators. Ver 2.0 - David Anderson 5/2009
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.D3SpotterV2 D3SpotterV2(bool activateAlertSound, Color dLC, int dLW, DashStyle dSD, int eMAPeriod1, int eMAPeriod2, int fast, double gamma, string lONGWAVFileName, int mDF, IndicatorMethod method, string myAlert1, string myAlert2, double offset, int period, int periodD, int periodK, D3PriceType pType, int range, int scanWidth, string sHORTWAVFileName, bool showArrows, bool showPaintbars, int slow, int sMIEMAPeriod, int smooth, int smooth1, int smooth2, int smooth3, int smoothing, int trigger, bool writeAlertText)
        {
            return _indicator.D3SpotterV2(Input, activateAlertSound, dLC, dLW, dSD, eMAPeriod1, eMAPeriod2, fast, gamma, lONGWAVFileName, mDF, method, myAlert1, myAlert2, offset, period, periodD, periodK, pType, range, scanWidth, sHORTWAVFileName, showArrows, showPaintbars, slow, sMIEMAPeriod, smooth, smooth1, smooth2, smooth3, smoothing, trigger, writeAlertText);
        }

        /// <summary>
        /// Divergence spotter incorporated with many common indicators. Ver 2.0 - David Anderson 5/2009
        /// </summary>
        /// <returns></returns>
        public Indicator.D3SpotterV2 D3SpotterV2(Data.IDataSeries input, bool activateAlertSound, Color dLC, int dLW, DashStyle dSD, int eMAPeriod1, int eMAPeriod2, int fast, double gamma, string lONGWAVFileName, int mDF, IndicatorMethod method, string myAlert1, string myAlert2, double offset, int period, int periodD, int periodK, D3PriceType pType, int range, int scanWidth, string sHORTWAVFileName, bool showArrows, bool showPaintbars, int slow, int sMIEMAPeriod, int smooth, int smooth1, int smooth2, int smooth3, int smoothing, int trigger, bool writeAlertText)
        {
            return _indicator.D3SpotterV2(input, activateAlertSound, dLC, dLW, dSD, eMAPeriod1, eMAPeriod2, fast, gamma, lONGWAVFileName, mDF, method, myAlert1, myAlert2, offset, period, periodD, periodK, pType, range, scanWidth, sHORTWAVFileName, showArrows, showPaintbars, slow, sMIEMAPeriod, smooth, smooth1, smooth2, smooth3, smoothing, trigger, writeAlertText);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Divergence spotter incorporated with many common indicators. Ver 2.0 - David Anderson 5/2009
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.D3SpotterV2 D3SpotterV2(bool activateAlertSound, Color dLC, int dLW, DashStyle dSD, int eMAPeriod1, int eMAPeriod2, int fast, double gamma, string lONGWAVFileName, int mDF, IndicatorMethod method, string myAlert1, string myAlert2, double offset, int period, int periodD, int periodK, D3PriceType pType, int range, int scanWidth, string sHORTWAVFileName, bool showArrows, bool showPaintbars, int slow, int sMIEMAPeriod, int smooth, int smooth1, int smooth2, int smooth3, int smoothing, int trigger, bool writeAlertText)
        {
            return _indicator.D3SpotterV2(Input, activateAlertSound, dLC, dLW, dSD, eMAPeriod1, eMAPeriod2, fast, gamma, lONGWAVFileName, mDF, method, myAlert1, myAlert2, offset, period, periodD, periodK, pType, range, scanWidth, sHORTWAVFileName, showArrows, showPaintbars, slow, sMIEMAPeriod, smooth, smooth1, smooth2, smooth3, smoothing, trigger, writeAlertText);
        }

        /// <summary>
        /// Divergence spotter incorporated with many common indicators. Ver 2.0 - David Anderson 5/2009
        /// </summary>
        /// <returns></returns>
        public Indicator.D3SpotterV2 D3SpotterV2(Data.IDataSeries input, bool activateAlertSound, Color dLC, int dLW, DashStyle dSD, int eMAPeriod1, int eMAPeriod2, int fast, double gamma, string lONGWAVFileName, int mDF, IndicatorMethod method, string myAlert1, string myAlert2, double offset, int period, int periodD, int periodK, D3PriceType pType, int range, int scanWidth, string sHORTWAVFileName, bool showArrows, bool showPaintbars, int slow, int sMIEMAPeriod, int smooth, int smooth1, int smooth2, int smooth3, int smoothing, int trigger, bool writeAlertText)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.D3SpotterV2(input, activateAlertSound, dLC, dLW, dSD, eMAPeriod1, eMAPeriod2, fast, gamma, lONGWAVFileName, mDF, method, myAlert1, myAlert2, offset, period, periodD, periodK, pType, range, scanWidth, sHORTWAVFileName, showArrows, showPaintbars, slow, sMIEMAPeriod, smooth, smooth1, smooth2, smooth3, smoothing, trigger, writeAlertText);
        }
    }
}
#endregion
