#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

namespace NinjaTrader.Indicator
{
    [Description("Constance Brown's Derivative Oscillator as pusblished in 'Technical Analysis for the Trading Professional' p. 293")]
    public class DerivativeOscillator : Indicator
    {
        #region Variables	
		
            private int period = 14; // Default setting for Period
            private int smooth1 = 5; // Default setting for Smooth1
            private int smooth2 = 3; // Default setting for Smooth2
            private int smooth3 = 9; // Default setting for Smooth3  
		
        #endregion
		
        protected override void Initialize()
        {
            Add(new Plot(Color.FromKnownColor(KnownColor.Blue), PlotStyle.Line, "DerivativeOscUp"));
			Add(new Plot(Color.FromKnownColor(KnownColor.Red), PlotStyle.Line, "DerivativeOscDn"));
			Add(new Plot(Color.FromKnownColor(KnownColor.Black), PlotStyle.Line, "DerivativeOsc"));
			Add(new Line(Color.FromKnownColor(KnownColor.DarkOliveGreen), 0, "Zero"));
			
            Overlay				= false;
            PriceTypeSupported	= true;
			PaintPriceMarkers   = true;
			VerticalGridLines   = false;
			
			Plots[0].Pen.Width = 2;
			Plots[1].Pen.Width = 2;
			Plots[2].Pen.Width = 2;			
			
			Plots[0].PlotStyle = PlotStyle.Bar;
			Plots[1].PlotStyle = PlotStyle.Bar;
			
			Plots[0].Min = 0;
			Plots[1].Max = 0;
		}

        protected override void OnBarUpdate()
        {
            double smooth1 = EMA(EMA(RSI(Input, Period, 1), Smooth1), Smooth2)[0];
			double smooth2 = SMA(EMA(EMA(RSI(Input, Period, 1), Smooth1), Smooth2), Smooth3)[0];
			
			DerivativeOsc.Set(smooth1 - smooth2);
            DerivativeOscUp.Set(smooth1 - smooth2);
			DerivativeOscDn.Set(smooth1 - smooth2);
			
        }

        #region Properties
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries DerivativeOscUp
        {
            get { return Values[0]; }
        }
		
		[Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries DerivativeOscDn
        {
            get { return Values[1]; }
        }
		
		[Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries DerivativeOsc
        {
            get { return Values[2]; }
        }
		
		[Description("Number of bars used in calcuations")]
        [Category("Parameters")]
        public int Period
        {
            get { return period; }
            set { period = Math.Max(2, value); }
        }

        [Description("Length of smoothing EMA 1")]
        [Category("Parameters")]
        public int Smooth1
        {
            get { return smooth1; }
            set { smooth1 = Math.Max(1, value); }
        }

        [Description("Length of smoothing EMA 2")]
        [Category("Parameters")]
        public int Smooth2
        {
            get { return smooth2; }
            set { smooth2 = Math.Max(1, value); }
        }

        [Description("Length of final smoothing SMA")]
        [Category("Parameters")]
        public int Smooth3
        {
            get { return smooth3; }
            set { smooth3 = Math.Max(1, value); }
        }
        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private DerivativeOscillator[] cacheDerivativeOscillator = null;

        private static DerivativeOscillator checkDerivativeOscillator = new DerivativeOscillator();

        /// <summary>
        /// Constance Brown's Derivative Oscillator as pusblished in 'Technical Analysis for the Trading Professional' p. 293
        /// </summary>
        /// <returns></returns>
        public DerivativeOscillator DerivativeOscillator(int period, int smooth1, int smooth2, int smooth3)
        {
            return DerivativeOscillator(Input, period, smooth1, smooth2, smooth3);
        }

        /// <summary>
        /// Constance Brown's Derivative Oscillator as pusblished in 'Technical Analysis for the Trading Professional' p. 293
        /// </summary>
        /// <returns></returns>
        public DerivativeOscillator DerivativeOscillator(Data.IDataSeries input, int period, int smooth1, int smooth2, int smooth3)
        {
            if (cacheDerivativeOscillator != null)
                for (int idx = 0; idx < cacheDerivativeOscillator.Length; idx++)
                    if (cacheDerivativeOscillator[idx].Period == period && cacheDerivativeOscillator[idx].Smooth1 == smooth1 && cacheDerivativeOscillator[idx].Smooth2 == smooth2 && cacheDerivativeOscillator[idx].Smooth3 == smooth3 && cacheDerivativeOscillator[idx].EqualsInput(input))
                        return cacheDerivativeOscillator[idx];

            lock (checkDerivativeOscillator)
            {
                checkDerivativeOscillator.Period = period;
                period = checkDerivativeOscillator.Period;
                checkDerivativeOscillator.Smooth1 = smooth1;
                smooth1 = checkDerivativeOscillator.Smooth1;
                checkDerivativeOscillator.Smooth2 = smooth2;
                smooth2 = checkDerivativeOscillator.Smooth2;
                checkDerivativeOscillator.Smooth3 = smooth3;
                smooth3 = checkDerivativeOscillator.Smooth3;

                if (cacheDerivativeOscillator != null)
                    for (int idx = 0; idx < cacheDerivativeOscillator.Length; idx++)
                        if (cacheDerivativeOscillator[idx].Period == period && cacheDerivativeOscillator[idx].Smooth1 == smooth1 && cacheDerivativeOscillator[idx].Smooth2 == smooth2 && cacheDerivativeOscillator[idx].Smooth3 == smooth3 && cacheDerivativeOscillator[idx].EqualsInput(input))
                            return cacheDerivativeOscillator[idx];

                DerivativeOscillator indicator = new DerivativeOscillator();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.Period = period;
                indicator.Smooth1 = smooth1;
                indicator.Smooth2 = smooth2;
                indicator.Smooth3 = smooth3;
                Indicators.Add(indicator);
                indicator.SetUp();

                DerivativeOscillator[] tmp = new DerivativeOscillator[cacheDerivativeOscillator == null ? 1 : cacheDerivativeOscillator.Length + 1];
                if (cacheDerivativeOscillator != null)
                    cacheDerivativeOscillator.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheDerivativeOscillator = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Constance Brown's Derivative Oscillator as pusblished in 'Technical Analysis for the Trading Professional' p. 293
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.DerivativeOscillator DerivativeOscillator(int period, int smooth1, int smooth2, int smooth3)
        {
            return _indicator.DerivativeOscillator(Input, period, smooth1, smooth2, smooth3);
        }

        /// <summary>
        /// Constance Brown's Derivative Oscillator as pusblished in 'Technical Analysis for the Trading Professional' p. 293
        /// </summary>
        /// <returns></returns>
        public Indicator.DerivativeOscillator DerivativeOscillator(Data.IDataSeries input, int period, int smooth1, int smooth2, int smooth3)
        {
            return _indicator.DerivativeOscillator(input, period, smooth1, smooth2, smooth3);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Constance Brown's Derivative Oscillator as pusblished in 'Technical Analysis for the Trading Professional' p. 293
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.DerivativeOscillator DerivativeOscillator(int period, int smooth1, int smooth2, int smooth3)
        {
            return _indicator.DerivativeOscillator(Input, period, smooth1, smooth2, smooth3);
        }

        /// <summary>
        /// Constance Brown's Derivative Oscillator as pusblished in 'Technical Analysis for the Trading Professional' p. 293
        /// </summary>
        /// <returns></returns>
        public Indicator.DerivativeOscillator DerivativeOscillator(Data.IDataSeries input, int period, int smooth1, int smooth2, int smooth3)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.DerivativeOscillator(input, period, smooth1, smooth2, smooth3);
        }
    }
}
#endregion
