// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// Exponential Moving Average. The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The anaAdaptiveTrendLines applies more weight to recent prices than the SMA.
	/// </summary>
	[Description("The Adaptice Trendlines include four digital FIR filters - FATL, SATL, RFTL and RSTL -, which were obtained by employing Maximum Entropy Spectral Analysis.")]
	public class anaAdaptiveTrendLines : Indicator
	{
		#region Variables
		private double		fastTrendline	= 0.0;
		private double 		slowTrendline 	= 0.0;
		private bool		showDots		= true;
		private int 		plot0Width 		= 1;
		private PlotStyle 	plot0Style		= PlotStyle.Line;
		private DashStyle	dash0Style		= DashStyle.Solid;
		private int 		plot1Width 		= 1;
		private PlotStyle 	plot1Style		= PlotStyle.Line;
		private DashStyle 	dash1Style		= DashStyle.Solid;
		private int 		plot4Width 		= 1;
		private PlotStyle 	plot4Style		= PlotStyle.Dot;
		private DashStyle 	dash4Style		= DashStyle.Dot;
		private Color 		fatlColor		= Color.PaleGoldenrod;
		private Color 		rftlColor		= Color.Red;
		private Color 		satlColor		= Color.CornflowerBlue;
		private Color 		rstlColor		= Color.RoyalBlue;
		
		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Plot(Color.Gray, PlotStyle.Line, "FATL"));
			Add(new Plot(Color.Gray, PlotStyle.Line, "RFTL"));
			Add(new Plot(Color.Gray, PlotStyle.Line, "SATL"));
			Add(new Plot(Color.Gray, PlotStyle.Line, "RSTL"));
			Add(new Plot(Color.Gray, PlotStyle.Dot, "FATLDots"));
			Add(new Plot(Color.Gray, PlotStyle.Dot, "SATLDots"));
			
			Overlay				= true;
			PlotsConfigurable 	= false;
		}
		
		protected override void OnStartUp()
		{
			Plots[0].Pen.Color = FATLColor;
			Plots[1].Pen.Color = RFTLColor;
			Plots[2].Pen.Color = SATLColor;
			Plots[3].Pen.Color = RSTLColor;
			Plots[4].Pen.Color = FATLColor;
			Plots[5].Pen.Color = SATLColor;
			Plots[0].Pen.Width = plot0Width;
			Plots[0].PlotStyle = plot0Style;
			Plots[0].Pen.DashStyle = dash0Style;
			Plots[1].Pen.Width = plot1Width;
			Plots[1].PlotStyle = plot1Style;
			Plots[1].Pen.DashStyle = dash1Style;
			Plots[2].Pen.Width = plot0Width;
			Plots[2].PlotStyle = plot0Style;
			Plots[2].Pen.DashStyle = dash0Style;
			Plots[3].Pen.Width = plot1Width;
			Plots[3].PlotStyle = plot1Style;
			Plots[3].Pen.DashStyle = dash1Style;
			Plots[4].Pen.Width = plot4Width;
			Plots[4].PlotStyle = plot4Style;
			Plots[4].Pen.DashStyle = dash4Style;
			Plots[5].Pen.Width = plot4Width;
			Plots[5].PlotStyle = plot4Style;
			Plots[5].Pen.DashStyle = dash4Style;
		}
			
			
		/// <summary>
		/// Called on each bar update event (incoming tick)
		/// </summary>
		protected override void OnBarUpdate()
		{
			if (CurrentBar < 90 )
				return;
			
			fastTrendline = 0.4360409450*Close[0] + 0.3658689069*Close[1] + 0.2460452079*Close[2] + 0.1104506886*Close[3] - 0.0054034585*Close[4] 
				-0.0760367731*Close[5] - 0.0933058722*Close[6] - 0.0670110374*Close[7] - 0.0190795053*Close[8] + 0.0259609206*Close[9] 
				+0.0502044896*Close[10] + 0.0477818607*Close[11] + 0.0249252327*Close[12] - 0.0047706151*Close[13] - 0.0272432537*Close[14] 
				-0.0338917071*Close[15] - 0.0244141482*Close[16] - 0.0055774838*Close[17] + 0.0128149838*Close[18] + 0.0226522218*Close[19] 
				+0.0208778257*Close[20] + 0.0100299086*Close[21] - 0.0036771622*Close[22] - 0.0136744850*Close[23] - 0.0160483392*Close[24] 
				-0.0108597376*Close[25] - 0.0016060704*Close[26] + 0.0069480557*Close[27] + 0.0110573605*Close[28] + 0.0095711419*Close[29] 
				+0.0040444064*Close[30] - 0.0023824623*Close[31] - 0.0067093714*Close[32] - 0.0072003400*Close[33] - 0.0047717710*Close[34] 
				+0.0005541115*Close[35] + 0.0007860160*Close[36] + 0.0130129076*Close[37] + 0.0040364019*Close[38]; 		
			
			FATL.Set(fastTrendline);
			
			RFTL.Set(-0.0025097319*Close[0] +0.0513007762*Close[1] + 0.1142800493*Close[2] + 0.1699342860*Close[3] + 0.2025269304*Close[4] 
				+0.2025269304*Close[5] + 0.1699342860*Close[6] + 0.1142800493*Close[7] + 0.0513007762*Close[8] - 0.0025097319*Close[9] 
				-0.0353166244*Close[10] - 0.0433375629*Close[11] -0.0311244617*Close[12] - 0.0088618137*Close[13] + 0.0120580088*Close[14] 
				+0.0233183633*Close[15] + 0.0221931304*Close[16] +0.0115769653*Close[17] - 0.0022157966*Close[18] - 0.0126536111*Close[19] 
				-0.0157416029*Close[20] - 0.0113395830*Close[21] -0.0025905610*Close[22] + 0.0059521459*Close[23] + 0.0105212252*Close[24] 
				+0.0096970755*Close[25] + 0.0046585685*Close[26] -0.0017079230*Close[27] - 0.0063513565*Close[28] - 0.0074539350*Close[29] 
				-0.0050439973*Close[30] - 0.0007459678*Close[31] +0.0032271474*Close[32] + 0.0051357867*Close[33] + 0.0044454862*Close[34] 
				+0.0018784961*Close[35] - 0.0011065767*Close[36] -0.0031162862*Close[37] - 0.0033443253*Close[38] - 0.0022163335*Close[39] 
				+0.0002573669*Close[40] + 0.0003650790*Close[41] +0.0060440751*Close[42] + 0.0018747783*Close[43]); 

			slowTrendline = 0.0982862174*Close[0] + 0.0975682269*Close[1] + 0.0961401078*Close[2] + 0.0940230544*Close[3] + 0.0912437090*Close[4] 
				+0.0878391006*Close[5] + 0.0838544303*Close[6] + 0.0793406350*Close[7] + 0.0743569346*Close[8] + 0.0689666682*Close[9] 
				+0.0632381578*Close[10] + 0.0572428925*Close[11] + 0.0510534242*Close[12] + 0.0447468229*Close[13] + 0.0383959950*Close[14] 
				+0.0320735368*Close[15] + 0.0258537721*Close[16] + 0.0198005183*Close[17] + 0.0139807863*Close[18] + 0.0084512448*Close[19] 
				+0.0032639979*Close[20] - 0.0015350359*Close[21] - 0.0059060082*Close[22] - 0.0098190256*Close[23] - 0.0132507215*Close[24] 
				-0.0161875265*Close[25] - 0.0186164872*Close[26] - 0.0205446727*Close[27] - 0.0219739146*Close[28] - 0.0229204861*Close[29] 
				-0.0234080863*Close[30] - 0.0234566315*Close[31] - 0.0231017777*Close[32] - 0.0223796900*Close[33] - 0.0213300463*Close[34] 
				-0.0199924534*Close[35] - 0.0184126992*Close[36] - 0.0166377699*Close[37] - 0.0147139428*Close[38] - 0.0126796776*Close[39] 
				-0.0105938331*Close[40] - 0.0084736770*Close[41] - 0.0063841850*Close[42] - 0.0043466731*Close[43] - 0.0023956944*Close[44] 
				-0.0005535180*Close[45] + 0.0011421469*Close[46] + 0.0026845693*Close[47] + 0.0040471369*Close[48] + 0.0052380201*Close[49] 
				+0.0062194591*Close[50] + 0.0070340085*Close[51] + 0.0076266453*Close[52] + 0.0080376628*Close[53] + 0.0083037666*Close[54] 
				+0.0083694798*Close[55] + 0.0082901022*Close[56] + 0.0080741359*Close[57] + 0.0077543820*Close[58] + 0.0073260526*Close[59] 
				+0.0068163569*Close[60] + 0.0062325477*Close[61] + 0.0056078229*Close[62] + 0.0049516078*Close[63] + 0.0161380976*Close[64]; 

			SATL.Set(slowTrendline);
			
			RSTL.Set(-0.0074151919*Close[0] - 0.0060698985*Close[1] - 0.0044979052*Close[2] - 0.0027054278*Close[3] - 0.0007031702*Close[4] 
				+0.0014951741*Close[5] + 0.0038713513*Close[6] + 0.0064043271*Close[7] + 0.0090702334*Close[8] + 0.0118431116*Close[9] 
				+0.0146922652*Close[10] + 0.0175884606*Close[11] + 0.0204976517*Close[12] + 0.0233865835*Close[13] + 0.0262218588*Close[14] 
				+0.0289681736*Close[15] + 0.0315922931*Close[16] + 0.0340614696*Close[17] + 0.0363444061*Close[18] + 0.0384120882*Close[19] 
				+0.0402373884*Close[20] + 0.0417969735*Close[21] + 0.0430701377*Close[22] + 0.0440399188*Close[23] + 0.0446941124*Close[24] 
				+0.0450230100*Close[25] + 0.0450230100*Close[26] + 0.0446941124*Close[27] + 0.0440399188*Close[28] + 0.0430701377*Close[29] 
				+0.0417969735*Close[30] + 0.0402373884*Close[31] + 0.0384120882*Close[32] + 0.0363444061*Close[33] + 0.0340614696*Close[34] 
				+0.0315922931*Close[35] + 0.0289681736*Close[36] + 0.0262218588*Close[37] + 0.0233865835*Close[38] + 0.0204976517*Close[39] 
				+0.0175884606*Close[40] + 0.0146922652*Close[41] + 0.0118431116*Close[42] + 0.0090702334*Close[43] + 0.0064043271*Close[44] 
				+0.0038713513*Close[45] + 0.0014951741*Close[46] - 0.0007031702*Close[47] - 0.0027054278*Close[48] - 0.0044979052*Close[49] 
				-0.0060698985*Close[50] - 0.0074151919*Close[51] - 0.0085278517*Close[52] - 0.0094111161*Close[53] - 0.0100658241*Close[54] 
				-0.0104994302*Close[55] - 0.0107227904*Close[56] - 0.0107450280*Close[57] - 0.0105824763*Close[58] - 0.0102517019*Close[59] 
				-0.0097708805*Close[60] - 0.0091581551*Close[61] - 0.0084345004*Close[62] - 0.0076214397*Close[63] - 0.0067401718*Close[64] 
				-0.0058083144*Close[65] - 0.0048528295*Close[66] - 0.0038816271*Close[67] - 0.0029244713*Close[68] - 0.0019911267*Close[69] 
				-0.0010974211*Close[70] - 0.0002535559*Close[71] + 0.0005231953*Close[72] + 0.0012297491*Close[73] + 0.0018539149*Close[74] 
				+0.0023994354*Close[75] + 0.0028490136*Close[76] + 0.0032221429*Close[77] + 0.0034936183*Close[78] + 0.0036818974*Close[79] 
				+0.0038037944*Close[80] + 0.0038338964*Close[81] + 0.0037975350*Close[82] + 0.0036986051*Close[83] + 0.0035521320*Close[84] 
				+0.0033559226*Close[85] + 0.0031224409*Close[86] + 0.0028550092*Close[87] + 0.0025688349*Close[88] + 0.0022682355*Close[89] 
				+0.0073925495*Close[90]); 		
			
			if (showDots)
			{
				FATLDots.Set(fastTrendline);
				SATLDots.Set(slowTrendline);
			}
			else
			{
				FATLDots.Reset();
				SATLDots.Reset();
			}
		}

		#region Properties
		[Description("Show FATL and SATLDots")]
        [Category("Plots")]
		[Gui.Design.DisplayName ("Show FATL and SATL Dots")]
        public bool ShowDots
        {
            get { return showDots; }
            set { showDots = value; }
        }

		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries FATL
		{
			get { return Values[0]; }
		}
		
		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries RFTL
		{
			get { return Values[1]; }
		}

		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries SATL
		{
			get { return Values[2]; }
		}
		
		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries RSTL
		{
			get { return Values[3]; }
		}
 
		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries FATLDots
		{
			get { return Values[4]; }
		}
		
		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries SATLDots
		{
			get { return Values[5]; }
		}
		
		/// <summary>
		/// </summary>
        [XmlIgnore()]		
		[Description("Select color for FATL")]
		[Category("Plots")]
		[Gui.Design.DisplayName("Color FATL")]
		public Color FATLColor
		{
			get { return fatlColor; }
			set { fatlColor = value; }
		}
		
		// Serialize Color object
		[Browsable(false)]
		public string FATLColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(fatlColor); }
			set { fatlColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		/// <summary>
		/// </summary>
        [XmlIgnore()]		
		[Description("Select color for RFTL")]
		[Category("Plots")]
		[Gui.Design.DisplayName("Color RFTL")]
		public Color RFTLColor
		{
			get { return rftlColor; }
			set { rftlColor = value; }
		}
		
		// Serialize Color object
		[Browsable(false)]
		public string RFTLColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(rftlColor); }
			set { rftlColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		/// <summary>
		/// </summary>
        [XmlIgnore()]		
		[Description("Select color for SATL")]
		[Category("Plots")]
		[Gui.Design.DisplayName("Color SATL")]
		public Color SATLColor
		{
			get { return satlColor; }
			set { satlColor = value; }
		}
		
		// Serialize Color object
		[Browsable(false)]
		public string SATLColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(satlColor); }
			set { satlColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}

		/// <summary>
		/// </summary>
        [XmlIgnore()]		
		[Description("Select color for RSTL")]
		[Category("Plots")]
		[Gui.Design.DisplayName("Color RSTL")]
		public Color RSTLColor
		{
			get { return rstlColor; }
			set { rstlColor = value; }
		}
		
		// Serialize Color object
		[Browsable(false)]
		public string RSTLColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(rstlColor); }
			set { rstlColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		/// <summary>
		/// </summary>
		[Description("Width for FATL and SATL Lines.")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Width FATL and SATL")]
		public int Plot0Width
		{
			get { return plot0Width; }
			set { plot0Width = Math.Max(1, value); }
		}
		
		/// <summary>
		/// </summary>
		[Description("PlotStyle for FATL and SATL Lines..")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Plot Style FATL and SATL")]
		public PlotStyle Plot0Style
		{
			get { return plot0Style; }
			set { plot0Style = value; }
		}
		
		/// <summary>
		/// </summary>
		[Description("DashStyle for FATL and SATL Lines..")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Dash Style FATL and SATL")]
		public DashStyle Dash0Style
		{
			get { return dash0Style; }
			set { dash0Style = value; }
		} 
		
		/// <summary>
		/// </summary>
		[Description("Width for RFTL and RSTL.")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Width RFTL and RSTL")]
		public int Plot1Width
		{
			get { return plot1Width; }
			set { plot1Width = Math.Max(1, value); }
		}
		
		/// <summary>
		/// </summary>
		[Description("PlotStyle for RFTL and RSTL.")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Plot Style RFTL and RSTL")]
		public PlotStyle Plot1Style
		{
			get { return plot1Style; }
			set { plot1Style = value; }
		}
		
		/// <summary>
		/// </summary>
		[Description("DashStyle for RFTL and RSTL.")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Dash Style RFTL and RSTL")]
		public DashStyle Dash1Style
		{
			get { return dash1Style; }
			set { dash1Style = value; }
		} 		
		
		/// <summary>
		/// </summary>
		[Description("Width for FATL and SATL Dots.")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Width Dots")]
		public int Plot4Width
		{
			get { return plot4Width; }
			set { plot4Width = Math.Max(1, value); }
		}
		#endregion
		
		#region Miscellaneous

		public override string FormatPriceMarker(double price)
		{
			double trunc = Math.Truncate(price);
			int fraction = Convert.ToInt32(320 * Math.Abs(price - trunc) - 0.0001); // rounding down for ZF and ZT
			string priceMarker = "";
			if (TickSize == 0.03125) 
			{
				fraction = fraction/10;
				if (fraction < 10)
					priceMarker = trunc.ToString() + "'0" + fraction.ToString();
				else 
					priceMarker = trunc.ToString() + "'" + fraction.ToString();
			}
			else if (TickSize == 0.015625 || TickSize == 0.0078125)
			{
				if (fraction < 10)
					priceMarker = trunc.ToString() + "'00" + fraction.ToString();
				else if (fraction < 100)
					priceMarker = trunc.ToString() + "'0" + fraction.ToString();
				else	
					priceMarker = trunc.ToString() + "'" + fraction.ToString();
			}
			else
				priceMarker = price.ToString(Gui.Globals.GetTickFormatString(TickSize));
			return priceMarker;
		}		
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private anaAdaptiveTrendLines[] cacheanaAdaptiveTrendLines = null;

        private static anaAdaptiveTrendLines checkanaAdaptiveTrendLines = new anaAdaptiveTrendLines();

        /// <summary>
        /// The Adaptice Trendlines include four digital FIR filters - FATL, SATL, RFTL and RSTL -, which were obtained by employing Maximum Entropy Spectral Analysis.
        /// </summary>
        /// <returns></returns>
        public anaAdaptiveTrendLines anaAdaptiveTrendLines()
        {
            return anaAdaptiveTrendLines(Input);
        }

        /// <summary>
        /// The Adaptice Trendlines include four digital FIR filters - FATL, SATL, RFTL and RSTL -, which were obtained by employing Maximum Entropy Spectral Analysis.
        /// </summary>
        /// <returns></returns>
        public anaAdaptiveTrendLines anaAdaptiveTrendLines(Data.IDataSeries input)
        {
            if (cacheanaAdaptiveTrendLines != null)
                for (int idx = 0; idx < cacheanaAdaptiveTrendLines.Length; idx++)
                    if (cacheanaAdaptiveTrendLines[idx].EqualsInput(input))
                        return cacheanaAdaptiveTrendLines[idx];

            lock (checkanaAdaptiveTrendLines)
            {
                if (cacheanaAdaptiveTrendLines != null)
                    for (int idx = 0; idx < cacheanaAdaptiveTrendLines.Length; idx++)
                        if (cacheanaAdaptiveTrendLines[idx].EqualsInput(input))
                            return cacheanaAdaptiveTrendLines[idx];

                anaAdaptiveTrendLines indicator = new anaAdaptiveTrendLines();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                Indicators.Add(indicator);
                indicator.SetUp();

                anaAdaptiveTrendLines[] tmp = new anaAdaptiveTrendLines[cacheanaAdaptiveTrendLines == null ? 1 : cacheanaAdaptiveTrendLines.Length + 1];
                if (cacheanaAdaptiveTrendLines != null)
                    cacheanaAdaptiveTrendLines.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheanaAdaptiveTrendLines = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// The Adaptice Trendlines include four digital FIR filters - FATL, SATL, RFTL and RSTL -, which were obtained by employing Maximum Entropy Spectral Analysis.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.anaAdaptiveTrendLines anaAdaptiveTrendLines()
        {
            return _indicator.anaAdaptiveTrendLines(Input);
        }

        /// <summary>
        /// The Adaptice Trendlines include four digital FIR filters - FATL, SATL, RFTL and RSTL -, which were obtained by employing Maximum Entropy Spectral Analysis.
        /// </summary>
        /// <returns></returns>
        public Indicator.anaAdaptiveTrendLines anaAdaptiveTrendLines(Data.IDataSeries input)
        {
            return _indicator.anaAdaptiveTrendLines(input);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// The Adaptice Trendlines include four digital FIR filters - FATL, SATL, RFTL and RSTL -, which were obtained by employing Maximum Entropy Spectral Analysis.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.anaAdaptiveTrendLines anaAdaptiveTrendLines()
        {
            return _indicator.anaAdaptiveTrendLines(Input);
        }

        /// <summary>
        /// The Adaptice Trendlines include four digital FIR filters - FATL, SATL, RFTL and RSTL -, which were obtained by employing Maximum Entropy Spectral Analysis.
        /// </summary>
        /// <returns></returns>
        public Indicator.anaAdaptiveTrendLines anaAdaptiveTrendLines(Data.IDataSeries input)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.anaAdaptiveTrendLines(input);
        }
    }
}
#endregion
