// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// Exponential Moving Average. The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The anaFTLMSTLM applies more weight to recent prices than the SMA.
	/// </summary>
	[Description("The Fast Trendline Momentum (FTLM) and Slow Trendline Momentum (STLM) are MACD type indicators that are calculated from Digital FIR filters.")]
	public class anaFTLMSTLM : Indicator
	{
		#region Variables
		private double		fastTrendlineMomentum	= 0.0;
		private double 		slowTrendlineMomentum 	= 0.0;
		private bool		showDots				= true;
		private int 		plot0Width 				= 1;
		private PlotStyle 	plot0Style				= PlotStyle.Dot;
		private DashStyle 	dash0Style				= DashStyle.Dot;
		private int 		plot1Width 				= 1;
		private PlotStyle 	plot1Style				= PlotStyle.Line;
		private DashStyle	dash1Style				= DashStyle.Solid;
		private int 		plot2Width 				= 3;
		private PlotStyle 	plot2Style				= PlotStyle.Bar;
		private DashStyle	dash2Style				= DashStyle.Solid;
		private Color 		ftlmColor				= Color.Yellow;
		private Color 		stlmColor				= Color.CornflowerBlue;
		
		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Plot(Color.Gray, PlotStyle.Dot, "FTLMDots"));
			Add(new Plot(Color.Gray, PlotStyle.Line, "FTLM"));
			Add(new Plot(Color.Gray, PlotStyle.Line, "STLM"));
			Add(new Line(Color.Gray, 0, "Zero line"));
			Overlay				= false;
			PlotsConfigurable 	= false;
		}
		
		protected override void OnStartUp()
		{
			Plots[0].Pen.Color = FTLMColor;
			Plots[1].Pen.Color = FTLMColor;
			Plots[2].Pen.Color = STLMColor;
			Plots[0].Pen.Width = plot0Width;
			Plots[0].PlotStyle = plot0Style;
			Plots[0].Pen.DashStyle = dash0Style;
			Plots[1].Pen.Width = plot1Width;
			Plots[1].PlotStyle = plot1Style;
			Plots[1].Pen.DashStyle = dash1Style;
			Plots[2].Pen.Width = plot2Width;
			Plots[2].PlotStyle = plot2Style;
			Plots[2].Pen.DashStyle = dash2Style;
		}
			
			
		/// <summary>
		/// Called on each bar update event (incoming tick)
		/// </summary>
		protected override void OnBarUpdate()
		{
			if (CurrentBar < 90 )
				return;
			fastTrendlineMomentum = anaAdaptiveTrendLines().FATL[0] - anaAdaptiveTrendLines().RFTL[0]; 
			slowTrendlineMomentum = anaAdaptiveTrendLines().SATL[0] - anaAdaptiveTrendLines().RSTL[0]; 
			if (showDots)
				FTLMDots.Set(fastTrendlineMomentum);
			else
				FTLMDots.Reset();
			FTLM.Set(fastTrendlineMomentum);
			STLM.Set(slowTrendlineMomentum);
			
		}

		#region Properties
		[Description("Show FTLM and STLM Dots")]
        [Category("Plots")]
		[Gui.Design.DisplayName ("Show Dots")]
        public bool ShowDots
        {
            get { return showDots; }
            set { showDots = value; }
        }

		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries FTLMDots
		{
			get { return Values[0]; }
		}
		
		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries FTLM
		{
			get { return Values[1]; }
		}

		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries STLM
		{
			get { return Values[2]; }
		}
		
		/// <summary>
		/// </summary>
        [XmlIgnore()]		
		[Description("Select color for FTLM")]
		[Category("Plots")]
		[Gui.Design.DisplayName("Color FTLM")]
		public Color FTLMColor
		{
			get { return ftlmColor; }
			set { ftlmColor = value; }
		}
		
		// Serialize Color object
		[Browsable(false)]
		public string FTLMColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(ftlmColor); }
			set { ftlmColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		/// <summary>
		/// </summary>
        [XmlIgnore()]		
		[Description("Select color for STLM")]
		[Category("Plots")]
		[Gui.Design.DisplayName("Color STLM")]
		public Color STLMColor
		{
			get { return stlmColor; }
			set { stlmColor = value; }
		}
		
		// Serialize Color object
		[Browsable(false)]
		public string STLMColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(stlmColor); }
			set { stlmColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		/// <summary>
		/// </summary>
		[Description("Width for FTLM Dots.")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Width FTLM Dots")]
		public int Plot0Width
		{
			get { return plot0Width; }
			set { plot0Width = Math.Max(1, value); }
		}
		
		/// <summary>
		/// </summary>
		[Description("Width for FTLM")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Width FTLM")]
		public int Plot1Width
		{
			get { return plot1Width; }
			set { plot1Width = Math.Max(1, value); }
		}
		
		/// <summary>
		/// </summary>
		[Description("PlotStyle for FTLM")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Plot Style FTLM")]
		public PlotStyle Plot1Style
		{
			get { return plot1Style; }
			set { plot1Style = value; }
		}
		
		/// <summary>
		/// </summary>
		[Description("DashStyle for FTLM")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Dash Style FTLM")]
		public DashStyle Dash1Style
		{
			get { return dash1Style; }
			set { dash1Style = value; }
		} 
		
		/// <summary>
		/// </summary>
		[Description("Width for STLM")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Width STLM")]
		public int Plot2Width
		{
			get { return plot2Width; }
			set { plot2Width = Math.Max(1, value); }
		}
		
		/// <summary>
		/// </summary>
		[Description("PlotStyle for STLM")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Plot Style STLM")]
		public PlotStyle Plot2Style
		{
			get { return plot2Style; }
			set { plot2Style = value; }
		}
		
		/// <summary>
		/// </summary>
		[Description("DashStyle for STLM")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Dash Style STLM")]
		public DashStyle Dash2Style
		{
			get { return dash2Style; }
			set { dash2Style = value; }
		} 
		#endregion
		
		#region Miscellaneous

		public override string FormatPriceMarker(double price)
		{
			double trunc = Math.Truncate(price);
			int fraction = Convert.ToInt32(320 * Math.Abs(price - trunc) - 0.0001); // rounding down for ZF and ZT
			string priceMarker = "";
			if (TickSize == 0.03125) 
			{
				fraction = fraction/10;
				if (fraction < 10)
					priceMarker = trunc.ToString() + "'0" + fraction.ToString();
				else 
					priceMarker = trunc.ToString() + "'" + fraction.ToString();
			}
			else if (TickSize == 0.015625 || TickSize == 0.0078125)
			{
				if (fraction < 10)
					priceMarker = trunc.ToString() + "'00" + fraction.ToString();
				else if (fraction < 100)
					priceMarker = trunc.ToString() + "'0" + fraction.ToString();
				else	
					priceMarker = trunc.ToString() + "'" + fraction.ToString();
			}
			else
				priceMarker = price.ToString(Gui.Globals.GetTickFormatString(TickSize));
			return priceMarker;
		}		
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private anaFTLMSTLM[] cacheanaFTLMSTLM = null;

        private static anaFTLMSTLM checkanaFTLMSTLM = new anaFTLMSTLM();

        /// <summary>
        /// The Fast Trendline Momentum (FTLM) and Slow Trendline Momentum (STLM) are MACD type indicators that are calculated from Digital FIR filters.
        /// </summary>
        /// <returns></returns>
        public anaFTLMSTLM anaFTLMSTLM()
        {
            return anaFTLMSTLM(Input);
        }

        /// <summary>
        /// The Fast Trendline Momentum (FTLM) and Slow Trendline Momentum (STLM) are MACD type indicators that are calculated from Digital FIR filters.
        /// </summary>
        /// <returns></returns>
        public anaFTLMSTLM anaFTLMSTLM(Data.IDataSeries input)
        {
            if (cacheanaFTLMSTLM != null)
                for (int idx = 0; idx < cacheanaFTLMSTLM.Length; idx++)
                    if (cacheanaFTLMSTLM[idx].EqualsInput(input))
                        return cacheanaFTLMSTLM[idx];

            lock (checkanaFTLMSTLM)
            {
                if (cacheanaFTLMSTLM != null)
                    for (int idx = 0; idx < cacheanaFTLMSTLM.Length; idx++)
                        if (cacheanaFTLMSTLM[idx].EqualsInput(input))
                            return cacheanaFTLMSTLM[idx];

                anaFTLMSTLM indicator = new anaFTLMSTLM();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                Indicators.Add(indicator);
                indicator.SetUp();

                anaFTLMSTLM[] tmp = new anaFTLMSTLM[cacheanaFTLMSTLM == null ? 1 : cacheanaFTLMSTLM.Length + 1];
                if (cacheanaFTLMSTLM != null)
                    cacheanaFTLMSTLM.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheanaFTLMSTLM = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// The Fast Trendline Momentum (FTLM) and Slow Trendline Momentum (STLM) are MACD type indicators that are calculated from Digital FIR filters.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.anaFTLMSTLM anaFTLMSTLM()
        {
            return _indicator.anaFTLMSTLM(Input);
        }

        /// <summary>
        /// The Fast Trendline Momentum (FTLM) and Slow Trendline Momentum (STLM) are MACD type indicators that are calculated from Digital FIR filters.
        /// </summary>
        /// <returns></returns>
        public Indicator.anaFTLMSTLM anaFTLMSTLM(Data.IDataSeries input)
        {
            return _indicator.anaFTLMSTLM(input);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// The Fast Trendline Momentum (FTLM) and Slow Trendline Momentum (STLM) are MACD type indicators that are calculated from Digital FIR filters.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.anaFTLMSTLM anaFTLMSTLM()
        {
            return _indicator.anaFTLMSTLM(Input);
        }

        /// <summary>
        /// The Fast Trendline Momentum (FTLM) and Slow Trendline Momentum (STLM) are MACD type indicators that are calculated from Digital FIR filters.
        /// </summary>
        /// <returns></returns>
        public Indicator.anaFTLMSTLM anaFTLMSTLM(Data.IDataSeries input)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.anaFTLMSTLM(input);
        }
    }
}
#endregion
