// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// Exponential Moving Average. The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The anaPCCI applies more weight to recent prices than the SMA.
	/// </summary>
	[Description("The Perfect Commodity Channel Index measures the distance of the Clocse from a digital FIR filter.")]
	public class anaPCCI : Indicator
	{
		#region Variables
		private double		perfectChannelIndex		= 0.0;
		private bool		showDots				= true;
		private int 		plot0Width 				= 1;
		private PlotStyle 	plot0Style				= PlotStyle.Line;
		private DashStyle	dash0Style				= DashStyle.Solid;
		private int 		plot1Width 				= 1;
		private PlotStyle 	plot1Style				= PlotStyle.Dot;
		private DashStyle 	dash1Style				= DashStyle.Dot;
		private Color 		pcciColor				= Color.OrangeRed;
		
		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Plot(Color.Gray, PlotStyle.Line, "PCCI"));
			Add(new Plot(Color.Gray, PlotStyle.Dot, "PCCIDots"));
			Add(new Line(Color.Gray, 0, "Zero line"));
			Overlay				= false;
			PlotsConfigurable 	= false;
		}
		
		protected override void OnStartUp()
		{
			Plots[0].Pen.Color = PCCIColor;
			Plots[1].Pen.Color = PCCIColor;
			Plots[0].Pen.Width = plot0Width;
			Plots[0].PlotStyle = plot0Style;
			Plots[0].Pen.DashStyle = dash0Style;
			Plots[1].Pen.Width = plot1Width;
			Plots[1].PlotStyle = plot1Style;
			Plots[1].Pen.DashStyle = dash1Style;
		}
			
			
		/// <summary>
		/// Called on each bar update event (incoming tick)
		/// </summary>
		protected override void OnBarUpdate()
		{
			if (CurrentBar < 90 )
				return;
			perfectChannelIndex = Close[0] - anaAdaptiveTrendLines().FATL[0]; 
			PCCI.Set(perfectChannelIndex);

			if (showDots)
				PCCIDots.Set(perfectChannelIndex);
			else
				PCCIDots.Reset();
		}

		#region Properties
		[Description("Show FTLM and STLM Dots")]
        [Category("Plots")]
		[Gui.Design.DisplayName ("Show Dots")]
        public bool ShowDots
        {
            get { return showDots; }
            set { showDots = value; }
        }

		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries PCCI
		{
			get { return Values[0]; }
		}
		
		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries PCCIDots
		{
			get { return Values[1]; }
		}

		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Upper1SD
		{
			get { return Values[2]; }
		}
		
		/// <summary>
		/// </summary>
        [XmlIgnore()]		
		[Description("Select color for PCCI")]
		[Category("Plots")]
		[Gui.Design.DisplayName("Color PCCI")]
		public Color PCCIColor
		{
			get { return pcciColor; }
			set { pcciColor = value; }
		}
		
		// Serialize Color object
		[Browsable(false)]
		public string PCCIColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(pcciColor); }
			set { pcciColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		/// <summary>
		/// </summary>
		[Description("Width for PCCI")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Width PCCI")]
		public int Plot0Width
		{
			get { return plot0Width; }
			set { plot0Width = Math.Max(1, value); }
		}
		
		/// <summary>
		/// </summary>
		[Description("PlotStyle for PCCI")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Plot Style PCCI")]
		public PlotStyle Plot0Style
		{
			get { return plot0Style; }
			set { plot0Style = value; }
		}
		
		/// <summary>
		/// </summary>
		[Description("DashStyle for PCCI")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Dash Style PCCI")]
		public DashStyle Dash0Style
		{
			get { return dash0Style; }
			set { dash0Style = value; }
		} 
		
		/// <summary>
		/// </summary>
		[Description("Width for PCCIDots")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Width PCCIDots")]
		public int Plot1Width
		{
			get { return plot1Width; }
			set { plot1Width = Math.Max(1, value); }
		}
		#endregion
		
		#region Miscellaneous

		public override string FormatPriceMarker(double price)
		{
			double trunc = Math.Truncate(price);
			int fraction = Convert.ToInt32(320 * Math.Abs(price - trunc) - 0.0001); // rounding down for ZF and ZT
			string priceMarker = "";
			if (TickSize == 0.03125) 
			{
				fraction = fraction/10;
				if (fraction < 10)
					priceMarker = trunc.ToString() + "'0" + fraction.ToString();
				else 
					priceMarker = trunc.ToString() + "'" + fraction.ToString();
			}
			else if (TickSize == 0.015625 || TickSize == 0.0078125)
			{
				if (fraction < 10)
					priceMarker = trunc.ToString() + "'00" + fraction.ToString();
				else if (fraction < 100)
					priceMarker = trunc.ToString() + "'0" + fraction.ToString();
				else	
					priceMarker = trunc.ToString() + "'" + fraction.ToString();
			}
			else
				priceMarker = price.ToString(Gui.Globals.GetTickFormatString(TickSize));
			return priceMarker;
		}		
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private anaPCCI[] cacheanaPCCI = null;

        private static anaPCCI checkanaPCCI = new anaPCCI();

        /// <summary>
        /// The Perfect Commodity Channel Index measures the distance of the Clocse from a digital FIR filter.
        /// </summary>
        /// <returns></returns>
        public anaPCCI anaPCCI()
        {
            return anaPCCI(Input);
        }

        /// <summary>
        /// The Perfect Commodity Channel Index measures the distance of the Clocse from a digital FIR filter.
        /// </summary>
        /// <returns></returns>
        public anaPCCI anaPCCI(Data.IDataSeries input)
        {
            if (cacheanaPCCI != null)
                for (int idx = 0; idx < cacheanaPCCI.Length; idx++)
                    if (cacheanaPCCI[idx].EqualsInput(input))
                        return cacheanaPCCI[idx];

            lock (checkanaPCCI)
            {
                if (cacheanaPCCI != null)
                    for (int idx = 0; idx < cacheanaPCCI.Length; idx++)
                        if (cacheanaPCCI[idx].EqualsInput(input))
                            return cacheanaPCCI[idx];

                anaPCCI indicator = new anaPCCI();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                Indicators.Add(indicator);
                indicator.SetUp();

                anaPCCI[] tmp = new anaPCCI[cacheanaPCCI == null ? 1 : cacheanaPCCI.Length + 1];
                if (cacheanaPCCI != null)
                    cacheanaPCCI.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheanaPCCI = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// The Perfect Commodity Channel Index measures the distance of the Clocse from a digital FIR filter.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.anaPCCI anaPCCI()
        {
            return _indicator.anaPCCI(Input);
        }

        /// <summary>
        /// The Perfect Commodity Channel Index measures the distance of the Clocse from a digital FIR filter.
        /// </summary>
        /// <returns></returns>
        public Indicator.anaPCCI anaPCCI(Data.IDataSeries input)
        {
            return _indicator.anaPCCI(input);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// The Perfect Commodity Channel Index measures the distance of the Clocse from a digital FIR filter.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.anaPCCI anaPCCI()
        {
            return _indicator.anaPCCI(Input);
        }

        /// <summary>
        /// The Perfect Commodity Channel Index measures the distance of the Clocse from a digital FIR filter.
        /// </summary>
        /// <returns></returns>
        public Indicator.anaPCCI anaPCCI(Data.IDataSeries input)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.anaPCCI(input);
        }
    }
}
#endregion
