// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// Exponential Moving Average. The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The anaRBCI applies more weight to recent prices than the SMA.
	/// </summary>
	[Description("The Range Bound Channel Index is an overall sum of market cycles and is calculated from two digital FIR filters.")]
	public class anaRBCI : Indicator
	{
		#region Variables
		private double		rangeBoundChannelIndex	= 0.0;
		private bool		showDots				= true;
		private int 		plot0Width 				= 1;
		private PlotStyle 	plot0Style				= PlotStyle.Line;
		private DashStyle	dash0Style				= DashStyle.Solid;
		private int 		plot1Width 				= 1;
		private PlotStyle 	plot1Style				= PlotStyle.Line;
		private DashStyle	dash1Style				= DashStyle.Dash;
		private int 		plot5Width 				= 1;
		private PlotStyle 	plot5Style				= PlotStyle.Dot;
		private DashStyle 	dash5Style				= DashStyle.Dot;
		private Color 		rbciColor				= Color.LimeGreen;
		private Color 		bandColor				= Color.LightSteelBlue;
		
		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Plot(Color.Gray, PlotStyle.Line, "RBCI"));
			Add(new Plot(Color.Gray, PlotStyle.Line, "Upper1SD"));
			Add(new Plot(Color.Gray, PlotStyle.Line, "Upper2SD"));
			Add(new Plot(Color.Gray, PlotStyle.Line, "Lower1SD"));
			Add(new Plot(Color.Gray, PlotStyle.Line, "Lower2SD"));
			Add(new Plot(Color.Gray, PlotStyle.Dot, "RBCIDots"));
			Overlay				= false;
			PlotsConfigurable 	= false;
		}
		
		protected override void OnStartUp()
		{
			Plots[0].Pen.Color = RBCIColor;
			Plots[1].Pen.Color = BandColor;
			Plots[2].Pen.Color = BandColor;
			Plots[3].Pen.Color = BandColor;
			Plots[4].Pen.Color = BandColor;
			Plots[5].Pen.Color = RBCIColor;
			Plots[0].Pen.Width = plot0Width;
			Plots[0].PlotStyle = plot0Style;
			Plots[0].Pen.DashStyle = dash0Style;
			Plots[1].Pen.Width = plot1Width;
			Plots[1].PlotStyle = plot1Style;
			Plots[1].Pen.DashStyle = dash1Style;
			Plots[2].Pen.Width = plot1Width;
			Plots[2].PlotStyle = plot1Style;
			Plots[2].Pen.DashStyle = dash1Style;
			Plots[3].Pen.Width = plot1Width;
			Plots[3].PlotStyle = plot1Style;
			Plots[3].Pen.DashStyle = dash1Style;
			Plots[4].Pen.Width = plot1Width;
			Plots[4].PlotStyle = plot1Style;
			Plots[4].Pen.DashStyle = dash1Style;
			Plots[5].Pen.Width = plot5Width;
			Plots[5].PlotStyle = plot5Style;
			Plots[5].Pen.DashStyle = dash5Style;
		}
			
			
		/// <summary>
		/// Called on each bar update event (incoming tick)
		/// </summary>
		protected override void OnBarUpdate()
		{
			if (CurrentBar < 90 )
				return;
			rangeBoundChannelIndex = anaAdaptiveTrendLines().FATL[0] - anaAdaptiveTrendLines().SATL[0]; 
			RBCI.Set(rangeBoundChannelIndex);
			double stdDev = StdDev(RBCI,54)[0];
			Upper2SD.Set(2*stdDev);
			Upper1SD.Set(stdDev);
			Lower1SD.Set(-stdDev);
			Lower2SD.Set(-2*stdDev);

			if (showDots)
				RBCIDots.Set(rangeBoundChannelIndex);
			else
				RBCIDots.Reset();
		}

		#region Properties
		[Description("Show FTLM and STLM Dots")]
        [Category("Plots")]
		[Gui.Design.DisplayName ("Show Dots")]
        public bool ShowDots
        {
            get { return showDots; }
            set { showDots = value; }
        }

		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries RBCI
		{
			get { return Values[0]; }
		}
		
		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Upper2SD
		{
			get { return Values[1]; }
		}

		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Upper1SD
		{
			get { return Values[2]; }
		}
		
		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Lower1SD
		{
			get { return Values[3]; }
		}

		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Lower2SD
		{
			get { return Values[4]; }
		}
		
		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries RBCIDots
		{
			get { return Values[5]; }
		}
		
		/// <summary>
		/// </summary>
        [XmlIgnore()]		
		[Description("Select color for RBCI")]
		[Category("Plots")]
		[Gui.Design.DisplayName("Color RBCI")]
		public Color RBCIColor
		{
			get { return rbciColor; }
			set { rbciColor = value; }
		}
		
		// Serialize Color object
		[Browsable(false)]
		public string RBCIColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(rbciColor); }
			set { rbciColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		/// <summary>
		/// </summary>
        [XmlIgnore()]		
		[Description("Select color for Bands")]
		[Category("Plots")]
		[Gui.Design.DisplayName("Color Bands")]
		public Color BandColor
		{
			get { return bandColor; }
			set { bandColor = value; }
		}
		
		// Serialize Color object
		[Browsable(false)]
		public string BandColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(bandColor); }
			set { bandColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		/// <summary>
		/// </summary>
		[Description("Width for RBCI")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Width RBCI")]
		public int Plot0Width
		{
			get { return plot0Width; }
			set { plot0Width = Math.Max(1, value); }
		}
		
		/// <summary>
		/// </summary>
		[Description("PlotStyle for RBCI")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Plot Style RBCI")]
		public PlotStyle Plot0Style
		{
			get { return plot0Style; }
			set { plot0Style = value; }
		}
		
		/// <summary>
		/// </summary>
		[Description("DashStyle for RBCI")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Dash Style RBCI")]
		public DashStyle Dash0Style
		{
			get { return dash0Style; }
			set { dash0Style = value; }
		} 
		
		/// <summary>
		/// </summary>
		[Description("Width for Bands")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Width Bands")]
		public int Plot1Width
		{
			get { return plot1Width; }
			set { plot1Width = Math.Max(1, value); }
		}
		
		/// <summary>
		/// </summary>
		[Description("PlotStyle for Bands")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Plot Style Bands")]
		public PlotStyle Plot1Style
		{
			get { return plot1Style; }
			set { plot1Style = value; }
		}
		
		/// <summary>
		/// </summary>
		[Description("DashStyle for Bands")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Dash Style Bands")]
		public DashStyle Dash1Style
		{
			get { return dash1Style; }
			set { dash1Style = value; }
		} 
		
		/// <summary>
		/// </summary>
		[Description("Width for RBCI Dots.")]
		[Category("Plots")]
		[Gui.Design.DisplayNameAttribute("Width RBCI Dots")]
		public int Plot5Width
		{
			get { return plot5Width; }
			set { plot5Width = Math.Max(1, value); }
		}
		#endregion
		
		#region Miscellaneous

		public override string FormatPriceMarker(double price)
		{
			double trunc = Math.Truncate(price);
			int fraction = Convert.ToInt32(320 * Math.Abs(price - trunc) - 0.0001); // rounding down for ZF and ZT
			string priceMarker = "";
			if (TickSize == 0.03125) 
			{
				fraction = fraction/10;
				if (fraction < 10)
					priceMarker = trunc.ToString() + "'0" + fraction.ToString();
				else 
					priceMarker = trunc.ToString() + "'" + fraction.ToString();
			}
			else if (TickSize == 0.015625 || TickSize == 0.0078125)
			{
				if (fraction < 10)
					priceMarker = trunc.ToString() + "'00" + fraction.ToString();
				else if (fraction < 100)
					priceMarker = trunc.ToString() + "'0" + fraction.ToString();
				else	
					priceMarker = trunc.ToString() + "'" + fraction.ToString();
			}
			else
				priceMarker = price.ToString(Gui.Globals.GetTickFormatString(TickSize));
			return priceMarker;
		}		
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private anaRBCI[] cacheanaRBCI = null;

        private static anaRBCI checkanaRBCI = new anaRBCI();

        /// <summary>
        /// The Range Bound Channel Index is an overall sum of market cycles and is calculated from two digital FIR filters.
        /// </summary>
        /// <returns></returns>
        public anaRBCI anaRBCI()
        {
            return anaRBCI(Input);
        }

        /// <summary>
        /// The Range Bound Channel Index is an overall sum of market cycles and is calculated from two digital FIR filters.
        /// </summary>
        /// <returns></returns>
        public anaRBCI anaRBCI(Data.IDataSeries input)
        {
            if (cacheanaRBCI != null)
                for (int idx = 0; idx < cacheanaRBCI.Length; idx++)
                    if (cacheanaRBCI[idx].EqualsInput(input))
                        return cacheanaRBCI[idx];

            lock (checkanaRBCI)
            {
                if (cacheanaRBCI != null)
                    for (int idx = 0; idx < cacheanaRBCI.Length; idx++)
                        if (cacheanaRBCI[idx].EqualsInput(input))
                            return cacheanaRBCI[idx];

                anaRBCI indicator = new anaRBCI();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                Indicators.Add(indicator);
                indicator.SetUp();

                anaRBCI[] tmp = new anaRBCI[cacheanaRBCI == null ? 1 : cacheanaRBCI.Length + 1];
                if (cacheanaRBCI != null)
                    cacheanaRBCI.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheanaRBCI = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// The Range Bound Channel Index is an overall sum of market cycles and is calculated from two digital FIR filters.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.anaRBCI anaRBCI()
        {
            return _indicator.anaRBCI(Input);
        }

        /// <summary>
        /// The Range Bound Channel Index is an overall sum of market cycles and is calculated from two digital FIR filters.
        /// </summary>
        /// <returns></returns>
        public Indicator.anaRBCI anaRBCI(Data.IDataSeries input)
        {
            return _indicator.anaRBCI(input);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// The Range Bound Channel Index is an overall sum of market cycles and is calculated from two digital FIR filters.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.anaRBCI anaRBCI()
        {
            return _indicator.anaRBCI(Input);
        }

        /// <summary>
        /// The Range Bound Channel Index is an overall sum of market cycles and is calculated from two digital FIR filters.
        /// </summary>
        /// <returns></returns>
        public Indicator.anaRBCI anaRBCI(Data.IDataSeries input)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.anaRBCI(input);
        }
    }
}
#endregion
