//
// Ratio of 2 Instruments - For NinjaTrader 7 & higher
//
// Calculates ratio of synthetic combination of 2 instruments
// Parameters are symbol of secondary instrument, 
// and true/false switch to "normalize" ratio in range of 0 to 1 (use ratio of Instrument1/(Instrument1+Instrument2)
// PointValues from the instrument manager are optionally used as multipliers.
//
// Kevin Doren  11/15/2009
//				1/11/2010:  Use synchronization methods; "Symbol2" now defaults to secondary chart instrument
//				1/28/2010:  Put in check to not reference Instrument2 until OnBarUpdate has been called for it at least once. (to compensate for possible NT7B7 bug)
//				2/5/2010:   Fixed synchronization problem with historical bars and CalculateOnBarClose=false
//				3/25/2010:  Fix possible exception when secondary instrument runs out of data; Format output to 4 decimal places; Force sync with Weekly/Monthly/Yearly bars as well as Daily

#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Ratio of Two Instruments
    /// </summary>
    [Description("Ratio of Two Instruments")]
    public class Ratio : Indicator
    {
		private const int Instrument1=0, Instrument2=1;
		
        #region Variables
            private string symbol2 = "";
			private bool normalize = false; // true for normalized ratio ->  P1/(P1 + P2)
			private bool useMultiplier = true;	//  Default parameter value; true = use contract multiplier in spread calculation
			private SyncedDataSeries Instrument2Close;
			private double mul1, mul2;		// storage for instrument multiplier values so we don't have to keep looking them up.
        #endregion		
		
		#region SynchronizationMethods
/*
  Synchronization of Multi-Instrument Indicators & Strategies - For NinjaTrader 7
  Kevin Doren - Version 3/25/2010

  Multi-instrument indicators (and strategies, for the same concepts apply) by their nature are looking
  at relationships between instruments.  Any calculations we do need to be time-synchronized - we want
  to be using data that occurred at the same time.  For certain kinds of analysis, such as intraday
  analysis of heavily-traded instruments during regular trading hours on historical data, the techniques
  shown here probably aren't necessary.  But for real time analysis, especially with instruments that
  might have missing bars, these techniques are necessary.

  There are 3 scenarios: Historical, real-time with (CalculateOnBarClose == true), 
  and real-time with (CalculateOnBarClose == false).  Each has its issues.

  The simplest case is Historical.  In this case, NT7 will synchronize the bars, which means that when
  OnBarUpdate is called for a Bar Series with a particular timestamp, any other Bar Series which
  has a bar with the same timestamp will have that bar set to be the current bar.  This means that
  Closes[0][0] and Closes[1][0] will both point to bars with the same timestamp IF Closes[1] has a
  bar with that timestamp.  If not, Close[1][0] will point to the most recent bar preceding it.
  So even Close[0][0] and Close[1][0] are not necessarily synchronized.  As you look back in time,
  it can get even more out of sync, because bars might be missing from either series.

  For an indicator like correlation, the calculation makes no sense unless the bars are all synchronized.
  The solution implemented here is to create 2 additional Data Series, each synchronized to the Primary 
  Instrument, because the indicator output is synced to the Primary Instrument.
  The first Data Series (Instrument2Close) is a copy of the secondary instrument, Closes[1].
  The second Data Series is a BoolSeries (Instrument2Synced) which flags the bars that are synced.

  When operating with Historical data, if a synchronized bar exists, it will be Closes[1][0].
  In real time, with (CalculateOnBarClose==false), it gets more complicated, because NT7 is event-driven,
  which mean that bars don't close because of time.  They close because a tick arrives with a timestamp of
  a later bar, at which time OnBarUpdate is called and the new bar starts forming.  This could take a long time.
  So when we are processing a bar for the primary instrument, we need to also check the unclosed bar of the secondary
  instrument, which we can do by referencing Times[1][-1]; this is safe to do because we know the bar's time
  has expired - we are running OnBarUpdate for the primary instrument for the same timestamp, which can
  only happen because a tick arrived for a later bar.

  If we use the unclosed bar (Closes[1][-1]), there is a very small chance that a late-arriving tick will change
  the value after we use it.  We can't change that calculation, but we can fix the value we stored earlier.
  This is done in OnBarUpdate for the secondary instrument.

  Also, in real time, it's possible that the secondary instrument has received bars after the time of the
  bar we are currently processing.  A synchronized bar could exist at Closes[1][1], Closes[1][2], or earlier.
  So we need to scan back through the seconday instrument's closes to see if the Synced bar is there.
  
  The Data Series "Instrument2Close" is filled from OnBarUpdate for the primary instrument, and it will hold the
  synchronized bars we found using the above techniques.  We set a flag in "Instrument2Synced" to show which
  locations are synced.  The routines "KDCorrelation" and "KDSMABool" are written to calculate correlation and SMA
  using only synced data.

  There is an additional issue when using daily bars.  As of this writing, daily bars carry a timestamp of the end of the
  NT7 session template.  This can change depending on which session template is applied to the instrument, either in the
  instrument manager or in the chart properties.  There is no issue if both instruments have the same session template,
  then their daily bars will have the same timestamp and thus be synchronized.  But if the secondary instrument has a timestamp
  later than the primary instrument, the days will be out of sync unless we strip away the times and just use the dates.
*/		
		private class SyncedDataSeries : DataSeries
		{
			public IndicatorBase IndBase;		// Base of instantiating indicator
			public int PrimaryInstrument;		// BarsArray Index of primary instrument (usually 0 in indicators)
												//   but could be higher than 0 for multi-instrument, multi-timeframe indicators
			public int SyncedInstrument;		// BarsArray Index of secondary instrument we want to synchronize to primary instrument
			public IDataSeries DataArray;		// Output Data Series synced to primary instrument, hold synchronized copy of secondary instrument
			public BoolSeries Synced;			// BoolSeries output, synced with primary instrument, holds flags
												//   true=output (DataArray) hold synchronized value, false = output does not hold synchronized value
			public int SynchronizedBarCount;	// Total number of synchronized bars held in output
												//   but some or all may be inaccessible if MaximumBarsLookBack is set to 256
			public bool DataStarted;			// True if we have data has started coming from BOTH instruments
			private bool dailyBars;
			private bool initializing;
			private bool lastSyncWasToUnclosedBar;
			private int lastCurrentBar;			//  needed to work around NT7 bug - FirstTickOfBar is broken in NT7B7
		
			public SyncedDataSeries (IndicatorBase indicator, int primaryInstrument, int syncedInstrument, IDataSeries dataArray) : base (indicator)
			{
				IndBase = indicator;
				PrimaryInstrument = primaryInstrument;
				SyncedInstrument = syncedInstrument;
				SynchronizedBarCount = 0;
				DataArray = dataArray;
				DataStarted = false;
				initializing = true;
				lastSyncWasToUnclosedBar = false;
				lastCurrentBar = -1;
				
				if (PrimaryInstrument == 0)
					Synced = new BoolSeries(IndBase);  // We are syncing to the primary instrument of the instantiating indicator
				else
					throw new ArgumentOutOfRangeException ("primaryInstrument", "primaryInstrument must = 0 if no syncedIndicator base is given.");
			}

			public SyncedDataSeries (IndicatorBase indicator, int primaryInstrument, int syncedInstrument, IDataSeries dataArray, IndicatorBase syncedIndicator) : base (syncedIndicator)

			{	
				IndBase = indicator;
				PrimaryInstrument = primaryInstrument;
				SyncedInstrument = syncedInstrument;
				SynchronizedBarCount = 0;
				DataArray = dataArray;
				DataStarted = false;
				initializing = true;
				lastSyncWasToUnclosedBar = false;
				Synced = new BoolSeries(syncedIndicator);	// Not syncing to primary instrument of instantiating indicator;
															// So create a new BoolSeries synced to the master instrument
			}
			
			public void Synchronize()
			{
				if ((IndBase.BarsInProgress != PrimaryInstrument) && (IndBase.BarsInProgress != SyncedInstrument))
						return;	// Bars being processed are not for us

				if (initializing)
				{
					dailyBars = ((IndBase.BarsPeriods[SyncedInstrument].Id == PeriodType.Day)
								|| (IndBase.BarsPeriods[SyncedInstrument].Id == PeriodType.Week)
								|| (IndBase.BarsPeriods[SyncedInstrument].Id == PeriodType.Month)
								|| (IndBase.BarsPeriods[SyncedInstrument].Id == PeriodType.Year));    // Save a flag which tells us if Bars are Daily
					initializing = false;
				}
				
				if (IndBase.CalculateOnBarClose == true)
				{
					if (IndBase.BarsInProgress == PrimaryInstrument) // Primary Instrument
					{
						if (!DataStarted)
							if (IndBase.BarsArray[SyncedInstrument].CurrentBar == -1)
							{
								Synced.Set(false);
								return;						// return if no data yet from synced instrument
							}
							else
								DataStarted = true;
							
						//	Scan through SyncedInstrument's bars, try to find one in sync with the current Primary instrument bar
						//	Start the scan with the unclosed bar [-1] (necessary in real time); if it has the same time as the Primary instrument's Time[0] (which is now closed), it should be done forming
						int barsBack;
						if (IndBase.BarsArray[SyncedInstrument].CurrentBar == IndBase.BarsArray[SyncedInstrument].Count-1)
							barsBack = 0;
						else
							barsBack = -1;
						
						if (dailyBars)  // if Daily bars, need to use just the Day portion
						{
							while ((barsBack < IndBase.BarsArray[SyncedInstrument].CurrentBar) && (IndBase.ToDay(IndBase.Times[PrimaryInstrument][0]) < IndBase.ToDay(IndBase.Times[SyncedInstrument][barsBack])))
								barsBack ++;  // ignore bars that occur in the future (this can happen in real time if primary instrument is missing bars)
							if ((IndBase.ToDay(IndBase.Times[PrimaryInstrument][0]) == IndBase.ToDay(IndBase.Times[SyncedInstrument][barsBack])))
								{	// Found a synchronized bar
									Synced.Set(true);
									SynchronizedBarCount = SynchronizedBarCount + 1;
								}
							else	// No synchronized bar found
								Synced.Set(false);
								
							base.Set(DataArray[barsBack]);	// set output to most recent bar that isn't in the future
						}
								
						else // Not daily bars, use entire timestamp
						{
							while ((barsBack < IndBase.BarsArray[SyncedInstrument].CurrentBar) && (IndBase.Times[PrimaryInstrument][0]) < IndBase.Times[SyncedInstrument][barsBack])
								barsBack ++;  // ignore bars that occur in the future (this can happen in real time if primary instrument is missing bars)
							if ((IndBase.Times[PrimaryInstrument][0] == IndBase.Times[SyncedInstrument][barsBack]))
								{	// Found a synchronized bar
									Synced.Set(true);
									SynchronizedBarCount = SynchronizedBarCount + 1;
								}
							else	// No synchronized bar found
								Synced.Set(false);
								
							base.Set(DataArray[barsBack]);	// set output to most recent bar that isn't in the future
						}
					}
					
					else // (IndBase.BarsInProgress == SyncedInstrument)
					{
						// Normally we don't have to do anything when (BarsInProgress == SyncedInstrument) if (CalculateOnBarClose == true)
						// Method output is synced with, and handled by, primary instrument when (BarsInProgress == PrimaryInstrument)
						//
						// Sometimes in a real-time OnBarUpdate for PrimaryIntrument, we find that SyncedInstrument has an unclosed bar (index of [-1]) with the same timestamp,
						// so we use it, which is safe to do because PrimaryInstrument's bar has already closed, so we know that the end of the bar period has passed.
						//
						// We are here because a SecondaryInstrument bar has been closed (in real time by an incoming tick belonging to a later bar, which could be MUCH later)
						// There is a small risk that the value of an unclosed bar we used earlier has changed since then, due to a late-arriving tick.
						// To cover this case, we store the final value in the proper location (the most recent synced bar)
						//
						if (!DataStarted)
							if (IndBase.BarsArray[PrimaryInstrument].CurrentBar == -1)
								return;							// return if no data yet from Primary instrument
							else
								DataStarted = true;
							
						if (lastSyncWasToUnclosedBar)
						{
							lastSyncWasToUnclosedBar = false;   // if there were any late arriving ticks, they are now included in the bar that just closed
							int barsBack=0;
							while (!Synced[barsBack])			// Scan for most recent synced bar
								barsBack++;
							base.Set(barsBack,DataArray[0]);	// Store the final value, in case it has changed since we stored the value from the unclosed bar
						}
					}
				}
				else  // (CalculateOnBarClose == false)
				{
					if (IndBase.BarsInProgress == PrimaryInstrument) // Primary Instrument
					{
						if (!DataStarted)
							if (IndBase.BarsArray[SyncedInstrument].CurrentBar == -1)
							{
								Synced.Set(false);
//								base.Reset(0);
								return;							// return if no data yet from synced instrument
							}
							else
								DataStarted = true;
								
						if (IndBase.BarsArray[PrimaryInstrument].CurrentBar != lastCurrentBar)
//						if (IndBase.FirstTickOfBar)			//broken in NT7B7
						{	// First tick of bar, need to set state of sync flag
							lastCurrentBar = IndBase.BarsArray[PrimaryInstrument].CurrentBar;
							if ((((!dailyBars) && (IndBase.Times[SyncedInstrument][0] == IndBase.Times[PrimaryInstrument][0]))
								|| (dailyBars && (IndBase.ToDay(IndBase.Times[SyncedInstrument][0]) == IndBase.ToDay(IndBase.Times[PrimaryInstrument][0])))))
							{
								Synced.Set(0,true);
								base.Set(0,DataArray[0]);
								SynchronizedBarCount = SynchronizedBarCount + 1;
							}
							else
								if ((((!dailyBars) && (IndBase.BarsArray[SyncedInstrument].CurrentBar > 0) && (IndBase.Times[SyncedInstrument][1] == IndBase.Times[PrimaryInstrument][0]))
									|| (dailyBars && (IndBase.BarsArray[SyncedInstrument].CurrentBar > 0) && (IndBase.ToDay(IndBase.Times[SyncedInstrument][1]) == IndBase.ToDay(IndBase.Times[PrimaryInstrument][0]))))
									)
								{
									Synced.Set(true);
									base.Set(0,DataArray[1]);
									SynchronizedBarCount = SynchronizedBarCount + 1;
								}
								else
								{
									Synced.Set(false);
									base.Set(0,DataArray[0]);  // store most recent value, in case no updates come for rest of bar
								}
						}
					}
					
					else // (IndBase.BarsInProgress == SyncedInstrument)
					{
						if (!DataStarted)
							if (IndBase.BarsArray[PrimaryInstrument].CurrentBar == -1)
								return;						// return if no data yet from primary instrument
							else
								DataStarted = true;
						
						if (((!dailyBars) && (IndBase.Times[SyncedInstrument][0] == IndBase.Times[PrimaryInstrument][0]))
							|| (dailyBars && (IndBase.ToDay(IndBase.Times[SyncedInstrument][0]) == IndBase.ToDay(IndBase.Times[PrimaryInstrument][0]))))
						{
							if ((!Synced[0]))
							{
								Synced.Set(0,true);
								SynchronizedBarCount = SynchronizedBarCount + 1;
							}
							base.Set(0,DataArray[0]);
						}
					}
				}
			}
		}
		#endregion
		
		private double KDRatio(double price1, double price2, bool normal)
		{
			if (normal)   // Normalized Ratio ->  P1/(P1 + P2)
				if ((price1 + price2) != 0)
					return (price1 / (price1 + price2));
				else
					return (0);
			else          // Standard Ratio -> P1/P2
				if (price2 != 0)
					return (price1 / price2);
				else
					return (0); // Return 0 rather than throw exception
		}	
		
        protected override void Initialize()
        {
			CalculateOnBarClose = false;	// Default to false for real-time display of ratio; set to false here or in indicator properties if not needed
			BarsRequired = 0; 				// OK to plot on first bar
			Add(new Plot(Color.FromKnownColor(KnownColor.ForestGreen), PlotStyle.Line, "Ratio"));
			
			// Add second instrument using "Symbol2" parameter
			// Use same BarsPeriod and Value as Primary instrument
			Add(Symbol2, BarsPeriod.Id, BarsPeriod.Value);
		}

		protected override void OnStartUp()
		{
			Instrument2Close = new SyncedDataSeries(this, Instrument1, Instrument2, Closes[Instrument2]);
			if (UseMultiplier)
			{
				mul1 = Instruments[Instrument1].MasterInstrument.PointValue;
				mul2 = Instruments[Instrument2].MasterInstrument.PointValue;
			}
			else
			{
				mul1 = 1;
				mul2 = 1;
			}
		}

        protected override void OnBarUpdate()
        {
			Instrument2Close.Synchronize();		// Call synchronization method
			if (!Instrument2Close.DataStarted)
				return;							// Return if we haven't seen data from BOTH instruments
			
			if ((Instrument2Close[0] != 0) || (Normalize && (Closes[Instrument1][0] != 0)))  // check for divide by zero
				Value.Set(KDRatio(Closes[Instrument1][0] * mul1, Instrument2Close[0] * mul2, Normalize));
			else	// avoid divide by zero exception
				if ((BarsArray[Instrument1].CurrentBar > 0) && (Value.ContainsValue(1)))		// this code is run when BarsInProgress is both 0 and 1.  Need to check the proper CurrentBar.
					Value.Set(Value[1]); // Repeat last output value
        }
		
		public override string ToString()
		{
			// Pretty up the chart label
			if (Normalize)
				return Name + "(" + Instruments[Instrument1].FullName + "," + Instruments[Instrument2].FullName + ",Norm)";
			else
				return Name + "(" + Instruments[Instrument1].FullName + "," + Instruments[Instrument2].FullName + ")";
		}
		
		public override string FormatPriceMarker(double price)
		{
			// Formats price values to 4 decimal places
			return price.ToString("N4");
		}

        #region Properties

        [Description("Denominator Symbol; i.e. SPY or ES 03-10\nDefault = Secondary chart instrument")]
        [GridCategory("Parameters")]
		// Force this parameter to be displayed first
		[Gui.Design.DisplayName ("\tSymbol 2")]
        public string Symbol2
        {
            get 
			{ 
				if ((symbol2 == "") && (ChartControl != null) && (Instruments != null)) 
					for(int i=0; i<ChartControl.BarsArray.Length; i++)
						if (ChartControl.BarsArray[i].Instrument.FullName != Instruments[0].FullName)
						{
							symbol2 = ChartControl.BarsArray[i].Instrument.FullName;
							break;
						}
				return symbol2; 
			}
            set { symbol2 = value.ToUpper(); }
		}
      
		[Description("False: Ratio = P1/P2\nTrue: Ratio = P1/(P1+P2)")]
        [GridCategory("Parameters")]
        public bool Normalize
        {
            get { return normalize; }
            set { normalize = value; }
        }
		
		[Description("True: Use Price * Contract Multiplier\nFalse: Use Price")]
        [GridCategory("Parameters")]
        public bool UseMultiplier
        {
            get { return useMultiplier; }
            set { useMultiplier = value; }
        }
		
        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private Ratio[] cacheRatio = null;

        private static Ratio checkRatio = new Ratio();

        /// <summary>
        /// Ratio of Two Instruments
        /// </summary>
        /// <returns></returns>
        public Ratio Ratio(bool normalize, string symbol2, bool useMultiplier)
        {
            return Ratio(Input, normalize, symbol2, useMultiplier);
        }

        /// <summary>
        /// Ratio of Two Instruments
        /// </summary>
        /// <returns></returns>
        public Ratio Ratio(Data.IDataSeries input, bool normalize, string symbol2, bool useMultiplier)
        {
            if (cacheRatio != null)
                for (int idx = 0; idx < cacheRatio.Length; idx++)
                    if (cacheRatio[idx].Normalize == normalize && cacheRatio[idx].Symbol2 == symbol2 && cacheRatio[idx].UseMultiplier == useMultiplier && cacheRatio[idx].EqualsInput(input))
                        return cacheRatio[idx];

            lock (checkRatio)
            {
                checkRatio.Normalize = normalize;
                normalize = checkRatio.Normalize;
                checkRatio.Symbol2 = symbol2;
                symbol2 = checkRatio.Symbol2;
                checkRatio.UseMultiplier = useMultiplier;
                useMultiplier = checkRatio.UseMultiplier;

                if (cacheRatio != null)
                    for (int idx = 0; idx < cacheRatio.Length; idx++)
                        if (cacheRatio[idx].Normalize == normalize && cacheRatio[idx].Symbol2 == symbol2 && cacheRatio[idx].UseMultiplier == useMultiplier && cacheRatio[idx].EqualsInput(input))
                            return cacheRatio[idx];

                Ratio indicator = new Ratio();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.Normalize = normalize;
                indicator.Symbol2 = symbol2;
                indicator.UseMultiplier = useMultiplier;
                Indicators.Add(indicator);
                indicator.SetUp();

                Ratio[] tmp = new Ratio[cacheRatio == null ? 1 : cacheRatio.Length + 1];
                if (cacheRatio != null)
                    cacheRatio.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheRatio = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Ratio of Two Instruments
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.Ratio Ratio(bool normalize, string symbol2, bool useMultiplier)
        {
            return _indicator.Ratio(Input, normalize, symbol2, useMultiplier);
        }

        /// <summary>
        /// Ratio of Two Instruments
        /// </summary>
        /// <returns></returns>
        public Indicator.Ratio Ratio(Data.IDataSeries input, bool normalize, string symbol2, bool useMultiplier)
        {
            return _indicator.Ratio(input, normalize, symbol2, useMultiplier);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Ratio of Two Instruments
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.Ratio Ratio(bool normalize, string symbol2, bool useMultiplier)
        {
            return _indicator.Ratio(Input, normalize, symbol2, useMultiplier);
        }

        /// <summary>
        /// Ratio of Two Instruments
        /// </summary>
        /// <returns></returns>
        public Indicator.Ratio Ratio(Data.IDataSeries input, bool normalize, string symbol2, bool useMultiplier)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.Ratio(input, normalize, symbol2, useMultiplier);
        }
    }
}
#endregion
