// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// The Tick Counter displays the current tick count of a bar.
	/// </summary>
	[Description("Displays the range countdown of a bar.")]
	public class RangeSound : Indicator
	{
		#region Variables
		private bool			countDown			= true;
		private bool			isRangeDerivate		= false;
		private bool			rangeDerivateChecked= false;
		private StringFormat	stringFormat		= new StringFormat();
		private SolidBrush		textBrush			= new SolidBrush(Color.Black);
		private Font			textFont			= new Font("Arial", 30);
		private float			textWidth			= 0;
		private float			textHeight			= 0;
		private string			noRangeMessage		= "Range Counter only works on Range bars.";
		private float			noRangeTextWidth	= 0;
		private float			noRangeTextHeight	= 0;
		private string	alertFile			= "Alert2.wav";
		private int		alertInterval		= 2;
		private int		alertTicks			= 1;
		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			ChartOnly			= true;
			Overlay				= true;
			CalculateOnBarClose = false;
		}

		/// <summary>
		/// </summary>
		protected override void OnBarUpdate()
		{
			if (rangeDerivateChecked)
				return;
			if (ChartControl == null || ChartControl.Bars == null || ChartControl.Bars.Length == 0)
				return;

			if (Data.BarsType.GetInstance(ChartControl.Bars[0].Period.Id).BuiltFrom == Data.PeriodType.Tick && ChartControl.Bars[0].Period.ToString().IndexOf("Range") >= 0)
				isRangeDerivate = true;
			rangeDerivateChecked = true;
		}

		public override void Plot(Graphics graphics, Rectangle bounds, double min, double max)
		{
			if (Bars == null)
				return;

			// Recalculate the proper string size should the chart control object font and axis color change
			if (textBrush.Color != ChartControl.AxisColor || textFont != ChartControl.Font)
			{
				textBrush.Color = ChartControl.AxisColor;
				textFont = ChartControl.Font;

				SizeF size = graphics.MeasureString((CountDown ? "Range remaining = %" : "Range count = %") + Bars.Period.Value, textFont);
				textWidth		= size.Width + 5;
				textHeight		= size.Height + 5;

				SizeF noTickSize = graphics.MeasureString(noRangeMessage, textFont);
				noRangeTextWidth = noTickSize.Width + 5;
				noRangeTextHeight = noTickSize.Height + 5;
			}

			// Plot the range count message to the lower right hand corner of the chart
			if (Bars.Period.Id == PeriodType.Range || isRangeDerivate)
			{
				int	actualRange	= (int) Math.Round(Math.Max(Close[0] - Low[0], High[0] - Close[0]) / Bars.Instrument.MasterInstrument.TickSize);
				int	rangeCount	= CountDown ? Bars.Period.Value - actualRange : actualRange;
				graphics.DrawString((CountDown ? "Range remaining = " + rangeCount : "Range count = " + rangeCount), ChartControl.Font, textBrush, bounds.X + bounds.Width - textWidth, bounds.Y + bounds.Height - textHeight, stringFormat);
				if (Bars.Period.Value - actualRange <= @AlertTicks )
				    Alert(CurrentBar.ToString(), NinjaTrader.Cbi.Priority.High, "RangeSound", @AlertFile, @AlertInterval, Color.LimeGreen, Color.White);
			}
			else
				graphics.DrawString(noRangeMessage, ChartControl.Font, textBrush, bounds.X + bounds.Width - noRangeTextWidth, bounds.Y + bounds.Height - noRangeTextHeight, stringFormat);
		}

		#region Properties
		/// <summary>
		/// </summary>
		[Gui.Design.VisualizationOnly()]
		[Description("Indicates if the indicator displays remaining ticks or current ticks of a bar.")]
		[Category("Parameters")]
		[Gui.Design.DisplayName("Count down")]
		public bool CountDown
		{
			get { return countDown; }
			set { countDown = value; }
		}

				[Description("How much ticks before Range finish?")]
				[Category("Sound and Display")]
				[Gui.Design.DisplayName("Ticks before end")]
				public int AlertTicks
				{
					get { return alertTicks; }
					set { alertTicks = value; }
				}

				[Description("How often do you want the alert to sound?")]
				[Category("Sound and Display")]
				[Gui.Design.DisplayName("Alert Interval (sec.)")]
				public int AlertInterval
				{
					get { return alertInterval; }
					set { alertInterval = value; }
				}
		
				[Description("Where is the alert sound file located?  File name of .wav file to play. Provide either the absolute file path or just the name if the file is located in NinjaTrader Installation Folder sounds folder")]
				[Category("Sound and Display")]
				[Gui.Design.DisplayName("Alert .wav File")]
				public string AlertFile
				{
					get { return alertFile; }
					set { alertFile = value; }
				}


        #endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private RangeSound[] cacheRangeSound = null;

        private static RangeSound checkRangeSound = new RangeSound();

        /// <summary>
        /// Displays the range countdown of a bar.
        /// </summary>
        /// <returns></returns>
        public RangeSound RangeSound(bool countDown)
        {
            return RangeSound(Input, countDown);
        }

        /// <summary>
        /// Displays the range countdown of a bar.
        /// </summary>
        /// <returns></returns>
        public RangeSound RangeSound(Data.IDataSeries input, bool countDown)
        {
            checkRangeSound.CountDown = countDown;
            countDown = checkRangeSound.CountDown;

            if (cacheRangeSound != null)
                for (int idx = 0; idx < cacheRangeSound.Length; idx++)
                    if (cacheRangeSound[idx].CountDown == countDown && cacheRangeSound[idx].EqualsInput(input))
                        return cacheRangeSound[idx];

            RangeSound indicator = new RangeSound();
            indicator.BarsRequired = BarsRequired;
            indicator.CalculateOnBarClose = CalculateOnBarClose;
            indicator.Input = input;
            indicator.CountDown = countDown;
            indicator.SetUp();

            RangeSound[] tmp = new RangeSound[cacheRangeSound == null ? 1 : cacheRangeSound.Length + 1];
            if (cacheRangeSound != null)
                cacheRangeSound.CopyTo(tmp, 0);
            tmp[tmp.Length - 1] = indicator;
            cacheRangeSound = tmp;
            Indicators.Add(indicator);

            return indicator;
        }

    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Displays the range countdown of a bar.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.RangeSound RangeSound(bool countDown)
        {
            return _indicator.RangeSound(Input, countDown);
        }

        /// <summary>
        /// Displays the range countdown of a bar.
        /// </summary>
        /// <returns></returns>
        public Indicator.RangeSound RangeSound(Data.IDataSeries input, bool countDown)
        {
            return _indicator.RangeSound(input, countDown);
        }

    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Displays the range countdown of a bar.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.RangeSound RangeSound(bool countDown)
        {
            return _indicator.RangeSound(Input, countDown);
        }

        /// <summary>
        /// Displays the range countdown of a bar.
        /// </summary>
        /// <returns></returns>
        public Indicator.RangeSound RangeSound(Data.IDataSeries input, bool countDown)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.RangeSound(input, countDown);
        }

    }
}
#endregion
