// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// Exponential Moving Average. The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The AdaptiveLF applies more weight to recent prices than the SMA.
	/// </summary>
	[Description ("Adaptive Laguerre Filter.")]
	public class AdaptiveLF : Indicator
	{
		#region Variables
		private int			period		= 14;
		#endregion

		DataSeries _diff;
		FloatSeries L0;
		FloatSeries L1;
		FloatSeries L2;
		FloatSeries L3;

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Plot(Color.Orange, "AdaptiveLF"));

			Overlay				= true;
			PriceTypeSupported	= true;

			_diff = new DataSeries (this);

			L0 = new FloatSeries (this);
			L1 = new FloatSeries (this);
			L2 = new FloatSeries (this);
			L3 = new FloatSeries (this);
		}
		
		/// <summary>
		/// Called on each bar update event (incoming tick)
		/// </summary>
		protected override void OnBarUpdate ()
		{
			if (CurrentBar == 0)
				return;

			_diff.Set (Math.Abs (Input [0] - Value [1]));

			double alpha = 0;
			if (CurrentBar >= Period && CurrentBar >= 5)
			{
				double max = MAX (_diff, Period) [0];
				double min = MIN (_diff, Period) [0];
				if (max != min)
					alpha = (MedianIndicator (_diff, 5) [0] - min) / (max - min);
			}

			L0.Set ((float)(alpha * Input [0] + (1.0 - alpha) * L0 [1]));
			L1.Set ((float)(-(1.0 - alpha) * L0 [0] + L0 [1] + (1.0 - alpha) * L1 [1]));
			L2.Set ((float)(-(1.0 - alpha) * L1 [0] + L1 [1] + (1.0 - alpha) * L2 [1]));
			L3.Set ((float)(-(1.0 - alpha) * L2 [0] + L2 [1] + (1.0 - alpha) * L3 [1]));

			Value.Set ((L0 [0] + 2 * L1 [0] + 2 * L2 [0] + L3 [0]) / 6);

		}

		#region Properties
		/// <summary>
		/// </summary>
		[Description ("Numbers of bars used for calculations")]
		[Category("Parameters")]
		public int Period
		{
			get { return period; }
			set { period = Math.Max(1, value); }
		}
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// </summary>
    public partial class Indicator : IndicatorBase
    {
        private AdaptiveLF[] cacheAdaptiveLF = null;
        private static AdaptiveLF checkAdaptiveLF = new AdaptiveLF();

        /// <summary>
        /// The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The AdaptiveLF applies more weight to recent prices than the SMA.
        /// </summary>
        /// <returns></returns>
        public AdaptiveLF AdaptiveLF(int period)
        {
            return AdaptiveLF(Input, period);
        }

        /// <summary>
        /// The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The AdaptiveLF applies more weight to recent prices than the SMA.
        /// </summary>
        /// <returns></returns>
        public AdaptiveLF AdaptiveLF(Data.IDataSeries input, int period)
        {
            checkAdaptiveLF.Period = period;
            period = checkAdaptiveLF.Period;

            if (cacheAdaptiveLF != null)
                for (int idx = 0; idx < cacheAdaptiveLF.Length; idx++)
                    if (cacheAdaptiveLF[idx].Period == period && cacheAdaptiveLF[idx].EqualsInput(input))
                        return cacheAdaptiveLF[idx];

            AdaptiveLF indicator = new AdaptiveLF();
			indicator.BarsRequired = BarsRequired;
            indicator.CalculateOnBarClose = CalculateOnBarClose;
            indicator.Input = input;
            indicator.Period = period;
            indicator.SetUp();

            AdaptiveLF[] tmp = new AdaptiveLF[cacheAdaptiveLF == null ? 1 : cacheAdaptiveLF.Length + 1];
            if (cacheAdaptiveLF != null)
                cacheAdaptiveLF.CopyTo(tmp, 0);
            tmp[tmp.Length - 1] = indicator;
            cacheAdaptiveLF = tmp;
            Indicators.Add(indicator);

            return indicator;
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    /// <summary>
    /// </summary>
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The AdaptiveLF applies more weight to recent prices than the SMA.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.AdaptiveLF AdaptiveLF(int period)
        {
            return _indicator.AdaptiveLF(Input, period);
        }

        /// <summary>
        /// The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The AdaptiveLF applies more weight to recent prices than the SMA.
        /// </summary>
        /// <returns></returns>
        public Indicator.AdaptiveLF AdaptiveLF(Data.IDataSeries input, int period)
        {
            return _indicator.AdaptiveLF(input, period);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    /// <summary>
    /// </summary>
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The AdaptiveLF applies more weight to recent prices than the SMA.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.AdaptiveLF AdaptiveLF(int period)
        {
            return _indicator.AdaptiveLF(Input, period);
        }

        /// <summary>
        /// The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The AdaptiveLF applies more weight to recent prices than the SMA.
        /// </summary>
        /// <returns></returns>
        public Indicator.AdaptiveLF AdaptiveLF(Data.IDataSeries input, int period)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.AdaptiveLF(input, period);
        }
    }
}
#endregion
