#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// Enter the description of your new custom indicator here
	/// </summary>
	[Description ("Adaptive Relative Vigor Index (from Cybernetic Analysis by John Ehlers)")]
	public class AdaptiveRVI : Indicator
	{
		#region Variables
		// Wizard generated variables
		private double alpha = 0.07; // Default setting for Alpha
		// User defined variables (add any user defined variables below)
		#endregion

		HilbertTransform2 _ht;
		WMA _wmaPeriod;
		DataSeries Value1;
		DataSeries Value2;

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize ()
		{
			Add (new Line (Color.DarkGray, 0, "ZeroLine"));

			Add (new Plot (Color.FromKnownColor (KnownColor.Red), PlotStyle.Line, "RVI"));
			Add (new Plot (Color.FromKnownColor (KnownColor.Blue), PlotStyle.Line, "Lag"));

			CalculateOnBarClose = true;
			Overlay = false;
			PriceTypeSupported = true;

			_ht = HilbertTransform2 (Input, Alpha);
			_wmaPeriod = WMA (_ht.Period, 4);

			Value1 = new DataSeries (this);
			Value2 = new DataSeries (this);
		}

		/// <summary>
		/// Called on each bar update event (incoming tick)
		/// </summary>
		protected override void OnBarUpdate ()
		{
			if (CurrentBar < 4)
				return;

			int length = (int) Math.Floor (_wmaPeriod [0] / 2);
			if (CurrentBar < length)
				return;

			Value1.Set (((Close [0] - Open [0]) + 2 * (Close [1] - Open [1]) + 2 * (Close [2] - Open [2]) + (Close [3] - Open [3])) / 6);
			Value2.Set (((High [0] - Low [0]) + 2 * (High [1] - Low [1]) + 2 * (High [2] - Low [2]) + (High [3] - Low [3])) / 6);

			double num = 0;
			double denom = 0;
			for (int i = 0; i < length; i++)
			{
				num += Value1 [i];
				denom += Value2 [i];
			}

			if (denom != 0)
				Value.Set (num / denom);

			Lag.Set (Value [1]);
		}

		#region Properties

		[Browsable (false)]
		[XmlIgnore ()]
		public DataSeries Lag
		{
			get { return Values [1]; }
		}

		[Description ("")]
		[Category ("Parameters")]
		public double Alpha
		{
			get { return alpha; }
			set { alpha = Math.Max (0.001, value); }
		}
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	public partial class Indicator : IndicatorBase
	{
		private AdaptiveRVI [] cacheAdaptiveRVI = null;

		private static AdaptiveRVI checkAdaptiveRVI = new AdaptiveRVI ();

		/// <summary>
		/// Enter the description of your new custom indicator here
		/// </summary>
		/// <returns></returns>
		public AdaptiveRVI AdaptiveRVI (double alpha)
		{
			return AdaptiveRVI (Input, alpha);
		}

		/// <summary>
		/// Enter the description of your new custom indicator here
		/// </summary>
		/// <returns></returns>
		public AdaptiveRVI AdaptiveRVI (Data.IDataSeries input, double alpha)
		{
			checkAdaptiveRVI.Alpha = alpha;
			alpha = checkAdaptiveRVI.Alpha;

			if (cacheAdaptiveRVI != null)
				for (int idx = 0; idx < cacheAdaptiveRVI.Length; idx++)
					if (Math.Abs (cacheAdaptiveRVI [idx].Alpha - alpha) <= double.Epsilon && cacheAdaptiveRVI [idx].EqualsInput (input))
						return cacheAdaptiveRVI [idx];

			AdaptiveRVI indicator = new AdaptiveRVI ();
			indicator.BarsRequired = BarsRequired;
			indicator.CalculateOnBarClose = CalculateOnBarClose;
			indicator.Input = input;
			indicator.Alpha = alpha;
			indicator.SetUp ();

			AdaptiveRVI [] tmp = new AdaptiveRVI [cacheAdaptiveRVI == null ? 1 : cacheAdaptiveRVI.Length + 1];
			if (cacheAdaptiveRVI != null)
				cacheAdaptiveRVI.CopyTo (tmp, 0);
			tmp [tmp.Length - 1] = indicator;
			cacheAdaptiveRVI = tmp;
			Indicators.Add (indicator);

			return indicator;
		}

	}
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
	public partial class Column : ColumnBase
	{
		/// <summary>
		/// Enter the description of your new custom indicator here
		/// </summary>
		/// <returns></returns>
		[Gui.Design.WizardCondition ("Indicator")]
		public Indicator.AdaptiveRVI AdaptiveRVI (double alpha)
		{
			return _indicator.AdaptiveRVI (Input, alpha);
		}

		/// <summary>
		/// Enter the description of your new custom indicator here
		/// </summary>
		/// <returns></returns>
		public Indicator.AdaptiveRVI AdaptiveRVI (Data.IDataSeries input, double alpha)
		{
			return _indicator.AdaptiveRVI (input, alpha);
		}

	}
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
	public partial class Strategy : StrategyBase
	{
		/// <summary>
		/// Enter the description of your new custom indicator here
		/// </summary>
		/// <returns></returns>
		[Gui.Design.WizardCondition ("Indicator")]
		public Indicator.AdaptiveRVI AdaptiveRVI (double alpha)
		{
			return _indicator.AdaptiveRVI (Input, alpha);
		}

		/// <summary>
		/// Enter the description of your new custom indicator here
		/// </summary>
		/// <returns></returns>
		public Indicator.AdaptiveRVI AdaptiveRVI (Data.IDataSeries input, double alpha)
		{
			if (InInitialize && input == null)
				throw new ArgumentException ("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

			return _indicator.AdaptiveRVI (input, alpha);
		}

	}
}
#endregion
