// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// Exponential Moving Average. The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The DominantCycle applies more weight to recent prices than the SMA.
	/// </summary>
	[Description ("The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The DominantCycle applies more weight to recent prices than the SMA.")]
	public class DominantCycle : Indicator
	{
		#region Variables
		private int period = 14;
		#endregion

		const int MIN_PERIODS = 8;
		const int COUNT_PERIODS = 42;


		double [] Q = new double [COUNT_PERIODS];
		double [] I = new double [COUNT_PERIODS];
		double [] Real = new double [COUNT_PERIODS];
		double [] Imag = new double [COUNT_PERIODS];
		double [] Ampl = new double [COUNT_PERIODS];

		double [] OldQ = new double [COUNT_PERIODS];
		double [] OldI = new double [COUNT_PERIODS];
		double [] OldReal = new double [COUNT_PERIODS];
		double [] OldImag = new double [COUNT_PERIODS];
		double [] OldAmpl = new double [COUNT_PERIODS];

		double [] OlderQ = new double [COUNT_PERIODS];
		double [] OlderI = new double [COUNT_PERIODS];
		double [] OlderReal = new double [COUNT_PERIODS];
		double [] OlderImag = new double [COUNT_PERIODS];
		double [] OlderAmpl = new double [COUNT_PERIODS];

		DataSeries _highPass;
		DataSeries _highPassSmoothed;
		DataSeries _domCyc;
		DataSeries _domCycMedian;

		FloatSeries [] _rgSeries;

		public IDataSeries HighPassSmoothed { get { return _highPassSmoothed; } }

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize ()
		{
			Add (new Plot (Color.CornflowerBlue, "DominantCycle"));

			Overlay = true;
			PriceTypeSupported = false;

			DrawOnPricePanel = false;
			Overlay = false;

			_highPass = new DataSeries (this);
			_highPassSmoothed = new DataSeries (this);
			_domCyc = new DataSeries (this);

			_domCycMedian = MedianIndicator (_domCyc, this.period).Value;

			_rgSeries = new FloatSeries [COUNT_PERIODS];

			for (int N = 0; N < COUNT_PERIODS; N++)
				_rgSeries [N] = new FloatSeries (this);

			//AutoScale = false;

			//VerticalGridLines = false;
			//HorizontalGridLines = false;

			//this.ChartControl.ChartObjects [this.Panel]

		}


		public override void GetMinMaxValues (ChartControl chartControl, ref double min, ref double max)
		{
			base.GetMinMaxValues (chartControl, ref min, ref max);

			min = MIN_PERIODS;
			max = MIN_PERIODS + COUNT_PERIODS - 1;
		}

		/// <summary>
		/// Called on each bar update event (incoming tick)
		/// </summary>
		protected override void OnBarUpdate ()
		{
			//Value.Set (CurrentBar % 2 == 0 ? 8 : 50);

			//this.DrawSquare ("a" + CurrentBar, 0, Input [0], Color.Azure);

			//DrawHorizontalLine ("min", 0, Color.Blue);
			//DrawHorizontalLine ("max", COUNT_BARS, Color.Blue);


			double alpha1 = (1.0 - Math.Sin (Math.PI * 2 / 40.0)) / Math.Cos (Math.PI * 2 / 40.0);

			if (CurrentBar <= 1)
				_highPass.Set (0);
			else
				_highPass.Set (.5 * (1.0 + alpha1) * (Input [0] - Input [1]) + alpha1 * _highPass [1]);


			if (CurrentBar < 1)
				_highPassSmoothed.Set (0);
			else if (CurrentBar < 7)
				_highPassSmoothed.Set (Input [0] - Input [1]);
			else
				_highPassSmoothed.Set ((_highPass [0] + 2 * _highPass [1] + 3 * _highPass [2] + 3 * _highPass [3] + 2 * _highPass [4] + _highPass [5]) / 12.0);


			double delta = -.015 * CurrentBar + .5;
			if (delta < .15)
				delta = .15;

			if (CurrentBar > 6)
			{
				for (int N = 0; N < COUNT_PERIODS; N++)
				{
					double angle = Math.PI * 2 / (N + MIN_PERIODS);

					double beta = Math.Cos (angle);
					double gamma = 1.0 / Math.Cos (angle * 2.0 * delta);
					double alpha = gamma - Math.Sqrt (gamma * gamma - 1);
					Q [N] = (_highPassSmoothed [0] - _highPassSmoothed [1]) / angle;
					I [N] = _highPassSmoothed [0];
					Real [N] = (1 - alpha) * (I [N] - OlderI [N]) / 2 + beta * (1 + alpha) * OldReal [N] - alpha * OlderReal [N];
					Imag [N] = (1 - alpha) * (Q [N] - OlderQ [N]) / 2 + beta * (1 + alpha) * OldImag [N] - alpha * OlderImag [N];
					Ampl [N] = (Real [N] * Real [N] + Imag [N] * Imag [N]);
				}
			}

			double maxAmpl = Ampl [0];
			for (int N = 1; N < COUNT_PERIODS; N++)
				if (Ampl [N] > maxAmpl)
					maxAmpl = Ampl [N];


			for (int N = 0; N < COUNT_PERIODS; N++)
			{
				double db;

				if (maxAmpl == 0 || (Ampl [N] / maxAmpl) <= 0)
					db = 0;
				else
				{
					//db = -0.5 * Math.Log (.01 / (1 - .99 * Ampl [N] / maxAmpl)) / Math.Log (10);
					db = -Math.Log (1.01 - Ampl [N] / maxAmpl) / Math.Log (100);
					//db = Ampl [N] / maxAmpl;
					//db = db * db;

					if (db < 0)
						db = 0;
				}

				_rgSeries [N].Set ((float) db);
			}

			double Num = 0;
			double Denom = 0;

			for (int N = 0; N < COUNT_PERIODS; N++)
			{
				double db = _rgSeries [N] [0];

				if (db >= (1 - 0.15))
				{
					Num += (N + MIN_PERIODS) * db;
					Denom += db;
				}

				if (Denom != 0)
					_domCyc.Set (Num / Denom);
			}

			Value.Set (_domCycMedian [0]);

			{
				double [] QTmp = OlderQ;
				double [] ITmp = OlderI;
				double [] RealTmp = OlderReal;
				double [] ImagTmp = OlderImag;
				double [] AmplTmp = OlderAmpl;

				OlderI = OldI;
				OldI = I;
				I = ITmp;

				OlderQ = OldQ;
				OldQ = Q;
				Q = QTmp;

				OlderReal = OldReal;
				OldReal = Real;
				Real = RealTmp;

				OlderImag = OldImag;
				OldImag = Imag;
				Imag = ImagTmp;

				OlderAmpl = OldAmpl;
				OldAmpl = Ampl;
				Ampl = AmplTmp;
			}
		}

		public override void Plot (Graphics graphics, Rectangle bounds, double min, double max)
		{
			GraphicsState state = graphics.Save ();
			try
			{
				graphics.TranslateTransform (
					bounds.Right - this.ChartControl.BarMarginRight - this.ChartControl.BarWidth - this.ChartControl.BarSpace / 2,
					bounds.Bottom);

				graphics.ScaleTransform (
					-this.ChartControl.BarSpace,
					-bounds.Height / (float) (max - min));

				graphics.TranslateTransform (
					0,
					(float) (MIN_PERIODS - min));

				SolidBrush brush = new SolidBrush (Color.Black);

				for (int N = 0; N < COUNT_PERIODS; N++)
				{
					FloatSeries periods = _rgSeries [N];

					for (int iBar = 0; iBar < this.FirstVisibleBar - this.LastVisibleBar; iBar++)
					{
						int bar = iBar + this.LastVisibleBar;
						if (bar >= Count - 1)
							break;

						float db = periods [iBar + this.LastVisibleBar];

						Color color;
						/*
						if (db < 1 / 3.0)
							color = Color.FromArgb (255, 255, (int) (255 * (1 - db * 3)));
						else if (db < 2 / 3.0)
							color = Color.FromArgb (255, (int) (255 * (2 - db * 3)), 0);
						else
							color = Color.FromArgb ((int) (255 * (3 - db * 3)), 0, 0);
						*/

						if (db <= 1 / 3.0)
							color = Color.FromArgb ((int) (255 * db * 3), 0, 0);
						else if (db <= 2 / 3.0)
							color = Color.FromArgb (255, (int) (255 * (db * 3 - 1)), 0);
						else
							color = Color.FromArgb (255, 255, (int) (255 * (db * 3 - 2)));

						brush.Color = color;
						graphics.FillRectangle (brush, iBar, N, 1, 1);
					}
				}
			}
			finally
			{
				graphics.Restore (state);
			}

			base.Plot (graphics, bounds, min, max);
		}

		#region Properties
		/// <summary>
		/// </summary>
		[Description ("Numbers of bars used for calculations")]
		[Category ("Parameters")]
		public int Period
		{
			get { return period; }
			set { period = Math.Max (1, value); }
		}
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// </summary>
	public partial class Indicator : IndicatorBase
	{
		private DominantCycle [] cacheDominantCycle = null;
		private static DominantCycle checkDominantCycle = new DominantCycle ();

		/// <summary>
		/// The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The DominantCycle applies more weight to recent prices than the SMA.
		/// </summary>
		/// <returns></returns>
		public DominantCycle DominantCycle (int period)
		{
			return DominantCycle (Input, period);
		}

		/// <summary>
		/// The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The DominantCycle applies more weight to recent prices than the SMA.
		/// </summary>
		/// <returns></returns>
		public DominantCycle DominantCycle (Data.IDataSeries input, int period)
		{
			checkDominantCycle.Period = period;
			period = checkDominantCycle.Period;

			if (cacheDominantCycle != null)
				for (int idx = 0; idx < cacheDominantCycle.Length; idx++)
					if (cacheDominantCycle [idx].Period == period && cacheDominantCycle [idx].EqualsInput (input))
						return cacheDominantCycle [idx];

			DominantCycle indicator = new DominantCycle ();
			indicator.BarsRequired = BarsRequired;
			indicator.CalculateOnBarClose = CalculateOnBarClose;
			indicator.Input = input;
			indicator.Period = period;
			indicator.SetUp ();

			DominantCycle [] tmp = new DominantCycle [cacheDominantCycle == null ? 1 : cacheDominantCycle.Length + 1];
			if (cacheDominantCycle != null)
				cacheDominantCycle.CopyTo (tmp, 0);
			tmp [tmp.Length - 1] = indicator;
			cacheDominantCycle = tmp;
			Indicators.Add (indicator);

			return indicator;
		}
	}
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
	/// <summary>
	/// </summary>
	public partial class Column : ColumnBase
	{
		/// <summary>
		/// The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The DominantCycle applies more weight to recent prices than the SMA.
		/// </summary>
		/// <returns></returns>
		[Gui.Design.WizardCondition ("Indicator")]
		public Indicator.DominantCycle DominantCycle (int period)
		{
			return _indicator.DominantCycle (Input, period);
		}

		/// <summary>
		/// The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The DominantCycle applies more weight to recent prices than the SMA.
		/// </summary>
		/// <returns></returns>
		public Indicator.DominantCycle DominantCycle (Data.IDataSeries input, int period)
		{
			return _indicator.DominantCycle (input, period);
		}
	}
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
	/// <summary>
	/// </summary>
	public partial class Strategy : StrategyBase
	{
		/// <summary>
		/// The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The DominantCycle applies more weight to recent prices than the SMA.
		/// </summary>
		/// <returns></returns>
		[Gui.Design.WizardCondition ("Indicator")]
		public Indicator.DominantCycle DominantCycle (int period)
		{
			return _indicator.DominantCycle (Input, period);
		}

		/// <summary>
		/// The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The DominantCycle applies more weight to recent prices than the SMA.
		/// </summary>
		/// <returns></returns>
		public Indicator.DominantCycle DominantCycle (Data.IDataSeries input, int period)
		{
			if (InInitialize && input == null)
				throw new ArgumentException ("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

			return _indicator.DominantCycle (input, period);
		}
	}
}
#endregion
