// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// The Aroon Oscillator is based upon his Aroon Indicator. Much like the Aroon Indicator, the Aroon Oscillator measures the strength of a trend.
	/// </summary>
	[Description("The Aroon Oscillator is based upon his Aroon Indicator. Much like the Aroon Indicator, the Aroon Oscillator measures the strength of a trend.")]
	public class DominantCycleOscillator : Indicator
	{
		#region Variables
		private int			period		= 14;
		#endregion

		DominantCycle _dc;

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Line(Color.DarkGray, 0, "Zero"));

			Add(new Plot(Color.Red, "Sin"));
			Add(new Plot(Color.Blue, "Cos"));

			CalculateOnBarClose = true;
			PriceTypeSupported	= false;

			_dc = DominantCycle (this.Period);
		}

		/// <summary>
		/// Called on each bar update event (incoming tick)
		/// </summary>
		protected override void OnBarUpdate()
		{

			double delta = -.015 * CurrentBar + .5;
			if (delta < .15)
				delta = .15;

			double dc = _dc [0];

			IDataSeries highPassSmoothed = _dc.HighPassSmoothed;

			if (CurrentBar > 2 && dc > 0)
			{
				double beta = Math.Cos (Math.PI * 2.0 / dc);
				double gamma = 1 / Math.Cos (Math.PI * 4.0 * delta / dc);
				double alpha = gamma - Math.Sqrt (gamma * gamma - 1);
				Values [0].Set (.5 * (1 - alpha) * (highPassSmoothed [0] - highPassSmoothed [1]) + beta * (1 + alpha) * Values [0] [1] - alpha * Values [0] [2]);
				Values [1].Set ((dc / (Math.PI * 2)) * (Values [0] [0] - Values [0] [1]));
			}
		}

		#region Properties
		/// <summary>
		/// </summary>
		[Description("Numbers of bars used for calculations")]
		[Category("Parameters")]
		public int Period
		{
			get { return period; }
			set { period = Math.Max(1, value); }
		}
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// </summary>
    public partial class Indicator : IndicatorBase
    {
        private DominantCycleOscillator[] cacheDominantCycleOscillator = null;
        private static DominantCycleOscillator checkDominantCycleOscillator = new DominantCycleOscillator();

        /// <summary>
        /// The Aroon Oscillator is based upon his Aroon Indicator. Much like the Aroon Indicator, the Aroon Oscillator measures the strength of a trend.
        /// </summary>
        /// <returns></returns>
        public DominantCycleOscillator DominantCycleOscillator(int period)
        {
            return DominantCycleOscillator(Input, period);
        }

        /// <summary>
        /// The Aroon Oscillator is based upon his Aroon Indicator. Much like the Aroon Indicator, the Aroon Oscillator measures the strength of a trend.
        /// </summary>
        /// <returns></returns>
        public DominantCycleOscillator DominantCycleOscillator(Data.IDataSeries input, int period)
        {
            checkDominantCycleOscillator.Period = period;
            period = checkDominantCycleOscillator.Period;

            if (cacheDominantCycleOscillator != null)
                for (int idx = 0; idx < cacheDominantCycleOscillator.Length; idx++)
                    if (cacheDominantCycleOscillator[idx].Period == period && cacheDominantCycleOscillator[idx].EqualsInput(input))
                        return cacheDominantCycleOscillator[idx];

            DominantCycleOscillator indicator = new DominantCycleOscillator();
			indicator.BarsRequired = BarsRequired;
            indicator.CalculateOnBarClose = CalculateOnBarClose;
            indicator.Input = input;
            indicator.Period = period;
            indicator.SetUp();

            DominantCycleOscillator[] tmp = new DominantCycleOscillator[cacheDominantCycleOscillator == null ? 1 : cacheDominantCycleOscillator.Length + 1];
            if (cacheDominantCycleOscillator != null)
                cacheDominantCycleOscillator.CopyTo(tmp, 0);
            tmp[tmp.Length - 1] = indicator;
            cacheDominantCycleOscillator = tmp;
            Indicators.Add(indicator);

            return indicator;
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    /// <summary>
    /// </summary>
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// The Aroon Oscillator is based upon his Aroon Indicator. Much like the Aroon Indicator, the Aroon Oscillator measures the strength of a trend.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.DominantCycleOscillator DominantCycleOscillator(int period)
        {
            return _indicator.DominantCycleOscillator(Input, period);
        }

        /// <summary>
        /// The Aroon Oscillator is based upon his Aroon Indicator. Much like the Aroon Indicator, the Aroon Oscillator measures the strength of a trend.
        /// </summary>
        /// <returns></returns>
        public Indicator.DominantCycleOscillator DominantCycleOscillator(Data.IDataSeries input, int period)
        {
            return _indicator.DominantCycleOscillator(input, period);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    /// <summary>
    /// </summary>
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// The Aroon Oscillator is based upon his Aroon Indicator. Much like the Aroon Indicator, the Aroon Oscillator measures the strength of a trend.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.DominantCycleOscillator DominantCycleOscillator(int period)
        {
            return _indicator.DominantCycleOscillator(Input, period);
        }

        /// <summary>
        /// The Aroon Oscillator is based upon his Aroon Indicator. Much like the Aroon Indicator, the Aroon Oscillator measures the strength of a trend.
        /// </summary>
        /// <returns></returns>
        public Indicator.DominantCycleOscillator DominantCycleOscillator(Data.IDataSeries input, int period)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.DominantCycleOscillator(input, period);
        }
    }
}
#endregion
