// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

//#define SHOW_PHASE

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// Exponential Moving Average. The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The HilbertSinewaveOscillator applies more weight to recent prices than the SMA.
	/// </summary>
	[Description ("Hilbert Sinewave Oscillator.")]
	public class HilbertSinewaveOscillator : Indicator
	{
		#region Variables
		private int period = 4;
		#endregion

		private HilbertTransform2 _ht;
		private DataSeries _phase;
#if !SHOW_PHASE
		private DataSeries _phaseChange;
#endif

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add (new Line (Color.DarkGray, 0.0, "Zero"));

			Add (new Plot (Color.Red, "Sine"));
			Add (new Plot (Color.Blue, "LeadSine"));

#if SHOW_PHASE
			Add (new Plot (Color.LightGray, "PhaseChange"));
#else
			_phaseChange = new DataSeries (this);
#endif

			_phase = new DataSeries (this);

			Overlay				= false;
			PriceTypeSupported	= true;

			_ht = HilbertTransform2 (0.07);
		}

		public override void SetUp ()
		{
			base.SetUp ();
		}
		
		/// <summary>
		/// Called on each bar update event (incoming tick)
		/// </summary>
		protected override void OnBarUpdate ()
		{
			if (CurrentBar < 1)
				return;

			double DCPeriod = Math.Floor (_ht.Period [0]);

			double re = 0;
			double im = 0;
			for (int iBar = 0; (iBar < DCPeriod) && (iBar < CurrentBar); iBar++)
			{
				double angle = Math.PI * 2 * iBar / DCPeriod;
				double amplitude = _ht.Cycle [iBar];
				re += Math.Cos (angle) * amplitude;
				im += Math.Sin (angle) * amplitude;
			}

			double phase = Math.Atan2 (im, re);

			_phase.Set (phase);
			double phaseChange = (_phase [0] - _phase [1]);
			if (phaseChange > 0.5* Math.PI)
				phaseChange -= 2 * Math.PI;
			else if (phaseChange < -0.5 * Math.PI)
				phaseChange += 2 * Math.PI;

			PhaseChange.Set (phaseChange);

			double DCPhase = phase + Math.PI / 4;

			Value.Set (Math.Cos (DCPhase));
			LeadSine.Set (Math.Cos (DCPhase + Math.PI / 8));
		}

		#region Properties
		/// <summary>
		/// </summary>
		[Description ("Numbers of bars used for calculations")]
		[Category("Parameters")]
		public int Period
		{
			get { return period; }
			set { period = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Browsable (false)]
		[XmlIgnore ()]
		public DataSeries LeadSine
		{
			get { return Values [1]; }
		}

		/// <summary>
		/// </summary>
		[Browsable (false)]
		[XmlIgnore ()]
		public DataSeries PhaseChange
		{
#if SHOW_PHASE
			get { return Values [2]; }
#else
			get { return _phaseChange; }
#endif
		}

		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// </summary>
    public partial class Indicator : IndicatorBase
    {
        private HilbertSinewaveOscillator[] cacheHilbertSinewaveOscillator = null;
        private static HilbertSinewaveOscillator checkHilbertSinewaveOscillator = new HilbertSinewaveOscillator();

        /// <summary>
        /// The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The HilbertSinewaveOscillator applies more weight to recent prices than the SMA.
        /// </summary>
        /// <returns></returns>
        public HilbertSinewaveOscillator HilbertSinewaveOscillator(int period)
        {
            return HilbertSinewaveOscillator(Input, period);
        }

        /// <summary>
        /// The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The HilbertSinewaveOscillator applies more weight to recent prices than the SMA.
        /// </summary>
        /// <returns></returns>
        public HilbertSinewaveOscillator HilbertSinewaveOscillator(Data.IDataSeries input, int period)
        {
            checkHilbertSinewaveOscillator.Period = period;
            period = checkHilbertSinewaveOscillator.Period;

            if (cacheHilbertSinewaveOscillator != null)
                for (int idx = 0; idx < cacheHilbertSinewaveOscillator.Length; idx++)
                    if (cacheHilbertSinewaveOscillator[idx].Period == period && cacheHilbertSinewaveOscillator[idx].EqualsInput(input))
                        return cacheHilbertSinewaveOscillator[idx];

            HilbertSinewaveOscillator indicator = new HilbertSinewaveOscillator();
			indicator.BarsRequired = BarsRequired;
            indicator.CalculateOnBarClose = CalculateOnBarClose;
            indicator.Input = input;
            indicator.Period = period;
            indicator.SetUp();

            HilbertSinewaveOscillator[] tmp = new HilbertSinewaveOscillator[cacheHilbertSinewaveOscillator == null ? 1 : cacheHilbertSinewaveOscillator.Length + 1];
            if (cacheHilbertSinewaveOscillator != null)
                cacheHilbertSinewaveOscillator.CopyTo(tmp, 0);
            tmp[tmp.Length - 1] = indicator;
            cacheHilbertSinewaveOscillator = tmp;
            Indicators.Add(indicator);

            return indicator;
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    /// <summary>
    /// </summary>
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The HilbertSinewaveOscillator applies more weight to recent prices than the SMA.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.HilbertSinewaveOscillator HilbertSinewaveOscillator(int period)
        {
            return _indicator.HilbertSinewaveOscillator(Input, period);
        }

        /// <summary>
        /// The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The HilbertSinewaveOscillator applies more weight to recent prices than the SMA.
        /// </summary>
        /// <returns></returns>
        public Indicator.HilbertSinewaveOscillator HilbertSinewaveOscillator(Data.IDataSeries input, int period)
        {
            return _indicator.HilbertSinewaveOscillator(input, period);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    /// <summary>
    /// </summary>
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The HilbertSinewaveOscillator applies more weight to recent prices than the SMA.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.HilbertSinewaveOscillator HilbertSinewaveOscillator(int period)
        {
            return _indicator.HilbertSinewaveOscillator(Input, period);
        }

        /// <summary>
        /// The Exponential Moving Average is an indicator that shows the average value of a security's price over a period of time. When calculating a moving average. The HilbertSinewaveOscillator applies more weight to recent prices than the SMA.
        /// </summary>
        /// <returns></returns>
        public Indicator.HilbertSinewaveOscillator HilbertSinewaveOscillator(Data.IDataSeries input, int period)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.HilbertSinewaveOscillator(input, period);
        }
    }
}
#endregion
