// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// The Commodity Channel Index (CCINoGrid) measures the variation of a security's price from its statistical mean. High values show that prices are unusually high compared to average prices whereas low values indicate that prices are unusually low.
	/// </summary>
	[Description("The Commodity Channel Index (CCINoGrid) measures the variation of a security's price from its statistical mean. High values show that prices are unusually high compared to average prices whereas low values indicate that prices are unusually low.")]
	public class CCI4RangeBars : Indicator
	{
		#region Variables
			private int			periods		= 14;
			private DataSeries	cCIDummySeries;
			private double projectedHigh 	= 0;
			private double projectedLow 	= 0;
			private double typUp = 0;//Last Bar Typical Price at high close
			private double typDown = 0;//Last Bar Typical Price at low close
			private double meanUp = 0;
			private double meanDown = 0;
			private double nMeanUp = 0;
			private double nMeanDown = 0;
			private double CCIup= 0;
			private double CCIdown=0;
			private double newCCIup= 0;
			private double newCCIdown=0;
			private double nCCIup= 0;
			private double nCCIdown=0;
			private double newTypUp = 0;
			private double newTypDown =0;
			private double oldTypUp = 0;
			private double oldTypDown =0;
			
			private int lineThickness = 2;
		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Plot(Color.Orange, "CCI4RangeBars"));
 			cCIDummySeries		= new DataSeries(this);
			HorizontalGridLines = false;
			DrawOnPricePanel = false;
			CalculateOnBarClose = false;
			}

		/// <summary>
		/// Calculates the indicator value(s) at the current index.
		/// </summary>
		protected override void OnBarUpdate()
		{
			if(Bars.Period.Id != PeriodType.Range) return;
			if(CurrentBar < periods) return;
			if (CurrentBar == 0)
				Value.Set(0);
			else
			{
			projectedHigh 	= Low[0]+Bars.Period.Value*TickSize;
			projectedLow 	= High[0]-Bars.Period.Value*TickSize;
			
			#region CCI
				double mean = 0;
				for (int idx = Math.Min(CurrentBar, Periods - 1); idx >= 0; idx--)
					mean += Math.Abs(Typical[idx] - SMA(Typical, Periods)[0]);
				cCIDummySeries.Set((Typical[0] - SMA(Typical, Periods)[0]) / (mean == 0 ? 1 : (0.015 * (mean / Math.Min(Periods, CurrentBar + 1)))));
			#endregion

				mean = 0;
				int n =  Periods - 1;
				double typUp = (Low[0] + (2*projectedHigh))/3;//Last Bar Typical Price at high close
				double typDown = (High[0] + (2*projectedLow))/3;//Last Bar Typical Price at low close
				///Sum of Typical Prices
				for (int i = (Periods - 2); i >= 1; i--)
				{
					typUp += Typical[i];
					typDown += Typical[i];
						
				}
				newTypUp = typUp + (Low[0] + (2*projectedHigh))/3;
				newTypDown =typDown + (High[0] + (2*projectedLow))/3;
				oldTypUp = typUp + Typical[n] ;
				oldTypDown = typDown + Typical[n];
				meanUp =  Math.Abs(Typical[n] - oldTypUp/Periods);
				meanDown = Math.Abs(Typical[n] - oldTypDown/Periods);
				meanUp += Math.Abs((Low[0] + (2*projectedHigh))/3 - oldTypUp/Periods);
				meanDown += Math.Abs((High[0] + (2*projectedLow))/3 - oldTypDown/Periods);
				nMeanUp = Math.Abs(projectedHigh - newTypUp/Periods);
				nMeanDown = Math.Abs(projectedLow - newTypDown/Periods);
				nMeanUp += Math.Abs((Low[0] + (2*projectedHigh))/3 - newTypUp/Periods);
				nMeanDown += Math.Abs((High[0] + (2*projectedLow))/3 - newTypDown/Periods);
				for (int idx = ( Periods - 2); idx >= 1; idx--)
				{
					meanUp += Math.Abs(Typical[idx] - oldTypUp/Periods);
					meanDown += Math.Abs(Typical[idx] - oldTypDown/Periods);
					nMeanUp += Math.Abs(Typical[idx] - newTypUp/Periods);
					nMeanDown += Math.Abs(Typical[idx] - newTypDown/Periods);
				}
			
				CCIup =((Low[0] + (2*projectedHigh))/3 - oldTypUp/Periods) / (meanUp == 0 ? 1 : (0.015 * (meanUp / Periods)));
				CCIdown =((High[0] + (2*projectedLow))/3 - oldTypDown/Periods) / (meanDown == 0 ? 1 : (0.015 * (meanDown / Periods)));

				if(FirstTickOfBar) Value.Set(1,cCIDummySeries[1]);
				DrawLine("CCI High",true,1,Value[1],0,CCIup,Color.Green,DashStyle.Solid,lineThickness);
				DrawLine("CCI Low",true,1,Value[1],0,CCIdown,Color.Red,DashStyle.Solid,lineThickness);
			
			
			}
		}

		#region Properties
		/// <summary>
		/// </summary>
		[Description("Numbers of bars used for calculations")]
		[GridCategory("Parameters")]
		public int Periods
		{
			get { return periods; }
			set { periods = Math.Max(1, value); }
		}
		[Description("Numbers of bars used for calculations")]
		[GridCategory("Parameters")]
		public int LineThickness
		{
			get { return lineThickness; }
			set { lineThickness = Math.Max(1, value); }
		}
		[Browsable(false)]
		[XmlIgnore()]
		[Description("CCI Dummy")]
		public DataSeries CCIDummy
		{
			get { Update();
			return cCIDummySeries; }
		}
		
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private CCI4RangeBars[] cacheCCI4RangeBars = null;

        private static CCI4RangeBars checkCCI4RangeBars = new CCI4RangeBars();

        /// <summary>
        /// The Commodity Channel Index (CCINoGrid) measures the variation of a security's price from its statistical mean. High values show that prices are unusually high compared to average prices whereas low values indicate that prices are unusually low.
        /// </summary>
        /// <returns></returns>
        public CCI4RangeBars CCI4RangeBars(int lineThickness, int periods)
        {
            return CCI4RangeBars(Input, lineThickness, periods);
        }

        /// <summary>
        /// The Commodity Channel Index (CCINoGrid) measures the variation of a security's price from its statistical mean. High values show that prices are unusually high compared to average prices whereas low values indicate that prices are unusually low.
        /// </summary>
        /// <returns></returns>
        public CCI4RangeBars CCI4RangeBars(Data.IDataSeries input, int lineThickness, int periods)
        {
            if (cacheCCI4RangeBars != null)
                for (int idx = 0; idx < cacheCCI4RangeBars.Length; idx++)
                    if (cacheCCI4RangeBars[idx].LineThickness == lineThickness && cacheCCI4RangeBars[idx].Periods == periods && cacheCCI4RangeBars[idx].EqualsInput(input))
                        return cacheCCI4RangeBars[idx];

            lock (checkCCI4RangeBars)
            {
                checkCCI4RangeBars.LineThickness = lineThickness;
                lineThickness = checkCCI4RangeBars.LineThickness;
                checkCCI4RangeBars.Periods = periods;
                periods = checkCCI4RangeBars.Periods;

                if (cacheCCI4RangeBars != null)
                    for (int idx = 0; idx < cacheCCI4RangeBars.Length; idx++)
                        if (cacheCCI4RangeBars[idx].LineThickness == lineThickness && cacheCCI4RangeBars[idx].Periods == periods && cacheCCI4RangeBars[idx].EqualsInput(input))
                            return cacheCCI4RangeBars[idx];

                CCI4RangeBars indicator = new CCI4RangeBars();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.LineThickness = lineThickness;
                indicator.Periods = periods;
                Indicators.Add(indicator);
                indicator.SetUp();

                CCI4RangeBars[] tmp = new CCI4RangeBars[cacheCCI4RangeBars == null ? 1 : cacheCCI4RangeBars.Length + 1];
                if (cacheCCI4RangeBars != null)
                    cacheCCI4RangeBars.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheCCI4RangeBars = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// The Commodity Channel Index (CCINoGrid) measures the variation of a security's price from its statistical mean. High values show that prices are unusually high compared to average prices whereas low values indicate that prices are unusually low.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.CCI4RangeBars CCI4RangeBars(int lineThickness, int periods)
        {
            return _indicator.CCI4RangeBars(Input, lineThickness, periods);
        }

        /// <summary>
        /// The Commodity Channel Index (CCINoGrid) measures the variation of a security's price from its statistical mean. High values show that prices are unusually high compared to average prices whereas low values indicate that prices are unusually low.
        /// </summary>
        /// <returns></returns>
        public Indicator.CCI4RangeBars CCI4RangeBars(Data.IDataSeries input, int lineThickness, int periods)
        {
            return _indicator.CCI4RangeBars(input, lineThickness, periods);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// The Commodity Channel Index (CCINoGrid) measures the variation of a security's price from its statistical mean. High values show that prices are unusually high compared to average prices whereas low values indicate that prices are unusually low.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.CCI4RangeBars CCI4RangeBars(int lineThickness, int periods)
        {
            return _indicator.CCI4RangeBars(Input, lineThickness, periods);
        }

        /// <summary>
        /// The Commodity Channel Index (CCINoGrid) measures the variation of a security's price from its statistical mean. High values show that prices are unusually high compared to average prices whereas low values indicate that prices are unusually low.
        /// </summary>
        /// <returns></returns>
        public Indicator.CCI4RangeBars CCI4RangeBars(Data.IDataSeries input, int lineThickness, int periods)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.CCI4RangeBars(input, lineThickness, periods);
        }
    }
}
#endregion
