#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Calculates slope of data series.
    /// </summary>
    [Description("Calculates slope of data series.")]
    public class SlopeSimple : Indicator
    {
        #region Variables
       
            private int period = 10;
            private bool displayRadians = false;
			private bool intervalBased = false;
		
			private SlopeType slopeType	= SlopeType.SMA;
			
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
            Add(new Plot(Color.FromKnownColor(KnownColor.Orange), PlotStyle.Line, "Slope"));
		}

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {	
        	if (CurrentBar < Period)
				return;
			
			double diff;
			double slope;
			double convert = 1;
			int interval = 1;
			
			if (DisplayRadians)
				convert = Math.PI / 180;
			
			if (IntervalBased)
				interval = BarsPeriods[0].Value;
			else
				interval = 1;
				
			switch (slopeType)
			{
				case SlopeType.Close:
				{
					diff = Close[0] - Close[Period];
					slope = diff / (Period * interval);
					Slope.Set(Math.Atan(slope) * convert);
					break;
				}
				
				case SlopeType.EMA:
				{
					diff = EMA(14)[0] - EMA(14)[Period];
					slope = diff / (Period * interval);
					Slope.Set(Math.Atan(slope) * convert);
					break;
				}
				
				case SlopeType.SMA:
				{
					diff = SMA(14)[0] - SMA(14)[Period];
					slope = diff / (Period * interval);
					Slope.Set(Math.Atan(slope) * convert);
					break;
				}
			}
					
        }

        #region Properties
        [Browsable(false)]	
        [XmlIgnore()]		
        public DataSeries Slope
        {
            get { return Values[0]; }
        }

        [Description("Number of bars used for the run, in rise over run")]
        [GridCategory("Parameters")]
        public int Period
        {
            get { return period; }
            set { period = Math.Max(1, value); }
        }

        [Description("Change plot output to read in Radians")]
        [GridCategory("Parameters")]
        public bool DisplayRadians
        {
            get { return displayRadians; }
            set { displayRadians = value; }
        }
		
		[Description("Set to true if you want to normalize to interval, for example, with a period of 10, uses 50 as the run value on a 5min chart")]
        [GridCategory("Parameters")]
        public bool IntervalBased
        {
            get { return intervalBased; }
            set { intervalBased = value; }
        }
		
		[Description("Choose a Moving Average type.")]
		[Category("Parameters")]
		public SlopeType SlopeType
		{
			get { return slopeType; }
			set { slopeType = value; }
		}
		
		
		
		
		
		
		
		
		
		
		
		
		
		
		
		
        #endregion
    }
}

public enum SlopeType
{
	Close,
	EMA,
	SMA,
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private SlopeSimple[] cacheSlopeSimple = null;

        private static SlopeSimple checkSlopeSimple = new SlopeSimple();

        /// <summary>
        /// Calculates slope of data series.
        /// </summary>
        /// <returns></returns>
        public SlopeSimple SlopeSimple(bool displayRadians, bool intervalBased, int period, SlopeType slopeType)
        {
            return SlopeSimple(Input, displayRadians, intervalBased, period, slopeType);
        }

        /// <summary>
        /// Calculates slope of data series.
        /// </summary>
        /// <returns></returns>
        public SlopeSimple SlopeSimple(Data.IDataSeries input, bool displayRadians, bool intervalBased, int period, SlopeType slopeType)
        {
            if (cacheSlopeSimple != null)
                for (int idx = 0; idx < cacheSlopeSimple.Length; idx++)
                    if (cacheSlopeSimple[idx].DisplayRadians == displayRadians && cacheSlopeSimple[idx].IntervalBased == intervalBased && cacheSlopeSimple[idx].Period == period && cacheSlopeSimple[idx].SlopeType == slopeType && cacheSlopeSimple[idx].EqualsInput(input))
                        return cacheSlopeSimple[idx];

            lock (checkSlopeSimple)
            {
                checkSlopeSimple.DisplayRadians = displayRadians;
                displayRadians = checkSlopeSimple.DisplayRadians;
                checkSlopeSimple.IntervalBased = intervalBased;
                intervalBased = checkSlopeSimple.IntervalBased;
                checkSlopeSimple.Period = period;
                period = checkSlopeSimple.Period;
                checkSlopeSimple.SlopeType = slopeType;
                slopeType = checkSlopeSimple.SlopeType;

                if (cacheSlopeSimple != null)
                    for (int idx = 0; idx < cacheSlopeSimple.Length; idx++)
                        if (cacheSlopeSimple[idx].DisplayRadians == displayRadians && cacheSlopeSimple[idx].IntervalBased == intervalBased && cacheSlopeSimple[idx].Period == period && cacheSlopeSimple[idx].SlopeType == slopeType && cacheSlopeSimple[idx].EqualsInput(input))
                            return cacheSlopeSimple[idx];

                SlopeSimple indicator = new SlopeSimple();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.DisplayRadians = displayRadians;
                indicator.IntervalBased = intervalBased;
                indicator.Period = period;
                indicator.SlopeType = slopeType;
                Indicators.Add(indicator);
                indicator.SetUp();

                SlopeSimple[] tmp = new SlopeSimple[cacheSlopeSimple == null ? 1 : cacheSlopeSimple.Length + 1];
                if (cacheSlopeSimple != null)
                    cacheSlopeSimple.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheSlopeSimple = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Calculates slope of data series.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.SlopeSimple SlopeSimple(bool displayRadians, bool intervalBased, int period, SlopeType slopeType)
        {
            return _indicator.SlopeSimple(Input, displayRadians, intervalBased, period, slopeType);
        }

        /// <summary>
        /// Calculates slope of data series.
        /// </summary>
        /// <returns></returns>
        public Indicator.SlopeSimple SlopeSimple(Data.IDataSeries input, bool displayRadians, bool intervalBased, int period, SlopeType slopeType)
        {
            return _indicator.SlopeSimple(input, displayRadians, intervalBased, period, slopeType);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Calculates slope of data series.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.SlopeSimple SlopeSimple(bool displayRadians, bool intervalBased, int period, SlopeType slopeType)
        {
            return _indicator.SlopeSimple(Input, displayRadians, intervalBased, period, slopeType);
        }

        /// <summary>
        /// Calculates slope of data series.
        /// </summary>
        /// <returns></returns>
        public Indicator.SlopeSimple SlopeSimple(Data.IDataSeries input, bool displayRadians, bool intervalBased, int period, SlopeType slopeType)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.SlopeSimple(input, displayRadians, intervalBased, period, slopeType);
        }
    }
}
#endregion
