// Multi-Timeframe by Gumphrie v0.1


#region Using declarations
using System;
using System.Diagnostics;
using System.Collections;
using System.Text;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Cbi;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
	/// Multiple Timeframes
    /// </summary>
    [Description("Multiple Timeframes")]
	[Gui.Design.DisplayName("Multiple Timeframes")]
	public class MTF : Indicator
    {
		
        #region Variables
		private DataSeries 	open;
		private DataSeries 	high;
		private DataSeries 	low;
		private DataSeries 	close;
		private DataSeries 	timestamp;
		
		private bool thickWick=false;
		private bool plotOverride=false;
		private bool priceScale=true;
		private bool modHA=false;
		private bool showShadow=true;
		private bool scalper=false;
		private bool showTriggers=false;
		private int shadowWidth=1;
		private int multiplier=2;
		private int currentBar;
		private int Digits=0;
		private int compBars=6;
		private double 	priceSize=1;
		
		private SolidBrush ScalpBarBrush;
		
		private Color barColorUp 	= Color.LightGreen;
		private Color barColorDown 	= Color.Red;
		
		private	Font					textFont		= new Font("Arial",8);
		private	SolidBrush				textBrush		= new SolidBrush(Color.White);
		private StringFormat			stringFormat	= new StringFormat();
		
		private MultiTimeFramePlots pricePlot = MultiTimeFramePlots.CandleStick;
		private MultiTimeFrameCandleScale candleScale = MultiTimeFrameCandleScale.One;
		
		public enum MultiTimeFramePlots
		{
				CandleStick,
				OHLC,
				HiLoBars,
				LineOnClose,
		}
		
		public enum MultiTimeFrameCandleScale
		{
				One,
				Two,
				Three,
		}
		
		#endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
			Add(new Plot(Color.Black, "Price"));
			
			ScalpBarBrush = new SolidBrush(Color.Lavender);
			
			open  = new DataSeries(this);
			high  = new DataSeries(this);
			low   = new DataSeries(this);
			close = new DataSeries(this);
			
			timestamp = new DataSeries(this);
			
			PlotsConfigurable 	= false;
			PaintPriceMarkers 	= true;
			Overlay				= false;		
			CalculateOnBarClose = false;
			
            if (TickSize.ToString().StartsWith("1E-"))
			{
				Digits=Convert.ToInt32(TickSize.ToString().Substring(3));
			}
			else if (TickSize.ToString().IndexOf(".")>0)
			{
				Digits=TickSize.ToString().Substring(TickSize.ToString().IndexOf("."),TickSize.ToString().Length-1).Length-1;
			}
			
			if (priceScale)
			{
				if (Digits==0) priceSize=100;
				else if (Digits>2) priceSize=Math.Pow(10,(double)Digits-2); 
				else if (Digits<2) priceSize=Math.Pow(10,(double)Digits); 
			}
			
			BarsRequired=1;
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {			
			if (CurrentBar == 0)
			{
				return;
			}
			
			TimeSpan ts = (new DateTime(Time[0].Ticks,DateTimeKind.Utc)- new DateTime(1970,1,1,0,0,0));
		 	timestamp.Set((int)ts.TotalSeconds);
			
            if (Digits<2)
			{
				open.Set(Open[0]/priceSize);
				high.Set(High[0]/priceSize);
				low.Set(Low[0]/priceSize);
				close.Set(Close[0]/priceSize);
				
				Value.Set(Close[0]/priceSize);
			}
			else
			{
				open.Set(Open[0]*priceSize);
				high.Set(High[0]*priceSize);
				low.Set(Low[0]*priceSize);
				close.Set(Close[0]*priceSize);
				
				Value.Set(Close[0]*priceSize);
			}
			currentBar=CurrentBar;
			
		}

		public override void GetMinMaxValues(ChartControl chartControl, ref double min, ref double max)
		{
			base.GetMinMaxValues(chartControl, ref min, ref max);
				
			int bars = base.ChartControl.BarsPainted;
            int barPaintWidth = base.ChartControl.ChartStyle.GetBarPaintWidth(base.ChartControl.BarWidth);			
			Exception caughtException;
			
            while (bars >= 0)
            {		
				int index = ((base.ChartControl.LastBarPainted - base.ChartControl.BarsPainted) + 1) + bars;
                if (base.ChartControl.ShowBarsRequired || ((index - base.Displacement) >= base.BarsRequired))
                {
					try
					{
						double thisHigh =  high.Get(index);
						double thisLow =  low.Get(index);
						
						if ((thisLow>0) && (thisHigh>0))
						{
						
							if (!priceScale || (Digits==2))
							{
								thisHigh+=(TickSize*2);
								thisLow-=(TickSize*2);
							}
							else if (Digits<2)
							{
								thisHigh+=((TickSize/priceSize)*2);
								thisLow-=((TickSize/priceSize)*2);
							}
							else
							{
								thisHigh+=((TickSize*priceSize)*2);
								thisLow-=((TickSize*priceSize)*2);
							}								
							
							if ((!double.IsNaN(thisHigh)) && (index>0))
							{
								if (thisHigh>max) max=thisHigh;
								if (thisHigh<min) min=thisHigh; 
							}
							
							if ((!double.IsNaN(thisLow)) && (index>0))
							{
								if (thisLow>max) max=thisLow;
								if (thisLow<min) min=thisLow; 
							}
						}
					}
					catch (Exception exception) {caughtException=exception;}
				}
				bars--;
			}
						
		}
		
		
		
		public override void Plot(Graphics graphics, Rectangle bounds, double min, double max)
		{
			// Default plotting in base class. 
			//base.Plot(graphics, bounds, min, max);
			if (base.Bars == null) return;
			
			ArrayList 	Mopen = new ArrayList();
			ArrayList 	Mhigh = new ArrayList();
		 	ArrayList 	Mlow = new ArrayList();
		 	ArrayList 	Mclose = new ArrayList();
			ArrayList 	Mtimestamp = new ArrayList();
			
			ArrayList 	HAopen = new ArrayList();
			ArrayList 	HAclose = new ArrayList();
			ArrayList 	HAcolor = new ArrayList();
			int[] colorArr = new int[compBars];
			
			int counter = 0;
			int ExtremeBar = 0;
			int lastBar=0;
			bool HighPainted = false;
			double ExtremeBarHigh = 0;
            double TriggerBarLow = 0;
			double ExtremeBarLow = 0;
            double TriggerBarHigh = 0;
			int TriggerBarLowNo=0;
			int TriggerBarHighNo=0;
			ArrayList 	ScalpBars = new ArrayList();
			
			string		barType="";
			int         barSize=1;
			
			if (Bars.Period.ToString()=="Daily") barType="Days";
			else if (Bars.Period.ToString()=="Weekly") barType="Weeks";
			else if ((Bars.Period.ToString()=="Monthly") || (Bars.Period.ToString()=="Yearly")) 
			{
				graphics.FillRectangle(new SolidBrush(Color.Black), bounds.X, bounds.Y + 23, 40, 13);
				graphics.DrawString("Timeframe not supported", textFont, textBrush, bounds.X, bounds.Y + 23, stringFormat);
				return;
			}
			else
			{
				barType = Bars.Period.ToString().Substring(Bars.Period.ToString().IndexOf(" ")+1);
				barSize = Convert.ToInt32(Bars.Period.ToString().Substring(0,(Bars.Period.ToString().IndexOf(" "))));
			}
			
			int         secondsGap=0;
			bool        timeBased=false;
			int   		shift=-1;

			MultiTimeFramePlots thisPlot;
			
			if (plotOverride)
				thisPlot=pricePlot;
			else
			{
				if (ChartControl.ChartStyleType.ToString()=="OHLC")
					thisPlot=MultiTimeFramePlots.OHLC;
				else if (ChartControl.ChartStyleType.ToString()=="HiLoBars")
					thisPlot=MultiTimeFramePlots.HiLoBars;
				else if (ChartControl.ChartStyleType.ToString()=="LineOnClose")
					thisPlot=MultiTimeFramePlots.LineOnClose;
				else
					thisPlot=MultiTimeFramePlots.CandleStick;
				barColorUp 		= ChartControl.ChartStyle.UpColor;
		        barColorDown 	= ChartControl.ChartStyle.DownColor;
			}
			
			
			
			// load from dataSeries
			if (barType=="Seconds")
			{
				secondsGap=barSize;
				timeBased=true;
			}
			else if (barType=="Min")
			{
				secondsGap=60*barSize;
				timeBased=true;
			}
			else if (barType=="Days")
			{
				secondsGap=86400*barSize;
				timeBased=true;
			}
			else if (barType=="Weeks")
			{
				secondsGap=604800*barSize;
				timeBased=true;
			}
			
			int lastTime=0;
			int lastColor=0;
			bool firstBar=true;
			int thisCurrentBar=currentBar;
			
			int beginBar=1;
			
			
			if (!timeBased)
			{
				for (int i=0;i<(thisCurrentBar/multiplier)+1;i++)
				{
					Mopen.Add(0.0);
					Mhigh.Add(0.0);
					Mlow.Add(0.0);
					Mclose.Add(0.0);
				}
			}
			else
			{
				if (thisCurrentBar>ChartControl.LastBarPainted) thisCurrentBar=ChartControl.LastBarPainted+1;
				beginBar=(ChartControl.LastBarPainted - ChartControl.BarsPainted)-compBars;			
				if (beginBar<1) beginBar=1;
			}
			
					
			for (int i=beginBar;i<=thisCurrentBar;i++)
			{
				bool bPopulated=false;
				
				if (timeBased)
				{
					int tm=(((int)timestamp.Get(i))/(secondsGap*multiplier));
					
				
					if (tm!=lastTime)
					{	
						Mopen.Add(open.Get(i));
						Mhigh.Add(high.Get(i));
						Mlow.Add(low.Get(i));
						Mclose.Add(close.Get(i));
						Mtimestamp.Add(tm);
						lastTime=tm;
						shift++;
						if (lastTime!=0) bPopulated=true;
					}
					else
					{
						if (low.Get(i)<(double)Mlow[shift])
							Mlow[shift]=low.Get(i);
						if (high.Get(i)>(double)Mhigh[shift])
							Mhigh[shift]=high.Get(i);
						Mclose[shift]=close.Get(i);
						bPopulated=false;
					}
				}
				else
				{
					if ((((double)i)/multiplier)==((double)(i/multiplier)))
					{
						Mopen[i/multiplier]=open.Get(i);
						Mhigh[i/multiplier]=high.Get(i);
						Mlow[i/multiplier]=low.Get(i);
						Mclose[i/multiplier]=close.Get(i);
						shift=i/multiplier;
						bPopulated=true;
					}
					else
					{
						if (low.Get(i)<(double)Mlow[i/multiplier])
							Mlow[i/multiplier]=low.Get(i);
						if (high.Get(i)>(double)Mhigh[i/multiplier])
							Mhigh[i/multiplier]=high.Get(i);
						Mclose[i/multiplier]=close.Get(i);
						bPopulated=false;
					}
				}
				
				if (bPopulated && shift>0)
				{				
					if (modHA)
					{
						double haClose=((double)Mopen[shift-1] + (double)Mclose[shift-1] + (double)Mhigh[shift-1] + (double)Mlow[shift-1])/4;
						double haOpen=0;	
						
						if (firstBar)
							haOpen=(double)Mopen[shift-1];
						else
							haOpen=((double)HAopen[shift-2] + (double)HAclose[shift-2])/2;
						
						HAopen.Add(haOpen);
						HAclose.Add(haClose);
								
						int BarColor = haClose < haOpen ? -1 : 1;
										
						
						if (((shift-1)-compBars)>-1)
						{
							for (int value1=1;value1<compBars;value1++) 
							{
								double pastOpen=(double)HAopen[(shift-1)-value1];
								double pastClose=(double)HAclose[(shift-1)-value1];
							
								if ((haOpen  <= Math.Max(pastOpen,pastClose)) &&
									(haOpen  >= Math.Min(pastOpen,pastClose)) &&
									(haClose <= Math.Max(pastOpen,pastClose)) &&
									(haClose >= Math.Min(pastOpen,pastClose)) )
								{
									BarColor=(int)colorArr[value1];
								}
							}
							
							
							
							for (int value1=1;value1<compBars;value1++) 
							{
								colorArr[value1]=(int)colorArr[value1-1];
							}
						}						
						
						colorArr[0]=BarColor;
						HAcolor.Add(BarColor);
						
						
						lastColor=shift-1;
					}
					
					if (scalper)
					{
						if ((shift-1) <= 4)
						{
							ScalpBars.Add((int)0);
							lastBar++;
						}
						else
						{
							
							//Up 
							if (HighPainted==false)
							{
			
								if (((double)Mhigh[shift-2] < (double)Mhigh[shift-3]) && (0==ExtremeBar))								 
								{
									ExtremeBar = 2; 
									ExtremeBarHigh = (double)Mhigh[shift-3]; 
									TriggerBarLow = (double)Mlow[shift-2]; 
									TriggerBarLowNo = shift-2;
								}
									
								if ((double)Mhigh[shift-1] > ExtremeBarHigh)
								{
									ExtremeBar = 0; 
									ExtremeBarHigh = 0; 
									TriggerBarLow = 0; 
									TriggerBarLowNo = 0;
									counter=-1; 
								}
			
								if (0!=ExtremeBar) counter++; 
			
			
								if (((double)Mclose[shift-1] < TriggerBarLow) && (0!=ExtremeBar))
								{
				
									ExtremeBar = ExtremeBar + counter;
									HighPainted = true; 	
									counter=-1;
								
									ScalpBars.Add((shift)-ExtremeBar);
									lastBar++;
									ExtremeBar=0; 
									
								}
								
							}
			
							//Down
							if (HighPainted==true)
							{
								if (((double)Mlow[shift-2] > (double)Mlow[shift-3]) &&  (0==ExtremeBar))
								{
									ExtremeBar = 2; 
									ExtremeBarLow = (double)Mlow[shift-3]; 
									TriggerBarHigh = (double)Mhigh[shift-2]; 
									TriggerBarHighNo = shift-2; 
								}
			
								if ((double)Mlow[shift-1] < ExtremeBarLow)
								{ 
									ExtremeBar = 0; 
									ExtremeBarLow = 0; 
									TriggerBarHigh = 0; 
									TriggerBarHighNo = 0; 
									counter=-1; 
								}
								
								if (0!=ExtremeBar) counter++; 
								
								if (((double)Mclose[shift-1] > TriggerBarHigh) && (0!=ExtremeBar))
								{
									ExtremeBar = ExtremeBar + counter;                 
									HighPainted = false;
									counter=-1; 
									
								
									ScalpBars.Add((shift)-ExtremeBar);
									lastBar++;
									ExtremeBar=0;	
									
								}
							
							}
						}
						
					}
					
					firstBar=false;
				}
			}
			
        	int index = -1;
			Exception caughtException;
			
				
            int barPaintWidth = ChartControl.ChartStyle.GetBarPaintWidth(ChartControl.BarWidth);
			
			if (thisPlot==MultiTimeFramePlots.CandleStick && candleScale==MultiTimeFrameCandleScale.Two)
				barPaintWidth=barPaintWidth*(multiplier/2);
			if (thisPlot==MultiTimeFramePlots.CandleStick && candleScale==MultiTimeFrameCandleScale.Three)
				barPaintWidth=barPaintWidth*(multiplier);
			
			int bars = ChartControl.BarsPainted;
			int ii=shift;
			float lastX=0;
			float lastY=0;
			
            while (bars >= 0)
            {
				index = ((ChartControl.LastBarPainted - ChartControl.BarsPainted) ) + bars;
                if (ChartControl.ShowBarsRequired || ((index - base.Displacement) >= base.BarsRequired))
                {
					try
					{
						float x1 = (((ChartControl.CanvasRight - ChartControl.BarMarginRight) - (barPaintWidth / 2)) - ((base.ChartControl.BarsPainted - 1) * base.ChartControl.BarSpace)) + (bars * (base.ChartControl.BarSpace));
						
						float y1=0,y2=0,y3=0,y4=0;
						bool drawThis=true;
						bool ScalpBar=false;
						double yHigh=0, yLow=0;
						Color BarColor=Color.Transparent;
						
						if (timeBased)
						{
							for (ii=shift;ii>0;ii--)
							{
								if ((int)Mtimestamp[ii]==(((int)timestamp.Get(index))/(secondsGap*multiplier)))
								{
									if (modHA && showShadow)
									{
										yHigh=(double)Mhigh[ii];
										yLow = (double)Mlow[ii];
									}
									y1 = (bounds.Y + bounds.Height) - ((int) ((((double)Mhigh[ii] - min) / ChartControl.MaxMinusMin(max, min)) * bounds.Height));
									y2 = (bounds.Y + bounds.Height) - ((int) ((((double)Mclose[ii] - min) / ChartControl.MaxMinusMin(max, min)) * bounds.Height));
									y3 = (bounds.Y + bounds.Height) - ((int) ((((double)Mopen[ii] - min) / ChartControl.MaxMinusMin(max, min)) * bounds.Height));
									y4 = (bounds.Y + bounds.Height) - ((int) ((((double)Mlow[ii] - min) / ChartControl.MaxMinusMin(max, min)) * bounds.Height));
									drawThis=true;
									shift=ii;
									break;
								}
								drawThis=false;
							}							
						}
						else
						{
							shift=index/multiplier;
							
							if (modHA && showShadow)
							{
								yHigh=(double)Mhigh[shift];
								yLow = (double)Mlow[shift];
							}
							
							y1 = (bounds.Y + bounds.Height) - ((int) ((((double)Mhigh[shift] - min) / ChartControl.MaxMinusMin(max, min)) * bounds.Height));
							y2 = (bounds.Y + bounds.Height) - ((int) ((((double)Mclose[shift] - min) / ChartControl.MaxMinusMin(max, min)) * bounds.Height));
							y3 = (bounds.Y + bounds.Height) - ((int) ((((double)Mopen[shift] - min) / ChartControl.MaxMinusMin(max, min)) * bounds.Height));
                         	y4 = (bounds.Y + bounds.Height) - ((int) ((((double)Mlow[shift] - min) / ChartControl.MaxMinusMin(max, min)) * bounds.Height));
							
                 		}
						
						if (drawThis)
						{
							BarColor = (y2 > y3 ? barColorDown : barColorUp);
									
							
							if (modHA)
							{
								if (((shift)>0) && shift<=lastColor)
								{
									if (1==(int)HAcolor[shift]) BarColor=barColorUp;
									else BarColor=barColorDown;
									
									if (showShadow)
									{
										yHigh = (bounds.Y + bounds.Height) - ((int) (((Math.Max(yHigh, (double)HAopen[shift]) - min) / ChartControl.MaxMinusMin(max, min)) * bounds.Height));
										yLow = (bounds.Y + bounds.Height) - ((int) (((Math.Min(yLow, (double)HAopen[shift]) - min) / ChartControl.MaxMinusMin(max, min)) * bounds.Height));
									}
								}	
							}
							
							if (scalper)
							{
								for (int scalpBar=0;scalpBar<lastBar;scalpBar++)
                        		{
									if ((int)ScalpBars[scalpBar]==(shift+1))
									{
										ScalpBar=true;
										break;
									}
								}
							}
							
							
							switch (thisPlot)
							{
							  	case MultiTimeFramePlots.OHLC:
									x1 -= (float)(barPaintWidth*1.5);
									graphics.DrawLine(new Pen(BarColor,(int)(ChartControl.BarWidth)),x1-((float)ChartControl.BarWidth*(float)1.4),y3,x1,y3);	
									graphics.DrawLine(new Pen(BarColor,(int)(ChartControl.BarWidth)),x1+((float)ChartControl.BarWidth*(float)1.2),y2,x1,y2);
									if (ScalpBar) BarColor=ScalpBarBrush.Color;
									graphics.DrawLine(new Pen(BarColor,(int)((((double)ChartControl.BarWidth)/3)*2.5)),x1,y1-((float)ChartControl.BarWidth/2),x1,y4+((((float)ChartControl.BarWidth)/6)*(float)2.5));						
									break;
									
								case MultiTimeFramePlots.HiLoBars:	
									x1 -= (float)(barPaintWidth*1.5);
									if (ScalpBar) BarColor=ScalpBarBrush.Color;
									graphics.DrawLine(new Pen(BarColor,ChartControl.BarWidth),x1,y1,x1,y4);
							        break;
									
								case MultiTimeFramePlots.LineOnClose:
									x1 -= (float)(barPaintWidth*1);
									if (lastX>0)
										graphics.DrawLine(new Pen(barColorUp,(int)(ChartControl.BarWidth)),x1,y2,lastX,lastY);
									lastX=x1;
									lastY=y2;
									break;
								
								case MultiTimeFramePlots.CandleStick:
									Pen pen;
									Pen wickpen;
									if (candleScale==MultiTimeFrameCandleScale.One) x1 -= (float)(barPaintWidth*2);
									if (!modHA || shift>lastColor || ScalpBar)
									{
										byte bRed = (byte)~(ChartControl.BackColor.R);
										byte bGreen = (byte)~(ChartControl.BackColor.G);
										byte bBlue = (byte)~(ChartControl.BackColor.B);
										pen = new Pen(Color.FromArgb(bRed,bGreen,bBlue), 1);
										
										if (thickWick) wickpen = new Pen(Color.FromArgb(bRed,bGreen,bBlue), 3);
										else wickpen = new Pen(Color.FromArgb(bRed,bGreen,bBlue), 1);
										if (y2>y3) BarColor=barColorDown;
										else BarColor=barColorUp;
									}
									else
									{
										pen=new Pen(BarColor, 1);
										wickpen=pen;
									}
									
									if (ScalpBar)
									{
										if (modHA && showShadow && shift<=lastColor)
										{
											byte bRed = (byte)~(ChartControl.BackColor.R);
											byte bGreen = (byte)~(ChartControl.BackColor.G);
											byte bBlue = (byte)~(ChartControl.BackColor.B);
											pen = new Pen(Color.FromArgb(bRed,bGreen,bBlue), ShadowWidth);
																		
											graphics.DrawLine(pen,x1,(float)yHigh,x1,(float)yLow);
										}
										graphics.FillRectangle(ScalpBarBrush,x1-(barPaintWidth/2)+1,y1,barPaintWidth-2,y4-y1);								
									}
									
									if (y2>y3)
									{
										graphics.DrawLine(wickpen,x1,y1,x1,y3);
										graphics.DrawLine(wickpen,x1,y2,x1,y4);
										if (!ScalpBar) graphics.FillRectangle(new SolidBrush(BarColor),x1-(barPaintWidth/2),y3,barPaintWidth,y2-y3);
									}
									else
									{
										graphics.DrawLine(wickpen,x1,y1,x1,y2);
										graphics.DrawLine(wickpen,x1,y3,x1,y4);
										if (!ScalpBar) graphics.FillRectangle(new SolidBrush(BarColor),x1-(barPaintWidth/2),y2,barPaintWidth,y3-y2);
									}
									graphics.DrawLine(pen,x1-(barPaintWidth/2),y3,x1+(barPaintWidth/2),y3);
									graphics.DrawLine(pen,x1-(barPaintWidth/2),y2,x1+(barPaintWidth/2),y2);
									graphics.DrawLine(pen,x1-(barPaintWidth/2),y2,x1-(barPaintWidth/2),y3);
									graphics.DrawLine(pen,x1+(barPaintWidth/2),y2,x1+(barPaintWidth/2),y3);
									
									if (!ScalpBar && modHA && showShadow && shift<=lastColor)
									{
										byte bRed = (byte)~(ChartControl.BackColor.R);
										byte bGreen = (byte)~(ChartControl.BackColor.G);
										byte bBlue = (byte)~(ChartControl.BackColor.B);
										pen = new Pen(Color.FromArgb(bRed,bGreen,bBlue), ShadowWidth);
																	
										graphics.DrawLine(pen,x1,(float)yHigh,x1,(float)yLow);
									}
									break;
							}
							if (scalper && !ScalpBar && showTriggers)
							{
								double offset=0;
								
								if (!priceScale || (Digits==2)) offset=TickSize*2;
								else if (Digits<2) offset=(TickSize/priceSize)*2;
								else offset=(TickSize*priceSize)*2;
								
								if ((TriggerBarLowNo>0) && (TriggerBarLowNo==shift))
								{
									y1 = (bounds.Y + bounds.Height) - ((int) (((((double)Mhigh[shift]+offset) - min) / ChartControl.MaxMinusMin(max, min)) * bounds.Height));
									
									DrawArrow((barPaintWidth/5)*4, graphics,(int)x1, (int)y1,ScalpBarBrush,false);
								}
								if ((TriggerBarHighNo>0) && (TriggerBarHighNo==shift))
								{
									y1 = (bounds.Y + bounds.Height) - ((int) (((((double)Mlow[shift]-offset) - min) / ChartControl.MaxMinusMin(max, min)) * bounds.Height));
									
									DrawArrow((barPaintWidth/5)*4, graphics,(int)x1,(int)y1,ScalpBarBrush,true);
								}
							}
						}
					}
					catch (Exception exception) { caughtException = exception; }
               	}
				bars-=multiplier;					
			}
			
			StringBuilder 	strTimeFrame = new StringBuilder();
			strTimeFrame.Append((barSize*multiplier).ToString());
			strTimeFrame.Append(" ");
			strTimeFrame.Append(barType);
			
			graphics.FillRectangle(new SolidBrush(Color.Black), bounds.X, bounds.Y + 23, 15+(strTimeFrame.ToString().Length*5), 13);
			graphics.DrawString(strTimeFrame.ToString(), textFont, textBrush, bounds.X, bounds.Y + 23, stringFormat);
		
			Mopen.Clear();
			Mlow.Clear();
			Mhigh.Clear();
			Mclose.Clear();
			Mtimestamp.Clear();
			HAopen.Clear();
			HAclose.Clear();
			HAcolor.Clear();
			ScalpBars.Clear();
		}
		
		
		private void DrawArrow(int arrowSize, Graphics graphics, int x, int y, SolidBrush brush, bool up)
    	{
        	Point[] points;
        	int halfSize = (int) Math.Floor((double) (((double) arrowSize) / 2));
        	Point[] upPoints = new Point[] { new Point(0, 0), new Point(arrowSize, arrowSize), new Point(halfSize, arrowSize), new Point(halfSize, arrowSize * 2), new Point(-halfSize, arrowSize * 2), new Point(-halfSize, arrowSize), new Point(-arrowSize, arrowSize), new Point(0, 0) };
        	Point[] dnPoints = new Point[] { new Point(-halfSize, -arrowSize * 2), new Point(halfSize, -arrowSize * 2), new Point(halfSize, -arrowSize), new Point(arrowSize, -arrowSize), new Point(0, 0), new Point(-arrowSize, -arrowSize), new Point(-halfSize, -arrowSize), new Point(-halfSize, -arrowSize * 2) };

			if (up)
        	{
            	points = upPoints;
        	}
        	else
        	{
				points = dnPoints;
        	}
        	for (int i = 0; i < points.Length; i++)
        	{
            	points[i].Offset(x, y);
        	}
			
			byte bRed 	= (byte)~(base.ChartControl.BackColor.R);
            byte bGreen = (byte)~(base.ChartControl.BackColor.G);
			byte bBlue 	= (byte)~(base.ChartControl.BackColor.B);

            graphics.FillPolygon(brush, points);
            graphics.DrawPolygon(new Pen(Color.FromArgb(bRed,bGreen,bBlue)), points);        	        
    	}

        #region Properties
		[Description("Timeframe multiplier.")]
        [Category("Parameters")]
        public int Multiplier
        {
            get { return multiplier; }
            set { multiplier = Math.Max(1, value); }
        }
		/// <summary>
		/// </summary>
		[Description("Whether to display a thick wick on candlestick price plot.")]
		[Category("Plot")]
		[NinjaTrader.Gui.Design.DisplayName("Candle thick wick")]
		public bool ThickWick
        {
            get { return thickWick; }
			set { thickWick = value; }
        }
		
		[Description("Scale to display candles relative to original prices.")]
        [Category("Plot")]
		[NinjaTrader.Gui.Design.DisplayName("Candle scale")]
		public MultiTimeFrameCandleScale CandleScale
		{
			get { return candleScale; }
            set { candleScale = value; }
		}
		
		[Description("Scale prices to 2 decimal places as all indicators are limited to always display values to 2 decimal places.")]
        [Category("Plot")]
		[NinjaTrader.Gui.Design.DisplayName("Price indicator scale")]
		public bool PriceScale
        {
            get { return priceScale; }
			set { priceScale = value; }
        }
		
		/// <summary>
		/// </summary>
		[Description("Whether to override the default price plot.")]
		[Category("Plot")]
		[NinjaTrader.Gui.Design.DisplayName("Plot Override")]
		public bool OverridePlot
        {
            get { return plotOverride; }
			set { plotOverride = value; }
        }
		
		[Description("Type of price plot if default is overridden.")]
        [Category("Plot")]
		[NinjaTrader.Gui.Design.DisplayName("Plot Override Type")]
		public MultiTimeFramePlots PricePlot
		{
			get { return pricePlot; }
            set { pricePlot = value; }
		}
		
		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Override Color of down bars.")]
		[Category("Plot")]
		[Gui.Design.DisplayNameAttribute("Plot Override Down color")]
		public Color BarColorDown
		{
			get { return barColorDown; }
			set { barColorDown = value; }
		}
		
		/// <summary>
		/// </summary>
		[Browsable(false)]
		public string BarColorDownSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(barColorDown); }
			set { barColorDown = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Override Color of up bars.")]
		[Category("Plot")]
		[Gui.Design.DisplayNameAttribute("Plot Override Up color")]
		public Color BarColorUp
		{
			get { return barColorUp; }
			set { barColorUp = value; }
		}

		/// <summary>
		/// </summary>
		[Browsable(false)]
		public string BarColorUpSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(barColorUp); }
			set { barColorUp = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		[Description("Enable embedded ModHeikenAshi style")]
        [Category("Style - ModHeikenAshi")]
		[NinjaTrader.Gui.Design.DisplayName("Enable")]
		public bool ModHA
        {
            get { return modHA; }
			set { modHA = value; }
        }
		
		[Description("Number of look back bars for the embedded ModHeikenAshi to compare.")]
        [Category("Style - ModHeikenAshi")]
		[NinjaTrader.Gui.Design.DisplayName("No. of Bars")]
        public int CompBars
        {
            get { return compBars; }
            set { compBars = Math.Max(1, value); }
        }
		
		[Description("Whether to hide the shadow line on CandleStick charts.")]
		[Category("Style - ModHeikenAshi")]
		[NinjaTrader.Gui.Design.DisplayName("Shadow lines on Candles")]
		public bool ShowShadow
        {
            get { return showShadow; }
			set { showShadow = value; }
        }
		
		[XmlIgnore()]
		[Description("Width of shadow line on CandleStick charts.")]
		[Category("Style - ModHeikenAshi")]
		[Gui.Design.DisplayNameAttribute("Shadow width")]
		public int ShadowWidth
		{
			get { return shadowWidth; }
			set { shadowWidth = Math.Max(value, 1); }
		}
		
		[Description("Enable embedded Scalper style")]
        [Category("Style - Scalper")]
		[NinjaTrader.Gui.Design.DisplayName("Enable")]
		public bool MTFScalper
        {
            get { return scalper; }
			set { scalper = value; }
        }
		
		[Browsable(false)]
    	public string ScalpBarColorSerialize
    	{
        		get { return NinjaTrader.Gui.Design.SerializableColor.ToString(this.ScalpBarColor); }
        		set { this.ScalpBarColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
    	}
		
		[Description("Default Colour for Scalp Bar")]
		[Category("Style - Scalper")]
		[NinjaTrader.Gui.Design.DisplayName("Scalp bar color")]
		public Color ScalpBarColor
        {
			get { return this.ScalpBarBrush.Color; }
        	set { this.ScalpBarBrush = new SolidBrush(value); }
        }
		
		[Description("Highlights the current trigger bars by displaying an arrow above or below them.")]
        [Category("Style - Scalper")]
		[NinjaTrader.Gui.Design.DisplayName("Highlight current trigger bars")]
		public bool MTFScalperTriggers
        {
            get { return showTriggers; }
			set { showTriggers = value; }
        }
		
        #endregion
    }
}


#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private MTF[] cacheMTF = null;

        private static MTF checkMTF = new MTF();

        /// <summary>
        /// Multiple Timeframes
        /// </summary>
        /// <returns></returns>
        public MTF MTF(int multiplier)
        {
            return MTF(Input, multiplier);
        }

        /// <summary>
        /// Multiple Timeframes
        /// </summary>
        /// <returns></returns>
        public MTF MTF(Data.IDataSeries input, int multiplier)
        {
            checkMTF.Multiplier = multiplier;
            multiplier = checkMTF.Multiplier;

            if (cacheMTF != null)
                for (int idx = 0; idx < cacheMTF.Length; idx++)
                    if (cacheMTF[idx].Multiplier == multiplier && cacheMTF[idx].EqualsInput(input))
                        return cacheMTF[idx];

            MTF indicator = new MTF();
            indicator.SetUp();
            indicator.CalculateOnBarClose = CalculateOnBarClose;
            indicator.Input = input;
            indicator.Multiplier = multiplier;

            MTF[] tmp = new MTF[cacheMTF == null ? 1 : cacheMTF.Length + 1];
            if (cacheMTF != null)
                cacheMTF.CopyTo(tmp, 0);
            tmp[tmp.Length - 1] = indicator;
            cacheMTF = tmp;
            Indicators.Add(indicator);

            return indicator;
        }

    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Multiple Timeframes
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.MTF MTF(int multiplier)
        {
            return _indicator.MTF(Input, multiplier);
        }

        /// <summary>
        /// Multiple Timeframes
        /// </summary>
        /// <returns></returns>
        public Indicator.MTF MTF(Data.IDataSeries input, int multiplier)
        {
            return _indicator.MTF(input, multiplier);
        }

    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Multiple Timeframes
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.MTF MTF(int multiplier)
        {
            return _indicator.MTF(Input, multiplier);
        }

        /// <summary>
        /// Multiple Timeframes
        /// </summary>
        /// <returns></returns>
        public Indicator.MTF MTF(Data.IDataSeries input, int multiplier)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.MTF(input, multiplier);
        }

    }
}
#endregion
