// A little help to label the bars with H1, H2, H3, ... and L1, L2, L3, ... 
// As discussed in the book "Reading Price Charts Bar by Bar" by Al Brooks.
// No, the "indicator" is not able to guess the correct numbers of H1, H2 ... 
// The rules therefore are not strict enough. But you know them...
// We are always waiting for the close of the bar. So CalculateOnBarClose is set to true.
// v0.2 Outsidebar depending on previous labeling and close of outsidebar.
// If depending is set to "false", the outsidebar will be labeled with both H and L and 
// starts there from a neutral position. If depending is set to "true" , it is 
// labeled only with H (L) if the last label was L (H) and the bar close > open (close < open), 
// and "counting" will then not be set to neutral.
// v0.3 Possibility to plot only the last x-labels
// v0.4 Compatibility with NT7

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui.Design;
using NinjaTrader.Cbi;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    [Description("LegHL")]
	[Gui.Design.DisplayName("LegHL")]
	public class LegHL : Indicator
    {
		
        #region Variables
		private int H = 0;
		private int L = 0;
		private int sufH = 0;
		private int sufHold = 0;
		private int sufL = 0;
		private int sufLold = 0;
		private string LabelH = "H";
		private string LabelL = "L";
		private bool xLabel = true; // xLabel on/off
		private int xLabels = 3; // x labels on Cart
		private bool outsidedepends = true; // Outsidebar labeling depends on prior bars on/off 
		private int textTicksOffset = 2; // Default setting for text ticks offset from bars
		private Font textFont;
		private int textSize = 5; // Default text size
		private Color textColorH = Color.DimGray; // Color of H
		private Color textColorL = Color.DimGray; // Color of L
		#endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
			PaintPriceMarkers 	= false;
			Overlay				= true;		
			CalculateOnBarClose = true;
			textFont = new Font("Arial", textSize, FontStyle.Regular);
		}

        protected override void OnBarUpdate()	
        {			
			if (CurrentBar == 0)
			{
				return;
			}

			if (xLabel == false) // xLabel off
			{
				LabelH = "H"+CurrentBar;
				LabelL = "L"+CurrentBar;
			}
			else // xLabel on
			{
				{
					if (sufHold == xLabels)
					{
						sufH = 1;
					}
					else
					{
						sufH = sufHold + 1;
					}
						LabelH = "H"+sufH;
				}
				{
					if (sufLold == xLabels)
					{
						sufL = 1;
					}
					else
					{
						sufL = sufLold + 1;
					}
						LabelL = "L"+sufL;
				}
			}
			if (High[0] > High[1] && Low[0] < Low[1] && FirstTickOfBar) // outside bar
			{
				if ( outsidedepends == true) // Outsidebar labeling depends on prior labeling
				{
					if (L == 1 && Close[0] > Open[0] && FirstTickOfBar) // outside H
					{
						DrawText(LabelH, false, "H", 0, High[0] + TextTicksOffset * TickSize, 0, textColorH, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						H = 1;
						L = 0;
						sufHold = sufH;
					}
					else if (H == 1 && Close[0] < Open[0] && FirstTickOfBar)// outside L
					{	
						DrawText(LabelL, false, "L", 0, Low[0] - TextTicksOffset * TickSize, 0, textColorL, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						H = 0;
						L = 1;
						sufLold = sufL;
					}
					else  if (FirstTickOfBar) // Outsidebar labeling starts neutral with H and L
					{
						DrawText(LabelH, false, "H", 0, High[0] + TextTicksOffset * TickSize, 0, textColorH, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						DrawText(LabelL, false, "L", 0, Low[0] - TextTicksOffset * TickSize, 0, textColorL, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
						H = 1;
						L = 1;
						sufHold = sufH;
						sufLold = sufL;	
					}
				}
				else if (FirstTickOfBar) // Outsidebar labeling starts neutral with H and L
				{
					DrawText(LabelH, false, "H", 0, High[0] + TextTicksOffset * TickSize, 0, textColorH, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
					DrawText(LabelL, false, "L", 0, Low[0] - TextTicksOffset * TickSize, 0, textColorL, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
					H = 1;
					L = 1;
					sufHold = sufH;	
					sufLold = sufL;	
				}	
			}
			else if (High[0] > High[1] && L == 1 && FirstTickOfBar) // H bar
			{
				DrawText(LabelH, false, "H", 0, High[0] + TextTicksOffset * TickSize, 0, textColorH, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
				H = 1;
				L = 0;
				sufHold = sufH;	
			}
			else if (Low[0] < Low[1] && H == 1 && FirstTickOfBar) // L bar
			{
				DrawText(LabelL, false, "L", 0, Low[0] - TextTicksOffset * TickSize, 0, textColorL, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
				H = 0;
				L = 1;
				sufLold = sufL;
			}
			
		}


        #region Properties
		
		[Description("Outsidebar labeling depends on prior labeling or not ")]
        [Category("Parameters")]
        public bool Outsidedepends
        {
            get { return outsidedepends; }
            set { outsidedepends = value; }
        }
		
		[Description("x-Label on/off ")]
        [Category("Parameters")]
        public bool XLabel
        {
            get { return xLabel; }
            set { xLabel = value; }
        }
		
		[Description("x-Labels (H and L) shown on chart")]
        [Category("Parameters")]
        public int XLabels
        {
            get { return xLabels; }
            set { xLabels = Math.Max(1, value); }
        }
		
		[Description("Text ticks away from bar")]
        [Category("Parameters")]
        public int TextTicksOffset
        {
            get { return textTicksOffset; }
            set { textTicksOffset = Math.Max(1, value); }
        }
		
		[Description("Text size")]
        [Category("Parameters")]
        public int TextSize
        {
            get { return textSize; }
            set { textSize = Math.Max(1, value); }
        }
		
		[Description("Color of H")]
        [Category("Visual")]
		[Gui.Design.DisplayName ("textColorH")]
        public Color TextColorH
        {
            get { return textColorH; }
            set { textColorH = value; }
        }
		[Browsable(false)]
		public string TextColorHSerialise
		{
			get { return Gui.Design.SerializableColor.ToString(textColorH); }
			set { textColorH = Gui.Design.SerializableColor.FromString(value); }
		}
		
		[Description("Color of L")]
        [Category("Visual")]
		[Gui.Design.DisplayName ("textColorL")]
        public Color TextColorL
        {
            get { return textColorL; }
            set { textColorL = value; }
		}
		[Browsable(false)]
		public string TextColorLSerialise
		{
			get { return Gui.Design.SerializableColor.ToString(textColorL); }
			set { textColorL = Gui.Design.SerializableColor.FromString(value); }
		}	
		#endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private LegHL[] cacheLegHL = null;

        private static LegHL checkLegHL = new LegHL();

        /// <summary>
        /// LegHL
        /// </summary>
        /// <returns></returns>
        public LegHL LegHL(bool outsidedepends, int textSize, int textTicksOffset, bool xLabel, int xLabels)
        {
            return LegHL(Input, outsidedepends, textSize, textTicksOffset, xLabel, xLabels);
        }

        /// <summary>
        /// LegHL
        /// </summary>
        /// <returns></returns>
        public LegHL LegHL(Data.IDataSeries input, bool outsidedepends, int textSize, int textTicksOffset, bool xLabel, int xLabels)
        {
            if (cacheLegHL != null)
                for (int idx = 0; idx < cacheLegHL.Length; idx++)
                    if (cacheLegHL[idx].Outsidedepends == outsidedepends && cacheLegHL[idx].TextSize == textSize && cacheLegHL[idx].TextTicksOffset == textTicksOffset && cacheLegHL[idx].XLabel == xLabel && cacheLegHL[idx].XLabels == xLabels && cacheLegHL[idx].EqualsInput(input))
                        return cacheLegHL[idx];

            lock (checkLegHL)
            {
                checkLegHL.Outsidedepends = outsidedepends;
                outsidedepends = checkLegHL.Outsidedepends;
                checkLegHL.TextSize = textSize;
                textSize = checkLegHL.TextSize;
                checkLegHL.TextTicksOffset = textTicksOffset;
                textTicksOffset = checkLegHL.TextTicksOffset;
                checkLegHL.XLabel = xLabel;
                xLabel = checkLegHL.XLabel;
                checkLegHL.XLabels = xLabels;
                xLabels = checkLegHL.XLabels;

                if (cacheLegHL != null)
                    for (int idx = 0; idx < cacheLegHL.Length; idx++)
                        if (cacheLegHL[idx].Outsidedepends == outsidedepends && cacheLegHL[idx].TextSize == textSize && cacheLegHL[idx].TextTicksOffset == textTicksOffset && cacheLegHL[idx].XLabel == xLabel && cacheLegHL[idx].XLabels == xLabels && cacheLegHL[idx].EqualsInput(input))
                            return cacheLegHL[idx];

                LegHL indicator = new LegHL();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.Outsidedepends = outsidedepends;
                indicator.TextSize = textSize;
                indicator.TextTicksOffset = textTicksOffset;
                indicator.XLabel = xLabel;
                indicator.XLabels = xLabels;
                Indicators.Add(indicator);
                indicator.SetUp();

                LegHL[] tmp = new LegHL[cacheLegHL == null ? 1 : cacheLegHL.Length + 1];
                if (cacheLegHL != null)
                    cacheLegHL.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheLegHL = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// LegHL
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.LegHL LegHL(bool outsidedepends, int textSize, int textTicksOffset, bool xLabel, int xLabels)
        {
            return _indicator.LegHL(Input, outsidedepends, textSize, textTicksOffset, xLabel, xLabels);
        }

        /// <summary>
        /// LegHL
        /// </summary>
        /// <returns></returns>
        public Indicator.LegHL LegHL(Data.IDataSeries input, bool outsidedepends, int textSize, int textTicksOffset, bool xLabel, int xLabels)
        {
            return _indicator.LegHL(input, outsidedepends, textSize, textTicksOffset, xLabel, xLabels);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// LegHL
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.LegHL LegHL(bool outsidedepends, int textSize, int textTicksOffset, bool xLabel, int xLabels)
        {
            return _indicator.LegHL(Input, outsidedepends, textSize, textTicksOffset, xLabel, xLabels);
        }

        /// <summary>
        /// LegHL
        /// </summary>
        /// <returns></returns>
        public Indicator.LegHL LegHL(Data.IDataSeries input, bool outsidedepends, int textSize, int textTicksOffset, bool xLabel, int xLabels)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.LegHL(input, outsidedepends, textSize, textTicksOffset, xLabel, xLabels);
        }
    }
}
#endregion
