#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
using System.Windows.Forms;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// ABC Fibs via 3 click alt-mouse events (once started, must finish sequence)
	/// Click below bar mid-value on 'A' click, will setup ABC as UP SWING (else Down Swing)
	///   (Up Swing will use low of A, High of B, and Low of C bar)
	/// V 1.0 - Dec 8, 2010, DeanV@nt
	/// 
    /// </summary>
    [Description("Alt-RightClick 3x to Draw ABC points and some related Fibs (A-bar click (above/below bar mid value) defines UP vs DOWN swings")]
    public class dDrawABC : Indicator
    {
        #region Variables
		
		private double adjustableFib = 1.786;
		private bool _fDragging = false;

		private ChartControl _chartControl;
		private Rectangle _chartBounds;
		private double _min, _max;

		//make these global and set on plot call, so we don't have to pass around.
		private int rightScrnBnum;	//right most bar# on screen.
		private int leftScrnBnum;	//left most bar# in display
		
		private int _iBarStart,_iBarEnd;
		private double _priceStart, _priceEnd;

		private Font 			textFont 		= new Font("Courier", 9, FontStyle.Regular);
		private Font 			textFontLabel	= new Font("Courier", 12, FontStyle.Regular);
		private int 			textOffset		= 15;
		private int 			textOffset2		= 30;
		private int 			textOffset3		= 45;
		private Color			colorUp			= Color.Green;
		private Color			colorDn			= Color.Red;
		private Color			swingColor		= Color.Black;

		private int swingDir	= 1;
		private int rightMaxBarNum = 0;

		private ILine lineAB = null;
		private ILine lineBC = null;
		
		//hidden saves for first time restore
		private DateTime _aPDate;
		private DateTime _bPDate;
		private DateTime _cPDate;
		private double _aPVal;
		private double _bPVal;
		private double _cPVal;
		
		private int _ABbarStart,_ABbarEnd;
		private double _ABpriceStart, _ABpriceEnd;
		private int _BCbarStart,_BCbarEnd;
		private double _BCpriceStart, _BCpriceEnd;
		private bool _fDrawAB = false;
		private bool _fDrawBC = false;

		private double[] FibPct = new double[14] ;	//prestuffed.
		private double[] FibVal = new double[14] ;
		
		private Color fibABColor		= Color.Gray;
		private Color fibBCColor		= Color.RoyalBlue;
		private int fibABRets = 3;
		private int fibBCRets = 1;
		private int fibABExtends = 5;
		private int fibBCExtends = 4;
		private int fibMMRets = 4;
		private int fibMMExtends = 5;
		private int fibMMReverse = 5;
		private bool fibABTextToRight = false;
		private bool fibBCTextToRight = true;
		private bool fibMMTextToRight = false;
		
		#endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
            Overlay				= true;
			//PlotsConfigurable = false;
			
			BarsRequired = 0;	//so we can draw with first 20 bars on screen, without error.
			
			//right most bar will be zero if calc@end of bar, or -1 if calc@each tick
			//Limit to existing bars so we can get Hi/Low info
			rightMaxBarNum = CalculateOnBarClose ? -1 : 0;
			
			//pre-stuff fib percentages, top to bottom.
			FibPct[0] = adjustableFib;	//!0 = show this level
			
			FibPct[1] = 2.0 ;
			FibPct[2] =  1.618 ;
			FibPct[3] = 1.272 ;	//continuation extentions

			FibPct[4] = 1.0 ;	// 100% of move
			//these are backwards for proper display, cause of later math.
			FibPct[5] = 0.764 ;	//not quite accurate, but close enough
			FibPct[6] = 0.618 ;
			FibPct[7] =  0.5 ;
			FibPct[8] = 0.382 ;
			FibPct[9] = 0.236 ;
			FibPct[10] = 0.0 ;	//reference point, low for up, hi for down.
			
			FibPct[11] = -0.272 ;	//retractions
			FibPct[12] = -0.618 ;
			FibPct[13] = -1.0 ;
			
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
			SetChartControl (this.ChartControl);
	
			if(lineAB == null) {
				RestoreDrawObjects();
			}

        }		
		
        #region Properties

        [Description("Adjustable Fib Level (+= Extend, -= Retract, -0.786r, 1.764e, 0=no). Effects All ABC points.")]
        [GridCategory("DisplayFibs")]
        public double AdjustableFib
        {
            get { return adjustableFib; }
            set { adjustableFib = value; }
        }

		[Description("Show AB Retract Levels (0=No, 1= 50%, 2= 38+61, 3=1+2, 4=2+78, 5=All.")]
        [GridCategory("DisplayFibs")]
        public int FibABRets
        {
            get { return fibABRets; }
            set { fibABRets = Math.Max(0, value); }
        }
        [Description("Show BC Retract Levels (0=No, 1= 50%, 2= 38+61, 3=1+2, 4=2+78, 5=All.")]
        [GridCategory("DisplayFibs")]
        public int FibBCRets
        {
            get { return fibBCRets; }
            set { fibBCRets = Math.Max(0, value); }
        }
        [Description("Show Extended Levels (0=No, 1= 127, 2= 161, 3=200, 4=161+200, 5=All 3.")]
        [GridCategory("DisplayFibs")]
        public int FibABExtends
        {
            get { return fibABExtends; }
            set { fibABExtends = Math.Max(0, value); }
        }
        [Description("Show Extended Levels (0=No, 1= 127, 2= 161, 3=200, 4=161+200, 5=All 3.")]
        [GridCategory("DisplayFibs")]
        public int FibBCExtends
        {
            get { return fibBCExtends; }
            set { fibBCExtends = Math.Max(0, value); }
        }
        [Description("Show MM Wrong Direction Levels (0=No, 1= -27, 2= -61, 3=-100, 4=61+100, 5=All 3.")]
        [GridCategory("DisplayFibs")]
        public int FibMMReverse
        {
            get { return fibMMReverse; }
            set { fibMMReverse = Math.Max(0, value); }
        }
        [Description("Show MM Retract Levels (0=No, 1= 50%, 2= 38+61, 3=1+2, 4=2+78, 5=All.")]
        [GridCategory("DisplayFibs")]
        public int FibMMRets
        {
            get { return fibMMRets; }
            set { fibMMRets = Math.Max(0, value); }
        }
        [Description("Show MM Extended Levels (0=No, 1= 127, 2= 161, 3=200, 4=161+200, 5=All")]
        [GridCategory("DisplayFibs")]
        public int FibMMExtends
        {
            get { return fibMMExtends; }
            set { fibMMExtends = Math.Max(0, value); }
        }
        [Description("Show AB Text Labels Right Justified")]
        [GridCategory("DisplayFibs")]
        public bool FibABTextToRight
        {
            get { return fibABTextToRight; }
            set { fibABTextToRight = value; }
        }
        [Description("Show BC Text Labels Right Justified")]
        [GridCategory("DisplayFibs")]
        public bool FibBCTextToRight
        {
            get { return fibBCTextToRight; }
            set { fibBCTextToRight = value; }
        }
        [Description("Show MM Text Labels Right Justified")]
        [GridCategory("DisplayFibs")]
        public bool FibMMTextToRight
        {
            get { return fibMMTextToRight; }
            set { fibMMTextToRight = value; }
        }

		/// <summary>
		/// Hidden previous save points for startup restore
		/// (no easy way to hide, so just move on...)
		/// </summary>
        [Category("saveme")]
        //[XmlInclude()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
		//[Browsable(false)]
        public DateTime APDate
        {
            get { return _aPDate; }
            set { _aPDate = value; }
        }
		// Serialize our object
		//[Browsable(false)]
		//public String _aPDateSerialize
		//{
		//	get { return _aPDate.ToString(); }
		//	set { _aPDate = DateTime.Parse(value); }
		//	//get { return _aPDate; }
		//	//set { _aPDate = (value); }
		//}
        [Category("saveme")]
		//[Browsable(false)]
        public double APVal
        {
            get { return _aPVal; }
            set { _aPVal = value; }
        }
        [Category("saveme")]
		//[Browsable(false)]
        public DateTime BPDate
        {
            get { return _bPDate; }
            set { _bPDate = value; }
        }
        [Category("saveme")]
		//[Browsable(false)]
        public double BPVal
        {
            get { return _bPVal; }
            set { _bPVal = value; }
        }
        [Category("saveme")]
		//[Browsable(false)]
        public DateTime CPDate
        {
            get { return _cPDate; }
            set { _cPDate = value; }
        }
        [Category("saveme")]
		//[Browsable(false)]
        public double CPVal
        {
            get { return _cPVal; }
            set { _cPVal = value; }
        }

		//Color stuff
		//
		[XmlIgnore()]
		[Description("Swing Color Up (also MM Fibs)")]
		[Category("Display Settings")]
		public Color SwingColorUp
		{
			get { return colorUp; }
			set { colorUp = value; }
		}
		// Serialize our Color object
		[Browsable(false)]
		public string ColorUpSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(colorUp); }
			set { colorUp = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		[XmlIgnore()]
		[Description("Swing Color Down (also MM Fibs)")]
		[Category("Display Settings")]
		public Color SwingColorDn
		{
			get { return colorDn; }
			set { colorDn = value; }
		}
		// Serialize our Color object
		[Browsable(false)]
		public string ColorDnSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(colorDn); }
			set { colorDn = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}

		[XmlIgnore()]
		[Description("AB Fib Color")]
		[Category("Display Settings")]
		public Color FibABColor
		{
			get { return fibABColor; }
			set { fibABColor = value; }
		}
		// Serialize our Color object
		[Browsable(false)]
		public string FibABColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(fibABColor); }
			set { fibABColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		[XmlIgnore()]
		[Description("BC Fib Color")]
		[Category("Display Settings")]
		public Color FibBCColor
		{
			get { return fibBCColor; }
			set { fibBCColor = value; }
		}
		// Serialize our Color object
		[Browsable(false)]
		public string FibBCColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(fibBCColor); }
			set { fibBCColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}

		#endregion
		
		int BarFromX (int x)
		{
			//returns BarsAgo
			int leftBarPix = _chartControl.GetXByBarIdx(Bars,leftScrnBnum);
			int leftOffset =  leftBarPix - ChartControl.BarSpace > 0 ? leftBarPix - ChartControl.BarSpace : 0;
			//Limit to existing left bar #, adjust for leftscreen whitespace as need.
			int myX = x < leftBarPix ? ChartControl.BarSpace : x-leftOffset;
			int xOffset = ((myX- (ChartControl.BarSpace/2)) /ChartControl.BarSpace);
			int thisBar = (CurrentBar - leftScrnBnum) -  xOffset;	//BarsAgo
			
			//Limit to existing right bar #
			if(thisBar < rightMaxBarNum) thisBar = rightMaxBarNum;	
								
			return  thisBar;
			
		}

		double PriceFromY (int y)
		{
			double bottom = _chartBounds.Bottom;
			return RoundInst(_min + (bottom - y) * (_max - _min) / _chartBounds.Height);
		}

		void ChartControl_MouseDown (object sender, System.Windows.Forms.MouseEventArgs e)
		{
			if ((Control.ModifierKeys & Keys.Alt) == 0) {
				//Print("not Alt");
				return;
			}

			if(_fDragging == false) {
				//start a new one
				RemoveDrawObjects();	//delete all object from this application
				_iBarStart = BarFromX (e.X);
				_priceStart = PriceFromY (e.Y);
				_fDrawAB = true;
				_fDrawBC = false;
				_fDragging = true;
				
				_ABbarStart = _iBarStart;
				//place 'A' text
				double midBarAdd = (High[_iBarStart] - Low[_iBarStart]) *0.5;
				swingDir = _priceStart >= Low[_iBarStart] + midBarAdd ? -1 : 1;
				swingColor = swingDir > 0 ? colorUp : colorDn;
				_priceStart = swingDir < 0 ? High[_iBarStart] : Low[_iBarStart];
				_ABpriceStart = _priceStart;
				
				DrawText("dDwALabel", AutoScale, "A",
					_iBarStart, _priceStart, textOffset2 * swingDir *-1, swingColor, textFontLabel,
					StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
				DrawText("dDwAPct", AutoScale, _priceStart.ToString(),
					_iBarStart, _priceStart, textOffset * swingDir *-1, swingColor, textFont,
					StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
				
			}
			else if(_fDrawAB == true) {
				//finish AB, start BC
				//save points for first time restore
				_aPDate = lineAB.StartTime;
				_aPVal = lineAB.StartY;
				_bPDate = lineAB.EndTime;
				_bPVal = lineAB.EndY;
				
				_fDrawAB = false;
				_fDrawBC = true;
				
				_ABbarEnd = _iBarEnd;	//current position from mousemove
				_ABpriceEnd = _priceEnd;
				_BCbarStart = _ABbarEnd;	//current position
				_BCpriceStart = _ABpriceEnd;
				//reset line start to draw BC
				_iBarStart = _BCbarStart;
				_priceStart = _BCpriceStart;
				
				//Place 'B' text on bar H or low
				double moveVal = RoundInst(Math.Abs(_ABpriceStart -_ABpriceEnd));
				
				DrawText("dDwBLabel", AutoScale, "B",
					_iBarStart, _priceStart, textOffset2 * swingDir, swingColor, textFontLabel,
					StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
				DrawText("dDwBPrice", AutoScale, moveVal.ToString() +"c - " + _priceStart.ToString(),
					_iBarStart, _priceStart, textOffset * swingDir, swingColor, textFont,
					StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
			}

			else if(_fDrawBC){
				_cPDate = lineBC.EndTime;
				_cPVal = lineBC.EndY;
				//finish BC draw
				_fDrawBC = false;
				_fDragging = false;
				_BCbarEnd = _iBarEnd;	//current position from mousemove
				_BCpriceEnd = _priceEnd;
				
				DoDrawFibs(1, fibABRets, fibABExtends, fibABColor);
				DoDrawFibs(2, fibBCRets, fibBCExtends, fibBCColor);
				DoDrawFibs(3, fibMMRets, fibMMExtends, swingColor);

				DrawText("dDwCLabel", AutoScale, "C",
					_iBarEnd, _priceEnd, textOffset3 * swingDir *-1, swingColor, textFontLabel,
					StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
				//move value and %
				double moveVal = RoundInst(Math.Abs(_BCpriceStart -_BCpriceEnd));
				double movePct = (moveVal / Math.Abs(_ABpriceStart -_ABpriceEnd)) * 100;
				
				DrawText("dDwCPct", AutoScale,  movePct.ToString("f1") + "%",
					_iBarEnd, _priceEnd, textOffset2 * swingDir *-1, swingColor, textFont,
					StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
				DrawText("dDwCPrice", AutoScale, moveVal.ToString() +"c - " + _priceEnd.ToString(),
					_iBarEnd, _priceEnd, textOffset * swingDir *-1, swingColor, textFont,
					StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
				
			}
			else {
				_fDragging = false; //error, turn off sequence
			}
			
		}

		void ChartControl_MouseMove (object sender, System.Windows.Forms.MouseEventArgs e)
		{
			if (!_fDragging)
				return;

			_iBarEnd = BarFromX (e.X);

			if(_fDrawAB) {
				_priceEnd = swingDir > 0 ? High[_iBarEnd] : Low[_iBarEnd];
				lineAB = UDrawLine("dDwABLine");
			}
			else {
				_priceEnd = swingDir < 0 ? High[_iBarEnd] : Low[_iBarEnd];
				lineBC = UDrawLine("dDwBCLine");
			}
			
		}


		public override void Plot (Graphics graphics, Rectangle bounds, double min, double max)
		{
			base.Plot (graphics, bounds, min, max);

			_chartBounds = bounds;
			_min = min;
			_max = max;

			rightScrnBnum = Math.Min(LastBarIndexPainted, CurrentBar); //limit to actual bars vs. white space on right.
			leftScrnBnum = Math.Min(FirstBarIndexPainted, CurrentBar); //not bigger than CurrentBar (ie. only 1 bar on screen).
		}

		void SetChartControl (ChartControl control)
		{
			if (_chartControl == control)
				return;

			if (_chartControl != null)
			{
				_chartControl.ChartPanel.MouseDown -= new System.Windows.Forms.MouseEventHandler (ChartControl_MouseDown);
				_chartControl.ChartPanel.MouseMove -= new System.Windows.Forms.MouseEventHandler (ChartControl_MouseMove);
				//_chartControl.ChartPanel.MouseUp -= new System.Windows.Forms.MouseEventHandler (ChartControl_MouseUp);
			}

			_chartControl = control;

			if (_chartControl != null)
			{
				_chartControl.ChartPanel.MouseDown += new System.Windows.Forms.MouseEventHandler (ChartControl_MouseDown);
				_chartControl.ChartPanel.MouseMove += new System.Windows.Forms.MouseEventHandler (ChartControl_MouseMove);
				//_chartControl.ChartPanel.MouseUp += new System.Windows.Forms.MouseEventHandler (ChartControl_MouseUp);
			}
		}

		public override void Dispose ()
		{
			base.Dispose ();
			SetChartControl (null);
		}
		
		//round to nearist tick size
		private double RoundInst(double tdbl) {
			
			if(Instrument != null) {	//check incase not called from OnBarUpdate().?
				return Instrument.MasterInstrument.Round2TickSize(tdbl);
			}
			
			return tdbl;
		}
		
		private ILine UDrawLine(string labelStr) {
			ILine Lrtn = null;
			
			//TriggerCustomEvent (
			//	delegate (object state) {
			//		try
			//		{
					Lrtn = DrawLine (
						labelStr,false,
						_iBarStart, _priceStart,
						_iBarEnd, _priceEnd,
						swingColor, DashStyle.Solid, 2);
					
			//		}
			//		catch { }
			//	},
			//	null
			//);
			
			return Lrtn;
		}


		private void DoDrawFibs(int fType, int rLevel, int eLevel, Color tColor) {
			double tSwgStartVal, tSwgEndVal;
			int tSwgStartBar, tSwgEndBar, lineEndBar;
			string fBaseStr = "dDwFib";
			int swingBarsHalf;
			int textpositionbar;
			StringAlignment textAlign;
			
			//setup per fType...
			if(fType == 3) {
				if(rLevel == 0 && eLevel == 0 && fibMMReverse == 0) return;	//nothing to do here.
				fBaseStr = fBaseStr + "MM";
				tSwgEndVal = _ABpriceEnd;
				tSwgStartVal = _ABpriceStart;
				tSwgEndBar = _BCbarEnd;
				tSwgStartBar = _ABbarStart;
				swingBarsHalf = (tSwgStartBar - tSwgEndBar) - 3; //so we start @ C point
				lineEndBar = _BCbarEnd - (int) Math.Abs((tSwgStartBar - tSwgEndBar) *1);
				textpositionbar = fibMMTextToRight ? lineEndBar : tSwgStartBar - swingBarsHalf;
				textAlign = fibMMTextToRight ? StringAlignment.Near : StringAlignment.Far;
			}
			else if(fType == 2) {
				if(rLevel == 0 && eLevel == 0) return;	//nothing to do here.
				fBaseStr = fBaseStr + "BC";
				tSwgEndVal = _BCpriceEnd;
				tSwgStartVal = _BCpriceStart;
				tSwgEndBar = _BCbarEnd;
				tSwgStartBar = _BCbarStart;
				swingBarsHalf = (int)((tSwgStartBar - tSwgEndBar) *0.5);
				lineEndBar = _BCbarEnd - (int) Math.Abs((tSwgStartBar - tSwgEndBar) *0.5);
				textpositionbar = fibBCTextToRight ? lineEndBar : tSwgStartBar - swingBarsHalf;
				textAlign = fibBCTextToRight ? StringAlignment.Near : StringAlignment.Far;
			}
			else { //if(fType == 1) {
				if(rLevel == 0 && eLevel == 0) return;	//nothing to do here.
				fBaseStr = fBaseStr + "AB";
				tSwgEndVal = _ABpriceEnd;
				tSwgStartVal = _ABpriceStart;
				tSwgEndBar = _ABbarEnd;
				tSwgStartBar = _ABbarStart;
				swingBarsHalf = (int)((tSwgStartBar - tSwgEndBar) *0.5);
				lineEndBar = _BCbarEnd;
				textpositionbar = fibABTextToRight ? lineEndBar : tSwgStartBar - swingBarsHalf;
				textAlign = fibABTextToRight ? StringAlignment.Near : StringAlignment.Far;
			}
			int swDirUp = (tSwgEndVal > tSwgStartVal ? 1 : -1);	//not correct for outside bar.

			//calc begin and end of lines
			double swgRange = Math.Abs(tSwgStartVal - tSwgEndVal);

			string fibString = null;
			for(int y = 0; y <= 13; y++) {
				//skip those not needed
				if(y == 0 && FibPct[0] == 0) continue;	//skip adjustable one
				
				if(rLevel < 5) {
					if(rLevel == 0) if(y > 3 && y < 11) continue;	//skip all retracts
					if(rLevel == 4 && y == 7) continue;	//skip 50%
					if(rLevel <= 3) if(y == 5 || y == 9) continue;	//skip .76 & .23
					if(rLevel == 2 && y == 7) continue;	//only show .38 & .81
					if(rLevel == 1) if(y == 6 || y == 8) continue;	//only show .5
				}
				if(eLevel < 5) { //do both sides
					if(eLevel == 0) if(y < 4 || y > 10) continue;	//skip all extends
					if(eLevel == 4) if(y == 3 || y == 11) continue;	//skip 1.27 both sides
					if(eLevel == 3) if(y == 3 || y == 11 || y == 12 || y == 2) continue;	//show  200 both sides
					if(eLevel == 2) if(y == 3 || y == 11 || y == 13 || y == 1) continue;	//show  161 both sides
					if(eLevel == 1) if(y == 2 || y == 12 || y == 13 || y == 1) continue;	//show  127 both sides
				}				
				if(fType == 3) {
					if(fibMMReverse < 5) { //skip somethings
						if(fibMMReverse == 0) if(y < 4 ) continue;	//skip all extends
						if(fibMMReverse == 4) if(y == 3) continue;	//skip -.27
						if(fibMMReverse == 3) if(y == 3 || y == 2) continue;	//show -100
						if(fibMMReverse == 2) if(y == 3 || y == 1) continue;	//show  -61
						if(fibMMReverse == 1) if(y == 2 || y == 1) continue;	//show  -27
					}				
						
					int x = 14 - y;	//reverse direction for this
					if(x == 14) x = 0;	//no 14, adjustable is at zero.
					FibVal[y] = _BCpriceEnd + ((swgRange * FibPct[x]) * swDirUp);
					fibString = ( (FibPct[x] * 100)).ToString() + "%m";
				}
				else {
					if(y == 0) {	//adjustable %
						double fadjust =  (swgRange * FibPct[y]);
						if(FibPct[y] < 0) fadjust = swgRange+(fadjust);
						else if(FibPct[y] < 1) fadjust = swgRange-(fadjust);
						FibVal[y] = tSwgStartVal + (fadjust * swDirUp);

						if(FibPct[y] < 0) fibString = ((FibPct[y] * -100)).ToString() + "%r";
						else if(FibPct[y] < 1) fibString = ((FibPct[y] * 100)).ToString() + "%r";
						else fibString = ((FibPct[y] * 100)).ToString() + "%e";//adjust extention labels
					}
					else {
					FibVal[y] = tSwgStartVal + ((swgRange * FibPct[y]) * swDirUp);
					if(y <= 3) fibString = ((FibPct[y] * 100)).ToString() + "%e";//adjust extention labels
					else fibString = (100 - (FibPct[y] * 100)).ToString() + "%r";
					}
				}
				
				DrawLine (fBaseStr + fibString, false, tSwgStartBar - swingBarsHalf, 
					FibVal[y], lineEndBar, FibVal[y], tColor, DashStyle.Dash, 1);
				DrawText(fBaseStr +"text" + fibString, false,  fibString,
					textpositionbar, FibVal[y], 2 , tColor, textFont,
					textAlign, Color.Transparent, Color.Transparent, 5);
			}
		}
			
		// Draw objects after any reload of data (if possible)
		// currently wont restore if one of points is -1 (unfinished bar, when calc per bar close)
		private void RestoreDrawObjects() {
			
			//check all dates for bars to validate they exist.
			//if(_aPDate == DateTime.MinValue) return;	//works, but could still be older than oldest bar.
			//make sure All our saved dates are available on chart (null save = 01/01/0001).
			DateTime oldestDate = Time[CurrentBar];
			if(oldestDate > _aPDate || oldestDate > _bPDate || oldestDate > _cPDate) return;
			if(Time[0] < _aPDate || Time[0] < _bPDate || Time[0] < _cPDate) return;
			
			swingDir = _aPVal >= _bPVal ? -1 : 1;
			swingColor = swingDir > 0 ? colorUp : colorDn;

			//Draw AB line and text, using date info
			lineAB = DrawLine ("dDwABLine",false, _aPDate, _aPVal  ,_bPDate, _bPVal, swingColor, DashStyle.Solid, 2);
			lineBC = DrawLine ("dDwBCLine",false, _bPDate, _bPVal  ,_cPDate, _cPVal, swingColor, DashStyle.Solid, 2);

			//Place 'A' text
			DrawText("dDwALabel", AutoScale, "A",
				_aPDate, _aPVal, textOffset2 * swingDir *-1, swingColor, textFontLabel,
				StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
			DrawText("dDwAPct", AutoScale, _aPVal.ToString(),
				_aPDate, _aPVal, textOffset * swingDir *-1, swingColor, textFont,
				StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
			
			//Place 'B' text
			double moveVal = RoundInst(Math.Abs(_aPVal -_bPVal));
			
			DrawText("dDwBLabel", AutoScale, "B",
				_bPDate, _bPVal, textOffset2 * swingDir, swingColor, textFontLabel,
				StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
			DrawText("dDwBPrice", AutoScale, moveVal.ToString() +"c - " + _bPVal.ToString(),
				_bPDate, _bPVal, textOffset * swingDir, swingColor, textFont,
				StringAlignment.Center, Color.Transparent, Color.Transparent, 0);

			//Draw 'C' text
			moveVal = RoundInst(Math.Abs(_bPVal -_cPVal));
			double movePct = (moveVal / Math.Abs(_aPVal -_bPVal)) * 100;

			DrawText("dDwCLabel", AutoScale, "C",
				_cPDate, _cPVal, textOffset3 * swingDir *-1, swingColor, textFontLabel,
				StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
			//move point value and %
			DrawText("dDwCPct", AutoScale,  movePct.ToString("f1") + "%",
				_cPDate, _cPVal, textOffset2 * swingDir *-1, swingColor, textFont,
				StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
			DrawText("dDwCPrice", AutoScale, moveVal.ToString() +"c - " + _cPVal.ToString(),
				_cPDate, _cPVal, textOffset * swingDir *-1, swingColor, textFont,
				StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
			
			//setup and call fib draws
			_ABbarStart = lineAB.StartBarsAgo;
			_ABbarEnd = lineAB.EndBarsAgo;
			_ABpriceStart = _aPVal;
			_ABpriceEnd = _bPVal;
			_BCbarStart = _ABbarEnd;
			_BCbarEnd = lineBC.EndBarsAgo;
			_BCpriceStart = _bPVal;
			_BCpriceEnd = _cPVal;
			
			DoDrawFibs(1, fibABRets, fibABExtends, fibABColor);
			DoDrawFibs(2, fibBCRets, fibBCExtends, fibBCColor);
			DoDrawFibs(3, fibMMRets, fibMMExtends, swingColor);
			
		}
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private dDrawABC[] cachedDrawABC = null;

        private static dDrawABC checkdDrawABC = new dDrawABC();

        /// <summary>
        /// Alt-RightClick 3x to Draw ABC points and some related Fibs (A-bar click (above/below bar mid value) defines UP vs DOWN swings
        /// </summary>
        /// <returns></returns>
        public dDrawABC dDrawABC(double adjustableFib, int fibABExtends, int fibABRets, bool fibABTextToRight, int fibBCExtends, int fibBCRets, bool fibBCTextToRight, int fibMMExtends, int fibMMRets, int fibMMReverse, bool fibMMTextToRight)
        {
            return dDrawABC(Input, adjustableFib, fibABExtends, fibABRets, fibABTextToRight, fibBCExtends, fibBCRets, fibBCTextToRight, fibMMExtends, fibMMRets, fibMMReverse, fibMMTextToRight);
        }

        /// <summary>
        /// Alt-RightClick 3x to Draw ABC points and some related Fibs (A-bar click (above/below bar mid value) defines UP vs DOWN swings
        /// </summary>
        /// <returns></returns>
        public dDrawABC dDrawABC(Data.IDataSeries input, double adjustableFib, int fibABExtends, int fibABRets, bool fibABTextToRight, int fibBCExtends, int fibBCRets, bool fibBCTextToRight, int fibMMExtends, int fibMMRets, int fibMMReverse, bool fibMMTextToRight)
        {
            if (cachedDrawABC != null)
                for (int idx = 0; idx < cachedDrawABC.Length; idx++)
                    if (Math.Abs(cachedDrawABC[idx].AdjustableFib - adjustableFib) <= double.Epsilon && cachedDrawABC[idx].FibABExtends == fibABExtends && cachedDrawABC[idx].FibABRets == fibABRets && cachedDrawABC[idx].FibABTextToRight == fibABTextToRight && cachedDrawABC[idx].FibBCExtends == fibBCExtends && cachedDrawABC[idx].FibBCRets == fibBCRets && cachedDrawABC[idx].FibBCTextToRight == fibBCTextToRight && cachedDrawABC[idx].FibMMExtends == fibMMExtends && cachedDrawABC[idx].FibMMRets == fibMMRets && cachedDrawABC[idx].FibMMReverse == fibMMReverse && cachedDrawABC[idx].FibMMTextToRight == fibMMTextToRight && cachedDrawABC[idx].EqualsInput(input))
                        return cachedDrawABC[idx];

            lock (checkdDrawABC)
            {
                checkdDrawABC.AdjustableFib = adjustableFib;
                adjustableFib = checkdDrawABC.AdjustableFib;
                checkdDrawABC.FibABExtends = fibABExtends;
                fibABExtends = checkdDrawABC.FibABExtends;
                checkdDrawABC.FibABRets = fibABRets;
                fibABRets = checkdDrawABC.FibABRets;
                checkdDrawABC.FibABTextToRight = fibABTextToRight;
                fibABTextToRight = checkdDrawABC.FibABTextToRight;
                checkdDrawABC.FibBCExtends = fibBCExtends;
                fibBCExtends = checkdDrawABC.FibBCExtends;
                checkdDrawABC.FibBCRets = fibBCRets;
                fibBCRets = checkdDrawABC.FibBCRets;
                checkdDrawABC.FibBCTextToRight = fibBCTextToRight;
                fibBCTextToRight = checkdDrawABC.FibBCTextToRight;
                checkdDrawABC.FibMMExtends = fibMMExtends;
                fibMMExtends = checkdDrawABC.FibMMExtends;
                checkdDrawABC.FibMMRets = fibMMRets;
                fibMMRets = checkdDrawABC.FibMMRets;
                checkdDrawABC.FibMMReverse = fibMMReverse;
                fibMMReverse = checkdDrawABC.FibMMReverse;
                checkdDrawABC.FibMMTextToRight = fibMMTextToRight;
                fibMMTextToRight = checkdDrawABC.FibMMTextToRight;

                if (cachedDrawABC != null)
                    for (int idx = 0; idx < cachedDrawABC.Length; idx++)
                        if (Math.Abs(cachedDrawABC[idx].AdjustableFib - adjustableFib) <= double.Epsilon && cachedDrawABC[idx].FibABExtends == fibABExtends && cachedDrawABC[idx].FibABRets == fibABRets && cachedDrawABC[idx].FibABTextToRight == fibABTextToRight && cachedDrawABC[idx].FibBCExtends == fibBCExtends && cachedDrawABC[idx].FibBCRets == fibBCRets && cachedDrawABC[idx].FibBCTextToRight == fibBCTextToRight && cachedDrawABC[idx].FibMMExtends == fibMMExtends && cachedDrawABC[idx].FibMMRets == fibMMRets && cachedDrawABC[idx].FibMMReverse == fibMMReverse && cachedDrawABC[idx].FibMMTextToRight == fibMMTextToRight && cachedDrawABC[idx].EqualsInput(input))
                            return cachedDrawABC[idx];

                dDrawABC indicator = new dDrawABC();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.AdjustableFib = adjustableFib;
                indicator.FibABExtends = fibABExtends;
                indicator.FibABRets = fibABRets;
                indicator.FibABTextToRight = fibABTextToRight;
                indicator.FibBCExtends = fibBCExtends;
                indicator.FibBCRets = fibBCRets;
                indicator.FibBCTextToRight = fibBCTextToRight;
                indicator.FibMMExtends = fibMMExtends;
                indicator.FibMMRets = fibMMRets;
                indicator.FibMMReverse = fibMMReverse;
                indicator.FibMMTextToRight = fibMMTextToRight;
                Indicators.Add(indicator);
                indicator.SetUp();

                dDrawABC[] tmp = new dDrawABC[cachedDrawABC == null ? 1 : cachedDrawABC.Length + 1];
                if (cachedDrawABC != null)
                    cachedDrawABC.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cachedDrawABC = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Alt-RightClick 3x to Draw ABC points and some related Fibs (A-bar click (above/below bar mid value) defines UP vs DOWN swings
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.dDrawABC dDrawABC(double adjustableFib, int fibABExtends, int fibABRets, bool fibABTextToRight, int fibBCExtends, int fibBCRets, bool fibBCTextToRight, int fibMMExtends, int fibMMRets, int fibMMReverse, bool fibMMTextToRight)
        {
            return _indicator.dDrawABC(Input, adjustableFib, fibABExtends, fibABRets, fibABTextToRight, fibBCExtends, fibBCRets, fibBCTextToRight, fibMMExtends, fibMMRets, fibMMReverse, fibMMTextToRight);
        }

        /// <summary>
        /// Alt-RightClick 3x to Draw ABC points and some related Fibs (A-bar click (above/below bar mid value) defines UP vs DOWN swings
        /// </summary>
        /// <returns></returns>
        public Indicator.dDrawABC dDrawABC(Data.IDataSeries input, double adjustableFib, int fibABExtends, int fibABRets, bool fibABTextToRight, int fibBCExtends, int fibBCRets, bool fibBCTextToRight, int fibMMExtends, int fibMMRets, int fibMMReverse, bool fibMMTextToRight)
        {
            return _indicator.dDrawABC(input, adjustableFib, fibABExtends, fibABRets, fibABTextToRight, fibBCExtends, fibBCRets, fibBCTextToRight, fibMMExtends, fibMMRets, fibMMReverse, fibMMTextToRight);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Alt-RightClick 3x to Draw ABC points and some related Fibs (A-bar click (above/below bar mid value) defines UP vs DOWN swings
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.dDrawABC dDrawABC(double adjustableFib, int fibABExtends, int fibABRets, bool fibABTextToRight, int fibBCExtends, int fibBCRets, bool fibBCTextToRight, int fibMMExtends, int fibMMRets, int fibMMReverse, bool fibMMTextToRight)
        {
            return _indicator.dDrawABC(Input, adjustableFib, fibABExtends, fibABRets, fibABTextToRight, fibBCExtends, fibBCRets, fibBCTextToRight, fibMMExtends, fibMMRets, fibMMReverse, fibMMTextToRight);
        }

        /// <summary>
        /// Alt-RightClick 3x to Draw ABC points and some related Fibs (A-bar click (above/below bar mid value) defines UP vs DOWN swings
        /// </summary>
        /// <returns></returns>
        public Indicator.dDrawABC dDrawABC(Data.IDataSeries input, double adjustableFib, int fibABExtends, int fibABRets, bool fibABTextToRight, int fibBCExtends, int fibBCRets, bool fibBCTextToRight, int fibMMExtends, int fibMMRets, int fibMMReverse, bool fibMMTextToRight)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.dDrawABC(input, adjustableFib, fibABExtends, fibABRets, fibABTextToRight, fibBCExtends, fibBCRets, fibBCTextToRight, fibMMExtends, fibMMRets, fibMMReverse, fibMMTextToRight);
        }
    }
}
#endregion
