#region Using declarations
using System;
using System.ComponentModel;
using System.Drawing;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// S&C April 2011
	/// </summary>
	[Description("S&C April 2011")]
	public class SemiCup : Indicator
	{
		#region Variables

		private DataSeries	dxPlus;
		private DataSeries	dxMinus;
		private DataSeries	filC;
		private int			lastB0;
		private int			lastB5;
		private double		lastBottom;
		private int			minimumBars	 = 20;
		private double		parameter = 1.5; 

		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Plot(new Pen(Color.Red, 3), PlotStyle.Line, "Semi-Cup"));

			MaximumBarsLookBack = MaximumBarsLookBack.Infinite;
			dxPlus			    = new DataSeries(this);
			dxMinus			    = new DataSeries(this);
			filC			    = new DataSeries(this);
			Overlay			    = true;
		}

		/// <summary>
		/// Called on each bar update event (incoming tick)
		/// </summary>
		protected override void OnBarUpdate()
		{
			filC.Set(Math.Log(Close[0]));

			if (CurrentBar == 0)
			{
				dxPlus .Set(0);
				dxMinus.Set(0);
				return;
			}

			dxPlus .Set(Math.Max(0, filC[0] - filC[1]));
			dxMinus.Set(Math.Max(0, filC[1] - filC[0]));

			lastBottom = filC[0];

			bool found = false;
			int i	   = 0;

			while (i < CurrentBar && !found)
			{
				i++;
				lastBottom = Math.Min(filC[i], lastBottom);

				if (Close[i] > Close[0] * parameter)
					found = true;
			}

			if (!found || i < minimumBars || ((CurrentBar - i) < lastB5 && (CurrentBar - i) != lastB0)) 
				return;

			if (!IsValidSemiCup(i)) 
				return;

			double a = (Close[i] - Math.Exp(lastBottom)) / Math.Pow(i, 10);

			for (int j = 0; j <= i; j++)
				Value.Set(j, Math.Min(Close[i], a * Math.Pow(j, 10) + 0.98 * Math.Exp(lastBottom)));
		}

		private bool IsValidSemiCup(int period)
		{
			int interval = (int)Math.Round(period * 0.2, 0);

			double smaDxPlus1  = 0;
			double smaDxMinus1 = 0;

			for (int j = period; j > period - 2 * interval; j--)
			{
				smaDxPlus1  += dxPlus[j];
				smaDxMinus1 += dxMinus[j];
			}
			smaDxPlus1  /= 2 * interval;
			smaDxMinus1 /= 2 * interval;

			double dx1 = 100 * Math.Abs(smaDxPlus1 - smaDxMinus1) / (smaDxPlus1 + smaDxMinus1);

			double smaDxPlus2  = 0;
			double smaDxMinus2 = 0;

			double l2 = lastBottom + (filC[period] - lastBottom) * 0.4;
			double l3 = lastBottom + (filC[period] - lastBottom) * 0.6;

			int b3 = period - interval * 3;

			for (int j = period - 2 * interval; j >= 0; j--)
			{
				double level = j < b3 ? l2 : l3;
				if (filC[j] > level)
					return false;

				smaDxPlus2  += dxPlus[j];
				smaDxMinus2 += dxMinus[j];
			}
			smaDxPlus2  /= 2 * interval;
			smaDxMinus2 /= 2 * interval;

			double dx2 = 100 * Math.Abs(smaDxPlus2 - smaDxMinus2) / (smaDxPlus2 + smaDxMinus2);

			if(dx1 <= 25 || dx2 >= 25)
				return false;

			lastB0 = CurrentBar - period;
			lastB5 = CurrentBar;
			return true;
		}

		#region Properties
		[Description("")]
		[GridCategory("Parameters")]
		public double Parameter
		{
			get { return parameter; }
			set { parameter = Math.Max(1, value); }
		}

		[Description("")]
		[GridCategory("Parameters")]
		public int MinimumBars
		{
			get { return minimumBars; }
			set { minimumBars = Math.Max(2, value); }
		}
		#endregion
	}
}
#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private SemiCup[] cacheSemiCup = null;

        private static SemiCup checkSemiCup = new SemiCup();

        /// <summary>
        /// S&C April 2011
        /// </summary>
        /// <returns></returns>
        public SemiCup SemiCup(int minimumBars, double parameter)
        {
            return SemiCup(Input, minimumBars, parameter);
        }

        /// <summary>
        /// S&C April 2011
        /// </summary>
        /// <returns></returns>
        public SemiCup SemiCup(Data.IDataSeries input, int minimumBars, double parameter)
        {
            if (cacheSemiCup != null)
                for (int idx = 0; idx < cacheSemiCup.Length; idx++)
                    if (cacheSemiCup[idx].MinimumBars == minimumBars && Math.Abs(cacheSemiCup[idx].Parameter - parameter) <= double.Epsilon && cacheSemiCup[idx].EqualsInput(input))
                        return cacheSemiCup[idx];

            lock (checkSemiCup)
            {
                checkSemiCup.MinimumBars = minimumBars;
                minimumBars = checkSemiCup.MinimumBars;
                checkSemiCup.Parameter = parameter;
                parameter = checkSemiCup.Parameter;

                if (cacheSemiCup != null)
                    for (int idx = 0; idx < cacheSemiCup.Length; idx++)
                        if (cacheSemiCup[idx].MinimumBars == minimumBars && Math.Abs(cacheSemiCup[idx].Parameter - parameter) <= double.Epsilon && cacheSemiCup[idx].EqualsInput(input))
                            return cacheSemiCup[idx];

                SemiCup indicator = new SemiCup();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.MinimumBars = minimumBars;
                indicator.Parameter = parameter;
                Indicators.Add(indicator);
                indicator.SetUp();

                SemiCup[] tmp = new SemiCup[cacheSemiCup == null ? 1 : cacheSemiCup.Length + 1];
                if (cacheSemiCup != null)
                    cacheSemiCup.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheSemiCup = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// S&C April 2011
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.SemiCup SemiCup(int minimumBars, double parameter)
        {
            return _indicator.SemiCup(Input, minimumBars, parameter);
        }

        /// <summary>
        /// S&C April 2011
        /// </summary>
        /// <returns></returns>
        public Indicator.SemiCup SemiCup(Data.IDataSeries input, int minimumBars, double parameter)
        {
            return _indicator.SemiCup(input, minimumBars, parameter);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// S&C April 2011
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.SemiCup SemiCup(int minimumBars, double parameter)
        {
            return _indicator.SemiCup(Input, minimumBars, parameter);
        }

        /// <summary>
        /// S&C April 2011
        /// </summary>
        /// <returns></returns>
        public Indicator.SemiCup SemiCup(Data.IDataSeries input, int minimumBars, double parameter)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.SemiCup(input, minimumBars, parameter);
        }
    }
}
#endregion
