#region Using declarations
using System;
using System.ComponentModel;
using System.Drawing;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// S&C April 2011
	/// </summary>
	[Description("S&C April 2011")]
	public class SemiCupInternal : Indicator
	{
		#region Variables

		private DataSeries	dxPlus;
		private DataSeries	dxMinus;

		private double		lastDx1		  = 0;
		private double		lastDx2		  = 0;

		private int			lastB0		  = 0;
		private int			lastB1		  = 0;
		private int			lastB2		  = 0;
		private int			lastB3		  = 0;
		private int			lastB4		  = 0;
		private int			lastB5		  = 0;

		private double		lastL0	 	  = 0;
		private double		lastL1	 	  = 0;
		private double		lastL2	 	  = 0;
		private double		lastL3	 	  = 0;
		private double		lastL4	 	  = 0;
		private double		lastL5	 	  = 0;
		private double		lastBottom	  = 0;
		private int			minimumBars	  = 20;
		private double		parameter 	  = 1.5;

		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Plot(new Pen(Color.Navy, 2), PlotStyle.Line, "FilC"));

			MaximumBarsLookBack = MaximumBarsLookBack.Infinite;
			dxPlus			    = new DataSeries(this);
			dxMinus			    = new DataSeries(this);
			Overlay			    = false;
			DrawOnPricePanel    = false;
		}

		/// <summary>
		/// Called on each bar update event (incoming tick)
		/// </summary>
		protected override void OnBarUpdate()
		{
			FilC.Set(Math.Log(Close[0]));

			if (CurrentBar == 0)
			{
				dxPlus .Set(0);
				dxMinus.Set(0);
				return;
			}
			dxPlus .Set(Math.Max(0, FilC[0] - FilC[1]));
			dxMinus.Set(Math.Max(0, FilC[1] - FilC[0]));

			lastBottom = FilC[0];
			bool found = false;
			int i	   = 0;

			while (i < CurrentBar && !found)
			{
				i++;
				lastBottom = Math.Min(FilC[i], lastBottom);

				if (Close[i] > Close[0] * parameter)
					found = true;
			}

			if (!found || i < minimumBars || CurrentBar - i < lastB5 && CurrentBar - i != lastB0) 
				return;

			if(!IsValidSemiCup(i)) 
				return;

			DrawRectangle(lastB0.ToString(), true, CurrentBar - lastB0, lastL0, 0, lastL5, Color.Blue, Color.Empty, 0);
			DrawLine(string.Format("{0}b1", lastB0), CurrentBar - lastB1, lastL0, CurrentBar - lastB1, lastL5, Color.Blue);
			DrawLine(string.Format("{0}b2", lastB0), CurrentBar - lastB2, lastL0, CurrentBar - lastB2, lastL5, Color.Blue);
			DrawLine(string.Format("{0}b3", lastB0), CurrentBar - lastB3, lastL0, CurrentBar - lastB3, lastL5, Color.Blue);
			DrawLine(string.Format("{0}b4", lastB0), CurrentBar - lastB4, lastL0, CurrentBar - lastB4, lastL5, Color.Blue);

			DrawLine(string.Format("{0}l1", lastB0), CurrentBar - lastB0, lastL1, 0, lastL1, Color.Blue);
			DrawLine(string.Format("{0}l2", lastB0), CurrentBar - lastB0, lastL2, 0, lastL2, Color.Blue);
			DrawLine(string.Format("{0}l3", lastB0), CurrentBar - lastB0, lastL3, 0, lastL3, Color.Blue);
			DrawLine(string.Format("{0}l4", lastB0), CurrentBar - lastB0, lastL4, 0, lastL4, Color.Blue);

			DrawRectangle(string.Format("{0}1", lastB0), true, CurrentBar - lastB2, lastL5, 0, lastL3, Color.Empty, Color.Yellow, 5);
			DrawRectangle(string.Format("{0}2", lastB0), true, CurrentBar - lastB3, lastL3, 0, lastL2, Color.Empty, Color.Yellow, 5);

			DrawText(string.Format("{0}dx1", lastB0), true, string.Format("DX1: {0}", lastDx1.ToString("0.00")), CurrentBar - lastB1, lastL0, 
				-(int)ChartControl.Font.GetHeight(), Color.Blue, ChartControl.Font, StringAlignment.Center, Color.Empty, Color.Empty, 0);
			DrawText(string.Format("{0}dx2", lastB0), true, string.Format("DX2: {0}", lastDx2.ToString("0.00")), CurrentBar - lastB3, lastL0, 
				-(int)ChartControl.Font.GetHeight(), Color.Blue, ChartControl.Font, StringAlignment.Near, Color.Empty, Color.Empty, 0);
		}

		private bool IsValidSemiCup(int period)
		{
			int interval = (int)Math.Round(period * 0.2, 0);

			double smaDxPlus1  = 0;
			double smaDxMinus1 = 0;

			for (int j = period; j > period - 2 * interval; j--)
			{
				smaDxPlus1  += dxPlus[j];
				smaDxMinus1 += dxMinus[j];
			}

			smaDxPlus1  /= 2 * interval;
			smaDxMinus1 /= 2 * interval;

			double dx1 = 100 * Math.Abs(smaDxPlus1 - smaDxMinus1) / (smaDxPlus1 + smaDxMinus1);

			double smaDxPlus2  = 0;
			double smaDxMinus2 = 0;

			double l2 = lastBottom + (FilC[period] - lastBottom) * 0.4;
			double l3 = lastBottom + (FilC[period] - lastBottom) * 0.6;

			int b3 = period - interval * 3;

			for (int j = period - 2 * interval; j >= 0; j--)
			{
				double level = j < b3 ? l2 : l3;

				if (FilC[j] > level)
					return false;

				smaDxPlus2  += dxPlus[j];
				smaDxMinus2 += dxMinus[j];
			}
			smaDxPlus2  /= 2 * interval;
			smaDxMinus2 /= 2 * interval;

			double dx2 = 100 * Math.Abs(smaDxPlus2 - smaDxMinus2) / (smaDxPlus2 + smaDxMinus2);

			if (dx1 <= 25 || dx2 >= 25)
				return false;

			lastB0 = CurrentBar - period;
			lastB1 = lastB0 + interval;
			lastB2 = lastB1 + interval;
			lastB3 = lastB2 + interval;
			lastB4 = lastB3 + interval;
			lastB5 = CurrentBar;

			lastL0 = lastBottom;
			lastL5 = FilC[period];
			lastL1 = lastL0 + (lastL5 - lastL0) * 0.2;
			lastL2 = lastL0 + (lastL5 - lastL0) * 0.4;
			lastL3 = lastL0 + (lastL5 - lastL0) * 0.6;
			lastL4 = lastL0 + (lastL5 - lastL0) * 0.8;

			lastDx1 = dx1;
			lastDx2 = dx2;

			return true;
		}

		#region Properties
		[Browsable(false)]
		[XmlIgnore]	
		public DataSeries FilC
		{
			get { return Values[0]; }
		}

		[Description("")]
		[GridCategory("Parameters")]
		public double Parameter
		{
			get { return parameter; }
			set { parameter = Math.Max(1, value); }
		}

		[Description("")]
		[GridCategory("Parameters")]
		public int MinimumBars
		{
			get { return minimumBars; }
			set { minimumBars = Math.Max(2, value); }
		}
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private SemiCupInternal[] cacheSemiCupInternal = null;

        private static SemiCupInternal checkSemiCupInternal = new SemiCupInternal();

        /// <summary>
        /// S&C April 2011
        /// </summary>
        /// <returns></returns>
        public SemiCupInternal SemiCupInternal(int minimumBars, double parameter)
        {
            return SemiCupInternal(Input, minimumBars, parameter);
        }

        /// <summary>
        /// S&C April 2011
        /// </summary>
        /// <returns></returns>
        public SemiCupInternal SemiCupInternal(Data.IDataSeries input, int minimumBars, double parameter)
        {
            if (cacheSemiCupInternal != null)
                for (int idx = 0; idx < cacheSemiCupInternal.Length; idx++)
                    if (cacheSemiCupInternal[idx].MinimumBars == minimumBars && Math.Abs(cacheSemiCupInternal[idx].Parameter - parameter) <= double.Epsilon && cacheSemiCupInternal[idx].EqualsInput(input))
                        return cacheSemiCupInternal[idx];

            lock (checkSemiCupInternal)
            {
                checkSemiCupInternal.MinimumBars = minimumBars;
                minimumBars = checkSemiCupInternal.MinimumBars;
                checkSemiCupInternal.Parameter = parameter;
                parameter = checkSemiCupInternal.Parameter;

                if (cacheSemiCupInternal != null)
                    for (int idx = 0; idx < cacheSemiCupInternal.Length; idx++)
                        if (cacheSemiCupInternal[idx].MinimumBars == minimumBars && Math.Abs(cacheSemiCupInternal[idx].Parameter - parameter) <= double.Epsilon && cacheSemiCupInternal[idx].EqualsInput(input))
                            return cacheSemiCupInternal[idx];

                SemiCupInternal indicator = new SemiCupInternal();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.MinimumBars = minimumBars;
                indicator.Parameter = parameter;
                Indicators.Add(indicator);
                indicator.SetUp();

                SemiCupInternal[] tmp = new SemiCupInternal[cacheSemiCupInternal == null ? 1 : cacheSemiCupInternal.Length + 1];
                if (cacheSemiCupInternal != null)
                    cacheSemiCupInternal.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheSemiCupInternal = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// S&C April 2011
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.SemiCupInternal SemiCupInternal(int minimumBars, double parameter)
        {
            return _indicator.SemiCupInternal(Input, minimumBars, parameter);
        }

        /// <summary>
        /// S&C April 2011
        /// </summary>
        /// <returns></returns>
        public Indicator.SemiCupInternal SemiCupInternal(Data.IDataSeries input, int minimumBars, double parameter)
        {
            return _indicator.SemiCupInternal(input, minimumBars, parameter);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// S&C April 2011
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.SemiCupInternal SemiCupInternal(int minimumBars, double parameter)
        {
            return _indicator.SemiCupInternal(Input, minimumBars, parameter);
        }

        /// <summary>
        /// S&C April 2011
        /// </summary>
        /// <returns></returns>
        public Indicator.SemiCupInternal SemiCupInternal(Data.IDataSeries input, int minimumBars, double parameter)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.SemiCupInternal(input, minimumBars, parameter);
        }
    }
}
#endregion
