// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

	public enum TypeBBMA
	{
		DEMA,
		EMA,
		HMA,
		SMA,
		TMA,
		VMA,
		WMA,
		ZLEMA
	}
	// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// Bollinger Bands are plotted at standard deviation levels above and below a moving average. Since standard deviation is a measure of volatility, the bands are self-adjusting: widening during volatile markets and contracting during calmer periods.	This implementation allows for use of MAs other than SMA.
	/// </summary>
	[Description("Bollinger Bands are plotted at standard deviation levels above and below a moving average. Since standard deviation is a measure of volatility, the bands are self-adjusting: widening during volatile markets and contracting during calmer periods.")]
	public class BollingerModified : Indicator
	{
		#region Variables
		private	double		numStdDev	= 2;
		private int			length		= 14;
		private TypeBBMA bollingerBandMA;
		private double maValue;
		private double stdDevValue;
		#endregion
		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Plot(Color.Orange, "Upper band"));
			Add(new Plot(Color.Orange, "Middle band"));
			Add(new Plot(Color.Orange, "Lower band"));

			Overlay				= true;
		}

		/// <summary>
		/// Called on each bar update event (incoming tick)
		/// </summary>
		protected override void OnBarUpdate()
		{
			switch (bollingerBandMA) 
			{
				case TypeBBMA.EMA:
					maValue = EMA(length)[0];
					break;
				case TypeBBMA.HMA:
					maValue = HMA(length)[0];
					break;
				case TypeBBMA.SMA:
					maValue = SMA(length)[0];
					break;
				case TypeBBMA.WMA:
					maValue = WMA(length)[0];
					break;
				case TypeBBMA.VMA:
					maValue = VMA(length,length)[0];
					break;
				case TypeBBMA.DEMA:
					maValue = DEMA(length)[0];
					break;
				case TypeBBMA.TMA:
					maValue = TMA(length)[0];
					break;
				case TypeBBMA.ZLEMA:
					maValue = ZLEMA(length)[0];
					break;
			}
		
			stdDevValue = StdDev(length)[0];
            Upper.Set(maValue + NumStdDev * stdDevValue);
            Middle.Set(maValue);
            Lower.Set(maValue - NumStdDev * stdDevValue);
		}

		#region Properties
		/// <summary>
		/// Gets the lower value.
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Lower
		{
			get { return Values[2]; }
		}
		
		/// <summary>
		/// Get the middle value.
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Middle
		{
			get { return Values[1]; }
		}

		/// <summary>
		/// Get the upper value.
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Upper
		{
			get { return Values[0]; }
		}

		/// <summary>
		/// </summary>
		[Description("Number of standard deviations")]
		[GridCategory("Parameters")]
		[Gui.Design.DisplayNameAttribute("# of std. dev.")]
		public double NumStdDev
		{
			get { return numStdDev; }
			set { numStdDev = Math.Max(0, value); }
		}

		/// <summary>
		/// </summary>
		[Description("Numbers of bars used for calculations")]
		[GridCategory("Parameters")]
		public int Period
		{
			get { return length; }
			set { length = Math.Max(1, value); }
		}

		[Description("The type of moving average the BB is derived from")]
        [GridCategory("Parameters")]
        public TypeBBMA BollingerBandMA
        {
            get { return bollingerBandMA; }
            set { bollingerBandMA = value; }
        }

		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private BollingerModified[] cacheBollingerModified = null;

        private static BollingerModified checkBollingerModified = new BollingerModified();

        /// <summary>
        /// Bollinger Bands are plotted at standard deviation levels above and below a moving average. Since standard deviation is a measure of volatility, the bands are self-adjusting: widening during volatile markets and contracting during calmer periods.
        /// </summary>
        /// <returns></returns>
        public BollingerModified BollingerModified(TypeBBMA bollingerBandMA, double numStdDev, int period)
        {
            return BollingerModified(Input, bollingerBandMA, numStdDev, period);
        }

        /// <summary>
        /// Bollinger Bands are plotted at standard deviation levels above and below a moving average. Since standard deviation is a measure of volatility, the bands are self-adjusting: widening during volatile markets and contracting during calmer periods.
        /// </summary>
        /// <returns></returns>
        public BollingerModified BollingerModified(Data.IDataSeries input, TypeBBMA bollingerBandMA, double numStdDev, int period)
        {
            if (cacheBollingerModified != null)
                for (int idx = 0; idx < cacheBollingerModified.Length; idx++)
                    if (cacheBollingerModified[idx].BollingerBandMA == bollingerBandMA && Math.Abs(cacheBollingerModified[idx].NumStdDev - numStdDev) <= double.Epsilon && cacheBollingerModified[idx].Period == period && cacheBollingerModified[idx].EqualsInput(input))
                        return cacheBollingerModified[idx];

            lock (checkBollingerModified)
            {
                checkBollingerModified.BollingerBandMA = bollingerBandMA;
                bollingerBandMA = checkBollingerModified.BollingerBandMA;
                checkBollingerModified.NumStdDev = numStdDev;
                numStdDev = checkBollingerModified.NumStdDev;
                checkBollingerModified.Period = period;
                period = checkBollingerModified.Period;

                if (cacheBollingerModified != null)
                    for (int idx = 0; idx < cacheBollingerModified.Length; idx++)
                        if (cacheBollingerModified[idx].BollingerBandMA == bollingerBandMA && Math.Abs(cacheBollingerModified[idx].NumStdDev - numStdDev) <= double.Epsilon && cacheBollingerModified[idx].Period == period && cacheBollingerModified[idx].EqualsInput(input))
                            return cacheBollingerModified[idx];

                BollingerModified indicator = new BollingerModified();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.BollingerBandMA = bollingerBandMA;
                indicator.NumStdDev = numStdDev;
                indicator.Period = period;
                Indicators.Add(indicator);
                indicator.SetUp();

                BollingerModified[] tmp = new BollingerModified[cacheBollingerModified == null ? 1 : cacheBollingerModified.Length + 1];
                if (cacheBollingerModified != null)
                    cacheBollingerModified.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheBollingerModified = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Bollinger Bands are plotted at standard deviation levels above and below a moving average. Since standard deviation is a measure of volatility, the bands are self-adjusting: widening during volatile markets and contracting during calmer periods.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.BollingerModified BollingerModified(TypeBBMA bollingerBandMA, double numStdDev, int period)
        {
            return _indicator.BollingerModified(Input, bollingerBandMA, numStdDev, period);
        }

        /// <summary>
        /// Bollinger Bands are plotted at standard deviation levels above and below a moving average. Since standard deviation is a measure of volatility, the bands are self-adjusting: widening during volatile markets and contracting during calmer periods.
        /// </summary>
        /// <returns></returns>
        public Indicator.BollingerModified BollingerModified(Data.IDataSeries input, TypeBBMA bollingerBandMA, double numStdDev, int period)
        {
            return _indicator.BollingerModified(input, bollingerBandMA, numStdDev, period);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Bollinger Bands are plotted at standard deviation levels above and below a moving average. Since standard deviation is a measure of volatility, the bands are self-adjusting: widening during volatile markets and contracting during calmer periods.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.BollingerModified BollingerModified(TypeBBMA bollingerBandMA, double numStdDev, int period)
        {
            return _indicator.BollingerModified(Input, bollingerBandMA, numStdDev, period);
        }

        /// <summary>
        /// Bollinger Bands are plotted at standard deviation levels above and below a moving average. Since standard deviation is a measure of volatility, the bands are self-adjusting: widening during volatile markets and contracting during calmer periods.
        /// </summary>
        /// <returns></returns>
        public Indicator.BollingerModified BollingerModified(Data.IDataSeries input, TypeBBMA bollingerBandMA, double numStdDev, int period)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.BollingerModified(input, bollingerBandMA, numStdDev, period);
        }
    }
}
#endregion
