//
// Spread3 Price of 3 Instruments - For NinjaTrader 7 & higher
//
// Calculates price of synthetic combination of 3 instruments
// Parameters are symbol of second & third instruments,
// and quantities of all instruments (positive for long, negative for short)
// PointValues from the instrument manager are optionally used as multipliers.
//
// The primary chart instrument (BarsInProgress == 0) is the "clock" to which the other instruments are synchronized.
// You should make sure the most active instrument is the one selected as the primary instrument.
// The output of this indicator changes only when the OnBarUpdate is called for the primary instrument.
//
// This indicator can be extended to 4 or more instruments.
// The code for an Instrument4 is in here but is commented out.
// For Instrument5, etc, you can copy and paste the Instrument4 code & change to 5, 6, etc.
//
// See my "Correlation" indicator for examples of how to do synchronized calculations with lookbacks 
//
// Kevin Doren  9/26/2010
//

#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Spread3 Price of Two Instruments
    /// </summary>
    [Description("Spread3 Price of Three Instruments")]
    public class Spread3 : Indicator
    {
		// Set constants for Bars-In-Progress indices for each instrument
		private const int Instrument1 = 0;
		private const int Instrument2 = 1;
		private const int Instrument3 = 2;
//		private const int Instrument4 = 3;
		
        #region Variables
            private string symbol2 = "";		//  Symbol for Instrument2 (full name, i.e. "ES 03-10" for futures)
			private string symbol3 = "";		//	Symbol for Instrument3
//			private string symbol4 = "";		//	Symbol for Instrument4
			private double qty1=1;				//  Default Quantity for Instrument1
			private double qty2=1;				//  Default Quantity for Instrument2
			private double qty3=1;				//  Default Quantity for Instrument3
//			private double qty4=1;				//  Default Quantity for Instrument4
			private bool useMultiplier = false;	//  Default parameter value; true = use contract multiplier in spread calculation
			private bool noOutputWhenMissingInput = false;  // true = produce blank plot if any input is missing a bar
															// BEWARE: this causes problems if this indicator is used as input to another indicator!
		
			private SyncedDataSeries Instrument2Close;	
			private SyncedDataSeries Instrument3Close;
//			private SyncedDataSeries Instrument4Close;
		
//			private SyncedDataSeries Instrument2Volume;	// If you need to synchronize High, Low, or Volume, declare it here
//			private SyncedDataSeries Instrument2High;	// If you need to synchronize High, Low, or Volume, declare it here
		
			private bool outputStarted = false;			// flag which is set to "true" when the indicator starts producing ouput
			private bool lastBarNotSynced = false;		// flag which is set to "true" in real time if the last bar had a missing input
		
        #endregion		
		
		#region SynchronizationMethods
/*
  Synchronization of Multi-Instrument Indicators & Strategies - For NinjaTrader 7
  Kevin Doren - Version 9/8/2010

  7/29/2010:  Added "SynchronizedBarCountToday" field
  9/8/2010:   Added "SyncedGet" method
			
  Multi-instrument indicators (and strategies, for the same concepts apply) by their nature are looking
  at relationships between instruments.  Any calculations we do need to be time-synchronized - we want
  to be using data that occurred at the same time.  For certain kinds of analysis, such as intraday
  analysis of heavily-traded instruments during regular trading hours on historical data, the techniques
  shown here probably aren't necessary.  But for real time analysis, especially with instruments that
  might have missing bars, these techniques are necessary.

  There are 3 scenarios: Historical, real-time with (CalculateOnBarClose == true), 
  and real-time with (CalculateOnBarClose == false).  Each has its issues.

  The simplest case is Historical.  In this case, NT7 will synchronize the bars, which means that when
  OnBarUpdate is called for a Bar Series with a particular timestamp, any other Bar Series which
  has a bar with the same timestamp will have that bar set to be the current bar.  This means that
  Closes[0][0] and Closes[1][0] will both point to bars with the same timestamp IF Closes[1] has a
  bar with that timestamp.  If not, Close[1][0] will point to the most recent bar preceding it.
  So even Close[0][0] and Close[1][0] are not necessarily synchronized.  As you look back in time,
  it can get even more out of sync, because bars might be missing from either series.

  For an indicator like correlation, the calculation makes no sense unless the bars are all synchronized.
  The solution implemented here is to create 2 additional Data Series, each synchronized to the Primary 
  Instrument, because the indicator output is synced to the Primary Instrument.
  The first Data Series (Instrument2Close) is a copy of the secondary instrument, Closes[1].
  The second Data Series is a BoolSeries (Instrument2Synced) which flags the bars that are synced.

  When operating with Historical data, if a synchronized bar exists, it will be Closes[1][0].
  In real time, with (CalculateOnBarClose==false), it gets more complicated, because NT7 is event-driven,
  which mean that bars don't close because of time.  They close because a tick arrives with a timestamp of
  a later bar, at which time OnBarUpdate is called and the new bar starts forming.  This could take a long time.
  So when we are processing a bar for the primary instrument, we need to also check the unclosed bar of the secondary
  instrument, which we can do by referencing Times[1][-1]; this is safe to do because we know the bar's time
  has expired - we are running OnBarUpdate for the primary instrument for the same timestamp, which can
  only happen because a tick arrived for a later bar.

  If we use the unclosed bar (Closes[1][-1]), there is a very small chance that a late-arriving tick will change
  the value after we use it.  We can't change that calculation, but we can fix the value we stored earlier.
  This is done in OnBarUpdate for the secondary instrument.

  Also, in real time, it's possible that the secondary instrument has received bars after the time of the
  bar we are currently processing.  A synchronized bar could exist at Closes[1][1], Closes[1][2], or earlier.
  So we need to scan back through the seconday instrument's closes to see if the Synced bar is there.
  
  The Data Series "Instrument2Close" is filled from OnBarUpdate for the primary instrument, and it will hold the
  synchronized bars we found using the above techniques.  We set a flag in "Instrument2Synced" to show which
  locations are synced.  The routines "KDCorrelation" and "KDSMABool" are written to calculate correlation and SMA
  using only synced data.

  There is an additional issue when using daily bars.  As of this writing, daily bars carry a timestamp of the end of the
  NT7 session template.  This can change depending on which session template is applied to the instrument, either in the
  instrument manager or in the chart properties.  There is no issue if both instruments have the same session template,
  then their daily bars will have the same timestamp and thus be synchronized.  But if the secondary instrument has a timestamp
  later than the primary instrument, the days will be out of sync unless we strip away the times and just use the dates.
*/		
		public class SyncedDataSeries : DataSeries
		{
			public IndicatorBase IndBase;
			public int PrimaryInstrument;		// BarsArray Index of primary instrument (usually 0 in indicators)
												//   but could be higher than 0 for multi-instrument, multi-timeframe indicators
			public int SyncedInstrument;		// BarsArray Index of secondary instrument we want to synchronize to primary instrument
			public IDataSeries DataArray;		// Output Data Series synced to primary instrument, hold synchronized copy of secondary instrument
			public BoolSeries Synced;			// BoolSeries output, synced with primary instrument, holds flags
												//   true=output (DataArray) hold synchronized value, false = output does not hold synchronized value
			public int SynchronizedBarCount;	// Total number of synchronized bars held in output
												//   but some or all may be inaccessible if MaximumBarsLookBack is set to 256
			public int SynchronizedBarCountToday;	// Total number of synchronized bars so far today
			public bool DataStarted;
			private bool dailyBars;
			private bool initializing;
			private bool lastSyncWasToUnclosedBar;
//			private int lastDay;				//  day of the previous bar
			private int lastCurrentBar;			//  needed to work around NT7 bug - FirstTickOfBar is broken in NT7B7
			
			public SyncedDataSeries (IndicatorBase indicator, int primaryInstrument, int syncedInstrument, IDataSeries dataArray) :
				this (indicator, primaryInstrument, syncedInstrument, dataArray, indicator.MaximumBarsLookBack)
			{
			}
			public SyncedDataSeries (IndicatorBase indicator, int primaryInstrument, int syncedInstrument, IDataSeries dataArray, MaximumBarsLookBack maxLookback) : base (indicator, maxLookback)
			{
				IndBase = indicator;
				PrimaryInstrument = primaryInstrument;
				SyncedInstrument = syncedInstrument;
				SynchronizedBarCount = 0;
				SynchronizedBarCountToday = 0;
				DataArray = dataArray;
				DataStarted = false;
				initializing = true;
				lastSyncWasToUnclosedBar = false;
//				lastDay = -1;
				lastCurrentBar = -1;
				
				if (PrimaryInstrument == 0)
					Synced = new BoolSeries(IndBase);  // We are syncing to the primary instrument of the instantiating indicator
				else
					throw new ArgumentOutOfRangeException ("primaryInstrument", "primaryInstrument must = 0 if no syncedIndicator base is given.");
			}
	
			public SyncedDataSeries (IndicatorBase indicator, int primaryInstrument, int syncedInstrument, IDataSeries dataArray, IndicatorBase syncedIndicator) :
				this (indicator, primaryInstrument, syncedInstrument, dataArray, syncedIndicator, indicator.MaximumBarsLookBack)
			{
			}
			public SyncedDataSeries (IndicatorBase indicator, int primaryInstrument, int syncedInstrument, IDataSeries dataArray, IndicatorBase syncedIndicator, MaximumBarsLookBack maxLookback) :base (syncedIndicator, maxLookback)
			{	
				IndBase = indicator;
				PrimaryInstrument = primaryInstrument;
				SyncedInstrument = syncedInstrument;
				SynchronizedBarCount = 0;
				SynchronizedBarCountToday = 0;
				DataArray = dataArray;
				DataStarted = false;
				initializing = true;
				lastSyncWasToUnclosedBar = false;
//				lastDay = -1;
				lastCurrentBar = -1;
				Synced = new BoolSeries(syncedIndicator);	// Not syncing to primary instrument of instantiating indicator;
															// So create a new BoolSeries synced to the master instrument
			}
			
			public double SyncedGet(int syncedBarsBack)
			{
				int i;
				if (syncedBarsBack >= SynchronizedBarCount)
					return 0;			// not enough synchronized bars - return 0 rather than throw exception
				return SyncedGet(syncedBarsBack, 0, out i);
			}
			
			public double SyncedGet(int syncedBarsBack, int initIndex, out int index)
			{
				index = initIndex;
				while (!Synced[index])  //Find most recent synced Bar
					index++;
				
				for (int counter=0; counter < Math.Min(syncedBarsBack,SynchronizedBarCount); counter++)
				{  //looking back past the first synced bar, keep looking for more
					index++;
					while (!Synced[index])  //Find previous synced Bar
						index++;
				}
				return base[index];
			}
			
			public void Synchronize()
			{
				if ((IndBase.BarsInProgress != PrimaryInstrument) && (IndBase.BarsInProgress != SyncedInstrument))
						return;	// Bars being processed are not for us

				if (initializing)
				{
					dailyBars = ((IndBase.BarsPeriods[SyncedInstrument].Id == PeriodType.Day)
								|| (IndBase.BarsPeriods[SyncedInstrument].Id == PeriodType.Week)
								|| (IndBase.BarsPeriods[SyncedInstrument].Id == PeriodType.Month)
								|| (IndBase.BarsPeriods[SyncedInstrument].Id == PeriodType.Year));    // Save a flag which tells us if Bars are Daily
					initializing = false;
				}
				
				if (IndBase.CalculateOnBarClose == true)
				{
					if (IndBase.BarsInProgress == PrimaryInstrument) // Primary Instrument
					{
						if (!DataStarted)
							if (IndBase.BarsArray[SyncedInstrument].CurrentBar == -1)
							{
								Synced.Set(false);
								return;						// return if no data yet from synced instrument
							}
							else
								DataStarted = true;
						
						
						if (IndBase.BarsArray[PrimaryInstrument].FirstBarOfSession)
							SynchronizedBarCountToday = 0;	// it's a new session, zero today's synced bar count
							
//						DateTime currentTime = IndBase.Times[PrimaryInstrument][0]
//						int currentDay = IndBase.ToDay(currentDateTime);
//						if (currentDay != lastDay)
//							SynchronizedBarCountToday = 0;	// it's a new day, zero today's synced bar count
//						lastDay = currentDay;
							
						//	Scan through SyncedInstrument's bars, try to find one in sync with the current Primary instrument bar
						//	Start the scan with the unclosed bar [-1] (necessary in real time); if it has the same time as the Primary instrument's Time[0] (which is now closed), it should be done forming
						int barsBack;
						if (IndBase.BarsArray[SyncedInstrument].CurrentBar == IndBase.BarsArray[SyncedInstrument].Count-1)
							barsBack = 0;
						else
							barsBack = -1;
						
						if (dailyBars)  // if Daily bars, need to use just the Day portion
						{
							while ((barsBack < IndBase.BarsArray[SyncedInstrument].CurrentBar) && (IndBase.ToDay(IndBase.Times[PrimaryInstrument][0]) < IndBase.ToDay(IndBase.Times[SyncedInstrument][barsBack])))
								barsBack ++;  // ignore bars that occur in the future (this can happen in real time if primary instrument is missing bars)
							if (IndBase.ToDay(IndBase.Times[PrimaryInstrument][0]) == IndBase.ToDay(IndBase.Times[SyncedInstrument][barsBack]))
								{	// Found a synchronized bar
									Synced.Set(true);
									SynchronizedBarCount++;
									SynchronizedBarCountToday++;
								}
							else	// No synchronized bar found
								Synced.Set(false);
							base.Set(DataArray[barsBack]);	// set output to most recent bar that isn't in the future
						}
								
						else // Not daily bars, use entire timestamp
						{
							while ((barsBack < IndBase.BarsArray[SyncedInstrument].CurrentBar) && (IndBase.Times[PrimaryInstrument][0]) < IndBase.Times[SyncedInstrument][barsBack])
								barsBack ++;  // ignore bars that occur in the future (this can happen in real time if primary instrument is missing bars)
							if (IndBase.Times[PrimaryInstrument][0] == IndBase.Times[SyncedInstrument][barsBack])
								{	// Found a synchronized bar
									Synced.Set(true);
									SynchronizedBarCount++;
									SynchronizedBarCountToday++;
								}
							else	// No synchronized bar found
								Synced.Set(false);
								
							base.Set(DataArray[barsBack]);	// set output to most recent bar that isn't in the future
						}
					}
					
					else // (IndBase.BarsInProgress == SyncedInstrument)
					{
						// Normally we don't have to do anything when (BarsInProgress == SyncedInstrument) if (CalculateOnBarClose == true)
						// Method output is synced with, and handled by, primary instrument when (BarsInProgress == PrimaryInstrument)
						//
						// Sometimes in a real-time OnBarUpdate for PrimaryIntrument, we find that SyncedInstrument has an unclosed bar (index of [-1]) with the same timestamp,
						// so we use it, which is safe to do because PrimaryInstrument's bar has already closed, so we know that the end of the bar period has passed.
						//
						// We are here because a SecondaryInstrument bar has been closed (in real time by an incoming tick belonging to a later bar, which could be MUCH later)
						// There is a small risk that the value of an unclosed bar we used earlier has changed since then, due to a late-arriving tick.
						// To cover this case, we store the final value in the proper location (the most recent synced bar)
						//
						if (!DataStarted)
							if (IndBase.BarsArray[PrimaryInstrument].CurrentBar == -1)
								return;							// return if no data yet from Primary instrument
							else
								DataStarted = true;
							
						if (lastSyncWasToUnclosedBar)
						{
							lastSyncWasToUnclosedBar = false;   // if there were any late arriving ticks, they are now included in the bar that just closed
							int barsBack=0;
							while (!Synced[barsBack])			// Scan for most recent synced bar
								barsBack++;
							base.Set(barsBack,DataArray[0]);	// Store the final value, in case it has changed since we stored the value from the unclosed bar
						}
					}
				}
				else  // (CalculateOnBarClose == false)
				{
					if (IndBase.BarsInProgress == PrimaryInstrument) // Primary Instrument
					{
						if (!DataStarted)
							if (IndBase.BarsArray[SyncedInstrument].CurrentBar == -1)
							{
								Synced.Set(false);
								base.Reset();
								return;							// return if no data yet from synced instrument
							}
							else
								DataStarted = true;
								
						if (IndBase.BarsArray[PrimaryInstrument].CurrentBar != lastCurrentBar)
//						if (IndBase.FirstTickOfBar)			//broken in NT7B7
						{	// First tick of bar, need to set state of sync flag
							lastCurrentBar = IndBase.BarsArray[PrimaryInstrument].CurrentBar;
							
							if (IndBase.BarsArray[PrimaryInstrument].FirstBarOfSession)
								SynchronizedBarCountToday = 0;	// it's a new session, zero today's synced bar count
							
							if (((!dailyBars) && (IndBase.Times[SyncedInstrument][0] == IndBase.Times[PrimaryInstrument][0]))
								|| (dailyBars && (IndBase.ToDay(IndBase.Times[SyncedInstrument][0]) == IndBase.ToDay(IndBase.Times[PrimaryInstrument][0]))))
							{
								Synced.Set(true);
								base.Set(DataArray[0]);
								SynchronizedBarCount++;
								SynchronizedBarCountToday++;
							}
							else
								if (((!dailyBars) && (IndBase.BarsArray[SyncedInstrument].CurrentBar > 0) && (IndBase.Times[SyncedInstrument][1] == IndBase.Times[PrimaryInstrument][0]))
									|| (dailyBars && (IndBase.BarsArray[SyncedInstrument].CurrentBar > 0) && (IndBase.ToDay(IndBase.Times[SyncedInstrument][1]) == IndBase.ToDay(IndBase.Times[PrimaryInstrument][0]))))
								{
									Synced.Set(true);
									base.Set(DataArray[1]);
									SynchronizedBarCount++;
									SynchronizedBarCountToday++;
								}
								else
								{
									Synced.Set(false);
									base.Set(DataArray[0]);  // store most recent value, in case no updates come for rest of bar
								}
						}
					}
					
					else // (IndBase.BarsInProgress == SyncedInstrument)
					{
						if (!DataStarted)
							if (IndBase.BarsArray[PrimaryInstrument].CurrentBar == -1)
								return;						// return if no data yet from primary instrument
							else
								DataStarted = true;
						
						if (((!dailyBars) && (IndBase.Times[SyncedInstrument][0] == IndBase.Times[PrimaryInstrument][0]))
							|| (dailyBars && (IndBase.ToDay(IndBase.Times[SyncedInstrument][0]) == IndBase.ToDay(IndBase.Times[PrimaryInstrument][0]))))
						{
							if (!Synced[0])
							{
								Synced.Set(true);
								SynchronizedBarCount++;
								SynchronizedBarCountToday++;
							}
							base.Set(DataArray[0]);
						}
					}
				}
			}
		}
		#endregion

		private bool allInputsSynced(int barsBack)
		{	// output is valid only when called with (BarsInProgress == Instrument1)
			if (	Instrument2Close.Synced[barsBack]
				&&	Instrument3Close.Synced[barsBack]
//				&&	Instrument4Close.Synced[barsBack]
				)
				return true;
			else
				return false;
		}
		
		private bool allInputsOpenedToday()
		{	// output is valid only when called with (BarsInProgress == Instrument1)
			if (	(Instrument2Close.SynchronizedBarCountToday != 0)
				&&	(Instrument3Close.SynchronizedBarCountToday != 0)
//				&&	(Instrument4Close.SynchronizedBarCountToday != 0)
				)
				return true;
			else
				return false;
		}
		
		private void produceMissingOutput(int barsBack)
		{
			if (NoOutputWhenMissingInput)
				Value.Reset(barsBack);			// Send "No Output" for the bar (be sure you never use this indicator as input to another indicator!
			else
			{	// Repeat the previous output if it exists
				if (outputStarted && (CurrentBar >= barsBack)) 
				{
					if (Value.ContainsValue(barsBack+1))
						Value.Set(barsBack,Value[barsBack+1]);	// repeat the previous output
					else  // previous bar contained nothing, do the same here
						Value.Reset(barsBack);
				}
			}
		}
		
		private double Spread3Calc(double price1, double q1, double price2, double q2, double price3, double q3 /* , double price4, double q4 */ )
		{
			if (UseMultiplier)
				return
				  (price1 * Instruments[Instrument1].MasterInstrument.PointValue * q1)
				+ (price2 * Instruments[Instrument2].MasterInstrument.PointValue * q2)
				+ (price3 * Instruments[Instrument3].MasterInstrument.PointValue * q3)
//				+ (price4 * Instruments[Instrument4].MasterInstrument.PointValue * q4)
				;
			else
				return 
				  (price1 * q1) 
				+ (price2 * q2) 
				+ (price3 * q3) 
//				+ (price4 * q4)
			;
		}
		
		/// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
		protected override void Initialize()
        {
			CalculateOnBarClose = true;
			BarsRequired = 0;  // OK to plot on first bar
			Add(new Plot(Color.FromKnownColor(KnownColor.ForestGreen), PlotStyle.Line, "Spread3"));
			
			// Add additional instruments
			// Use same BarsPeriod and Value as Primary instrument
			Add(Symbol2, BarsPeriod.Id, BarsPeriod.Value);
			Add(Symbol3, BarsPeriod.Id, BarsPeriod.Value);
//			Add(Symbol4, BarsPeriod.Id, BarsPeriod.Value);
		}

		protected override void OnStartUp()
		{
			Instrument2Close = new SyncedDataSeries(this, Instrument1, Instrument2, Closes[Instrument2]);
			Instrument3Close = new SyncedDataSeries(this, Instrument1, Instrument3, Closes[Instrument3]);
//			Instrument4Close = new SyncedDataSeries(this, Instrument1, Instrument4, Closes[Instrument4]);
			
//			If you need to synchronize High, Low or Volume, instantiate it here like this:			
//			Instrument2Volume = new SyncedDataSeries(this, Instrument1, Instrument2, Volumes[Instrument2]);
//			Instrument2High = new SyncedDataSeries(this, Instrument1, Instrument2, Highs[Instrument2]);

		}
		
		
        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {	
			Instrument2Close.Synchronize();		// Call synchronization method for Instrument2
			Instrument3Close.Synchronize();		// Call synchronization method for Instrument3
//			Instrument4Close.Synchronize();		// Call synchronization method for Instrument4
			
//			If you need to synchronize High, Low or Volume, call the Synchronize method here like this: 
//			Instrument2Volume.Synchronize();	// Call synchronization method for Instrument2 Volume
//			Instrument2High.Synchronize();		// Call synchronization method for Instrument2 High
			
			if (BarsInProgress != Instrument1)	// With more than 2 Instruments, 
				return;							// it's only safe to calculate indicator output when BarsInProgress==Instrument1
			
			if  (  (!Instrument2Close.DataStarted) 
				|| (!Instrument3Close.DataStarted)
//  			|| (!Instrument4Close.DataStarted)
				)
				return;							// Return if we haven't seen data from ALL instruments
			
			if (CalculateOnBarClose == true)
			{
				if (allInputsSynced(0))
				{
					//  We are here because this bar contains current data from Instrument1, Instrument2, & Instrument3  (& Instrument4)
					//	Current data is held in Closes[Instrument1][0], Instrument2Close[0], Instrument3Close[0]  (& Instrument4Close[0])
					//
					//  Do not use Closes[Instrument2][0], Closes[Instrument3][0] as these bars may refer to a different time.
					//
					//  If you need to do lookbacks, Closes[Instrument1][i] ("Close[i]" in this context), Instrument2Close[i], Instrument3Close[i] are all time-synchronized - 
					//     values are valid if (Instrument2Close.Synced[i]==true).
					//  Lookbacks all refer to the same time in each of these series.  Times are held in Times[Instrument1][i]  ("Time[i]" in this context)
					//
					//  If you to worry about missing bars, you can check Instrument2Close.Synced[i] to see if that bar contains
					//  synchronized (valid) data (false means a missing bar for that time).
					//
					//  If you have synchronized Volume, High, and/or Low in addition to Close, the current bar's data will be valid here
					//    i.e. Instrument2Close[0], Instrument2High[0], Instrument2Volume[0] will all refer to the same time as Close[0]
					//    and values will be valid because Instrument2Close.Synced[0]==true. 
					
					Value.Set(Spread3Calc(
					 Close[0],Qty1
					,Instrument2Close[0],Qty2
					,Instrument3Close[0],Qty3 
// 					,Instrument4Close[0],Qty4 
					));
					outputStarted = true;
				}
				else
				{
					// We are here if one of the inputs skipped a bar.
					// We have 3 options:   (1) Do the calculation anyway with the most recent data
					//						(2) repeat the last indicator output
					//						(3) we could also send "no ouput"  ( Value.Reset(0) ) but that causes problems if this indicator is input to another indicator 
					
					// We could just do the calculation using the most recent data; but that could produce (very) misleading output
//					Value.Set(Spread3Calc(Close[0],Qty1,Instrument2Close[0],Qty2,Instrument3Close[0],Qty3 /* ,Instrument4Close[0], Qty4 */ ));  
//					return;
					
					produceMissingOutput(0);
				}
			}
			else  //  (CalculateOnBarClose == false)
			{
				//  We are here if we are processing every tick of Instrument1 (because we want the chart to update in real time)
				//  Current data is held in Closes[Instrument1][0], Instrument2Close[0], Instrument3Close[0]  (& Instrument4Close[0])
				//
				//  Do not use Closes[Instrument2][0], Closes[Instrument3][0] as these bars may refer to a different time.
				//  (You would think that Closes[Instrument2][0] and Closes[Instrument3][0] would always hold the most recent value in this case -
				//   they do in real time, but on historical data there are times they do not)
				//
				//  If you need to do lookbacks, Closes[Instrument1] ("Close" in this context), Instrument2Close, Instrument3Close are all time-synchronized - 
				//  Lookbacks all refer to the same time in each of these series.  Times are held in Times[Instrument1]  ("Time" in this context)
				//
				//  Handling missing bars is more complicated in real time:
				//  Since we are processing every tick, we can't know if an instrument has skipped a bar until the bar is over.
				//
				//  Each SyncedDataSeries has a field named "SynchronizedBarCountToday" which tells how many synchronized bars it has seen today.
				//  A Value of 0 means no bars yet today (i.e. it hasn't opened).
				
				if ((!outputStarted) && (!allInputsSynced(0)))
						return;					// don't process anything until we've seen at least one synchronized bar
				
				if (!allInputsOpenedToday()) 
					{ // not all instruments have opened yet
						produceMissingOutput(0); 
						return; 
					}
				if (Historical)
				{	// Historical data, we missed a bar if the current bar wasn't synced
					if  (!allInputsSynced(0))
					{ 	// Missing an input in current bar, produce "Missing Output" until we see some synchronized data
						produceMissingOutput(0);
						return; 
					}
				}
				else // !Historical
				{	// Real-time tick data, we only know we missed a bar if the previous bar wasn't synced
					if (FirstTickOfBar)
					{  
						lastBarNotSynced = !allInputsSynced(1);
						if  (lastBarNotSynced)
							produceMissingOutput(1);	// Previous bar had a missing input, set it to "Missing Output"
					}
					if (lastBarNotSynced && (!allInputsSynced(0)))
					{	// No data in previous bar or current bar, produce "Missing Output" until we see some synchronized data
							produceMissingOutput(0);
							return; 
					}
				}

				//  The code below will use the last Close value of any instrument, no matter how old it is, if we get here without checking for missing bars.
				Value.Set(Spread3Calc(
				 Close[0],Qty1
				,Instrument2Close[0],Qty2
				,Instrument3Close[0],Qty3 
// 				,Instrument4Close[0],Qty4 
				));
				outputStarted = true;
			}
        }
		
		public override string ToString()
		{
			// Pretty up the chart label
				return Name
				+ "(" + Instruments[Instrument1].FullName + "," + Qty1 
				+ "," + Instruments[Instrument2].FullName + "," + Qty2
				+ "," + Instruments[Instrument3].FullName + "," + Qty3
//				+ "," + Instruments[Instrument4].FullName + "," + Qty4  
				+ ")";
		}

        #region Properties

		[Description("Quantity 1")]
        [GridCategory("Parameters")]
		// Force this parameter to be displayed first
		[Gui.Design.DisplayName ("\t\t\t\t\t\t\tQuantity 1")]
        public double Qty1
        {
            get { return qty1; }
            set { qty1 = value; }
        }
		
        [Description("Symbol 2; i.e. SPY or ES 03-10\nDefault = Second chart instrument")]
        [GridCategory("Parameters")]
		// Force this parameter to be displayed second
		[Gui.Design.DisplayName ("\t\t\t\t\t\tSymbol 2")]
        public string Symbol2
        {
            get 
			{   // symbol2 defaults to secondary chart data series
				if ((symbol2 == "") && (ChartControl != null) && (Instruments != null)) 
					for(int i=0; i<ChartControl.BarsArray.Length; i++)
						if (ChartControl.BarsArray[i].Instrument.FullName != Instruments[Instrument1].FullName)
						{
							symbol2 = ChartControl.BarsArray[i].Instrument.FullName;
							break;
						}
				return symbol2; 
			}
            set { symbol2 = value.ToUpper(); }
		}
		
		[Description("Quantity 2")]
        [GridCategory("Parameters")]
		[Gui.Design.DisplayName ("\t\t\t\t\tQuantity 2")]
        public double Qty2
        {
            get { return qty2; }
            set { qty2 = value; }
        }
		
		[Description("Symbol 3; i.e. SPY or ES 03-10\nDefault = Third chart instrument")]
        [GridCategory("Parameters")]
		// Force this parameter to be displayed fourth
		[Gui.Design.DisplayName ("\t\t\t\tSymbol 3")]
        public string Symbol3
        {
            get 
			{   // symbol3 defaults to third chart data series
				if ((symbol3 == "") && (ChartControl != null) && (Instruments != null)) 
					for(int i=0; i<ChartControl.BarsArray.Length; i++)
						if    ((ChartControl.BarsArray[i].Instrument.FullName != Instruments[Instrument1].FullName)
							&& (ChartControl.BarsArray[i].Instrument.FullName != Instruments[Instrument2].FullName))
						{
							symbol3 = ChartControl.BarsArray[i].Instrument.FullName;
							break;
						}
				return symbol3; 
			}
            set { symbol3 = value.ToUpper(); }
		}
		
		[Description("Quantity 3")]
        [GridCategory("Parameters")]
		[Gui.Design.DisplayName ("\t\t\tQuantity 3")]
        public double Qty3
        {
            get { return qty3; }
            set { qty3 = value; }
        }
		
//		[Description("Symbol 4; i.e. SPY or ES 03-10\nDefault = Fourth chart instrument")]
//		[GridCategory("Parameters")]
//		// Force this parameter to be displayed sixth
//		[Gui.Design.DisplayName ("\t\tSymbol 4")]
//		public string Symbol4
//		{
//			get 
//			{   // symbol4 defaults to fourth chart data series
//				if ((symbol4 == "") && (ChartControl != null) && (Instruments != null)) 
//					for(int i=0; i<ChartControl.BarsArray.Length; i++)
//						if    ((ChartControl.BarsArray[i].Instrument.FullName != Instruments[Instrument1].FullName)
//							&& (ChartControl.BarsArray[i].Instrument.FullName != Instruments[Instrument2].FullName)
//							&& (ChartControl.BarsArray[i].Instrument.FullName != Instruments[Instrument3].FullName))
//						{
//							symbol4 = ChartControl.BarsArray[i].Instrument.FullName;
//							break;
//						}
//				return symbol4; 
//			}
//			set { symbol4 = value.ToUpper(); }
//		}
		
//		[Description("Quantity 4")]
//		[GridCategory("Parameters")]
//		[Gui.Design.DisplayName ("\tQuantity 4")]
//		public double Qty4
//		{
//			get { return qty4; }
//			set { qty4 = value; }
//		}
		
		
		[Description("True: Use Price * Contract Multiplier\nFalse: Use Price")]
        [GridCategory("Parameters")]
        public bool UseMultiplier
        {
            get { return useMultiplier; }
            set { useMultiplier = value; }
        }
		
		[Description("True: No Plot on Missing Input")]
        [GridCategory("Parameters")]
        public bool NoOutputWhenMissingInput
        {
            get { return noOutputWhenMissingInput; }
            set { noOutputWhenMissingInput = value; }
        }
		
        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private Spread3[] cacheSpread3 = null;

        private static Spread3 checkSpread3 = new Spread3();

        /// <summary>
        /// Spread3 Price of Three Instruments
        /// </summary>
        /// <returns></returns>
        public Spread3 Spread3(bool noOutputWhenMissingInput, double qty1, double qty2, double qty3, string symbol2, string symbol3, bool useMultiplier)
        {
            return Spread3(Input, noOutputWhenMissingInput, qty1, qty2, qty3, symbol2, symbol3, useMultiplier);
        }

        /// <summary>
        /// Spread3 Price of Three Instruments
        /// </summary>
        /// <returns></returns>
        public Spread3 Spread3(Data.IDataSeries input, bool noOutputWhenMissingInput, double qty1, double qty2, double qty3, string symbol2, string symbol3, bool useMultiplier)
        {
            if (cacheSpread3 != null)
                for (int idx = 0; idx < cacheSpread3.Length; idx++)
                    if (cacheSpread3[idx].NoOutputWhenMissingInput == noOutputWhenMissingInput && Math.Abs(cacheSpread3[idx].Qty1 - qty1) <= double.Epsilon && Math.Abs(cacheSpread3[idx].Qty2 - qty2) <= double.Epsilon && Math.Abs(cacheSpread3[idx].Qty3 - qty3) <= double.Epsilon && cacheSpread3[idx].Symbol2 == symbol2 && cacheSpread3[idx].Symbol3 == symbol3 && cacheSpread3[idx].UseMultiplier == useMultiplier && cacheSpread3[idx].EqualsInput(input))
                        return cacheSpread3[idx];

            lock (checkSpread3)
            {
                checkSpread3.NoOutputWhenMissingInput = noOutputWhenMissingInput;
                noOutputWhenMissingInput = checkSpread3.NoOutputWhenMissingInput;
                checkSpread3.Qty1 = qty1;
                qty1 = checkSpread3.Qty1;
                checkSpread3.Qty2 = qty2;
                qty2 = checkSpread3.Qty2;
                checkSpread3.Qty3 = qty3;
                qty3 = checkSpread3.Qty3;
                checkSpread3.Symbol2 = symbol2;
                symbol2 = checkSpread3.Symbol2;
                checkSpread3.Symbol3 = symbol3;
                symbol3 = checkSpread3.Symbol3;
                checkSpread3.UseMultiplier = useMultiplier;
                useMultiplier = checkSpread3.UseMultiplier;

                if (cacheSpread3 != null)
                    for (int idx = 0; idx < cacheSpread3.Length; idx++)
                        if (cacheSpread3[idx].NoOutputWhenMissingInput == noOutputWhenMissingInput && Math.Abs(cacheSpread3[idx].Qty1 - qty1) <= double.Epsilon && Math.Abs(cacheSpread3[idx].Qty2 - qty2) <= double.Epsilon && Math.Abs(cacheSpread3[idx].Qty3 - qty3) <= double.Epsilon && cacheSpread3[idx].Symbol2 == symbol2 && cacheSpread3[idx].Symbol3 == symbol3 && cacheSpread3[idx].UseMultiplier == useMultiplier && cacheSpread3[idx].EqualsInput(input))
                            return cacheSpread3[idx];

                Spread3 indicator = new Spread3();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.NoOutputWhenMissingInput = noOutputWhenMissingInput;
                indicator.Qty1 = qty1;
                indicator.Qty2 = qty2;
                indicator.Qty3 = qty3;
                indicator.Symbol2 = symbol2;
                indicator.Symbol3 = symbol3;
                indicator.UseMultiplier = useMultiplier;
                Indicators.Add(indicator);
                indicator.SetUp();

                Spread3[] tmp = new Spread3[cacheSpread3 == null ? 1 : cacheSpread3.Length + 1];
                if (cacheSpread3 != null)
                    cacheSpread3.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheSpread3 = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Spread3 Price of Three Instruments
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.Spread3 Spread3(bool noOutputWhenMissingInput, double qty1, double qty2, double qty3, string symbol2, string symbol3, bool useMultiplier)
        {
            return _indicator.Spread3(Input, noOutputWhenMissingInput, qty1, qty2, qty3, symbol2, symbol3, useMultiplier);
        }

        /// <summary>
        /// Spread3 Price of Three Instruments
        /// </summary>
        /// <returns></returns>
        public Indicator.Spread3 Spread3(Data.IDataSeries input, bool noOutputWhenMissingInput, double qty1, double qty2, double qty3, string symbol2, string symbol3, bool useMultiplier)
        {
            return _indicator.Spread3(input, noOutputWhenMissingInput, qty1, qty2, qty3, symbol2, symbol3, useMultiplier);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Spread3 Price of Three Instruments
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.Spread3 Spread3(bool noOutputWhenMissingInput, double qty1, double qty2, double qty3, string symbol2, string symbol3, bool useMultiplier)
        {
            return _indicator.Spread3(Input, noOutputWhenMissingInput, qty1, qty2, qty3, symbol2, symbol3, useMultiplier);
        }

        /// <summary>
        /// Spread3 Price of Three Instruments
        /// </summary>
        /// <returns></returns>
        public Indicator.Spread3 Spread3(Data.IDataSeries input, bool noOutputWhenMissingInput, double qty1, double qty2, double qty3, string symbol2, string symbol3, bool useMultiplier)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.Spread3(input, noOutputWhenMissingInput, qty1, qty2, qty3, symbol2, symbol3, useMultiplier);
        }
    }
}
#endregion
