// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// The Stochastic Oscillator is made up of two lines that oscillate between a vertical scale of 0 to 100. The %K is the main line and it is drawn as a solid line. The second is the %D line and is a moving average of %K. The %D line is drawn as a dotted line. Use as a buy/sell signal generator, buying when fast moves above slow and selling when fast moves below slow.
	/// </summary>
	[Description("The Stochastic Oscillator is made up of two lines that oscillate between a vertical scale of 0 to 100. The %K is the main line and it is drawn as a solid line. The second is the %D line and is a moving average of %K. The %D line is drawn as a dotted line. Use as a buy/sell signal generator, buying when fast moves above slow and selling when fast moves below slow.")]
	public class StochasticsColors : Indicator
	{
		#region Variables
		
		private int				periodD	= 7;	// SlowDperiod
		private int				periodK	= 14;	// Kperiod
		private int				smooth	= 3;	// SlowKperiod
		
		private DataSeries		den;
		private DataSeries		nom;
		
		private int alertRearmSeconds = 10;
		private Color alertBackColor = Color.Blue;
		private Color alertForeColor = Color.Yellow;
		private bool alertBool = false;
		
		private DataSeries		k;
		
		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Plot(Color.Violet, "DUpper"));
			Add(new Plot(Color.Aqua, "DMiddle"));
			Add(new Plot(Color.LightGreen, "DLower"));
			
			Add(new Plot(Color.Red, "KUpper"));
			Add(new Plot(Color.Blue, "KMiddle"));
			Add(new Plot(Color.Lime, "KLower"));
			
			Plots[3].Pen.Width = 3;
			Plots[4].Pen.Width = 3;
			Plots[5].Pen.Width = 3;
			
			Add(new Line(Color.Red, 20, "Lower"));
			Add(new Line(Color.Lime, 80, "Upper"));

			den			= new DataSeries(this);
			nom			= new DataSeries(this);
			
			k			= new DataSeries(this);
						
		}

		/// <summary>
		/// Calculates the indicator value(s) at the current index.
		/// </summary>
		protected override void OnBarUpdate()
		{
			nom.Set(Close[0] - MIN(Low, PeriodK)[0]);
			den.Set(MAX(High, PeriodK)[0] - MIN(Low, PeriodK)[0]);										
			
			k.Set(100 * SUM(nom, Smooth)[0] / (SUM(den, Smooth)[0] == 0 ? 1.0 : SUM(den, Smooth)[0]));	
						
			KUpper.Set(100 * SUM(nom, Smooth)[0] / (SUM(den, Smooth)[0] == 0 ? 1.0 : SUM(den, Smooth)[0]));	
			DUpper.Set(SMA(KUpper, PeriodD)[0]);																	
			
			KMiddle.Set(100 * SUM(nom, Smooth)[0] / (SUM(den, Smooth)[0] == 0 ? 1.0 : SUM(den, Smooth)[0]));	
			DMiddle.Set(SMA(KMiddle, PeriodD)[0]);	
			
			KLower.Set(100 * SUM(nom, Smooth)[0] / (SUM(den, Smooth)[0] == 0 ? 1.0 : SUM(den, Smooth)[0]));	
			DLower.Set(SMA(KLower, PeriodD)[0]);	
			
			
			
			Plots[0].Min = Lines[1].Value; 
			Plots[1].Max = Lines[1].Value;
			Plots[1].Min = Lines[0].Value;
			Plots[2].Max = Lines[0].Value; 
			
			Plots[3].Min = Lines[1].Value; 
			Plots[4].Max = Lines[1].Value;
			Plots[4].Min = Lines[0].Value;
			Plots[5].Max = Lines[0].Value;
			
			if (AlertBool)
			{
				if (CrossAbove(k,Lines[0].Value,1))
					Alert("CrossAbove", NinjaTrader.Cbi.Priority.Medium, Instrument.FullName + " Stochastics moved out of oversold" , "Alert4.wav", AlertRearmSeconds, AlertBackColor, AlertForeColor);
			
				if (CrossBelow(k,Lines[1].Value,1))
					Alert("CrossBelow", NinjaTrader.Cbi.Priority.Medium, Instrument.FullName + " Stochastics moved out of overbought", "Alert4.wav", AlertRearmSeconds, AlertBackColor, AlertForeColor);
			}
			
		}

		#region Properties
		/// <summary>
		/// Gets the slow D value.
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries DUpper
		{
			get { return Values[0]; }
		}

		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries DMiddle
		{
			get { return Values[1]; }
		}
		
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries DLower
		{
			get { return Values[2]; }
		}
		
		/// <summary>
		/// Gets the slow K value.
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries KUpper
		{
			get { return Values[3]; }
		}
		
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries KMiddle
		{
			get { return Values[4]; }
		}
		
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries KLower
		{
			get { return Values[5]; }
		}
		
		/// <summary>
		/// </summary>
		[Description("Numbers of bars used for moving average over K values")]
		[GridCategory("Parameters")]
		public int PeriodD
		{
			get { return periodD; }
			set { periodD = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Description("Numbers of bars used for calculating the K values")]
		[GridCategory("Parameters")]
		public int PeriodK
		{
			get { return periodK; }
			set { periodK = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Description("Number of bars for smoothing the slow K values")]
		[GridCategory("Parameters")]
		public int Smooth
		{
			get { return smooth; }
			set { smooth = Math.Max(1, value); }
		}
		
		[Description("Seconds to wait until alert rearms")]
        [Category("Alert")]
        public int AlertRearmSeconds
        {
            get { return alertRearmSeconds; }
            set { alertRearmSeconds = Math.Max(1, value); }
        }
		
		[XmlIgnore()]
        [Description("Alert back color.")]
        [Category("Alert")]
        public Color AlertBackColor
        {
            get { return alertBackColor; }
            set { alertBackColor = value; }
        }
		
		// Serialize our Color object
		[Browsable(false)]
		public string BorderColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(alertBackColor); }
			set { alertBackColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}

		[XmlIgnore()]
		[Description("Alert fore color.")]
        [Category("Alert")]
        public Color AlertForeColor
        {
            get { return alertForeColor; }
            set {alertForeColor = value; }
        }
		
		[Browsable(false)]
		public string FillColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(alertForeColor); }
			set { alertForeColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		[Gui.Design.DisplayNameAttribute("Alert?")]
        [Description("Send an alert to Alerts window when the K plot moves out of oversold or overbought.")]
		[Category("Alert")]
        public bool AlertBool
        {
            get { return alertBool; }
            set { alertBool = value; }
        }
		
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private StochasticsColors[] cacheStochasticsColors = null;

        private static StochasticsColors checkStochasticsColors = new StochasticsColors();

        /// <summary>
        /// The Stochastic Oscillator is made up of two lines that oscillate between a vertical scale of 0 to 100. The %K is the main line and it is drawn as a solid line. The second is the %D line and is a moving average of %K. The %D line is drawn as a dotted line. Use as a buy/sell signal generator, buying when fast moves above slow and selling when fast moves below slow.
        /// </summary>
        /// <returns></returns>
        public StochasticsColors StochasticsColors(int periodD, int periodK, int smooth)
        {
            return StochasticsColors(Input, periodD, periodK, smooth);
        }

        /// <summary>
        /// The Stochastic Oscillator is made up of two lines that oscillate between a vertical scale of 0 to 100. The %K is the main line and it is drawn as a solid line. The second is the %D line and is a moving average of %K. The %D line is drawn as a dotted line. Use as a buy/sell signal generator, buying when fast moves above slow and selling when fast moves below slow.
        /// </summary>
        /// <returns></returns>
        public StochasticsColors StochasticsColors(Data.IDataSeries input, int periodD, int periodK, int smooth)
        {
            if (cacheStochasticsColors != null)
                for (int idx = 0; idx < cacheStochasticsColors.Length; idx++)
                    if (cacheStochasticsColors[idx].PeriodD == periodD && cacheStochasticsColors[idx].PeriodK == periodK && cacheStochasticsColors[idx].Smooth == smooth && cacheStochasticsColors[idx].EqualsInput(input))
                        return cacheStochasticsColors[idx];

            lock (checkStochasticsColors)
            {
                checkStochasticsColors.PeriodD = periodD;
                periodD = checkStochasticsColors.PeriodD;
                checkStochasticsColors.PeriodK = periodK;
                periodK = checkStochasticsColors.PeriodK;
                checkStochasticsColors.Smooth = smooth;
                smooth = checkStochasticsColors.Smooth;

                if (cacheStochasticsColors != null)
                    for (int idx = 0; idx < cacheStochasticsColors.Length; idx++)
                        if (cacheStochasticsColors[idx].PeriodD == periodD && cacheStochasticsColors[idx].PeriodK == periodK && cacheStochasticsColors[idx].Smooth == smooth && cacheStochasticsColors[idx].EqualsInput(input))
                            return cacheStochasticsColors[idx];

                StochasticsColors indicator = new StochasticsColors();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.PeriodD = periodD;
                indicator.PeriodK = periodK;
                indicator.Smooth = smooth;
                Indicators.Add(indicator);
                indicator.SetUp();

                StochasticsColors[] tmp = new StochasticsColors[cacheStochasticsColors == null ? 1 : cacheStochasticsColors.Length + 1];
                if (cacheStochasticsColors != null)
                    cacheStochasticsColors.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheStochasticsColors = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// The Stochastic Oscillator is made up of two lines that oscillate between a vertical scale of 0 to 100. The %K is the main line and it is drawn as a solid line. The second is the %D line and is a moving average of %K. The %D line is drawn as a dotted line. Use as a buy/sell signal generator, buying when fast moves above slow and selling when fast moves below slow.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.StochasticsColors StochasticsColors(int periodD, int periodK, int smooth)
        {
            return _indicator.StochasticsColors(Input, periodD, periodK, smooth);
        }

        /// <summary>
        /// The Stochastic Oscillator is made up of two lines that oscillate between a vertical scale of 0 to 100. The %K is the main line and it is drawn as a solid line. The second is the %D line and is a moving average of %K. The %D line is drawn as a dotted line. Use as a buy/sell signal generator, buying when fast moves above slow and selling when fast moves below slow.
        /// </summary>
        /// <returns></returns>
        public Indicator.StochasticsColors StochasticsColors(Data.IDataSeries input, int periodD, int periodK, int smooth)
        {
            return _indicator.StochasticsColors(input, periodD, periodK, smooth);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// The Stochastic Oscillator is made up of two lines that oscillate between a vertical scale of 0 to 100. The %K is the main line and it is drawn as a solid line. The second is the %D line and is a moving average of %K. The %D line is drawn as a dotted line. Use as a buy/sell signal generator, buying when fast moves above slow and selling when fast moves below slow.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.StochasticsColors StochasticsColors(int periodD, int periodK, int smooth)
        {
            return _indicator.StochasticsColors(Input, periodD, periodK, smooth);
        }

        /// <summary>
        /// The Stochastic Oscillator is made up of two lines that oscillate between a vertical scale of 0 to 100. The %K is the main line and it is drawn as a solid line. The second is the %D line and is a moving average of %K. The %D line is drawn as a dotted line. Use as a buy/sell signal generator, buying when fast moves above slow and selling when fast moves below slow.
        /// </summary>
        /// <returns></returns>
        public Indicator.StochasticsColors StochasticsColors(Data.IDataSeries input, int periodD, int periodK, int smooth)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.StochasticsColors(input, periodD, periodK, smooth);
        }
    }
}
#endregion
