#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Timers;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
using System.IO;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Detects and alerts you to divergence between price and fast stochastics
	/// Updated by; Tony Alcamo Feb 24 2011 talcamo@gmail.com
	/// changes:
	/// 1) updated DrawText statements to conform to NT7 format
	/// 2) moved initialize of DataSeries to onbarupdate method to conform to new Nt7 format
	/// 3) changed Name to NT7GbDivergence3
    /// </summary>
    [Description("Detects and alerts you to divergence between price and fast stochastics")]
    public class NT7GbDivergence3 : Indicator
    {
        #region Variables
		private int				periodD	= 3;
		private int				periodK	= 8;
		private DataSeries		den;
		private DataSeries		nom;

	// Wizard generated variables
		private string lineSelected = "D";
		private int periods = 45; // Default setting for Periods
		private int peakMargin = 10; // Default setting for PeakRange

	// User defined variables (add any user defined variables below)
		private static peakDetail[] hiPeaks;
		private static peakDetail[] loPeaks;
		private int hiCount;
		private int loCount;
		private double sto;		// temp var used for current sto value
		private Font myFont;
		private string period;
		
		
		private bool firsttime	=true;
        #endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Print("=====================================");
			Print("GbDivergence2 Initialize()");

			Add(new Plot(new Pen(Color.Red,4), "D"));
			Add(new Plot(new Pen(Color.Blue,2), "K"));

			Add(new Line(new Pen(Color.DarkViolet,2), 20, "Lower"));
			Add(new Line(new Pen(Color.YellowGreen,2), 80, "Upper"));

			den		= new DataSeries(this);
			nom		= new DataSeries(this);

			CalculateOnBarClose	= false;
			//Overlay				= false;
			//PriceTypeSupported	= false;
			DrawOnPricePanel	= false;

			Name = " NT7GbDivergence3";
			
		}

		/// <summary>
		/// Called on each bar update event (incoming tick)
		/// </summary>
		//===========================================================================
		protected override void OnBarUpdate()
		{
			
			if (firsttime==true)
			{
				
				myFont = new Font("Arial Bold",8);

			period = Bars.Period.Value.ToString();

			hiPeaks = new peakDetail[6];
			loPeaks = new peakDetail[6];
			
				
				for (int index = 0; index<hiPeaks.GetUpperBound(0); index++)
				{
					hiPeaks[index] = new peakDetail(-1);
					loPeaks[index] = new peakDetail(101);
				}
				firsttime=false;
			}
			
			
			nom.Set(Close[0] - MIN(Low, PeriodK)[0]);
			den.Set(MAX(High, PeriodK)[0] - MIN(Low, PeriodK)[0]);

			K.Set(100 * nom[0] / (den[0] == 0 ? 1.0 : den[0])); 
			D.Set(100 * SUM(nom, PeriodD)[0] / (SUM(den, PeriodD)[0] == 0 ? 1.0 : SUM(den, PeriodD)[0]));

			try
			{
				if (period=="1" || period=="5")
				{
					// If extreme Sto hit, create an empty file so custom GbBollinger indicator can detect it
					if (K[0] == 0.0)
					{
						if (!File.Exists("Sto"+period+"Min000.gb"))
						{
							File.WriteAllText("Sto"+period+"Min000.gb","");	// indicate a 0 hit on Sto
							Print(period+"Min 0 written");
						}
					}
					else if (K[0] == 100.0)
					{
						if (!File.Exists("Sto"+period+"Min100.gb"))
						{
							File.WriteAllText("Sto"+period+"Min100.gb","");	// indicate a 100 hit on Sto
							Print(period+"Min 100 written");
						}
					}
				}
				
				if (FirstTickOfBar)			// calculate peaks once per bar
				{
					Print("============== 1st tick ==================");
					Print(string.Format("GbDivergence2 on {0} {1} bar chart at {2}", Bars.Period.Value, Bars.Period.Id, DateTime.Now));
	
					hiCount = 0;
					loCount = 0;
	
					// scan recent stochastic values to identify peaks
					for (int BarsBack=periods; BarsBack>1; BarsBack--)
					{
						//Print(BarsBack.ToString("BarsBack {0}"));
	
						sto = (lineSelected=="K") ? K[BarsBack] : D[BarsBack];
	
						// look for an extreme top peak (or plateau)
						if ( sto>=100-peakMargin && isHiPeak(sto,BarsBack) )
						{
							hiCount = 0;		// reset hiCount so this will be a new #1 peak
							AddHiPeak(sto,BarsBack);	// add the peak to the list
						}
						// look for other hi peaks (must be above 50)
						else if (hiCount>0 && sto>50 && isHiPeak(sto,BarsBack) )
						{
							AddHiPeak(sto,BarsBack);	// add the peak to the list
						}
	
						// look for an extreme lower peak
						else if ( sto<=peakMargin && isLoPeak(sto,BarsBack) )
						{
							loCount = 0;		// reset hiCount so this will be a new #1 peak
							AddLoPeak(sto,BarsBack);	// add the peak to the list
						}
						// look for other lo peaks (must be below 50)
						else if (loCount>0 && sto<50 && isLoPeak(sto,BarsBack) )
						{
							AddLoPeak(sto,BarsBack);		// add the peak to the list
						}
					}	// end of looking for peaks
	
					//RemoveDrawObjects();					// remove all extra marks
	
					
					// Label HI peaks and print list in output window
					Print(string.Format("Final: {0} Hi peaks",hiCount ));
					for (int index=0; index<hiCount; index++)
					{
						Print(string.Format("{0})  time={1:HH:mm}  sto={2:0.0}  Close={3:0.00}",
							index+1, Time[hiPeaks[index].BarsBack].TimeOfDay, hiPeaks[index].sto, hiPeaks[index].Close));
						double y = (hiPeaks[index].sto>95) ? hiPeaks[index].sto - 8 : hiPeaks[index].sto + 7;
						DrawText(string.Format("hi{0}",true,Time[hiPeaks[index].BarsBack].TimeOfDay),true,string.Format("{0}",index+1), 
							hiPeaks[index].BarsBack, y, 0,Color.White, myFont,StringAlignment.Center,Color.Red,Color.Red,60 );
					}
	
					// Label LO peaks and print list in output window
					Print(string.Format("Final: {0} Lo peaks",loCount));
					for (int index=0; index<loCount; index++)
					{
						Print(string.Format("{0})  time={1:t}  sto={2:0.0}  Close={3:0.00}",
							index+1, Time[loPeaks[index].BarsBack].TimeOfDay, loPeaks[index].sto, loPeaks[index].Close));
						double y = (loPeaks[index].sto<5) ? loPeaks[index].sto + 6 : loPeaks[index].sto - 8;
						DrawText(string.Format("lo{0}",true,Time[loPeaks[index].BarsBack].TimeOfDay),true,string.Format("{0}",index+1), 
							loPeaks[index].BarsBack, y,0, Color.White, myFont,StringAlignment.Center,Color.Green,Color.Green,60 );
					}
					
					// check for RECENT divergence on hi peaks
					if (hiCount>=2)		// if we have at least 2 peaks
					{
						// loop thru peaks and check for divergence
						for (int peakIndex=hiCount-1; peakIndex>0; peakIndex--)
						{
							Print(string.Format("Checking for divergence across hi peaks 1 and {0}",peakIndex+1));
							double run = hiPeaks[0].BarsBack - hiPeaks[peakIndex].BarsBack;
							double slopeSto = (hiPeaks[peakIndex].sto - hiPeaks[0].sto) / run;
							double slopePrice = (hiPeaks[peakIndex].Close - hiPeaks[0].Close) / run;
							Print(string.Format("\t*** SlopeSto={0:0.00}  SlopePrice={1:0.00}",slopeSto,slopePrice));
							if ( slopeSto<0 && slopePrice>=0 )
							{
								Print("DIVERGENCE SHORT");
								// Draw a line from the first peak to the last peak of divergence
								DrawLine(string.Format("divHi{0}-{1}",hiPeaks[0].BarsBack,hiPeaks[peakIndex].BarsBack),
									true, hiPeaks[0].BarsBack, hiPeaks[0].sto,
									hiPeaks[peakIndex].BarsBack, hiPeaks[peakIndex].sto, Color.Red, DashStyle.Dot, 3);

								DrawOnPricePanel	= true;
								// draw the starting point # on price chart
								DrawText(string.Format("divPr{0}",true,Time[hiPeaks[0].BarsBack].TimeOfDay),true,"1",
									hiPeaks[0].BarsBack, hiPeaks[0].Close,0, Color.White, myFont,StringAlignment.Center,Color.Black,Color.Black,60);
								// draw the ending point # on price chart
								DrawText(string.Format("divPr{0}",true,Time[hiPeaks[peakIndex].BarsBack].TimeOfDay),true,string.Format("{0}",peakIndex+1),
									hiPeaks[peakIndex].BarsBack, hiPeaks[peakIndex].Close,0, Color.White, myFont,StringAlignment.Center,Color.Black,Color.Black,60);
								DrawOnPricePanel	= false;

								// if this is a recent divergence  and it's for Minute bars
								if (hiPeaks[peakIndex].BarsBack<=4 && Bars.Period.Id.ToString().StartsWith("Min"))
								{
									if (Bars.Period.Value==1)
										PlaySound("DivergenceShort1.wav");
									else if (Bars.Period.Value==5)
										PlaySound("DivergenceShort5.wav");
								}
							}
						}	// end for
					}	// end if
					
					// check for RECENT divergence on lo peaks
					if (loCount>=2)		// if we have at least 2 peaks
					{
						// loop thru peaks and check for divergence
						for (int peakIndex=loCount-1; peakIndex>0; peakIndex--)
						{
							Print(string.Format("Checking for divergence across lo peaks 1 and {0}",peakIndex+1));
							double run = loPeaks[0].BarsBack - loPeaks[peakIndex].BarsBack;
							double slopeSto = (loPeaks[peakIndex].sto - loPeaks[0].sto) / run;
							double slopePrice = ( loPeaks[peakIndex].Close - loPeaks[0].Close) / run;
							Print(string.Format("*** SlopeSto={0:0.00}  SlopePrice={1:0.00}",slopeSto,slopePrice));
							if ( slopeSto>0 && slopePrice<=0 )
							{
								Print("DIVERGENCE LONG");
								// Draw a line from the first peak to the last peak of divergence
								DrawLine(string.Format("divLo{0}-{1}",loPeaks[0].BarsBack,loPeaks[peakIndex].BarsBack),
									true, loPeaks[0].BarsBack, loPeaks[0].sto,
									loPeaks[peakIndex].BarsBack, loPeaks[peakIndex].sto, Color.Green, DashStyle.Dot, 3);
								
								DrawOnPricePanel	= true;
								// draw the starting point # on price chart
								DrawText(string.Format("divPr{0}",true,Time[loPeaks[0].BarsBack].TimeOfDay),true,"1",
									loPeaks[0].BarsBack, loPeaks[0].Close,0, Color.White, myFont,StringAlignment.Center,Color.Black,Color.Black,60);
								// draw the ending point # on price chart
								DrawText(string.Format("divPr{0}",true,Time[loPeaks[peakIndex].BarsBack].TimeOfDay),true,string.Format("{0}",peakIndex+1),
									loPeaks[peakIndex].BarsBack, loPeaks[peakIndex].Close,0, Color.White, myFont,StringAlignment.Center,Color.Black,Color.Black,60);
								DrawOnPricePanel	= false;

								// if this is a recent divergence  and  it's for Minute bars
								if (loPeaks[peakIndex].BarsBack<=4 && Bars.Period.Id.ToString().StartsWith("Min"))
								{
									//Print(string.Format("Timer started from LONG: {0:T}",DateTime.Now));
									// determine time base to play different messages
									if (Bars.Period.Value==1)
										PlaySound("DivergenceLong1.wav");
									else if (Bars.Period.Value==5)
										PlaySound("DivergenceLong5.wav");
								}
							}
						}	// end for
					}	// end if
				}	// end try
			}
			catch (Exception ex)
			{
				Print(string.Format("ERROR in GbDivergence2: {0}\n{1}\n{2}",ex.Message,ex.InnerException,ex.StackTrace ));
			}
		}

		//===========================================================================
		private bool isHiPeak(double sto, int BarsBack)
		{
			if (lineSelected=="K")
			{
				return (sto>K[BarsBack-1] &&  ( sto>K[BarsBack+1] ||
				( sto>=K[BarsBack+1] && sto>K[BarsBack+2] ) ||
				( sto>=K[BarsBack+1] && sto>=K[BarsBack+2] && sto>K[BarsBack+3] ) ) );
			}
			else
			{
				return (sto>D[BarsBack-1] &&  ( sto>D[BarsBack+1] ||
				( sto>=D[BarsBack+1] && sto>D[BarsBack+2] ) ||
				( sto>=D[BarsBack+1] && sto>=D[BarsBack+2] && sto>D[BarsBack+3] ) ) );
			}
		}
		
		//===========================================================================
		//= Pass sto into this function to check
		private bool isLoPeak(double sto, int BarsBack)
		{
			if (lineSelected=="K")
			{
				return (sto<K[BarsBack-1] &&  ( sto<K[BarsBack+1] ||
				( sto<=K[BarsBack+1] && sto<K[BarsBack+2] ) ||
				( sto<=K[BarsBack+1] && sto-1<=K[BarsBack+2] && sto<K[BarsBack+3] ) ) );
			}
			else
			{
				return (sto<D[BarsBack-1] &&  ( sto<D[BarsBack+1] ||
				( sto<=D[BarsBack+1] && sto<D[BarsBack+2] ) ||
				( sto<=D[BarsBack+1] && sto<=D[BarsBack+2] && sto<D[BarsBack+3] ) ) );
			}
		}
		
		//===========================================================================
		private void AddHiPeak(double sto, int BarsBack)
		{
			if (hiCount<hiPeaks.GetUpperBound(0))
			{
				//Print(string.Format("Adding hi peak #{0} for {1:t} K={2:#.#} Pr={3:#.##}",hiCount+1,Time[BarsBack].TimeOfDay,K,Close[BarsBack] ));
				hiPeaks[hiCount].sto = sto;
				hiPeaks[hiCount].BarsBack = BarsBack;
				hiPeaks[hiCount].Close = Close[BarsBack];
				hiCount++;
			}
		}
		
		//===========================================================================
		private void AddLoPeak(double sto, int BarsBack)
		{
			if (loCount<loPeaks.GetUpperBound(0))
			{
				//Print(string.Format("Adding low peak #{0} for {1:t} K={2:#.#} Pr={3:#.##}",loCount+1,Time[BarsBack].TimeOfDay,K,Close[BarsBack] ));
				loPeaks[loCount].sto = sto;
				loPeaks[loCount].BarsBack = BarsBack;
				loPeaks[loCount].Close = Close[BarsBack];
				loCount++;
			}
		}

		
        #region Properties
		[Browsable(false)]
		[XmlIgnore()]
		/// <summary>
		/// Gets the fast D value.
		/// </summary>
		public DataSeries D
		{
			get { return Values[0]; }
		}

		/// <summary>
		/// Gets the fast K value.
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries K
		{
			get { return Values[1]; }
		}
		
		[Description("Numbers of bars used for the moving average over K values")]
		[Category("Parameters")]
		public int PeriodD
		{
			get { return periodD; }
			set { periodD = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Description("Numbers of bars used for calculating the K values")]
		[Category("Parameters")]
		public int PeriodK
		{
			get { return periodK; }
			set { periodK = Math.Max(1, value); }
		}


		[Description("Number of periods to look back")]
		[Category("Parameters")]
		public int Periods
		{
			get { return periods; }
			set { periods = Math.Max(7, value); }
		}

		[Description("How near to extremes must #1 peak be? (ie. if set to 10... >=90 and <=10 are peaks)")]
		[Category("Parameters")]
		public int PeakMargin
		{
			get { return peakMargin; }
			set { peakMargin = Math.Max(5, Math.Min(value,20)); }
		}

		[Description("Which stochastics line (D or K) to check for divergence?")]
		[Category("Parameters")]
		public string LineSelected
		{
			get { return lineSelected; }
			set { lineSelected = value; }
		}
	#endregion
	}


#region peakDetail class
	public class peakDetail
	{
		// private vars
		private double	m_sto;
		private int	m_BarsBack; 	// # of bars back
		private double	m_Close;	// closing price
		
		public peakDetail() {
		}

		public peakDetail(double sto) {
			m_sto = sto;
		}
		
        public double sto {
            get { return m_sto; }
            set { m_sto = value; }
        }

        public int BarsBack {
            get { return m_BarsBack; }
            set { m_BarsBack = value; }
        }

        public double Close {
            get { return m_Close; }
            set { m_Close = value; }
        }
	}	// end class peakDetail
#endregion
}


#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private NT7GbDivergence3[] cacheNT7GbDivergence3 = null;

        private static NT7GbDivergence3 checkNT7GbDivergence3 = new NT7GbDivergence3();

        /// <summary>
        /// Detects and alerts you to divergence between price and fast stochastics
        /// </summary>
        /// <returns></returns>
        public NT7GbDivergence3 NT7GbDivergence3(string lineSelected, int peakMargin, int periodD, int periodK, int periods)
        {
            return NT7GbDivergence3(Input, lineSelected, peakMargin, periodD, periodK, periods);
        }

        /// <summary>
        /// Detects and alerts you to divergence between price and fast stochastics
        /// </summary>
        /// <returns></returns>
        public NT7GbDivergence3 NT7GbDivergence3(Data.IDataSeries input, string lineSelected, int peakMargin, int periodD, int periodK, int periods)
        {
            if (cacheNT7GbDivergence3 != null)
                for (int idx = 0; idx < cacheNT7GbDivergence3.Length; idx++)
                    if (cacheNT7GbDivergence3[idx].LineSelected == lineSelected && cacheNT7GbDivergence3[idx].PeakMargin == peakMargin && cacheNT7GbDivergence3[idx].PeriodD == periodD && cacheNT7GbDivergence3[idx].PeriodK == periodK && cacheNT7GbDivergence3[idx].Periods == periods && cacheNT7GbDivergence3[idx].EqualsInput(input))
                        return cacheNT7GbDivergence3[idx];

            lock (checkNT7GbDivergence3)
            {
                checkNT7GbDivergence3.LineSelected = lineSelected;
                lineSelected = checkNT7GbDivergence3.LineSelected;
                checkNT7GbDivergence3.PeakMargin = peakMargin;
                peakMargin = checkNT7GbDivergence3.PeakMargin;
                checkNT7GbDivergence3.PeriodD = periodD;
                periodD = checkNT7GbDivergence3.PeriodD;
                checkNT7GbDivergence3.PeriodK = periodK;
                periodK = checkNT7GbDivergence3.PeriodK;
                checkNT7GbDivergence3.Periods = periods;
                periods = checkNT7GbDivergence3.Periods;

                if (cacheNT7GbDivergence3 != null)
                    for (int idx = 0; idx < cacheNT7GbDivergence3.Length; idx++)
                        if (cacheNT7GbDivergence3[idx].LineSelected == lineSelected && cacheNT7GbDivergence3[idx].PeakMargin == peakMargin && cacheNT7GbDivergence3[idx].PeriodD == periodD && cacheNT7GbDivergence3[idx].PeriodK == periodK && cacheNT7GbDivergence3[idx].Periods == periods && cacheNT7GbDivergence3[idx].EqualsInput(input))
                            return cacheNT7GbDivergence3[idx];

                NT7GbDivergence3 indicator = new NT7GbDivergence3();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.LineSelected = lineSelected;
                indicator.PeakMargin = peakMargin;
                indicator.PeriodD = periodD;
                indicator.PeriodK = periodK;
                indicator.Periods = periods;
                Indicators.Add(indicator);
                indicator.SetUp();

                NT7GbDivergence3[] tmp = new NT7GbDivergence3[cacheNT7GbDivergence3 == null ? 1 : cacheNT7GbDivergence3.Length + 1];
                if (cacheNT7GbDivergence3 != null)
                    cacheNT7GbDivergence3.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheNT7GbDivergence3 = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Detects and alerts you to divergence between price and fast stochastics
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.NT7GbDivergence3 NT7GbDivergence3(string lineSelected, int peakMargin, int periodD, int periodK, int periods)
        {
            return _indicator.NT7GbDivergence3(Input, lineSelected, peakMargin, periodD, periodK, periods);
        }

        /// <summary>
        /// Detects and alerts you to divergence between price and fast stochastics
        /// </summary>
        /// <returns></returns>
        public Indicator.NT7GbDivergence3 NT7GbDivergence3(Data.IDataSeries input, string lineSelected, int peakMargin, int periodD, int periodK, int periods)
        {
            return _indicator.NT7GbDivergence3(input, lineSelected, peakMargin, periodD, periodK, periods);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Detects and alerts you to divergence between price and fast stochastics
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.NT7GbDivergence3 NT7GbDivergence3(string lineSelected, int peakMargin, int periodD, int periodK, int periods)
        {
            return _indicator.NT7GbDivergence3(Input, lineSelected, peakMargin, periodD, periodK, periods);
        }

        /// <summary>
        /// Detects and alerts you to divergence between price and fast stochastics
        /// </summary>
        /// <returns></returns>
        public Indicator.NT7GbDivergence3 NT7GbDivergence3(Data.IDataSeries input, string lineSelected, int peakMargin, int periodD, int periodK, int periods)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.NT7GbDivergence3(input, lineSelected, peakMargin, periodD, periodK, periods);
        }
    }
}
#endregion
