/*

Modified: Oct 14 2010 
Modification: added logic to reset the Plots when the CalculateOnBarClose=false;
              previously the bars would not paint correctly.
              enjoy


Barry Taylor's Better Volume Indicator as described at
http://emini-watch.com/free-stuff/volume-indicator/
Coded for NinjaTrader by Alex Matulich, 2009-10-08, Unicorn Research Corporation.
This indicator optionally uses True Range rather than Range, it plots an exponential
moving average of the volume instead of simple moving average for efficiency,
and it optionally paints the price bars when BetterVolume2 indicates something
other than the normal color.

Watch for these at different stages of the market:

                  | Bottom | Start |Continue|  Top   |  Start  |Continue
                  |        |uptrend|uptrend |        |downtrend|downtrend
------------------+--------+-------+--------+--------+---------+---------
Volume Climax Up  |        |  Yes  |        |  Yes   |         |   Yes     red
Volume Climax Down|  Yes   |       |  Yes   |        |  Yes    |           white/black
High Volume Churn |  Yes   |       |        |  Yes   |         |           green/blue
Climax + Hi Churn |  Yes   |       |        |  Yes   |         |           magenta
Low Volume        |  Yes   |       |  Yes   |  Yes   |         |   Yes     yellow
-------------------------------------------------------------------------

This indicator works the same on daily or intraday bars. The original Tradestaion
code makes use of up volume and down volume data for intraday bars, and for daily
bars it calculates up volume and down volume from the relative positions of open,
high, low, and close. We use the latter method here, because NinjaTrader does not
provide up volume and down volume data.
*/
#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Better Volume indicator from emini-watch.com/free-stuff/volume-indicator/ using true range instead of range in calculations.
    /// </summary>
    [Description("Better Volume indicator from emini-watch.com/free-stuff/volume-indicator/ using true range instead of range in calculations.")]
    public class BetterVolume3 : Indicator
    {
        #region Variables
        // Wizard generated variables
            private int lookback = 20; // Default setting for Lookback
			private bool use2bars = true; // Default setting for Use2Bars
			private bool paintbars = true; // Default setting for Paintbars
			private bool useTrueRange = true; // Default setting for UseTrueRange
        // User defined variables (add any user defined variables below)
            private int barwidth = 2;
			private DataSeries[] value;
			private double vwt;
			private bool eod = true;
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
			int hilight=barwidth+1;
            Add(new Plot(new Pen(Color.FromKnownColor(KnownColor.Silver), barwidth), PlotStyle.Bar, "VBar"));
            Add(new Plot(new Pen(Color.FromKnownColor(KnownColor.Goldenrod), hilight), PlotStyle.Bar, "LowVol"));
            Add(new Plot(new Pen(Color.FromKnownColor(KnownColor.Red), hilight), PlotStyle.Bar, "ClimaxUp"));
            Add(new Plot(new Pen(Color.FromKnownColor(KnownColor.Black), hilight), PlotStyle.Bar, "ClimaxDown"));
            Add(new Plot(new Pen(Color.FromKnownColor(KnownColor.Green), hilight), PlotStyle.Bar, "HiChurn"));
            Add(new Plot(new Pen(Color.FromKnownColor(KnownColor.Magenta), hilight), PlotStyle.Bar, "ClimaxChurn"));
			Add(new Plot(new Pen(Color.FromKnownColor(KnownColor.Red), 1), PlotStyle.Line, "VolEMA"));

            CalculateOnBarClose	= false;
            Overlay				= false;
            PriceTypeSupported	= false;

			value = new DataSeries[23];
			for (int i = 1; i <= 22; ++i) value[i] = new DataSeries(this); // value1 to value22
			vwt = 2.0 / (5.0 * lookback + 1.0); // bar average period is 5*lookback
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
			double range, netvol;
			bool cond1, cond2, cond3, cond4, cond5, cond6, cond7, cond8, cond9, cond10,
				cond11=false, cond12=false, cond13=false, cond14=false, cond15=false, cond16=false, cond17=false, cond18=false, cond19=false, cond20=false,
				hiclose, loclose;
			// reset the current bars plots at the top of OnBarUpdate
			// NT7 will otherwise give a multi-colored bars since the
			//  values of the plots will be different depending on the volume 
			VBar.Reset();   		// vbar 0 			Oct 14 2010 
			LowVol.Reset();			// lowvolume 1		Oct 14 2010 
			ClimaxUp.Reset();		// climaxup 2		Oct 14 2010 
			ClimaxDown.Reset();		// climaxdown 3		Oct 14 2010 
			HiChurn.Reset();		// hi churn 4		Oct 14 2010 
			ClimaxChurn.Reset();	// climaxchurn 5	Oct 14 2010 
			
			
			if (CurrentBar <= 1) {
				range = High[0] - Low[0];
				VolEMA.Set(Volume[0]);
				eod = (Bars.Period.Id == PeriodType.Day || Bars.Period.Id == PeriodType.Week || Bars.Period.Id == PeriodType.Month || Bars.Period.Id == PeriodType.Year);
				return;
			}

			// calculate up volume and down volume

			range = useTrueRange ?
				Math.Max(High[0], Close[1]) - Math.Min(Low[0], Close[1]) // use true range
				: High[0] - Low[0]; // use normal range	
			if (range == 0.0)
				value[1][0] = value[2][0] = 0.5 * Volume[0];
			else {
				if (Close[0] > Open[0])
					value[1][0] = range / (2.0*range + Open[0]-Close[0]) * Volume[0];
				else if (Close[0] < Open[0])
					value[1][0] = (range+Close[0]-Open[0]) / (2.0*range + Close[0]-Open[0]) * Volume[0];
				else value[1][0] = 0.5 * Volume[0];
				value[2][0] = Volume[0] - value[1][0];
			}
			
			netvol = value[1][0] - value[2][0];
			value[3][0] = Volume[0]; //Math.Abs(value[1][0]+value[2][0]);
			value[4][0] = value[1][0]*range;
			value[5][0] = netvol*range;
			value[6][0] = value[2][0]*range;
			value[7][0] = -netvol*range;
			if (range != 0.0) {
				value[8][0] = value[1][0]/range;
				value[9][0] = netvol/range;
				value[10][0] = value[2][0]/range;
				value[11][0] = -netvol/range;
				value[12][0] = value[3][0]/range;
			}
			if (use2bars) {
				double hh = MAX(High, 2)[0], ll = MIN(Low, 2)[0];
				range = useTrueRange ? Math.Max(hh, Close[2]) - Math.Min(ll, Close[2]) // use true range
					: hh - ll; // use normal range
				value[13][0] = value[3][0]+value[3][1];
				value[14][0] = (value[1][0]+value[1][1])*range;
				netvol = value[1][0] + value[1][1] - value[2][0] - value[2][1];
				value[15][0] = netvol*range;
				value[16][0] = (value[2][0]+value[2][1])*range;
				value[17][0] = -netvol*range;
				if (range != 0.0) {
					value[18][0] = (value[1][0]+value[1][1])/range;
					value[19][0] = netvol/range;
					value[20][0] = (value[2][0]+value[2][1])/range;
					value[21][0] = -netvol/range;
					value[22][0] = value[13][0]/range;
				} else { // set to previous values
					for (int i = 18; i <= 22; ++i) value[i][0] = value[i][1];
				}
			}

			hiclose = (Close[0] > Open[0]);
			loclose = (Close[0] < Open[0]);
			cond1 = (value[3][0] == MIN(value[3],lookback)[0]);
			cond2 = (value[4][0] == MAX(value[4],lookback)[0] && hiclose);
			cond3 = (value[5][0] == MAX(value[5],lookback)[0] && hiclose);
			cond4 = (value[6][0] == MAX(value[6],lookback)[0] && loclose);
			cond5 = (value[7][0] == MAX(value[7],lookback)[0] && loclose);
			cond6 = (value[8][0] == MIN(value[8],lookback)[0] && loclose);
			cond7 = (value[9][0] == MIN(value[9],lookback)[0] && loclose);
			cond8 = (value[10][0] == MIN(value[10],lookback)[0] && hiclose);
			cond9 = (value[11][0] == MIN(value[11],lookback)[0] && hiclose);
			cond10 = (value[12][0] == MAX(value[12],lookback)[0]);
			if (use2bars) {
				hiclose = (Close[0] > Open[0] && Close[1] > Open[1]);
				loclose = (Close[0] < Open[0] && Close[1] < Open[1]);
				cond11 = (value[13][0] == MIN(value[13],lookback)[0]);
				cond12 = (value[14][0] == MAX(value[14],lookback)[0] && hiclose);
				cond13 = (value[15][0] == MAX(value[15],lookback)[0] && hiclose);
				cond14 = (value[16][0] == MAX(value[16],lookback)[0] && loclose);
				cond15 = (value[17][0] == MAX(value[17],lookback)[0] && loclose);
				cond16 = (value[18][0] == MIN(value[18],lookback)[0] && loclose);
				cond17 = (value[19][0] == MIN(value[19],lookback)[0] && loclose);
				cond18 = (value[20][0] == MIN(value[20],lookback)[0] && hiclose);
				cond19 = (value[21][0] == MIN(value[21],lookback)[0] && hiclose);
				cond20 = (value[22][0] == MAX(value[22],lookback)[0]);
			}

			// bc = bar color = index number of plot created in Initialize()

			bool sameday = eod ? true : (Time[0].Day == Time[1].Day);
			int bc = 0; // default 0 (normal volume bar)
			if (cond1 || (cond11 && sameday)) bc = 1; //LowVol
			if (cond2 || cond3 || cond8 || cond9 || ((cond12 || cond13 || cond18 || cond19) && sameday)) bc = 2; //ClimaxUp
			if (cond4 || cond5 || cond6 || cond7 || ((cond14 || cond15 || cond16 || cond17) && sameday)) bc = 3; //ClimaxDown
			if (cond10 || (cond20 && sameday)) bc = 4; //Churn
			if (bc == 4 && (cond2 || cond3 || cond4 || cond5 || cond6 || cond7 || cond8 || cond9 || (cond12 || cond13 || cond14 || cond15 || cond16 || cond17 || cond18 || cond19) && sameday))
				bc = 5; //ClimaxChurn

			// plot stuff
			Values[bc].Set(Volume[0]); //draw the volume bar we found
			if (paintbars && bc > 0) BarColor = Plots[bc].Pen.Color; // paint the price bar
			
			VolEMA.Set(vwt * (Volume[0] - VolEMA[1]) + VolEMA[1]); // plot average; faster than SMA
        }

        #region Properties
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries VBar
        {
            get { return Values[0]; }
        }

        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries LowVol
        {
            get { return Values[1]; }
        }

        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries ClimaxUp
        {
            get { return Values[2]; }
        }

        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries ClimaxDown
        {
            get { return Values[3]; }
        }

        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries HiChurn
        {
            get { return Values[4]; }
        }

        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries ClimaxChurn
        {
            get { return Values[5]; }
        }
		
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries VolEMA
        {
            get { return Values[6]; }
        }

        [Description("Volume bars to look back for min/max values")]
        [Category("Parameters")]
        public int Lookback
        {
            get { return lookback; }
            set { lookback = Math.Max(2, value); }
        }

		[Description("True=use last 2 bars for calculations, false=use current bar")]
        [Category("Parameters")]
        public bool Use2bars
        {
            get { return use2bars; }
            set { use2bars = value; }
        }

		[Description("True=show paintbars, false=leave price bars alone.")]
        [Category("Parameters")]
        public bool Paintbars
        {
            get { return paintbars; }
            set { paintbars = value; }
        }

		[Description("True=use true range in calculations, false=use High-Low range.")]
        [Category("Parameters")]
        public bool UseTrueRange
        {
            get { return useTrueRange; }
            set { useTrueRange = value; }
        }
        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private BetterVolume3[] cacheBetterVolume3 = null;

        private static BetterVolume3 checkBetterVolume3 = new BetterVolume3();

        /// <summary>
        /// Better Volume indicator from emini-watch.com/free-stuff/volume-indicator/ using true range instead of range in calculations.
        /// </summary>
        /// <returns></returns>
        public BetterVolume3 BetterVolume3(int lookback, bool paintbars, bool use2bars, bool useTrueRange)
        {
            return BetterVolume3(Input, lookback, paintbars, use2bars, useTrueRange);
        }

        /// <summary>
        /// Better Volume indicator from emini-watch.com/free-stuff/volume-indicator/ using true range instead of range in calculations.
        /// </summary>
        /// <returns></returns>
        public BetterVolume3 BetterVolume3(Data.IDataSeries input, int lookback, bool paintbars, bool use2bars, bool useTrueRange)
        {
            if (cacheBetterVolume3 != null)
                for (int idx = 0; idx < cacheBetterVolume3.Length; idx++)
                    if (cacheBetterVolume3[idx].Lookback == lookback && cacheBetterVolume3[idx].Paintbars == paintbars && cacheBetterVolume3[idx].Use2bars == use2bars && cacheBetterVolume3[idx].UseTrueRange == useTrueRange && cacheBetterVolume3[idx].EqualsInput(input))
                        return cacheBetterVolume3[idx];

            lock (checkBetterVolume3)
            {
                checkBetterVolume3.Lookback = lookback;
                lookback = checkBetterVolume3.Lookback;
                checkBetterVolume3.Paintbars = paintbars;
                paintbars = checkBetterVolume3.Paintbars;
                checkBetterVolume3.Use2bars = use2bars;
                use2bars = checkBetterVolume3.Use2bars;
                checkBetterVolume3.UseTrueRange = useTrueRange;
                useTrueRange = checkBetterVolume3.UseTrueRange;

                if (cacheBetterVolume3 != null)
                    for (int idx = 0; idx < cacheBetterVolume3.Length; idx++)
                        if (cacheBetterVolume3[idx].Lookback == lookback && cacheBetterVolume3[idx].Paintbars == paintbars && cacheBetterVolume3[idx].Use2bars == use2bars && cacheBetterVolume3[idx].UseTrueRange == useTrueRange && cacheBetterVolume3[idx].EqualsInput(input))
                            return cacheBetterVolume3[idx];

                BetterVolume3 indicator = new BetterVolume3();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.Lookback = lookback;
                indicator.Paintbars = paintbars;
                indicator.Use2bars = use2bars;
                indicator.UseTrueRange = useTrueRange;
                Indicators.Add(indicator);
                indicator.SetUp();

                BetterVolume3[] tmp = new BetterVolume3[cacheBetterVolume3 == null ? 1 : cacheBetterVolume3.Length + 1];
                if (cacheBetterVolume3 != null)
                    cacheBetterVolume3.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheBetterVolume3 = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Better Volume indicator from emini-watch.com/free-stuff/volume-indicator/ using true range instead of range in calculations.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.BetterVolume3 BetterVolume3(int lookback, bool paintbars, bool use2bars, bool useTrueRange)
        {
            return _indicator.BetterVolume3(Input, lookback, paintbars, use2bars, useTrueRange);
        }

        /// <summary>
        /// Better Volume indicator from emini-watch.com/free-stuff/volume-indicator/ using true range instead of range in calculations.
        /// </summary>
        /// <returns></returns>
        public Indicator.BetterVolume3 BetterVolume3(Data.IDataSeries input, int lookback, bool paintbars, bool use2bars, bool useTrueRange)
        {
            return _indicator.BetterVolume3(input, lookback, paintbars, use2bars, useTrueRange);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Better Volume indicator from emini-watch.com/free-stuff/volume-indicator/ using true range instead of range in calculations.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.BetterVolume3 BetterVolume3(int lookback, bool paintbars, bool use2bars, bool useTrueRange)
        {
            return _indicator.BetterVolume3(Input, lookback, paintbars, use2bars, useTrueRange);
        }

        /// <summary>
        /// Better Volume indicator from emini-watch.com/free-stuff/volume-indicator/ using true range instead of range in calculations.
        /// </summary>
        /// <returns></returns>
        public Indicator.BetterVolume3 BetterVolume3(Data.IDataSeries input, int lookback, bool paintbars, bool use2bars, bool useTrueRange)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.BetterVolume3(input, lookback, paintbars, use2bars, useTrueRange);
        }
    }
}
#endregion
