#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
using System.IO;
using System.Globalization;
using System.Collections.Generic;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
    /// Cumulative Delta
    /// </summary>
	/// 
	/// 
	public  class DeltaCalculator
	{
		private double lastprice=0;
		private int lastdirection=0;
		private bool startlookingforreversal=false;
		
		
		public int CalcDelta(TickTypeEnum tickType,double price,int volume,GomCDCalculationModeType calcmode,GomBackupModeType backupmode,int filtersize,GomFilterModeType filtermode )
			
		{
			int delta=0;
			int direction=lastdirection;
						
			
			if ((calcmode==GomCDCalculationModeType.BidAsk) && (tickType!=TickTypeEnum.Unknown))
			{	
				if ((tickType==TickTypeEnum.BelowBid)||(tickType==TickTypeEnum.AtBid))
					delta =-volume;
				else if ((tickType==TickTypeEnum.AboveAsk)||(tickType==TickTypeEnum.AtAsk))
					delta=volume;
			}
			else if ((calcmode==GomCDCalculationModeType.UpDownTick) || ((calcmode==GomCDCalculationModeType.BidAsk) && (backupmode==GomBackupModeType.UseUpDownTick)))
			{
				if (lastprice != 0)
				{
					if (price>lastprice) delta=volume;
					if (price<lastprice) delta=-volume;
				}
			}
			else if ((calcmode==GomCDCalculationModeType.UpDownTickWithContinuation) || ((calcmode==GomCDCalculationModeType.BidAsk) && (backupmode==GomBackupModeType.UseWithContinuation)))
			{	
				if(price > lastprice)  //normal uptick/dn tick
					direction = 1;
				else if(price < lastprice)
					direction = -1;
				
				delta=direction*volume;
			}
			
			// added	
			
			else if ((calcmode==GomCDCalculationModeType.Hybrid) )
			{	
				
				if(price > lastprice)  //normal uptick/dn tick
				{ 
					direction = 1;
					//price changed, we reinit the startlookingforreversal bool.
				    startlookingforreversal=false;	
				}
				else if(price < lastprice)
				{
					direction = -1;
				    startlookingforreversal=false;	
				}
			
				
				if (!startlookingforreversal)
					if (direction==1)
						//if going up, we want to be hitting bid to be able to start to spot reversals (hitting the ask)
						startlookingforreversal=(tickType==TickTypeEnum.AtBid)||(tickType==TickTypeEnum.BelowBid);	
					else 
						startlookingforreversal=(tickType==TickTypeEnum.AtAsk)||(tickType==TickTypeEnum.AboveAsk);	

				//what happens when price is same
				if (price==lastprice)
				{	
					//if going up, and we have already hit the bid (startlookingforreversal is true) at a price level, 
					// and start hitting the ask, let's reverse
					
					if ((direction==1)&& startlookingforreversal && ((tickType==TickTypeEnum.AtAsk)||(tickType==TickTypeEnum.BetweenBidAsk)))		
						direction =-1;
					
					else if ((direction==-1)&&startlookingforreversal&&((tickType==TickTypeEnum.AtBid) ||(tickType==TickTypeEnum.BetweenBidAsk)))
						direction = 1;	//buyers take control of ask
				}
				
				
				delta=direction*volume;				
			
			}
			
			lastprice=price;
			lastdirection=direction;
		
			if ((filtermode==GomFilterModeType.OnlyLargerThan) && (volume<filtersize))
				delta=0;
			
			if ((filtermode==GomFilterModeType.OnlySmallerThan) && (volume>filtersize))
				delta=0;
			
			return delta;
			
		}
			
	
		
	}
	
    [Description("Cumulative Delta")]
    public class GomCD : GomRecorderIndicator
           {
        #region Variables
        // Wizard generated variables
        // User defined variables (add any user defined variables below)

		private GomPaintType PtType=GomPaintType.UpDown;
		private GomCDChartType chart = GomCDChartType.CumulativeChart;
		private GomCDCalculationModeType calcMode = GomCDCalculationModeType.BidAsk;
		private GomBackupModeType backupMode=GomBackupModeType.DoNothing;
		private GomFilterModeType  filterMode=GomFilterModeType.None;
		private bool ShowOutline=false;
		private bool EnhanceHL =true;

			
		private bool ReinitSession=false;
		private bool ForceHiLo=true;
			
		private int ForcedHiLoBS=2;
		
		protected int totalvolume=0,hi,lo;
			
		private double ask=0.0,bid=0.0;
		
		protected DataSeries dsOpen,dsHigh,dsLow,dsClose;
				
		protected int startbar=-1;
		protected int lastcalcbar=-1;
		
		private  DeltaCalculator deltaCalc=new DeltaCalculator();

			
		/// added
		
		private int filterSize=1;
			
		protected bool useplot=true;
		
		private bool initdone=false;
						
        #endregion

		
        protected override void GomInitialize()
        {
			
            CalculateOnBarClose	= false;
            Overlay				= false;
            PriceTypeSupported	= false;
			
				if (Instrument != null)
				{
					#if !NT7
						dsOpen 	= new DataSeries(this);
						dsHigh 	= new DataSeries(this);
						dsLow 	= new DataSeries(this);
						dsClose = new DataSeries(this);
					#else
						dsOpen 	= new DataSeries(this,MaximumBarsLookBack.Infinite);
						dsHigh 	= new DataSeries(this,MaximumBarsLookBack.Infinite);
						dsLow 	= new DataSeries(this,MaximumBarsLookBack.Infinite);
						dsClose = new DataSeries(this,MaximumBarsLookBack.Infinite);
					#endif
				}

			
        }

		//translation for non cumulative and session break
		
		protected void Reinit(int previoustotalvolume)
		{
			totalvolume -= previoustotalvolume;
			hi -= previoustotalvolume;
			lo -= previoustotalvolume;
		}
			
	

			
		protected override void GomOnBarUpdate()
		{			
			if ((FirstTickOfBar))   
			{									
				if ((chart==GomCDChartType.NonCumulativeChart)|| (ReinitSession && Bars.SessionBreak))
				{
					totalvolume=0;
				}

				dsOpen.Set(totalvolume);
				dsHigh.Set(totalvolume);
				dsLow.Set(totalvolume);
				dsClose.Set(totalvolume);
				hi=totalvolume;
				lo=totalvolume;
			}
			
		}
			
				
		public int CalcDelta(TickTypeEnum tickType,double price,int volume)	
		{
			return deltaCalc.CalcDelta(tickType,price,volume,calcMode,backupMode,filterSize,filterMode);
		}
		
		
		//void ProcessDelta(int delta, bool firstTick)
		protected override void GomOnMarketData(TickTypeEnum tickType,double price,int volume,bool firstTickOfBar)	
		{									
			int delta=CalcDelta(tickType,price,volume);

			totalvolume += delta;

			hi=Math.Max(hi,totalvolume);
			lo=Math.Min(lo,totalvolume);
								
			dsClose.Set(totalvolume);
			dsHigh.Set(hi);
			dsLow.Set(lo);
	
			if (startbar==-1)
				startbar=CurrentBar;

			lastcalcbar=CurrentBar;
			DeltaValue.Set(dsClose.Get(CurrentBar));
		
		}

		public override void GetMinMaxValues(ChartControl chartControl, ref double min, ref double max)
		{	
			if (Bars==null)	return;
			
				if (useplot)
				{
					#if NT7
						int lastBar		= Math.Min(this.LastBarIndexPainted, Bars.Count - 1);
						int firstBar	= this.FirstBarIndexPainted;
					#else
						int lastBar		= Math.Min(ChartControl.LastBarPainted, Bars.Count - 1);
						int firstBar	= Math.Max((lastBar - ChartControl.BarsPainted) + 1,0);
					#endif			
					min=Double.MaxValue;
					max=Double.MinValue;
														
					for(int index=firstBar;index<=lastBar;index++)
					{		
						if ((index<=lastcalcbar) && (index >=Math.Max(1,startbar)))
						{   
							min=Math.Min(min,dsLow.Get(index));
							max=Math.Max(max,dsHigh.Get(index));
						}
					}
				}
				
			else
				base.GetMinMaxValues(chartControl, ref  min, ref  max);
	
		}
		
	
		public override void Plot(Graphics graphics, Rectangle bounds, double min, double max)
		{
			if (Bars==null)	return;

			if (useplot)
			{
			#if NT7
				int lastBar		= Math.Min(this.LastBarIndexPainted, Bars.Count - 1);
				int firstBar	= this.FirstBarIndexPainted;
			#else
				int lastBar		= Math.Min(ChartControl.LastBarPainted, Bars.Count - 1);
				int firstBar	= Math.Max((lastBar - ChartControl.BarsPainted) + 1,0);
			#endif
								
			int barPaintWidth =  ChartControl.ChartStyle.GetBarPaintWidth(ChartControl.BarWidth);
			int barwidth=ChartControl.BarWidth;
	
			#if !NT7
				double scalingfactor= (double)bounds.Height/ChartControl.MaxMinusMin(max, min);
			#endif				
			//Color ColorNeutral=Color.FromArgb(255,255-ChartControl.BackColor.R,255-ChartControl.BackColor.G,255-ChartControl.BackColor.B);
			
			#if NT7
				Pen outlinePen=ChartControl.ChartStyle.Pen;			
			#else				
				Pen outlinePen=ChartControl.BarOutlinePen;
			#endif

			Color ColorNeutral=outlinePen.Color;

			Pen HiLoPen= (Pen)outlinePen.Clone();

			if (EnhanceHL)
				HiLoPen.Width *=2;
				
			SolidBrush drawBrush=new SolidBrush(Color.Transparent);
			SolidBrush neutralBrush=new SolidBrush(ColorNeutral);

			int penSize;		
			if (ChartControl.ChartStyle.ChartStyleType==ChartStyleType.OHLC)
				penSize=Math.Max(0,barwidth-2);
			else if (ChartControl.ChartStyle.ChartStyleType==ChartStyleType.HiLoBars)
				penSize=barwidth;
			else if (chart==GomCDChartType.NonCumulativeChart && ForceHiLo)
				penSize=ForcedHiLoBS;
			else
				penSize=1;
			
			Pen drawPen=new Pen(Color.Transparent,penSize);
			
			int x, yHigh, yClose, yOpen, yLow;	
			int direction;			
			
			//zero line
			#if NT7
				int y0=ChartControl.GetYByValue(this,0.0);
			#else
				int y0=(bounds.Y + bounds.Height) + Convert.ToInt32(min * scalingfactor);
			#endif
			
			
			if ((chart==GomCDChartType.NonCumulativeChart)||ReinitSession)
				graphics.DrawLine(new Pen(Color.Blue),bounds.X,y0,bounds.X+bounds.Width,y0);
			
			//whow recording mode
			graphics.DrawString(recordingMessage, ChartControl.Font, new SolidBrush(ColorNeutral), bounds.Left, bounds.Bottom-22,new StringFormat());
			
            for(int index=firstBar;index<=lastBar;index++)
            {

				direction=0;
				
				if ((index<=lastcalcbar) && (index >=Math.Max(1,startbar)))
				
				{	
					#if NT7 
						x=ChartControl.GetXByBarIdx(BarsArray[0],index);
						yHigh = ChartControl.GetYByValue(this,dsHigh.Get(index));
						yClose = ChartControl.GetYByValue(this,dsClose.Get(index));
						yOpen = ChartControl.GetYByValue(this,dsOpen.Get(index)) ;
                    	yLow = ChartControl.GetYByValue(this,dsLow.Get(index)) ;

					#else
						x=ChartControl.GetXByBarIdx(index);
						yHigh = (bounds.Y + bounds.Height) - Convert.ToInt32((dsHigh.Get(index) - min) * scalingfactor);
						yClose = (bounds.Y + bounds.Height) - Convert.ToInt32((dsClose.Get(index) - min)  * scalingfactor);
						yOpen = (bounds.Y + bounds.Height) - Convert.ToInt32((dsOpen.Get(index) - min)  * scalingfactor) ;
                    	yLow = (bounds.Y + bounds.Height) - Convert.ToInt32((dsLow.Get(index) - min) * scalingfactor) ;
					#endif
										
					if (PtType==GomPaintType.StrongUpDown)							
					{
						if ( dsClose.Get(index)<dsLow.Get(index-1) )
							direction=-1;
						else if ( dsClose.Get(index)>dsHigh.Get(index-1)  )
							direction=1;
					}
					else if (PtType==GomPaintType.UpDown)	
					{
						if ( dsClose.Get(index)<dsOpen.Get(index) )
							direction=-1;
						else if ( dsClose.Get(index)>dsOpen.Get(index)  )
							direction=1;
					}						
						
					
					if (direction==1)
					{
						drawPen.Color=ChartControl.UpColor;
					}
					else if (direction==-1)
					{
						drawPen.Color=ChartControl.DownColor;
					}
					else
					{
						drawPen.Color=ColorNeutral;
					}
							
					drawBrush.Color=drawPen.Color;	
										

					if ((ChartControl.ChartStyle.ChartStyleType==ChartStyleType.HiLoBars)|| (chart==GomCDChartType.NonCumulativeChart && ForceHiLo))
					{	
						graphics.DrawLine(drawPen,x,yHigh,x,yLow);
					}
					
					else if (ChartControl.ChartStyle.ChartStyleType==ChartStyleType.CandleStick)
					{						
						graphics.DrawLine(HiLoPen,x,yLow,x,yHigh);

						if (yClose==yOpen)
							graphics.DrawLine(outlinePen,x-barwidth-outlinePen.Width,yClose,x+barwidth+outlinePen.Width,yClose);
						
						else
						{	
							
							graphics.FillRectangle(drawBrush,x-barwidth-outlinePen.Width,Math.Min(yClose,yOpen)+1,2*(barwidth+outlinePen.Width)+1,Math.Abs(yClose-yOpen)-1);
	
							if (ShowOutline)
							{	
							//	graphics.FillRectangle(neutralBrush,x-barwidth-outlinepenwidth,Math.Min(yClose,yOpen)+1,2*(barwidth+outlinepenwidth)+1,Math.Abs(yClose-yOpen)-1);
							//	graphics.FillRectangle(drawBrush,x-barwidth,Math.Min(yClose,yOpen)+1,2*barwidth+1,Math.Abs(yClose-yOpen)-1);

							graphics.DrawRectangle(outlinePen,x-barwidth-outlinePen.Width/2,Math.Min(yClose,yOpen),2*(barwidth)+outlinePen.Width+1,Math.Abs(yClose-yOpen));
							}
							
						}
					}
						
					else	
					{
						graphics.DrawLine(drawPen,x,yLow+penSize/2,x,yHigh-penSize/2);
						graphics.DrawLine(drawPen,x,yClose,x+barwidth,yClose);							
						graphics.DrawLine(drawPen,x-barwidth,yOpen,x,yOpen);
					}
				}	
			}	
			}
			else
				base.Plot(graphics,bounds,min, max);
			
		}
		
	#if	NT7
		public override void Dispose()
		{
			if (dsClose != null)
				dsClose.Dispose();
			
			if (dsOpen != null)
				dsOpen.Dispose();
			
			if (dsHigh != null)
				dsHigh.Dispose();
					
			if (dsLow != null)
				dsLow.Dispose();
			
			base.Dispose();
		}
	#endif
		
				
        #region Properties
		
		[Description("Type of delta calculation - Cumulative or Non")]
        [Category("Parameters")]
		[Gui.Design.DisplayName("Delta Calculation")]
        public GomCDChartType Chart
        {
            get { return chart; }
            set { chart = value; }
        }
		[Description("Paint Type")]
        [Category("Settings")]
		[Gui.Design.DisplayNameAttribute("Paint Type")]
        public GomPaintType ptType 
        {
            get { return PtType; }
            set { PtType = value; }
        }
		
		[Description("CandleSticks : Show Outline")]
        [Category("Settings")]
		[Gui.Design.DisplayNameAttribute("CandleSticks : Show Outline")]
        public bool showOutline 
        {
            get { return ShowOutline; }
            set { ShowOutline = value; }
        }
		
		[Description("CandleSticks : Enhance HiLo Bar")]
        [Category("Settings")]
		[Gui.Design.DisplayNameAttribute("CandleSticks : Enhance HiLo Bar")]
        public bool enhanceHL 
        {
            get { return EnhanceHL; }
            set { EnhanceHL = value; }
        }
		
		
		[Description("Force HiLo on noncumulative chart")]
        [Category("Settings")]
		[Gui.Design.DisplayNameAttribute("Force HiLo for Non cumulative")]
        public bool forceHiLo 
        {
            get { return ForceHiLo; }
            set { ForceHiLo = value; }
        }
		
		[Description("Size of HiLo bars when force hilo mode is used")]
        [Category("Settings")]
		[Gui.Design.DisplayNameAttribute("Forced HiLo Bar Size")]
        public int forcedHiLoBS 
        {
            get { return ForcedHiLoBS; }
            set { ForcedHiLoBS = value; }
        }
		
		[Description("Reinit on session break")]
        [Category("Parameters")]
		[Gui.Design.DisplayNameAttribute("Reinit on session break")]
        public bool reinitSession 
        {
            get { return ReinitSession; }
            set { ReinitSession = value; }
        }
		
		[Description("UpDownTick : volume is up if price>lastprice, down if price<lastprice.\nUpDownTickWithContinuation : volume is up if price>lastprice or\nprice=lastprice and last direction was up, same for downside")]
        [Category("Parameters")]
		[Gui.Design.DisplayNameAttribute("Calculation Mode")]
        public GomCDCalculationModeType CalcMode 
        {
            get { return calcMode; }
            set { calcMode = value; }
        }

		[Description("Volume Filter")]
        [Category("Parameters")]
		[Gui.Design.DisplayNameAttribute("Volume Filter Size")]
        public int FilterSize 
        {
            get { return filterSize; }
            set { filterSize = value; }
		}
		
		[Description("Bid/Ask Backup Mode : what to do if we find no Bid/Ask in file but tick data")]
        [Category("Settings")]
		[Gui.Design.DisplayNameAttribute("Bid/Ask Backup mode")]
        public GomBackupModeType BackupMode 
        {
            get { return backupMode; }
            set { backupMode = value; }
		}
		
		[Description("Filter Mode")]
        [Category("Parameters")]
		[Gui.Design.DisplayNameAttribute("Size Filter Mode")]
        public GomFilterModeType FilterMode 
        {
            get { return filterMode; }
            set { filterMode = value; }
		}
		
		[Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries DeltaValue
        {
            get { return Values[0]; }
        }

		
	
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries DeltaClose
		{
			get 
			{ 
				return dsClose; 
			}
		}
		
				/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries DeltaOpen
		{
			get 
			{ 
				return dsOpen; 
			}
		}
		
				/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries DeltaHigh
		{
			get 
			{ 
				return dsHigh; 
			}
		}
		
				/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries DeltaLow
		{
			get 
			{ 
				return dsLow; 
			}
		}
				
		[Browsable(false)]
        [Category("Parameters")]
		 public GomFileFormat FileFormat
		{
			get { return recFormat; }
			set {  recFormat=value;}
		}

		
        #endregion
    }
}

public enum GomCDCalculationModeType
	{
		BidAsk,
		UpDownTick,
		UpDownTickWithContinuation,
		Hybrid
	}
		
public enum GomPaintType
	{
		None,
		UpDown,
		StrongUpDown
	}
		
public enum GomCDChartType
	{
		CumulativeChart,
		NonCumulativeChart,
	}
	
public enum GomBackupModeType
	{
		DoNothing,
		UseUpDownTick,
		UseWithContinuation
	}

public enum GomFilterModeType
	{
		OnlyLargerThan,
		OnlySmallerThan,
		None
	}
	

		

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private GomCD[] cacheGomCD = null;

        private static GomCD checkGomCD = new GomCD();

        /// <summary>
        /// Cumulative Delta
        /// </summary>
        /// <returns></returns>
        public GomCD GomCD(GomCDCalculationModeType calcMode, GomCDChartType chart, GomFileFormat fileFormat, GomFilterModeType filterMode, int filterSize, bool reinitSession)
        {
            return GomCD(Input, calcMode, chart, fileFormat, filterMode, filterSize, reinitSession);
        }

        /// <summary>
        /// Cumulative Delta
        /// </summary>
        /// <returns></returns>
        public GomCD GomCD(Data.IDataSeries input, GomCDCalculationModeType calcMode, GomCDChartType chart, GomFileFormat fileFormat, GomFilterModeType filterMode, int filterSize, bool reinitSession)
        {
            if (cacheGomCD != null)
                for (int idx = 0; idx < cacheGomCD.Length; idx++)
                    if (cacheGomCD[idx].CalcMode == calcMode && cacheGomCD[idx].Chart == chart && cacheGomCD[idx].FileFormat == fileFormat && cacheGomCD[idx].FilterMode == filterMode && cacheGomCD[idx].FilterSize == filterSize && cacheGomCD[idx].reinitSession == reinitSession && cacheGomCD[idx].EqualsInput(input))
                        return cacheGomCD[idx];

            lock (checkGomCD)
            {
                checkGomCD.CalcMode = calcMode;
                calcMode = checkGomCD.CalcMode;
                checkGomCD.Chart = chart;
                chart = checkGomCD.Chart;
                checkGomCD.FileFormat = fileFormat;
                fileFormat = checkGomCD.FileFormat;
                checkGomCD.FilterMode = filterMode;
                filterMode = checkGomCD.FilterMode;
                checkGomCD.FilterSize = filterSize;
                filterSize = checkGomCD.FilterSize;
                checkGomCD.reinitSession = reinitSession;
                reinitSession = checkGomCD.reinitSession;

                if (cacheGomCD != null)
                    for (int idx = 0; idx < cacheGomCD.Length; idx++)
                        if (cacheGomCD[idx].CalcMode == calcMode && cacheGomCD[idx].Chart == chart && cacheGomCD[idx].FileFormat == fileFormat && cacheGomCD[idx].FilterMode == filterMode && cacheGomCD[idx].FilterSize == filterSize && cacheGomCD[idx].reinitSession == reinitSession && cacheGomCD[idx].EqualsInput(input))
                            return cacheGomCD[idx];

                GomCD indicator = new GomCD();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.CalcMode = calcMode;
                indicator.Chart = chart;
                indicator.FileFormat = fileFormat;
                indicator.FilterMode = filterMode;
                indicator.FilterSize = filterSize;
                indicator.reinitSession = reinitSession;
                Indicators.Add(indicator);
                indicator.SetUp();

                GomCD[] tmp = new GomCD[cacheGomCD == null ? 1 : cacheGomCD.Length + 1];
                if (cacheGomCD != null)
                    cacheGomCD.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheGomCD = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Cumulative Delta
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.GomCD GomCD(GomCDCalculationModeType calcMode, GomCDChartType chart, GomFileFormat fileFormat, GomFilterModeType filterMode, int filterSize, bool reinitSession)
        {
            return _indicator.GomCD(Input, calcMode, chart, fileFormat, filterMode, filterSize, reinitSession);
        }

        /// <summary>
        /// Cumulative Delta
        /// </summary>
        /// <returns></returns>
        public Indicator.GomCD GomCD(Data.IDataSeries input, GomCDCalculationModeType calcMode, GomCDChartType chart, GomFileFormat fileFormat, GomFilterModeType filterMode, int filterSize, bool reinitSession)
        {
            return _indicator.GomCD(input, calcMode, chart, fileFormat, filterMode, filterSize, reinitSession);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Cumulative Delta
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.GomCD GomCD(GomCDCalculationModeType calcMode, GomCDChartType chart, GomFileFormat fileFormat, GomFilterModeType filterMode, int filterSize, bool reinitSession)
        {
            return _indicator.GomCD(Input, calcMode, chart, fileFormat, filterMode, filterSize, reinitSession);
        }

        /// <summary>
        /// Cumulative Delta
        /// </summary>
        /// <returns></returns>
        public Indicator.GomCD GomCD(Data.IDataSeries input, GomCDCalculationModeType calcMode, GomCDChartType chart, GomFileFormat fileFormat, GomFilterModeType filterMode, int filterSize, bool reinitSession)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.GomCD(input, calcMode, chart, fileFormat, filterMode, filterSize, reinitSession);
        }
    }
}
#endregion
