// SMI Candle Paintbar
// Modified by Prospectus (readtheprospectus@yahoo.com)
// This indicator paints the candles to match the same colors as the SMI.
// Also contains an audio alert and signal arrows.
// 
// Author: Mathew Wyatt (mathew.wyatt at gmail.com)
// This indicator is the Stochastic Momentum Index by William Blau. Formula for the indicator was found - http://www.purebytes.com/archives/metastock/1998/msg06283.html.
//
// And modified by TheWizard and renamed SMI5 - added alert lines (+20 / -20) to indicate possible price direction change area / chop area
// rearranged the plots so that it plots more like the Eco2 indicator (a little easier to read than the prior plots)
// changed the way the candlesticks draw so that they are "hollow" instead of filled in
// added background coloring to indicate uptrend or downtrend
// added the ability to draw a vertical line on the price panel indicating uptrend or downtrend
// added the message "Possible Chop" when dots enter the +20 / -20 zone
// added the message "Long Breakout" when the dots exit up from the +20 zone
// added the message "Short Breakout" when the dots exit down from the -20 zone
//
// Cory Nguyen contributed code to add a "chop zone" +20 / -20 area on the indicator panel highlighting area of possible chop activity - March 7, 2010
//
// eDanny contributed / modified code April 2010 to clean up existing code and to remove errant vertical lines & arrows & also stabilize the background coloring.

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// The Stochastic Momentum Index is made up of two lines that oscillate between a vertical scale of -100 to 100.
	/// </summary>
	[Description("The Stochastic Momentum Index is made up of two lines that oscillate between a vertical scale of -100 to 100.")]
	public class SMI5: Indicator
	{
		#region Variables
		private int				range	= 10;
		private int				emaperiod1	= 17;
		private int				emaperiod2	= 1;
		private int 			smiemaperiod = 2;
		private int				opacity = 75;  // ADDED BY TheWIZARD 10/21/09
		
		private DataSeries		sms;
		private DataSeries		hls;
		private DataSeries 		smis;
		private DataSeries 		esmis;
		
		private DataSeries 		line20;  // Added by TheWizard March 7, 2010 with CODE CONTRIBUTED BY CORY NGUYEN
		
		private bool 			activatealertsound = true;
		private bool 			showarrows = false;
		private double  		offset = 1;
		private bool 			colorbackground = true;  // added by TheWizard 02/06/10
		private bool			colorALLbackgrounds = false; // added by TheWizard April 11, 2010
		
		//For Alerts
		private double 			chopzone = 20;    // Added by TheWizard March 7, 2010 with CODE CONTRIBUTED BY CORY NG
		private double			alertlevel = 40;
		private double 			overValue = 80;  // ADDED BY TheWizard 10/21/09

		private string 			longwavfilename = "long.wav";
		private string 			shortwavfilename="short.wav";
		private bool 			writealerttext = false;
		private int 			BarAtLastCross = 0;
		private bool 			newbarsound=false;  // Added by TheWizard 02/06/10
		private string 			newbarwavfilename = "AutoTrail.wav";  // Added by TheWizard 02/06/10

		private double 			chopfilter = 0.0;
		
		private bool 			colorpricebars = true;
		private 				Color barColorUp = Color.DodgerBlue;
		private 				Color barColorDown = Color.Red;
		private 				Color barColorNeutral = Color.Olive;
		private 				Color backColorUp = Color.RoyalBlue;  // added by TheWizard 02/06/10
		private 				Color backColorDn = Color.DeepPink;  // added by TheWizard 02/06/10
		private 				Color czColor = Color.DimGray; // added by TheWizard March 13, 2010
		
		private bool 			drawVerticalLineOnPricePanel = false; // Added by TheWizard December 22, 2009
		private 				Color vlineUpColor = Color.DodgerBlue; // added by TheWizard December 23, 2009
		private 				Color vlineDnColor = Color.Red; // added by TheWizard December 23, 2009
		private int 			vlineWidth = 1; // added by TheWizard December 23, 2009
		//private bool 			cValue = false;// <-- re-enable if needed for something else
		
		// Text Box
		private bool			showText = true;
		private bool			textOnLeft = true;
		private bool			textOnTop = true;
		private 				TextPosition bPosition = TextPosition.BottomLeft;
		private 				TextPosition tPosition = TextPosition.TopLeft;
		private					Font textFontMed = new Font("Arial", 14, FontStyle.Bold);
		
		string					textMessage;
		
		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Line(Color.White, 0, "07. Zero line"));
			Add(new Line(Color.Red, overValue, "05. Over bought"));  // ADDED BY TheWizard 10/21/09
			Add(new Line(Color.DodgerBlue, -overValue, "06. Over sold"));  // ADDED BY TheWizard 10/21/09
			Add(new Line(Color.Red, alertlevel, "03. Alert line"));
			Add(new Line(Color.DodgerBlue, -alertlevel, "04. Alert line"));
			Add(new Line(Color.White, chopzone, "01. Chop Zone Upper"));    // Added by TheWizard March 7, 2010 with CODE CONTRIBUTED BY CORY NG
			Add(new Line(Color.White, -chopzone, "02. Chop Zone Lower"));    // Added by TheWizard March 7, 2010 with CODE CONTRIBUTED BY CORY NG
			 
			Lines[0].Pen.DashStyle = DashStyle.Solid;
			Lines[0].Pen.Width = 1;
			Lines[1].Pen.DashStyle = DashStyle.DashDotDot;
			Lines[1].Pen.Width = 3;
			Lines[2].Pen.DashStyle = DashStyle.DashDotDot;
			Lines[2].Pen.Width = 3;
			Lines[3].Pen.DashStyle = DashStyle.Dash;
			Lines[3].Pen.Width = 1;
			Lines[4].Pen.DashStyle = DashStyle.Dash;
			Lines[4].Pen.Width = 1;
			Lines[5].Pen.DashStyle = DashStyle.Dash;
			Lines[5].Pen.Width = 1;
			Lines[6].Pen.DashStyle = DashStyle.Dash;
			Lines[6].Pen.Width = 1;

			Add(new Plot(Color.Blue, PlotStyle.Line, "SMI"));
			Add(new Plot(Color.Gold, PlotStyle.Line, "SMIEMA"));
			Add(new Plot(Color.FromKnownColor(KnownColor.Red), PlotStyle.Dot, "SMIdn"));
			Add(new Plot(Color.FromKnownColor(KnownColor.DodgerBlue), PlotStyle.Dot, "SMIup")); 
	
		    Plots[0].Pen.Width = 2;    // SMI
			Plots[1].Pen.Width = 1;    // smiema
			Plots[2].Pen.Width = 2;    // dn
			Plots[3].Pen.Width = 2;    // up
			
			Add(new Plot(Color.FromKnownColor(KnownColor.Transparent), PlotStyle.Line, "eSMIdn"));//Can ditch these two plots here and in properties
			Add(new Plot(Color.FromKnownColor(KnownColor.Transparent), PlotStyle.Line, "eSMIup"));//if not used in code below for chop
			
			Plots[4].Pen.Width = 2;  // eSMIdn
			Plots[5].Pen.Width = 2;  // eSMIup

			Add(new Plot(new Pen(Color.DarkRed, 4), PlotStyle.Bar, "HistTickDown"));
			Add(new Plot(new Pen(Color.Blue, 4), PlotStyle.Bar, "HistTickUp"));

			//stochastic momentums
			sms			= new DataSeries(this);
			//high low diffs
			hls			= new DataSeries(this);
			//stochastic momentum indexes
			smis		= new DataSeries(this);
			
			esmis		= new DataSeries(this);
			
			line20		= new DataSeries(this);

			
			CalculateOnBarClose = false;

		}

		/// <summary>
		/// Calculates the indicator value(s) at the current index.
		/// </summary>
		protected override void OnBarUpdate()
		{
			//color 20 -20 region (chop area)  // Added by TheWizard March 7, 2010 with CODE CONTRIBUTED BY CORY NG
			if (CZColor != Color.Transparent)
			{line20.Set(20);
			DrawOnPricePanel = false;
			DrawRegion("band1", CurrentBar, 0, line20,-20,Color.Transparent, CZColor,  5);
			DrawOnPricePanel = true;
			}

			if (newbarsound) 
			{
				if (FirstTickOfBar) PlaySound(newbarwavfilename);  // Added by TheWizard 02/06/10
			}
			if (( CurrentBar < emaperiod2) | ( CurrentBar < emaperiod1)) 
			{
				DiffUp.Set(0);
				DiffDown.Set(0);
				return;
			}
			
			//Stochastic Momentum = SM {distance of close - midpoint}
		 	sms.Set(Close[0] - 0.5 * ((MAX(High, range)[0] + MIN(Low, range)[0])));
			
			//High low diffs
			hls.Set(MAX(High, range)[0] - MIN(Low, range)[0]);

			//Stochastic Momentum Index = SMI
			double denom = 0.5*WMA(WMA(hls,emaperiod1),emaperiod2)[0];
 			smis.Set(100*(WMA(WMA(sms,emaperiod1),emaperiod2))[0] / (denom ==0 ? 1 : denom  ));
			
			if (Falling(smis))
			{
//				if (smis[1] >= smis[2] && Plots[2].PlotStyle == PlotStyle.Line)  // CHANGED by TheWizard 02/08/10 to stop repainting once a price bar has closed
//					SMIdn.Set(1, smis[1]);
				SMIdn.Set(smis[0]);
				// Sets the bar color to chosen UP bar color
				if (colorpricebars)  // Changed code to allow for 'hollow' UP price bars & 'filled' DOWN price bars
					if (ChartControl.ChartStyleType == ChartStyleType.CandleStick)  // modified by TheWizard Feb 5, 2010 to allow for hollow-style candlesticks
					if(Close[0] < Open[0])
					{
					BarColor= barColorDown;
					CandleOutlineColor = barColorDown;
					}
					else
					{
					BarColor = Color.Transparent;
					CandleOutlineColor = barColorDown;
					}
			}	
			if (Rising(smis))
			{
//				if (smis[1] <= smis[2] && Plots[3].PlotStyle == PlotStyle.Line)  // CHANGED by TheWizard 02/08/10 to stop repainting once a price bar has closed
//					SMIup.Set(1, smis[1]);
				SMIup.Set(smis[0]);
				// Sets the bar color to chosen DOWN bar color
				if (colorpricebars)
					if (ChartControl.ChartStyleType == ChartStyleType.CandleStick)  // modified by TheWizard Feb 5, 2010 to allow for hollow-style candlesticks
					if(Close[0] < Open[0])
					{
					BarColor= barColorUp;
					CandleOutlineColor = barColorUp;
					}
					else
					{
					BarColor = Color.Transparent;
					CandleOutlineColor = barColorUp;	
					}
			}
			
			//Set the line value for the SMIEMA by taking the EMA of the SMI
			SMIEMA.Set(WMA(smis, smiemaperiod)[0]);
			esmis.Set(WMA(smis, smiemaperiod)[0]);
			if (Falling(esmis))
			{
//				eSMIdn.Set(1, esmis[1]);  // CHANGED by TheWizard 02/08/10 to stop repainting once a price bar has closed
				eSMIdn.Set(esmis[0]);
			
			}	
			if (Rising(esmis))
			{
//				eSMIup.Set(1, esmis[1]);  // CHANGED by TheWizard 02/08/10 to stop repainting once a price bar has closed
				eSMIup.Set(esmis[0]);
			
			}
				
			if(CrossBelow(smis, SMIEMA,1))
			{
					if (showarrows)
						DrawArrowDown("SMIcrossd"+CurrentBar,false,0,High[0]+offset*TickSize,vlineDnColor);
					if(drawVerticalLineOnPricePanel) DrawVerticalLine("MyVerticalLine"+CurrentBar, 0, vlineDnColor,DashStyle.Dash,vlineWidth);
			}
				else if(CrossAbove(smis, SMIEMA,1))
			{
					if (showarrows)
						DrawArrowUp("SMIcrossu"+CurrentBar,false,0,Low[0]-offset*TickSize,vlineUpColor);
					if(drawVerticalLineOnPricePanel) DrawVerticalLine("MyVerticalLine"+CurrentBar, 0, vlineUpColor,DashStyle.Dash,vlineWidth);	
			}
			else
			{
			//Remove bad arrows - CODE MODIFIED / ADDED BY eDanny April 2010.
			if(showarrows)
			{
				RemoveDrawObject("SMIcrossd"+CurrentBar);
				RemoveDrawObject("SMIcrossu"+CurrentBar);
			}
			//Remove bad lines	- CODE MODIFIED / ADDED BY eDanny April 2010.
			if(drawVerticalLineOnPricePanel)
				RemoveDrawObject("MyVerticalLine"+CurrentBar);
			}

			if(colorbackground)
			{
			if(smis[0] > SMIEMA[0])
			{
				BackColor = Color.FromArgb(opacity,BackColorUp);
				if(colorALLbackgrounds) BackColorAll = Color.FromArgb(opacity,BackColorUp);
			}
			else if(smis[0] < SMIEMA[0])
			{
				BackColor = Color.FromArgb(opacity,BackColorDn);
				if(colorALLbackgrounds) BackColorAll = Color.FromArgb(opacity,BackColorDn);
			}
			else
			{
				BackColor = Color.Empty;// for rare case scenario - CODE MODIFIED / ADDED BY eDanny April 2010.
			}
			}
			
			if(FirstTickOfBar)
			{
				if(smis[1] < SMIEMA[1] && smis[2] > SMIEMA[2])// I like this better than CrossAbove/Below - CODE MODIFIED / ADDED BY eDanny April 2010.
				{
					if(activatealertsound)
					PlaySound(shortwavfilename);
					textMessage = "SMI crossed below signal line ";
					BarAtLastCross = CurrentBar;
				}
				
				if(smis[1] > SMIEMA[1] && smis[2] < SMIEMA[2])
				{
					if(activatealertsound)
					PlaySound(longwavfilename);
					textMessage = "SMI crossed above signal line ";
					BarAtLastCross = CurrentBar;
				}
					
			}
			
			if(writealerttext) // Simpler way to do this I believe - CODE MODIFIED / ADDED BY eDanny April 2010.
				DrawTextFixed("SMIalert","" + textMessage + (CurrentBar - (BarAtLastCross - 1)) + " bars ago",TextPosition.BottomLeft);
					
			if(textOnLeft)			
			{
				if(textOnTop)
				tPosition = TextPosition.TopLeft;
				else 
				bPosition = TextPosition.BottomLeft;
				
			}
			else
			{
				if(!textOnTop)
				bPosition = TextPosition.BottomRight;
				else
				tPosition = TextPosition.TopRight;
			}
						
			if(showText)
			{
			
			if(CrossAbove(SMIEMA, -20, 1))
			DrawTextFixed("Possible Chop", " POSSIBLE CHOP ", tPosition, Color.White, textFontMed, Color.Black, Color.Magenta, 10);
				else
			RemoveDrawObject("Possible Chop");
			
			if(CrossBelow(SMIEMA, 20, 1))
			DrawTextFixed("Possible Chop", " POSSIBLE CHOP ", tPosition, Color.White, textFontMed, Color.Black, Color.Magenta, 10);
				else
			RemoveDrawObject("Possible Chop");
			
			if(CrossAbove(SMIEMA, 20, 1))
			DrawTextFixed("Long Breakout", " LONG BREAKOUT ", tPosition, Color.White, textFontMed, Color.Black, Color.Blue, 10);
				else
			RemoveDrawObject("Long Breakout");
			
			if(CrossBelow(SMIEMA, -20, 1))
			DrawTextFixed("Short Breakout", " SHORT BREAKOUT ", tPosition, Color.White, textFontMed, Color.Black, Color.Red, 10);
				else
			RemoveDrawObject("Short Breakout");
			}	
		}

		
		#region Properties
		/// <summary>
		/// Gets the fast D value.
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries SMI
		{
			get { return Values[0]; }
		}
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries SMIdn
		{
			get { return Values[2]; }
		}
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries SMIup
		{
			get { return Values[3]; }
		}

		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries eSMIdn
		{
			get { return Values[6]; }
		}
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries eSMIup
		{
			get { return Values[7]; }
		}
		/// <summary>
		/// Gets the fast K value.
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries SMIEMA
		{
			get { return Values[1]; }
		}

		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries DiffUp
		{
			get { return Values[4]; }
		}
		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries DiffDown
		{
			get { return Values[5]; }
		}
		/// <summary>
		/// </summary>
		[Browsable(false)]
		[Description("Chop Range filter")]
		[Category("Parameters")]
		public double Chopfilter
		{
			get { return chopfilter; }
			set { chopfilter = value; }
		}
		[Description("Range for momentum calculation. ( Q )")]
		[Category("Parameters")]
		public int Range
		{
			get { return range; }
			set { range = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Description("1st ema smothing period. ( R )")]
		[Category("Parameters")]
		public int EMAPeriod1
		{
			get { return emaperiod1; }
			set { emaperiod1 = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Description("2nd ema smoothing period. ( S )")]
		[Category("Parameters")]
		public int EMAPeriod2
		{
			get { return emaperiod2; }
			set { emaperiod2 = Math.Max(1, value); }
		}
		
		/// <summary>
		/// </summary>
		[Description("SMI EMA smoothing period.")]
		[Category("Parameters")]
		public int SMIEMAPeriod
		{
			get { return smiemaperiod; }
			set { smiemaperiod = Math.Max(1, value); }
		}
		[Description("Activate Alert Sounds?")]
		[Category("Parameters")]
		public bool ActivateAlertSound
		{
			get { return activatealertsound; }
			set { activatealertsound = value; }
		}
		[Description("Sound file to play for long alert.")]
		[Category("Parameters")]
		public string LONGWAVFileName
		{
			get { return longwavfilename; }
			set { longwavfilename = value; }
		}
		[Description("Sound file to play for short alert.")]
		[Category("Parameters")]
		public string SHORTWAVFileName
		{
			get { return shortwavfilename; }
			set { shortwavfilename = value; }
		}
		[Description(" New Bar Alert Sound")]  // Added by TheWizard 02/06/10
        [Category("Parameters")]  // Added by TheWizard 02/06/10
        public bool NewBarSound  // Added by TheWizard 02/06/10
        {
            get { return newbarsound; }  // Added by TheWizard 02/06/10
            set { newbarsound = value; }  // Added by TheWizard 02/06/10
        }
		[Description("Sound file to play for new bar alert.")]  // Added by TheWizard 02/06/10
		[Category("Parameters")]  // Added by TheWizard 02/06/10
		public string NewBarWavFileName  // Added by TheWizard 02/06/10
		{
			get { return newbarwavfilename; }  // Added by TheWizard 02/06/10
			set { newbarwavfilename = value; }  // Added by TheWizard 02/06/10
		}
		[Description("Write alert text message to screen?")]
		[Category("Parameters")]
		public bool WriteAlertText
		{
			get { return writealerttext; }
			set { writealerttext = value; }
		}
		[Description("Show Arrow Signals?")]
		[Category("Parameters")]
		public bool ShowArrows
		{
			get { return showarrows; }
			set { showarrows = value; }
		}
		[Description("Arrow Signal Offset (in points)")]
		[Category("Parameters")]
		public double Offset
		{
			get { return offset; }
			set { offset = value; }
		}
		[Description("Color Price Bars?")]
		[Category("Coloring")]
		[Gui.Design.DisplayNameAttribute("01. Color Price Bars?")]
		public bool Colorpricebars
		{
			get { return colorpricebars; }
			set { colorpricebars = value; }
		}
        [Description("Color of Up bars")]
        [Category("Coloring")]
        [Gui.Design.DisplayNameAttribute("02. Up bar color")]
        public Color BarColorUp
        {
            get { return barColorUp; }
            set { barColorUp = value; }
        }
        [Browsable(false)]
        public string barColorUpSerialize
        {
            get { return NinjaTrader.Gui.Design.SerializableColor.ToString(barColorUp); }
            set { barColorUp = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
        }
		[XmlIgnore()]
        [Description("Color of Down bars")]
        [Category("Coloring")]
        [Gui.Design.DisplayNameAttribute("03. Down bar color")]
        public Color BarColorDown
        {
            get { return barColorDown; }
            set { barColorDown = value; }
        }
        [Browsable(false)]
        public string barColorDownSerialize
        {
            get { return NinjaTrader.Gui.Design.SerializableColor.ToString(barColorDown); }
            set { barColorDown = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
        }
		[XmlIgnore()]
		[Browsable(false)]
        [Description("Color of neutral bars")]
        [Category("Coloring")]
        [Gui.Design.DisplayNameAttribute("04. Neutral bar color")]
        public Color BarColorNeutral
        {
            get { return barColorNeutral; }
            set { barColorNeutral = value; }
        }
        [Browsable(false)]
        public string barColorNeutralSerialize
        {
            get { return NinjaTrader.Gui.Design.SerializableColor.ToString(barColorNeutral); }
            set { barColorNeutral = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
        }
		[Description("Color Background?")]  // added this section by TheWizard 02/06/10
		[Category("Coloring")]
		[Gui.Design.DisplayNameAttribute("05. Color Background?")]
		public bool ColorBackground
		{
			get { return colorbackground; }
			set { colorbackground = value; }
		}
		[Description("Color ALL Backgrounds?")]  // added this section by TheWizard 02/06/10
		[Category("Coloring")]
		[Gui.Design.DisplayNameAttribute("06. Color ALL Backgrounds?")]
		public bool ColorALLBackgrounds
		{
			get { return colorALLbackgrounds; }
			set { colorALLbackgrounds = value; }
		}
		[Description("Back Up Color")]
        [Category("Coloring")]
        [Gui.Design.DisplayNameAttribute("07. Background Up Color")]
        public Color BackColorUp
        {
            get { return backColorUp; }
            set { backColorUp = value; }
        }
        [Browsable(false)]
        public string backColorUpSerialize
        {
            get { return NinjaTrader.Gui.Design.SerializableColor.ToString(backColorUp); }
            set { backColorUp = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
        }
		[XmlIgnore()]
        [Description("Back Down COlor")]
        [Category("Coloring")]
        [Gui.Design.DisplayNameAttribute("08. Background Down Color")]
        public Color BackColorDn
        {
            get { return backColorDn; }
            set { backColorDn = value; }
        }
        [Browsable(false)]
        public string backColorDownSerialize
        {
            get { return NinjaTrader.Gui.Design.SerializableColor.ToString(backColorDn); }
            set { backColorDn = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
        }
		[Description("Background Opacity")]
		[Category("Coloring")]
		[Gui.Design.DisplayNameAttribute("09. Background Opacity")]
		public int Opacity
		{
			get { return opacity; }
			set { opacity = value; }
		}
		 // The Following lines added by TheWizard December 22, 2009 
		[Description("Draw Vertical Line on Price Panel?")]
        [Category("Coloring")]
        [Gui.Design.DisplayNameAttribute("10. DrawVerticalLineOnPricePanel?")]
        public bool DrawVerticalLineOnPricePanel
        {
            get { return drawVerticalLineOnPricePanel; }
            set { drawVerticalLineOnPricePanel = value; }
        }
		[Description("VLineUpColor")]
        [Category("Coloring")]
		[Gui.Design.DisplayNameAttribute("11. VLineUpColor")]
        public Color VlineUpColor
        {
            get { return vlineUpColor; }
            set { vlineUpColor = value; }
        }

        /// <summary>
        /// </summary>
        [Browsable(false)]
        public string vlineUpColorSerialize
        {
            get { return NinjaTrader.Gui.Design.SerializableColor.ToString(vlineUpColor); }
            set { vlineUpColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
        }
		[Description("VLineDNColor")]
        [Category("Coloring")]
        [Gui.Design.DisplayNameAttribute("12. VLineDnColor")]
        public Color VlineDnColor
        {
            get { return vlineDnColor; }
            set { vlineDnColor = value; }
        }

        /// <summary>
        /// </summary>
        [Browsable(false)]
        public string vlineDnColorSerialize
        {
            get { return NinjaTrader.Gui.Design.SerializableColor.ToString(vlineDnColor); }
            set { vlineDnColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
        }
		[Description("VLineWidth")]
        [Category("Coloring")]
        [Gui.Design.DisplayNameAttribute("13. VLineWidth")]
		    public int VLineWidth
        {
            get { return vlineWidth; }
            set { vlineWidth = value; }
        }
		[Description("Chop Zone Color")]
        [Category("Coloring")]
        [Gui.Design.DisplayNameAttribute("14. Chop Zone Color")]
        public Color CZColor
        {
            get { return czColor; }
            set { czColor = value; }
        }
        [Browsable(false)]
        public string czColorSerialize
        {
            get { return NinjaTrader.Gui.Design.SerializableColor.ToString(czColor); }
            set { czColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
        }
		[Description("Show Text Info.")]
        [Category("Drawing Objects")]
        public bool ShowText
        {
            get { return showText; }
            set { showText = value; }
        }
		
		[Description("Text on left of chart? For those with no right margin.")]
        [Category("Drawing Objects")]
        public bool TextOnLeft
        {
            get { return textOnLeft; }
            set { textOnLeft = value; }
        }
		[Description("Text on top of chart?")]
        [Category("Drawing Objects")]
        public bool TextOnTop
        {
            get { return textOnTop; }
            set { textOnTop = value; }
        }
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private SMI5[] cacheSMI5 = null;

        private static SMI5 checkSMI5 = new SMI5();

        /// <summary>
        /// The Stochastic Momentum Index is made up of two lines that oscillate between a vertical scale of -100 to 100.
        /// </summary>
        /// <returns></returns>
        public SMI5 SMI5(bool activateAlertSound, double chopfilter, int eMAPeriod1, int eMAPeriod2, string lONGWAVFileName, bool newBarSound, string newBarWavFileName, double offset, int range, string sHORTWAVFileName, bool showArrows, int sMIEMAPeriod, bool writeAlertText)
        {
            return SMI5(Input, activateAlertSound, chopfilter, eMAPeriod1, eMAPeriod2, lONGWAVFileName, newBarSound, newBarWavFileName, offset, range, sHORTWAVFileName, showArrows, sMIEMAPeriod, writeAlertText);
        }

        /// <summary>
        /// The Stochastic Momentum Index is made up of two lines that oscillate between a vertical scale of -100 to 100.
        /// </summary>
        /// <returns></returns>
        public SMI5 SMI5(Data.IDataSeries input, bool activateAlertSound, double chopfilter, int eMAPeriod1, int eMAPeriod2, string lONGWAVFileName, bool newBarSound, string newBarWavFileName, double offset, int range, string sHORTWAVFileName, bool showArrows, int sMIEMAPeriod, bool writeAlertText)
        {
            checkSMI5.ActivateAlertSound = activateAlertSound;
            activateAlertSound = checkSMI5.ActivateAlertSound;
            checkSMI5.Chopfilter = chopfilter;
            chopfilter = checkSMI5.Chopfilter;
            checkSMI5.EMAPeriod1 = eMAPeriod1;
            eMAPeriod1 = checkSMI5.EMAPeriod1;
            checkSMI5.EMAPeriod2 = eMAPeriod2;
            eMAPeriod2 = checkSMI5.EMAPeriod2;
            checkSMI5.LONGWAVFileName = lONGWAVFileName;
            lONGWAVFileName = checkSMI5.LONGWAVFileName;
            checkSMI5.NewBarSound = newBarSound;
            newBarSound = checkSMI5.NewBarSound;
            checkSMI5.NewBarWavFileName = newBarWavFileName;
            newBarWavFileName = checkSMI5.NewBarWavFileName;
            checkSMI5.Offset = offset;
            offset = checkSMI5.Offset;
            checkSMI5.Range = range;
            range = checkSMI5.Range;
            checkSMI5.SHORTWAVFileName = sHORTWAVFileName;
            sHORTWAVFileName = checkSMI5.SHORTWAVFileName;
            checkSMI5.ShowArrows = showArrows;
            showArrows = checkSMI5.ShowArrows;
            checkSMI5.SMIEMAPeriod = sMIEMAPeriod;
            sMIEMAPeriod = checkSMI5.SMIEMAPeriod;
            checkSMI5.WriteAlertText = writeAlertText;
            writeAlertText = checkSMI5.WriteAlertText;

            if (cacheSMI5 != null)
                for (int idx = 0; idx < cacheSMI5.Length; idx++)
                    if (cacheSMI5[idx].ActivateAlertSound == activateAlertSound && Math.Abs(cacheSMI5[idx].Chopfilter - chopfilter) <= double.Epsilon && cacheSMI5[idx].EMAPeriod1 == eMAPeriod1 && cacheSMI5[idx].EMAPeriod2 == eMAPeriod2 && cacheSMI5[idx].LONGWAVFileName == lONGWAVFileName && cacheSMI5[idx].NewBarSound == newBarSound && cacheSMI5[idx].NewBarWavFileName == newBarWavFileName && Math.Abs(cacheSMI5[idx].Offset - offset) <= double.Epsilon && cacheSMI5[idx].Range == range && cacheSMI5[idx].SHORTWAVFileName == sHORTWAVFileName && cacheSMI5[idx].ShowArrows == showArrows && cacheSMI5[idx].SMIEMAPeriod == sMIEMAPeriod && cacheSMI5[idx].WriteAlertText == writeAlertText && cacheSMI5[idx].EqualsInput(input))
                        return cacheSMI5[idx];

            SMI5 indicator = new SMI5();
            indicator.BarsRequired = BarsRequired;
            indicator.CalculateOnBarClose = CalculateOnBarClose;
            indicator.Input = input;
            indicator.ActivateAlertSound = activateAlertSound;
            indicator.Chopfilter = chopfilter;
            indicator.EMAPeriod1 = eMAPeriod1;
            indicator.EMAPeriod2 = eMAPeriod2;
            indicator.LONGWAVFileName = lONGWAVFileName;
            indicator.NewBarSound = newBarSound;
            indicator.NewBarWavFileName = newBarWavFileName;
            indicator.Offset = offset;
            indicator.Range = range;
            indicator.SHORTWAVFileName = sHORTWAVFileName;
            indicator.ShowArrows = showArrows;
            indicator.SMIEMAPeriod = sMIEMAPeriod;
            indicator.WriteAlertText = writeAlertText;
            indicator.SetUp();

            SMI5[] tmp = new SMI5[cacheSMI5 == null ? 1 : cacheSMI5.Length + 1];
            if (cacheSMI5 != null)
                cacheSMI5.CopyTo(tmp, 0);
            tmp[tmp.Length - 1] = indicator;
            cacheSMI5 = tmp;
            Indicators.Add(indicator);

            return indicator;
        }

    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// The Stochastic Momentum Index is made up of two lines that oscillate between a vertical scale of -100 to 100.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.SMI5 SMI5(bool activateAlertSound, double chopfilter, int eMAPeriod1, int eMAPeriod2, string lONGWAVFileName, bool newBarSound, string newBarWavFileName, double offset, int range, string sHORTWAVFileName, bool showArrows, int sMIEMAPeriod, bool writeAlertText)
        {
            return _indicator.SMI5(Input, activateAlertSound, chopfilter, eMAPeriod1, eMAPeriod2, lONGWAVFileName, newBarSound, newBarWavFileName, offset, range, sHORTWAVFileName, showArrows, sMIEMAPeriod, writeAlertText);
        }

        /// <summary>
        /// The Stochastic Momentum Index is made up of two lines that oscillate between a vertical scale of -100 to 100.
        /// </summary>
        /// <returns></returns>
        public Indicator.SMI5 SMI5(Data.IDataSeries input, bool activateAlertSound, double chopfilter, int eMAPeriod1, int eMAPeriod2, string lONGWAVFileName, bool newBarSound, string newBarWavFileName, double offset, int range, string sHORTWAVFileName, bool showArrows, int sMIEMAPeriod, bool writeAlertText)
        {
            return _indicator.SMI5(input, activateAlertSound, chopfilter, eMAPeriod1, eMAPeriod2, lONGWAVFileName, newBarSound, newBarWavFileName, offset, range, sHORTWAVFileName, showArrows, sMIEMAPeriod, writeAlertText);
        }

    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// The Stochastic Momentum Index is made up of two lines that oscillate between a vertical scale of -100 to 100.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.SMI5 SMI5(bool activateAlertSound, double chopfilter, int eMAPeriod1, int eMAPeriod2, string lONGWAVFileName, bool newBarSound, string newBarWavFileName, double offset, int range, string sHORTWAVFileName, bool showArrows, int sMIEMAPeriod, bool writeAlertText)
        {
            return _indicator.SMI5(Input, activateAlertSound, chopfilter, eMAPeriod1, eMAPeriod2, lONGWAVFileName, newBarSound, newBarWavFileName, offset, range, sHORTWAVFileName, showArrows, sMIEMAPeriod, writeAlertText);
        }

        /// <summary>
        /// The Stochastic Momentum Index is made up of two lines that oscillate between a vertical scale of -100 to 100.
        /// </summary>
        /// <returns></returns>
        public Indicator.SMI5 SMI5(Data.IDataSeries input, bool activateAlertSound, double chopfilter, int eMAPeriod1, int eMAPeriod2, string lONGWAVFileName, bool newBarSound, string newBarWavFileName, double offset, int range, string sHORTWAVFileName, bool showArrows, int sMIEMAPeriod, bool writeAlertText)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.SMI5(input, activateAlertSound, chopfilter, eMAPeriod1, eMAPeriod2, lONGWAVFileName, newBarSound, newBarWavFileName, offset, range, sHORTWAVFileName, showArrows, sMIEMAPeriod, writeAlertText);
        }

    }
}
#endregion
