/*
ZigZagUTC_TL_BarCount_Fibs - another zig zag indicator
by Alex Matulich, anachronist@gmail.com
Unicorn Research Corporation, September 2008

I derived this from the verbal description on page 39 of
_The Ultimate Trading Guide_ by John R. Hill, George Pruitt, and Lundy Hill
published (2000) by John Wiley and Sons, Inc, ISBN 0-471-38135-7

Instead of using an arbitrary pullback amount to identify a swing high
or a swing low, this zigzag indicator uses an objective definition:
A swing high bar is the highest high prior to penetration of the low
of that highest high bar. A swing low bar is the lowest low prior to
penetration of the high of that lowest low bar.

This indicator goes a bit further by introducing a 'span' parameter.
When span=1, the indicator behaves as described in the book. When
span>1, the highs and lows are calculated as the highest and lowest in
a sliding window 'span' bars wide. Increasing the size of 'span' causes
the indicator to ignore the smaller swings. When span=6, this indicator
matches Larry Williams' zigzag description in his "Money Tree" course.

If you set "Use Highs and Lows" to false, then only input prices (such
as closes) will be used. In this case, you should increase span to 2 or
higher to avoid identifying nearly every price as a swing.

This indicator has a number of public properties that can be accessed
from other indicators.
*/
#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// ZigZag indicator from the Ultimate Trading Guide by Hill, Pruitt, and Hill. A swing high occurs when price takes out the low of the highest bar. A swing low occurs when price takes out the high of the lowest bar. Span setting determines how many bars to look back for highest highs and lowest lows; i.e. span=5 on daily bars find weekly swings.
	/// add HH/LL trend lines and rays 11/14/09 -Cory
	/// add fib suite 11/25/09 - cory
    /// </summary>
    [Description("ZigZag indicator from the Ultimate Trading Guide by Hill, Pruitt, and Hill. A swing high occurs when price takes out the low of the highest bar. A swing low occurs when price takes out the high of the lowest bar. Span setting determines how many bars to look back for highest highs and lowest lows; i.e. span=5 on daily bars find weekly swings.")]
    public class ZigZagUTC_TL_BarCount_Fibs_v5_1 : Indicator
    {
        #region Variables
        // Wizard generated variables
            private int span = 2; // Default setting for Span
			private bool useHiLo = true; // Default setting for UseHiLo
			private Color zigZagColor = Color.Green; // Default setting for zigZagColor
			private int show = 3; // Default setting for Show
        // User defined variables (add any user defined variables below)
			private DataSeries hi, lo;
			private bool drawlines, showdots;
			private int linewidth = 2;
		// Useful public properties
			public int dir;         // direction of current zigzag trend, 1=up, -1=down
			public int lasthibar;   // bar number of last swing high (bars ago = CurrentBar-lasthibar)
			public int lastlobar;   // bar number of last swing low
			public double lasthi;   // value of last swing high
			public double lastlo;   // value of last swing low
		
			public double prevhi = 0;   // value of previous swing high
			public double prevlo = 0;   // value of previous swing low
			public int prevhibar = 0;   // bar number of previous swing high (bars ago = CurrentBar-lasthibar)
			public int prevlobar = 0;   // bar number of previous swing low
			public double prevhi2 = 0;   // value of previous swing high
			public double prevlo2 = 0;   // value of previous swing low
			public int prevhibar2 = 0;   // bar number of previous swing high (bars ago = CurrentBar-lasthibar)
			public int prevlobar2 = 0;   // bar number of previous swing low
		
		    private int hicount = 0;
		    private int locount = 0;
		    private Color hhllColor = Color.Brown;
		    private bool drawRays  = true;
		    private Font 			textFont 			= new Font("Courier", 11, FontStyle.Bold);
			private Color 			textColor			= Color.Red;
		    private int 			textOffset			 = 3;
			private bool			showfibE				= true;
			private bool			showfibR				= true;	
			private bool			showfibT				= true;
			private bool			showfibC				= true;
		    private bool			showfibR2				= true;	
            private bool		    drawbracket		    	= true;	
			private Color colorBZ = Color.DarkGreen;
		
		
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
            Add(new Plot(new Pen(Color.FromKnownColor(KnownColor.DarkOrchid), 2), PlotStyle.Dot, "ZigZagDot"));
            CalculateOnBarClose	= false;
            Overlay				= true;
            PriceTypeSupported	= true;

			hi = new DataSeries(this);
			lo = new DataSeries(this);
			dir = 0;
			drawlines = ((show & 1) > 0);
			showdots = ((show & 2) > 0);
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
			if (CurrentBar == 0) { // first bar initialization
				lasthi = hi[0] = useHiLo ? High[0] : Input[0];
				lastlo = lo[0] = useHiLo ? Low[0] : Input[0];
				lasthibar = lastlobar = 0;
				return;
			}

			// get high and low for last 'span' number of bars
			hi.Set(MAX(useHiLo ? High : Input, span)[0]);
			lo.Set(MIN(useHiLo ? Low : Input, span)[0]);

			// set initial trend direction (dir will become non-zero after the first couple of bars)
			if (dir == 0) { // set initial direction
				if (hi[0] > lasthi) {
					lasthi = hi[0]; lasthibar = CurrentBar;
					if (lo[0] > lo[1]) dir = 1;
				}
				if (lo[0] < lastlo) {
					lastlo = lo[0]; lastlobar = CurrentBar;
					if (hi[0] < hi[1]) dir = -1;
				}
			}

			// look for swing points and draw lines
			if (dir > 0) { // trend is up, look for new swing high
				if (hi[0] > lasthi) { // found a higher high
					lasthi = hi[0]; lasthibar = CurrentBar;
					if (drawlines) // draw/re-draw upward (current trend) line from lastlobar
#if NT7
						DrawLine(lastlobar.ToString(),AutoScale, CurrentBar-lastlobar, lastlo, CurrentBar-lasthibar, lasthi, zigZagColor, DashStyle.Solid, linewidth); 
				        DrawText("TextLast",AutoScale, ( - prevlobar + lasthibar ).ToString(), CurrentBar-lasthibar, lasthi +  textOffset*TickSize,0, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
#else				
						DrawLine(lastlobar.ToString(), CurrentBar-lastlobar, lastlo, CurrentBar-lasthibar, lasthi, zigZagColor, DashStyle.Solid, linewidth); 
				        DrawText("TextLast", ( - prevlobar + lasthibar ).ToString(), CurrentBar-lasthibar, lasthi +  textOffset*TickSize, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
#endif
					if (showfibR2)
						DrawFibonacciRetracements("fibR2", true, (CurrentBar - lasthibar), lasthi, (CurrentBar - prevlobar), prevlo);
				}
				else if (hi[0] < lasthi && lo[0] < lo[1]) { // found a swing high
					if (drawlines) // redraw the upward line from lastlobar to new swing high
#if NT7
						DrawLine(lastlobar.ToString(),AutoScale, CurrentBar-lastlobar, lastlo, CurrentBar-lasthibar, lasthi, zigZagColor, DashStyle.Solid, linewidth); 
					    DrawText("TextLast",AutoScale, ( - prevlobar + lasthibar ).ToString(), CurrentBar-lasthibar, lasthi +  textOffset*TickSize,0, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
#else				
						DrawLine(lastlobar.ToString(), CurrentBar-lastlobar, lastlo, CurrentBar-lasthibar, lasthi, zigZagColor, DashStyle.Solid, linewidth); 
					    DrawText("TextLast", ( - prevlobar + lasthibar ).ToString(), CurrentBar-lasthibar, lasthi +  textOffset*TickSize, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
#endif
				
					dir = -1;                               // trend direction is now down
					lastlo = lo[0]; lastlobar = CurrentBar; // now seeking new lows
					if (drawlines) // start new trendline from new swing high to most recent low
#if NT7
						DrawLine(lasthibar.ToString(),AutoScale, CurrentBar-lasthibar, lasthi, CurrentBar-lastlobar, lastlo, zigZagColor, DashStyle.Solid, linewidth);
#else				
						DrawLine(lasthibar.ToString(), CurrentBar-lasthibar, lasthi, CurrentBar-lastlobar, lastlo, zigZagColor, DashStyle.Solid, linewidth);
#endif
					if (showfibR)
					//	DrawFibonacciRetracements("fibR", true, (CurrentBar - prevlobar), prevlo, (CurrentBar - lasthibar), lasthi);
						DrawFibonacciRetracements("fibR", true, (CurrentBar - lasthibar), lasthi, (CurrentBar - prevlobar), prevlo);
					
					if (showfibE)
						DrawFibonacciExtensions("fibE", true, CurrentBar - prevhibar, prevhi, (CurrentBar - prevlobar), prevlo, (CurrentBar - lasthibar), lasthi);
					if (showfibC)
#if NT7
						DrawFibonacciCircle("fibC", true, false,(CurrentBar - prevlobar), prevlo, (CurrentBar - lasthibar), lasthi);	
#else				
						DrawFibonacciCircle("fibC", true, (CurrentBar - prevlobar), prevlo, (CurrentBar - lasthibar), lasthi);	
#endif
					if (showfibT)
						DrawFibonacciTimeExtensions("fibT", (CurrentBar - prevlobar), prevlo, (CurrentBar - lasthibar), lasthi);	
					if (showdots) ZigZagDot.Set(CurrentBar-lasthibar, lasthi);
#if NT7
					    DrawText("TextLast",AutoScale,  ( - prevhibar + lastlobar ).ToString(), CurrentBar-lastlobar, lastlo -  textOffset*TickSize,0, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
#else				
					    DrawText("TextLast",( - prevhibar + lastlobar ).ToString(), CurrentBar-lastlobar, lastlo -  textOffset*TickSize, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
#endif
				
		
				hicount =  1;
				if (prevhi == 0 )
				{
				prevhi = lasthi;
				prevhibar = lasthibar;
				hicount = 0;
				}	
				}

			} else { // dir < 0, trend is down, look for new swing low
				if (lo[0] < lastlo) { // found a lower low
					lastlo = lo[0]; lastlobar = CurrentBar;
					if (drawlines) // draw/re-draw downward (current trend) line from lasthibar
#if NT7
						DrawLine(lasthibar.ToString(),AutoScale, CurrentBar-lasthibar, lasthi, CurrentBar-lastlobar, lastlo, zigZagColor, DashStyle.Solid, linewidth);
#else				
						DrawLine(lasthibar.ToString(), CurrentBar-lasthibar, lasthi, CurrentBar-lastlobar, lastlo, zigZagColor, DashStyle.Solid, linewidth);
#endif

#if NT7
					DrawText("TextLast",AutoScale, ( - prevhibar + lastlobar ).ToString(), CurrentBar-lastlobar, lastlo -  textOffset*TickSize,0, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
#else				
					DrawText("TextLast", ( - prevhibar + lastlobar ).ToString(), CurrentBar-lastlobar, lastlo -  textOffset*TickSize, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
#endif
					
					if (showfibR2)
						DrawFibonacciRetracements("fibR2", true, (CurrentBar - lastlobar), lastlo, (CurrentBar - prevhibar), prevhi);
				}
				else if (lo[0] > lastlo && hi[0] > hi[1]) { // found a swing low
					if (drawlines) // redraw the downward line from lastlobar to new swing low
#if NT7
						DrawLine(lasthibar.ToString(),AutoScale, CurrentBar-lasthibar, lasthi, CurrentBar-lastlobar, lastlo, zigZagColor, DashStyle.Solid, linewidth);
#else				
						DrawLine(lasthibar.ToString(), CurrentBar-lasthibar, lasthi, CurrentBar-lastlobar, lastlo, zigZagColor, DashStyle.Solid, linewidth);
#endif

#if NT7
					DrawText("TextLast",AutoScale, ( - prevhibar + lastlobar ).ToString(), CurrentBar-lastlobar, lastlo -  textOffset*TickSize,0, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
#else				
					DrawText("TextLast", ( - prevhibar + lastlobar ).ToString(), CurrentBar-lastlobar, lastlo -  textOffset*TickSize, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
#endif
				
					dir = 1;                                // trend direction is now up
					lasthi = hi[0]; lasthibar = CurrentBar; // now seeking new highs
					if (drawlines) // start new trendline from new swing low to most recent high
#if NT7
						DrawLine(lastlobar.ToString(),AutoScale, CurrentBar-lastlobar, lastlo, CurrentBar-lasthibar, lasthi, zigZagColor, DashStyle.Solid, linewidth);
#else				
						DrawLine(lastlobar.ToString(), CurrentBar-lastlobar, lastlo, CurrentBar-lasthibar, lasthi, zigZagColor, DashStyle.Solid, linewidth);
#endif
					if (showfibR)
				//	DrawFibonacciRetracements("fibR", true, (CurrentBar - prevhibar), prevhi, (CurrentBar - lastlobar), lastlo);
					DrawFibonacciRetracements("fibR", true, (CurrentBar - lastlobar), lastlo, (CurrentBar - prevhibar), prevhi);
					
					if (showfibE)
						DrawFibonacciExtensions("fibE", true, CurrentBar - prevlobar, prevlo, (CurrentBar - prevhibar), prevhi, (CurrentBar - lastlobar), lastlo);
					if (showfibC)
#if NT7
						DrawFibonacciCircle("fibC", true,false, (CurrentBar - prevhibar), prevhi, (CurrentBar - lastlobar), lastlo);
#else				
						DrawFibonacciCircle("fibC", true,false, (CurrentBar - prevhibar), prevhi, (CurrentBar - lastlobar), lastlo);
#endif
					if (showfibT)
						DrawFibonacciTimeExtensions("fibT", (CurrentBar - prevhibar), prevhi, (CurrentBar - lastlobar), lastlo);
					if (showdots) 
						ZigZagDot.Set(CurrentBar-lastlobar, lastlo);
#if NT7
					DrawText("TextLast",AutoScale, ( - prevlobar + lasthibar ).ToString(), CurrentBar-lasthibar, lasthi +  textOffset*TickSize,0, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
#else				
					DrawText("TextLast", ( - prevlobar + lasthibar ).ToString(), CurrentBar-lasthibar, lasthi +  textOffset*TickSize, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
#endif
				
		
				locount = 1;
				if (prevlo == 0 )
				{
				prevlo = lastlo;
				prevlobar = lastlobar;
				locount = 0;
				}				
				}
			}
			
			// draw line connects hi to hi, low to low

			 if (prevlo > 0 && locount == 1)
			{
#if NT7
				DrawLine("Low Line"+CurrentBar,AutoScale, CurrentBar-prevlobar, prevlo, CurrentBar-lastlobar, lastlo, hhllColor, DashStyle.Solid, linewidth);
#else				
				DrawLine("Low Line"+CurrentBar, CurrentBar-prevlobar, prevlo, CurrentBar-lastlobar, lastlo, hhllColor, DashStyle.Solid, linewidth);
#endif
		//		DrawText("Text" + CurrentBar, (CurrentBar - prevhibar -(CurrentBar-lastlobar )).ToString(), CurrentBar-lastlobar, lastlo -  textOffset*TickSize, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
#if NT7
				DrawText("Text" + CurrentBar,AutoScale, ( - prevhibar + lastlobar ).ToString(), CurrentBar-lastlobar, lastlo -  textOffset*TickSize,0, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
#else				
				DrawText("Text" + CurrentBar, ( - prevhibar + lastlobar ).ToString(), CurrentBar-lastlobar, lastlo -  textOffset*TickSize, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
#endif
					
				if (drawRays)
#if NT7
				    DrawRay("rayLow",AutoScale, CurrentBar-prevlobar, prevlo, CurrentBar-lastlobar, lastlo,hhllColor,DashStyle.Dash,2);
#else				
				    DrawRay("rayLow", CurrentBar-prevlobar, prevlo, CurrentBar-lastlobar, lastlo,hhllColor,DashStyle.Dash,2);
#endif

        		if (drawbracket)	
#if NT7
					DrawRay("ray1",AutoScale, 2, lastlo, 0, lastlo, colorBZ,DashStyle.Dash,3);
#else				
					DrawRay("ray1", 2, lastlo, 0, lastlo, colorBZ,DashStyle.Dash,3);
#endif
				
				prevlo2 = prevlo;
				prevlo = lastlo;
				prevlobar2 = prevlobar;
				prevlobar = lastlobar;
				locount = 0;
			}
		
 			if (prevhi > 0 && hicount == 1)
			{
#if NT7
			    DrawLine("High Line"+CurrentBar,AutoScale,  CurrentBar-prevhibar, prevhi, CurrentBar-lasthibar, lasthi, hhllColor, DashStyle.Solid, linewidth);
#else				
			    DrawLine("High Line"+CurrentBar,  CurrentBar-prevhibar, prevhi, CurrentBar-lasthibar, lasthi, hhllColor, DashStyle.Solid, linewidth);
#endif
		//		DrawText("Text" + CurrentBar, (CurrentBar - prevlobar -(CurrentBar - lasthibar) ).ToString(), CurrentBar-lasthibar, lasthi +  textOffset*TickSize, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
#if NT7
				DrawText("Text" + CurrentBar,AutoScale, ( - prevlobar + lasthibar ).ToString(), CurrentBar-lasthibar, lasthi +  textOffset*TickSize,0, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
#else				
				DrawText("Text" + CurrentBar, ( - prevlobar + lasthibar ).ToString(), CurrentBar-lasthibar, lasthi +  textOffset*TickSize, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
#endif
					
				if (drawRays)
#if NT7
				    DrawRay("rayHi", AutoScale, CurrentBar-prevhibar, prevhi, CurrentBar-lasthibar, lasthi,hhllColor,DashStyle.Dash,2);
#else				
				    DrawRay("rayHi", CurrentBar-prevhibar, prevhi, CurrentBar-lasthibar, lasthi,hhllColor,DashStyle.Dash,2);
#endif
				
				if (drawbracket)	
#if NT7
					DrawRay("ray2",AutoScale, 2, lasthi, 0, lasthi, colorBZ,DashStyle.Dash,3);
#else				
					DrawRay("ray2", 2, lasthi, 0, lasthi, colorBZ,DashStyle.Dash,3);
#endif
				
				prevhi2 = prevhi;
				prevhi = lasthi;
				prevhibar2 = prevhibar;
				prevhibar = lasthibar;
				hicount = 0;
			}			
	
			
        }

        #region Properties
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries ZigZagDot
        {
            get { return Values[0]; }
        }

        [Description("Bar span to consider for highest and lowest values")]
        [Category("Parameters")]
        public int Span
        {
            get { return span; }
            set { span = Math.Max(1, value); }
        }
		[XmlIgnore()]
		[Description("Color of zigzag lines")]
		[Category("Parameters")]
		public Color ZigZagColor
		{
			get { return zigZagColor; }
			set { zigZagColor = value; }
		}
	//	[Browsable(false)]
	//	public string PaintColorSerialize
	//	{
    // 		get { return NinjaTrader.Gui.Design.SerializableColor.ToString(zigZagColor); }
    // 		set { zigZagColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
	//	}
		[XmlIgnore()]
		[Description("Color of HH/LL trend lines")]
		[Category("Visual")]
		public Color TrendLineColor
		{
			get { return hhllColor; }
			set { hhllColor = value; }
		}
	//	[Browsable(false)]
	//	public string hhllColorSerialize
	//	{
    // 		get { return NinjaTrader.Gui.Design.SerializableColor.ToString(hhllColor); }
    // 		set { hhllColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
	//	}
		[Description("Draw HH/LL Ray Lines?")]
		[Category("Parameters")]
		public bool ShowDiagonalRays
		{
			get { return drawRays; }
			set { drawRays = value; }
		}
		[XmlIgnore()]
		[Description("Text Font")]
        [Category("Visual")]
		[NinjaTrader.Gui.Design.DisplayName("Text font")]
        public Font TextFont
        {
            get { return textFont; }
            set { textFont = value; }
		}
	//	[Browsable(false)]
	//	public string textFontSerialize
	//	{
	//		get { return NinjaTrader.Gui.Design.SerializableFont.ToString(textFont); }
	//		set { textFont = NinjaTrader.Gui.Design.SerializableFont.FromString(value); }
	//	}
		
		[Description("Text Offset in multiple of Tick Size")]
		[Category("Visual")]
		[Gui.Design.DisplayNameAttribute("Text Offset")]
		public int TextOffset
		{
			get { return textOffset; }
			set { textOffset = value; }
		}
		[XmlIgnore()]
        [Description("Text Color ")]
        [Category("Visual")]
        [Gui.Design.DisplayNameAttribute("Text Color")]
        public Color TextColor
        {
            get { return textColor; }
            set { textColor = value; }
        }
		[Description("Draw bracket of flat rays?")]
		[Category("Parameters")]
		public bool ShowBracketofFlatRays
		{
			get { return drawbracket; }
			set { drawbracket = value; }
		}
		[XmlIgnore()]
        [Description("Color of flat ray")]
        [Category("Visual")]
        [Gui.Design.DisplayNameAttribute("Color of Flat Rays")]
        public Color ColorBZ
        {
            get { return colorBZ; }
            set { colorBZ = value; }
        }
		
        [Browsable(false)]
        public string colorBZSerialize
        {
            get { return NinjaTrader.Gui.Design.SerializableColor.ToString(colorBZ); }
            set { colorBZ = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
        }
        [Browsable(false)]
        public string textColorSerialize
        {
            get { return NinjaTrader.Gui.Design.SerializableColor.ToString(textColor); }
            set { textColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
        }
		[Description("Toggle for Fib Retracements")]
        [Category("Parameters")]
        public bool ShowFibRetrace
        {
            get { return showfibR; }
            set { showfibR = value; }
        }
		[Description("Toggle for Still In Develoment Fib Retracements")]
        [Category("Parameters")]
        public bool ShowFibRetraceInProgress
        {
            get { return showfibR2; }
            set { showfibR2 = value; }
        }
		[Description("Toggle for Fib Extensions")]
        [Category("Parameters")]
        public bool ShowFibExtend
        {
            get { return showfibE; }
            set { showfibE = value; }
        }
		[Description("Toggle for Fib Circles")]
        [Category("Parameters")]
        public bool ShowFibCircle
        {
            get { return showfibC; }
            set { showfibC = value; }
        }
		[Description("Toggle for Fib Time Extensions")]
        [Category("Parameters")]
        public bool ShowFibTime
        {
            get { return showfibT; }
            set { showfibT = value; }
        }
		[Description("true = use Highs and Lows for swings, false = use price input")]
        [Category("Parameters")]
		[Gui.Design.DisplayName("Use high and low")]
		[RefreshProperties(RefreshProperties.All)]
        public bool UseHiLo
        {
            get { return useHiLo; }
            set { useHiLo = value; PriceTypeSupported = !value; }
        }

		[Description("What to show: 1=zigzag lines, 2=swing dots, 3=both")]
        [Category("Parameters")]
        public int Show
        {
            get { return show; }
            set { show = Math.Max(1, Math.Min(3, value)); }
        }
        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private ZigZagUTC_TL_BarCount_Fibs_v5_1[] cacheZigZagUTC_TL_BarCount_Fibs_v5_1 = null;

        private static ZigZagUTC_TL_BarCount_Fibs_v5_1 checkZigZagUTC_TL_BarCount_Fibs_v5_1 = new ZigZagUTC_TL_BarCount_Fibs_v5_1();

        /// <summary>
        /// ZigZag indicator from the Ultimate Trading Guide by Hill, Pruitt, and Hill. A swing high occurs when price takes out the low of the highest bar. A swing low occurs when price takes out the high of the lowest bar. Span setting determines how many bars to look back for highest highs and lowest lows; i.e. span=5 on daily bars find weekly swings.
        /// </summary>
        /// <returns></returns>
        public ZigZagUTC_TL_BarCount_Fibs_v5_1 ZigZagUTC_TL_BarCount_Fibs_v5_1(int show, bool showBracketofFlatRays, bool showDiagonalRays, bool showFibCircle, bool showFibExtend, bool showFibRetrace, bool showFibRetraceInProgress, bool showFibTime, int span, bool useHiLo, Color zigZagColor)
        {
            return ZigZagUTC_TL_BarCount_Fibs_v5_1(Input, show, showBracketofFlatRays, showDiagonalRays, showFibCircle, showFibExtend, showFibRetrace, showFibRetraceInProgress, showFibTime, span, useHiLo, zigZagColor);
        }

        /// <summary>
        /// ZigZag indicator from the Ultimate Trading Guide by Hill, Pruitt, and Hill. A swing high occurs when price takes out the low of the highest bar. A swing low occurs when price takes out the high of the lowest bar. Span setting determines how many bars to look back for highest highs and lowest lows; i.e. span=5 on daily bars find weekly swings.
        /// </summary>
        /// <returns></returns>
        public ZigZagUTC_TL_BarCount_Fibs_v5_1 ZigZagUTC_TL_BarCount_Fibs_v5_1(Data.IDataSeries input, int show, bool showBracketofFlatRays, bool showDiagonalRays, bool showFibCircle, bool showFibExtend, bool showFibRetrace, bool showFibRetraceInProgress, bool showFibTime, int span, bool useHiLo, Color zigZagColor)
        {
            if (cacheZigZagUTC_TL_BarCount_Fibs_v5_1 != null)
                for (int idx = 0; idx < cacheZigZagUTC_TL_BarCount_Fibs_v5_1.Length; idx++)
                    if (cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].Show == show && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].ShowBracketofFlatRays == showBracketofFlatRays && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].ShowDiagonalRays == showDiagonalRays && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].ShowFibCircle == showFibCircle && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].ShowFibExtend == showFibExtend && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].ShowFibRetrace == showFibRetrace && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].ShowFibRetraceInProgress == showFibRetraceInProgress && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].ShowFibTime == showFibTime && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].Span == span && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].UseHiLo == useHiLo && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].ZigZagColor == zigZagColor && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].EqualsInput(input))
                        return cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx];

            lock (checkZigZagUTC_TL_BarCount_Fibs_v5_1)
            {
                checkZigZagUTC_TL_BarCount_Fibs_v5_1.Show = show;
                show = checkZigZagUTC_TL_BarCount_Fibs_v5_1.Show;
                checkZigZagUTC_TL_BarCount_Fibs_v5_1.ShowBracketofFlatRays = showBracketofFlatRays;
                showBracketofFlatRays = checkZigZagUTC_TL_BarCount_Fibs_v5_1.ShowBracketofFlatRays;
                checkZigZagUTC_TL_BarCount_Fibs_v5_1.ShowDiagonalRays = showDiagonalRays;
                showDiagonalRays = checkZigZagUTC_TL_BarCount_Fibs_v5_1.ShowDiagonalRays;
                checkZigZagUTC_TL_BarCount_Fibs_v5_1.ShowFibCircle = showFibCircle;
                showFibCircle = checkZigZagUTC_TL_BarCount_Fibs_v5_1.ShowFibCircle;
                checkZigZagUTC_TL_BarCount_Fibs_v5_1.ShowFibExtend = showFibExtend;
                showFibExtend = checkZigZagUTC_TL_BarCount_Fibs_v5_1.ShowFibExtend;
                checkZigZagUTC_TL_BarCount_Fibs_v5_1.ShowFibRetrace = showFibRetrace;
                showFibRetrace = checkZigZagUTC_TL_BarCount_Fibs_v5_1.ShowFibRetrace;
                checkZigZagUTC_TL_BarCount_Fibs_v5_1.ShowFibRetraceInProgress = showFibRetraceInProgress;
                showFibRetraceInProgress = checkZigZagUTC_TL_BarCount_Fibs_v5_1.ShowFibRetraceInProgress;
                checkZigZagUTC_TL_BarCount_Fibs_v5_1.ShowFibTime = showFibTime;
                showFibTime = checkZigZagUTC_TL_BarCount_Fibs_v5_1.ShowFibTime;
                checkZigZagUTC_TL_BarCount_Fibs_v5_1.Span = span;
                span = checkZigZagUTC_TL_BarCount_Fibs_v5_1.Span;
                checkZigZagUTC_TL_BarCount_Fibs_v5_1.UseHiLo = useHiLo;
                useHiLo = checkZigZagUTC_TL_BarCount_Fibs_v5_1.UseHiLo;
                checkZigZagUTC_TL_BarCount_Fibs_v5_1.ZigZagColor = zigZagColor;
                zigZagColor = checkZigZagUTC_TL_BarCount_Fibs_v5_1.ZigZagColor;

                if (cacheZigZagUTC_TL_BarCount_Fibs_v5_1 != null)
                    for (int idx = 0; idx < cacheZigZagUTC_TL_BarCount_Fibs_v5_1.Length; idx++)
                        if (cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].Show == show && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].ShowBracketofFlatRays == showBracketofFlatRays && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].ShowDiagonalRays == showDiagonalRays && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].ShowFibCircle == showFibCircle && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].ShowFibExtend == showFibExtend && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].ShowFibRetrace == showFibRetrace && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].ShowFibRetraceInProgress == showFibRetraceInProgress && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].ShowFibTime == showFibTime && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].Span == span && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].UseHiLo == useHiLo && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].ZigZagColor == zigZagColor && cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx].EqualsInput(input))
                            return cacheZigZagUTC_TL_BarCount_Fibs_v5_1[idx];

                ZigZagUTC_TL_BarCount_Fibs_v5_1 indicator = new ZigZagUTC_TL_BarCount_Fibs_v5_1();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.Show = show;
                indicator.ShowBracketofFlatRays = showBracketofFlatRays;
                indicator.ShowDiagonalRays = showDiagonalRays;
                indicator.ShowFibCircle = showFibCircle;
                indicator.ShowFibExtend = showFibExtend;
                indicator.ShowFibRetrace = showFibRetrace;
                indicator.ShowFibRetraceInProgress = showFibRetraceInProgress;
                indicator.ShowFibTime = showFibTime;
                indicator.Span = span;
                indicator.UseHiLo = useHiLo;
                indicator.ZigZagColor = zigZagColor;
                Indicators.Add(indicator);
                indicator.SetUp();

                ZigZagUTC_TL_BarCount_Fibs_v5_1[] tmp = new ZigZagUTC_TL_BarCount_Fibs_v5_1[cacheZigZagUTC_TL_BarCount_Fibs_v5_1 == null ? 1 : cacheZigZagUTC_TL_BarCount_Fibs_v5_1.Length + 1];
                if (cacheZigZagUTC_TL_BarCount_Fibs_v5_1 != null)
                    cacheZigZagUTC_TL_BarCount_Fibs_v5_1.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheZigZagUTC_TL_BarCount_Fibs_v5_1 = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// ZigZag indicator from the Ultimate Trading Guide by Hill, Pruitt, and Hill. A swing high occurs when price takes out the low of the highest bar. A swing low occurs when price takes out the high of the lowest bar. Span setting determines how many bars to look back for highest highs and lowest lows; i.e. span=5 on daily bars find weekly swings.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.ZigZagUTC_TL_BarCount_Fibs_v5_1 ZigZagUTC_TL_BarCount_Fibs_v5_1(int show, bool showBracketofFlatRays, bool showDiagonalRays, bool showFibCircle, bool showFibExtend, bool showFibRetrace, bool showFibRetraceInProgress, bool showFibTime, int span, bool useHiLo, Color zigZagColor)
        {
            return _indicator.ZigZagUTC_TL_BarCount_Fibs_v5_1(Input, show, showBracketofFlatRays, showDiagonalRays, showFibCircle, showFibExtend, showFibRetrace, showFibRetraceInProgress, showFibTime, span, useHiLo, zigZagColor);
        }

        /// <summary>
        /// ZigZag indicator from the Ultimate Trading Guide by Hill, Pruitt, and Hill. A swing high occurs when price takes out the low of the highest bar. A swing low occurs when price takes out the high of the lowest bar. Span setting determines how many bars to look back for highest highs and lowest lows; i.e. span=5 on daily bars find weekly swings.
        /// </summary>
        /// <returns></returns>
        public Indicator.ZigZagUTC_TL_BarCount_Fibs_v5_1 ZigZagUTC_TL_BarCount_Fibs_v5_1(Data.IDataSeries input, int show, bool showBracketofFlatRays, bool showDiagonalRays, bool showFibCircle, bool showFibExtend, bool showFibRetrace, bool showFibRetraceInProgress, bool showFibTime, int span, bool useHiLo, Color zigZagColor)
        {
            return _indicator.ZigZagUTC_TL_BarCount_Fibs_v5_1(input, show, showBracketofFlatRays, showDiagonalRays, showFibCircle, showFibExtend, showFibRetrace, showFibRetraceInProgress, showFibTime, span, useHiLo, zigZagColor);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// ZigZag indicator from the Ultimate Trading Guide by Hill, Pruitt, and Hill. A swing high occurs when price takes out the low of the highest bar. A swing low occurs when price takes out the high of the lowest bar. Span setting determines how many bars to look back for highest highs and lowest lows; i.e. span=5 on daily bars find weekly swings.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.ZigZagUTC_TL_BarCount_Fibs_v5_1 ZigZagUTC_TL_BarCount_Fibs_v5_1(int show, bool showBracketofFlatRays, bool showDiagonalRays, bool showFibCircle, bool showFibExtend, bool showFibRetrace, bool showFibRetraceInProgress, bool showFibTime, int span, bool useHiLo, Color zigZagColor)
        {
            return _indicator.ZigZagUTC_TL_BarCount_Fibs_v5_1(Input, show, showBracketofFlatRays, showDiagonalRays, showFibCircle, showFibExtend, showFibRetrace, showFibRetraceInProgress, showFibTime, span, useHiLo, zigZagColor);
        }

        /// <summary>
        /// ZigZag indicator from the Ultimate Trading Guide by Hill, Pruitt, and Hill. A swing high occurs when price takes out the low of the highest bar. A swing low occurs when price takes out the high of the lowest bar. Span setting determines how many bars to look back for highest highs and lowest lows; i.e. span=5 on daily bars find weekly swings.
        /// </summary>
        /// <returns></returns>
        public Indicator.ZigZagUTC_TL_BarCount_Fibs_v5_1 ZigZagUTC_TL_BarCount_Fibs_v5_1(Data.IDataSeries input, int show, bool showBracketofFlatRays, bool showDiagonalRays, bool showFibCircle, bool showFibExtend, bool showFibRetrace, bool showFibRetraceInProgress, bool showFibTime, int span, bool useHiLo, Color zigZagColor)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.ZigZagUTC_TL_BarCount_Fibs_v5_1(input, show, showBracketofFlatRays, showDiagonalRays, showFibCircle, showFibExtend, showFibRetrace, showFibRetraceInProgress, showFibTime, span, useHiLo, zigZagColor);
        }
    }
}
#endregion
