#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using System.Collections.Generic;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

/*
changelog:

v2.0:
- combined different depths with same price (good for stocks)
- removed highlighting of 3rd cumulative depth price (complications with combining depths)
- removed clearing DOM when scrolling back, if DOM is out of sync, can just refresh by pressing F5
- realigned position of smaller bars better with chart price
*/

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    [Description("Speculator Ed's Level 2 indicator inspired from John Thom's jtRealStats indicator.  Press F5 to refresh indicators if level 2 becomes out of sync. edsfreedom.blogspot.com")]
    public class EdsLevel2 : Indicator
    {
		public override string ToString()
		{
			return "Eds_Level2_v2.0";
		}
		
        #region Variables
		private	List<DOMRow> askRows = new List<DOMRow>();
		private	List<DOMRow> bidRows = new List<DOMRow>();
		private List<DOMRow> oneDOMRow = null;
		private Dictionary<double, int> askRowsSorted = new Dictionary<double, int>();
		private Dictionary<double, int> bidRowsSorted = new Dictionary<double, int>();
		
		private Font textFont;
		private StringFormat stringFormat;
		private Color textColor = Color.Black;
		private Brush textBrush;
		private Color textColor2 = Color.Gray;
		private Brush textBrush2;
		private Color histAskBarColor = Color.Plum;
		private Brush histAskBarBrush;
		private Color histBidBarColor = Color.LightGreen;
		private Brush histBidBarBrush;
		private Rectangle rect;
		private int x;
		private int y;
		private int totalY;
		private double maxMinusMin;
		private int barLength;
		private int barYOffset;
		private double barLengthScale = 1;
		private int barHeight = 1;
		private bool isCramped = false;
		
		private int histBarOpacity = 60;
		private DateTime lastRefresh = DateTime.MinValue;
		private int refreshDelay = 300;

		private int totalAskVolume = 0;
		private int totalBidVolume = 0;
		private int DOMLargestVolume = 1;
		private int i = 0;
		
		private int rowVolume = 0;
		private int totalRowVolume = 0;
		private string rowVolumeString = "";
		private string totalRowVolumeString = "";
		#endregion

		#region Properties
		[XmlIgnore()]
		[Description("Histogram opacity percentage.")]	
		[Category("Level 2 Histogram")]
		[NinjaTrader.Gui.Design.DisplayName("Bar Opacity")]		
		public int HistogramOpacity
		{
			get{ return histBarOpacity; }
			set{ histBarOpacity = Math.Max(Math.Min(value, 100), 0); }
		}
		
		[XmlIgnore()]
		[Description("Histogram ask bar color.")]	
		[Category("Level 2 Histogram")]
		[NinjaTrader.Gui.Design.DisplayName("Ask Bar Color")]		
		public Color HistogramAskBarColor
		{
			get{ return histAskBarColor; }
			set{ histAskBarColor = value; }
		}
		
		[XmlIgnore()]
		[Description("Histogram bid bar color.")]	
		[Category("Level 2 Histogram")]
		[NinjaTrader.Gui.Design.DisplayName("Bid Bar Color")]		
		public Color HistogramBidBarColor
		{
			get{ return histBidBarColor; }
			set{ histBidBarColor = value; }
		}
		
		[XmlIgnore()]
		[Description("Level 2 refresh delay in milliseconds to avoid overloading CPU.")]	
		[Category("Level 2 Histogram")]
		[NinjaTrader.Gui.Design.DisplayName("Refresh Delay")]		
		public int Level2RefreshDelay
		{
			get{ return refreshDelay; }
			set{ refreshDelay = value; }
		}
        #endregion
		
		#region Initialize
        protected override void Initialize()
        {
            CalculateOnBarClose	= true;
            Overlay				= true;
			DrawOnPricePanel	= true;
			PaintPriceMarkers 	= false;
            PriceTypeSupported	= false;
			PlotsConfigurable 	= false;
			
			textFont = new Font(FontFamily.GenericSansSerif, 9, FontStyle.Regular, GraphicsUnit.Pixel);
			stringFormat = new StringFormat();
			stringFormat.Alignment = StringAlignment.Far;
			textBrush = new SolidBrush(textColor);
			textBrush2 = new SolidBrush(textColor2);
			histAskBarBrush = new SolidBrush(Color.FromArgb(histBarOpacity * 255 / 100, histAskBarColor));
			histBidBarBrush = new SolidBrush(Color.FromArgb(histBarOpacity * 255 / 100, histBidBarColor));
		}
		#endregion
	
		#region On Event Updates
        protected override void OnBarUpdate()
        {
        }
		
		protected override void OnMarketDepth(MarketDepthEventArgs e)
        {
			// Checks to see if the Market Data is of the Ask type
			if (e.MarketDataType == MarketDataType.Ask)
				oneDOMRow = askRows;
			
			// Checks to see if the Market Data is of the Bid type
			else if (e.MarketDataType == MarketDataType.Bid)
				oneDOMRow = bidRows;
			
			if (oneDOMRow == null)
				return;

			// Checks to see if the action taken by the Ask data was an insertion into the ladder
			if (e.Operation == Operation.Insert)
				oneDOMRow.Insert(e.Position, new DOMRow(e.Price, (int) e.Volume));
			
			/* Checks to see if the action taken by the Ask data was a removal of itself from the ladder
			Note: Due to the multi threaded architecture of the NT core, race conditions could occur
			-> check if e.Position is within valid range */
			else if (e.Operation == Operation.Remove && e.Position < oneDOMRow.Count)
				oneDOMRow.RemoveAt(e.Position);
			
			/* Checks to see if the action taken by the Ask data was to update a data already on the ladder
			Note: Due to the multi threaded architecture of the NT core, race conditions could occur
			-> check if e.Position is within valid range */
			else if (e.Operation == Operation.Update && e.Position < oneDOMRow.Count)
			{
				oneDOMRow[e.Position].Price = e.Price;
				oneDOMRow[e.Position].Volume = (int) e.Volume;
			}
			
			try	{
			if (DateTime.Now > lastRefresh.AddMilliseconds(refreshDelay))
			{
				ChartControl.Refresh();
				lastRefresh = DateTime.Now;
			}
			} catch {}
        }
		#endregion
		
		#region Draw Plot
		public override void Plot(Graphics graphics, Rectangle bounds, double min, double max)
		{	
			// if right-most bar is not a real-time updating bar, then do not display level 2 histogram
			if (ChartControl.LastBarPainted < Bars.Count - 1)
				return;
			
			totalAskVolume = 0;
			totalBidVolume = 0;
			
			x = ChartControl.CanvasRight;
			totalY = bounds.Y + bounds.Height;
			maxMinusMin = ChartControl.MaxMinusMin(max, min);

			// find largest volume, copy arrays, and combine depths of same price
			DOMLargestVolume = 1;
			askRowsSorted.Clear();
			bidRowsSorted.Clear();
			for (i = 0; i < askRows.Count; i++)
			{
				if (askRowsSorted.ContainsKey(askRows[i].Price) == false)
					askRowsSorted.Add(askRows[i].Price, askRows[i].Volume);
				else
					askRowsSorted[askRows[i].Price] += askRows[i].Volume;
				if (askRowsSorted[askRows[i].Price] > DOMLargestVolume)
					DOMLargestVolume = askRowsSorted[askRows[i].Price];
			}
			for (i = 0; i < bidRows.Count; i++)
			{
				if (bidRowsSorted.ContainsKey(bidRows[i].Price) == false)
					bidRowsSorted.Add(bidRows[i].Price, bidRows[i].Volume);
				else
					bidRowsSorted[bidRows[i].Price] += bidRows[i].Volume;
				if (bidRowsSorted[bidRows[i].Price] > DOMLargestVolume)
					DOMLargestVolume = bidRowsSorted[bidRows[i].Price];
			}
			barLengthScale = 100.0 / DOMLargestVolume;
			
			// if cramped, draw one histogram bar per two depths
			isCramped = (((totalY - (int) (((min) / maxMinusMin) * bounds.Height)) -
						  (totalY - (int) (((min + TickSize) / maxMinusMin) * bounds.Height))) < 8);

			barHeight = isCramped ? 3 : 7;
			barYOffset = isCramped ? 5 : 3;
			
			foreach (KeyValuePair<double, int> key in askRowsSorted)
			{
				if (key.Value > 0)
				{
					y = totalY - ((int) ((( key.Key - min) / maxMinusMin) * bounds.Height)) - 7;
					totalAskVolume += key.Value;
					barLength = (int) (key.Value * barLengthScale);
					rect = new Rectangle(x - barLength, y + barYOffset, barLength, barHeight);
					graphics.FillRectangle(histAskBarBrush, rect);
					
					if (isCramped == false)
					{
						graphics.DrawString(RowVolumeString(key.Value), textFont, textBrush, x, y, stringFormat);
						graphics.DrawString(RowVolumeString(totalAskVolume), textFont, textBrush2, x - 26, y, stringFormat);			
					}
					else
					{
						if (askRows.Count > 0)
							if ((key.Key == askRows[0].Price) || (key.Key == askRows[askRows.Count - 1].Price))
								graphics.DrawString(RowVolumeString(totalAskVolume), textFont, textBrush, x - 26, y, stringFormat);
					}
				}
			}
			
			foreach (KeyValuePair<double, int> key in bidRowsSorted)
			{
				if (key.Value > 0)
				{
					y = totalY - ((int) ((( key.Key - min) / maxMinusMin) * bounds.Height)) - 7;
					totalBidVolume += key.Value;					
					barLength = (int) (key.Value * barLengthScale);
					rect = new Rectangle(x - barLength, y + barYOffset, barLength, barHeight);
					graphics.FillRectangle(histBidBarBrush, rect);
					
					if (isCramped == false)
					{
						graphics.DrawString(RowVolumeString(key.Value), textFont, textBrush, x, y, stringFormat);
						graphics.DrawString(RowVolumeString(totalBidVolume), textFont, textBrush2, x - 26, y, stringFormat);
					}
					else
					{
						if (bidRows.Count > 0)
						{
							if (key.Key == bidRows[0].Price)
								graphics.DrawString(RowVolumeString(totalBidVolume), textFont, textBrush, x, y, stringFormat);								
							else if (key.Key == bidRows[bidRows.Count - 1].Price)
								graphics.DrawString(RowVolumeString(totalBidVolume), textFont, textBrush, x - 26, y, stringFormat);
						}
					}
				}
			}
		}
		#endregion
		
		#region Support Methods
		private string RowVolumeString(int volume)
		{
			if (volume >= 1000000)
				return (volume / 1000000).ToString() + "M";
			else if (volume >= 100000)
				return (volume / 1000).ToString() + "K";
			else
				return volume.ToString();
		}
		
		private class DOMRow
		{
			public double Price;
			public int Volume;

			public DOMRow(double myPrice, int myVolume)
			{
				Price = myPrice;
				Volume = myVolume;
			}
		}
		#endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private EdsLevel2[] cacheEdsLevel2 = null;

        private static EdsLevel2 checkEdsLevel2 = new EdsLevel2();

        /// <summary>
        /// Speculator Ed's Level 2 indicator inspired from John Thom's jtRealStats indicator.  Press F5 to refresh indicators if level 2 becomes out of sync. edsfreedom.blogspot.com
        /// </summary>
        /// <returns></returns>
        public EdsLevel2 EdsLevel2()
        {
            return EdsLevel2(Input);
        }

        /// <summary>
        /// Speculator Ed's Level 2 indicator inspired from John Thom's jtRealStats indicator.  Press F5 to refresh indicators if level 2 becomes out of sync. edsfreedom.blogspot.com
        /// </summary>
        /// <returns></returns>
        public EdsLevel2 EdsLevel2(Data.IDataSeries input)
        {
            if (cacheEdsLevel2 != null)
                for (int idx = 0; idx < cacheEdsLevel2.Length; idx++)
                    if (cacheEdsLevel2[idx].EqualsInput(input))
                        return cacheEdsLevel2[idx];

            lock (checkEdsLevel2)
            {
                if (cacheEdsLevel2 != null)
                    for (int idx = 0; idx < cacheEdsLevel2.Length; idx++)
                        if (cacheEdsLevel2[idx].EqualsInput(input))
                            return cacheEdsLevel2[idx];

                EdsLevel2 indicator = new EdsLevel2();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                Indicators.Add(indicator);
                indicator.SetUp();

                EdsLevel2[] tmp = new EdsLevel2[cacheEdsLevel2 == null ? 1 : cacheEdsLevel2.Length + 1];
                if (cacheEdsLevel2 != null)
                    cacheEdsLevel2.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheEdsLevel2 = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Speculator Ed's Level 2 indicator inspired from John Thom's jtRealStats indicator.  Press F5 to refresh indicators if level 2 becomes out of sync. edsfreedom.blogspot.com
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.EdsLevel2 EdsLevel2()
        {
            return _indicator.EdsLevel2(Input);
        }

        /// <summary>
        /// Speculator Ed's Level 2 indicator inspired from John Thom's jtRealStats indicator.  Press F5 to refresh indicators if level 2 becomes out of sync. edsfreedom.blogspot.com
        /// </summary>
        /// <returns></returns>
        public Indicator.EdsLevel2 EdsLevel2(Data.IDataSeries input)
        {
            return _indicator.EdsLevel2(input);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Speculator Ed's Level 2 indicator inspired from John Thom's jtRealStats indicator.  Press F5 to refresh indicators if level 2 becomes out of sync. edsfreedom.blogspot.com
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.EdsLevel2 EdsLevel2()
        {
            return _indicator.EdsLevel2(Input);
        }

        /// <summary>
        /// Speculator Ed's Level 2 indicator inspired from John Thom's jtRealStats indicator.  Press F5 to refresh indicators if level 2 becomes out of sync. edsfreedom.blogspot.com
        /// </summary>
        /// <returns></returns>
        public Indicator.EdsLevel2 EdsLevel2(Data.IDataSeries input)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.EdsLevel2(input);
        }
    }
}
#endregion
