// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// Linear Regression. The Linear Regression is an indicator that 'predicts' the value of a security's price.
	/// </summary>
	[Description("The Linear Regression Predictor is an indicator that 'predicts' the value of a security's price.")]
	public class LinRegPredictor : Indicator
	{
		#region Variables
		private int				period	= 14;
		private	DataSeries		y;
		private	DataSeries		yHigh;
		private	DataSeries		yLow;
		private	DataSeries		lRHighCloseSeries;
		private	DataSeries		lRLowCloseSeries;
		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Plot(Color.Orange, "WoodiesLRPredictor"));
	
			y					= new DataSeries(this);
			yHigh				= new DataSeries(this);
			yLow				= new DataSeries(this);
			lRHighCloseSeries	= new DataSeries(this);
			lRLowCloseSeries	= new DataSeries(this);
			Overlay				= true;
			PriceTypeSupported	= true;
			CalculateOnBarClose	= false;
		}

		/// <summary>
		/// Called on each bar update event (incoming tick)
		/// </summary>
		protected override void OnBarUpdate()
		{
			double	sumX	= (double) Period * (Period - 1) * 0.5;
			double	divisor = sumX * sumX - (double) Period * Period * (Period - 1) * (2 * Period - 1) / 6;
			double	sumXY	= 0;

			for (int count = 0; count < Period && CurrentBar - count >= 0; count++)
				sumXY += count * Input[count];
			
			y.Set(Input[0]);
			yHigh.Set(Low[0] + Bars.Period.Value*TickSize);
			yLow.Set(High[0] - Bars.Period.Value*TickSize);
			double	slope		= ((double) Period * sumXY - sumX * SUM(y, Period)[0]) / divisor;
			double	intercept	= (SUM(y, Period)[0] - slope * sumX) / Period;
			double	slopeHigh		= ((double) Period * sumXY - (sumX * SUM(y, Period-1)[Math.Min(CurrentBar,1)] + sumX * yHigh[0])) / divisor;
			double	interceptHigh	= ((SUM(y, Period-1)[Math.Min(CurrentBar,1)] + yHigh[0]) - slopeHigh * sumX) / Period;
			double	slopeLow		= ((double) Period * sumXY - (sumX * SUM(y, Period-1)[Math.Min(CurrentBar,1)]+sumX*yLow[0])) / divisor;
			double	interceptLow	= ((SUM(y, Period-1)[Math.Min(CurrentBar,1)] + yLow[0]) - slopeLow * sumX) / Period;
			
			//Value.Set(intercept + slope * (Period - 1));
			lRHighCloseSeries.Set(interceptHigh + slopeHigh * (Period-1));
			lRLowCloseSeries.Set(interceptLow + slopeLow * (Period-1));
		}

		#region Properties
		/// <summary>
		/// </summary>
		[Description("Numbers of bars used for calculations")]
		[Category("Parameters")]
		public int Period
		{
			get { return period; }
			set { period = Math.Max(2, value); }
		}
		[Browsable(false)]
		[XmlIgnore()]
		[Description("LSMA at a High Bar Close")]
		public DataSeries LRHighClose
		{
			get { Update();
			return lRHighCloseSeries; }
		}

		[Browsable(false)]
		[XmlIgnore()]
		[Description("Woodies Pattern, ZLR, FAM, VT, GB, TT, GST")]
		public DataSeries LRLowClose
		{
			get { Update();
			return lRLowCloseSeries; }
		}
	
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private LinRegPredictor[] cacheLinRegPredictor = null;

        private static LinRegPredictor checkLinRegPredictor = new LinRegPredictor();

        /// <summary>
        /// The Linear Regression Predictor is an indicator that 'predicts' the value of a security's price.
        /// </summary>
        /// <returns></returns>
        public LinRegPredictor LinRegPredictor(int period)
        {
            return LinRegPredictor(Input, period);
        }

        /// <summary>
        /// The Linear Regression Predictor is an indicator that 'predicts' the value of a security's price.
        /// </summary>
        /// <returns></returns>
        public LinRegPredictor LinRegPredictor(Data.IDataSeries input, int period)
        {
            if (cacheLinRegPredictor != null)
                for (int idx = 0; idx < cacheLinRegPredictor.Length; idx++)
                    if (cacheLinRegPredictor[idx].Period == period && cacheLinRegPredictor[idx].EqualsInput(input))
                        return cacheLinRegPredictor[idx];

            lock (checkLinRegPredictor)
            {
                checkLinRegPredictor.Period = period;
                period = checkLinRegPredictor.Period;

                if (cacheLinRegPredictor != null)
                    for (int idx = 0; idx < cacheLinRegPredictor.Length; idx++)
                        if (cacheLinRegPredictor[idx].Period == period && cacheLinRegPredictor[idx].EqualsInput(input))
                            return cacheLinRegPredictor[idx];

                LinRegPredictor indicator = new LinRegPredictor();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.Period = period;
                Indicators.Add(indicator);
                indicator.SetUp();

                LinRegPredictor[] tmp = new LinRegPredictor[cacheLinRegPredictor == null ? 1 : cacheLinRegPredictor.Length + 1];
                if (cacheLinRegPredictor != null)
                    cacheLinRegPredictor.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheLinRegPredictor = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// The Linear Regression Predictor is an indicator that 'predicts' the value of a security's price.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.LinRegPredictor LinRegPredictor(int period)
        {
            return _indicator.LinRegPredictor(Input, period);
        }

        /// <summary>
        /// The Linear Regression Predictor is an indicator that 'predicts' the value of a security's price.
        /// </summary>
        /// <returns></returns>
        public Indicator.LinRegPredictor LinRegPredictor(Data.IDataSeries input, int period)
        {
            return _indicator.LinRegPredictor(input, period);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// The Linear Regression Predictor is an indicator that 'predicts' the value of a security's price.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.LinRegPredictor LinRegPredictor(int period)
        {
            return _indicator.LinRegPredictor(Input, period);
        }

        /// <summary>
        /// The Linear Regression Predictor is an indicator that 'predicts' the value of a security's price.
        /// </summary>
        /// <returns></returns>
        public Indicator.LinRegPredictor LinRegPredictor(Data.IDataSeries input, int period)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.LinRegPredictor(input, period);
        }
    }
}
#endregion
