#region Using declarations
using System;
using System.ComponentModel;
using System.Drawing;
using System.Xml.Serialization;
#endregion

// v0.3 If LabelOnPricePanel == true --> labels will be printed on price panel. Set panel = 1 (standard setup)
// If LabelOnPricePanel == false --> set panel other than 1 (labels will be displayed in an indicator panel)
// v0.4 Compatibility with NT7

namespace NinjaTrader.Indicator
{
    /// <summary>
    /// </summary>
    [Gui.Design.DisplayName("BarOfSession")]
    [Description("Indicator to label bars with numbers since start of session.")]
    public class BarOfSession : Indicator
    {
        #region Variables
            private int offset = 3; // Default setting for Offset
			private Color textColor = Color.DimGray;
			private Font textFont;
			private int textSize = 6; // Default text size
			private Placement numberPosition = Placement.Bottom;
			private bool labelOnPricePanel = true; // Label on Price Panel
			private double position;
			private int interval = 5;
			private int barCounter	= 0; 
			private int labelold	= 0; 
			private int labelnew	= 0; 
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
            CalculateOnBarClose	= true;
            Overlay				= true;
            PriceTypeSupported	= false;
			textFont = new Font("Arial", textSize, FontStyle.Regular);
			DrawOnPricePanel = labelOnPricePanel;
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
			if (CurrentBar == 0)
			{
				return;
			}	
			if (Bars.FirstBarOfSession && FirstTickOfBar)
			{
				barCounter = 0;
				labelold = 0;
				labelnew = 0;
			}
			else 
			{
				barCounter++;
				labelold = labelnew;
				if (labelold + interval == barCounter)
				{
					labelnew = labelold + interval;
					if (numberPosition == Placement.Bottom)
					{
						position = (Low[0]-(TickSize*offset));
						if (DrawOnPricePanel == false)
						{
							position = 0.3;
						}
						else
						{
							position = (Low[0]-(TickSize*offset));
						}
					}
					else
					{
						position = (High[0] +(TickSize*offset));
						if (DrawOnPricePanel == false)
						{
							position = 0.3;
						}
						else
						{
							position = (High[0] +(TickSize*offset));
						}
					}
					DrawText("BarOfSession" + CurrentBar, false, Convert.ToString(labelnew + 1), 0, position, 0, textColor, textFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
				}
			}
        }

        #region Properties

		public enum Placement { Top, Bottom };
		
        [Description("Put numbers at top or bottom of each bar?")]
        [Category("Bar Numbers")]
		[Gui.Design.DisplayNameAttribute("Position")]
        public Placement NumberPosition
        {
            get { return numberPosition; }
            set { numberPosition = value; }
        }
				
		[Description("Text size")]
        [Category("Bar Numbers")]
        public int TextSize
        {
            get { return textSize; }
            set { textSize = Math.Max(1, value); }
        }

		[Description("LabelOnPricePanel true/false. If false Panel has to be other than 1 ")]
        [Category("Bar Numbers")]
        public bool LabelOnPricePanel
        {
            get { return labelOnPricePanel; }
            set { labelOnPricePanel = value; }
        }
		
		[Description("Number of ticks offset from bar")]
        [Category("Bar Numbers")]
        public int Offset
        {
            get { return offset; }
            set { offset = Math.Max(0, value); }
        }
		
		[Description("Interval of bars")]
        [Category("Bar Numbers")]
        public int Interval
        {
            get { return interval; }
            set { interval = Math.Max(0, value); }
        }
				
		[XmlIgnore()]
		[Description("Color of the bar numbers.")]
		[Category("Bar Numbers")]
		[Gui.Design.DisplayNameAttribute("Color")]
		public Color TextColor
		{
			get { return textColor; }
			set { textColor = value; }
		}

		/// <summary>
		/// </summary>
		[Browsable(false)]
		public string TextColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(textColor); }
			set { textColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private BarOfSession[] cacheBarOfSession = null;

        private static BarOfSession checkBarOfSession = new BarOfSession();

        /// <summary>
        /// Indicator to label bars with numbers since start of session.
        /// </summary>
        /// <returns></returns>
        public BarOfSession BarOfSession()
        {
            return BarOfSession(Input);
        }

        /// <summary>
        /// Indicator to label bars with numbers since start of session.
        /// </summary>
        /// <returns></returns>
        public BarOfSession BarOfSession(Data.IDataSeries input)
        {
            if (cacheBarOfSession != null)
                for (int idx = 0; idx < cacheBarOfSession.Length; idx++)
                    if (cacheBarOfSession[idx].EqualsInput(input))
                        return cacheBarOfSession[idx];

            lock (checkBarOfSession)
            {
                if (cacheBarOfSession != null)
                    for (int idx = 0; idx < cacheBarOfSession.Length; idx++)
                        if (cacheBarOfSession[idx].EqualsInput(input))
                            return cacheBarOfSession[idx];

                BarOfSession indicator = new BarOfSession();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                Indicators.Add(indicator);
                indicator.SetUp();

                BarOfSession[] tmp = new BarOfSession[cacheBarOfSession == null ? 1 : cacheBarOfSession.Length + 1];
                if (cacheBarOfSession != null)
                    cacheBarOfSession.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheBarOfSession = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Indicator to label bars with numbers since start of session.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.BarOfSession BarOfSession()
        {
            return _indicator.BarOfSession(Input);
        }

        /// <summary>
        /// Indicator to label bars with numbers since start of session.
        /// </summary>
        /// <returns></returns>
        public Indicator.BarOfSession BarOfSession(Data.IDataSeries input)
        {
            return _indicator.BarOfSession(input);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Indicator to label bars with numbers since start of session.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.BarOfSession BarOfSession()
        {
            return _indicator.BarOfSession(Input);
        }

        /// <summary>
        /// Indicator to label bars with numbers since start of session.
        /// </summary>
        /// <returns></returns>
        public Indicator.BarOfSession BarOfSession(Data.IDataSeries input)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.BarOfSession(input);
        }
    }
}
#endregion
