#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Originally by John Ehlers
    /// </summary>
    [Description("Laguerre Adaptive Filter")]
    public class LaguerreAdaptiveFilter : Indicator
    {
        #region Variables
		private int			Length	= 20; // Input default
		
		// DataSeries would need less coding, but Arrays, are faster and in this case use far less memory.
		
		private double[]	HH		= new double[] { 0, 0, 0, 0, 0 };
		private double[]	LL		= new double[] { 0, 0, 0, 0, 0 };
		private double[]	DLH		= new double[] { 0, 0, 0, 0, 0 };
		private double[]	Med		= new double[5];
		private double[]	L0		= new double[2];
		private double[]	L1		= new double[2];
		private double[]	L2		= new double[2];
		private double[]	L3		= new double[2];
		private DataSeries	Diff; // Easier to use when the length is not known.
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
            Add(new Plot(Color.FromKnownColor(KnownColor.Blue), PlotStyle.Line, "Filt"));
			Diff				= new DataSeries(this);
			PriceTypeSupported	= true;
			PriceType			= PriceType.Median;
            Overlay				= true;
			CalculateOnBarClose = true;
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
			// We have to move things along the arrays, but only do it once per bar.
			
			if ( FirstTickOfBar )
			{
				for ( int count = 4; count > 0; count-- )
				{
					HH[count] = HH[count-1];
					LL[count] = LL[count-1];
					DLH[count] = DLH[count-1];
				}
				L0[1] = L0[0];
				L1[1] = L1[0];
				L2[1] = L2[0];
				L3[1] = L3[0]; 
			}
			
			if (CurrentBar < Length)
				return;
			
			Diff.Set(Math.Abs(Input[0] - Filt[1]));
			HH[0] = Diff[0];
			LL[0] = Diff[0];			
						
			for (int count = 0; count < Length; count++)
			{
				if (Diff[count] > HH[0])
					HH[0] = Diff[count];
				if (Diff[count] < LL[0])
					LL[0] = Diff[count];
			}
			if (Diff[0] - LL[0] != 0 && HH[0] - LL[0] != 0) 
				DLH[0] = (Diff[0] - LL[0]) / (HH[0] - LL[0]);
			else
				DLH[0] = (0);
			
			for (int count = 0; count < 5; count++)
				Med[count] = DLH[count];
			
			// Find the median.
			
			bool swapped = true;
			double temp;
			while (swapped)
			{
				swapped = false;
				for (int count = 0; count < 4; count++)
				{
					if (Med[count] > Med[count + 1])
					{
						temp = Med[count];
						Med[count] = Med[count + 1];
						Med[count + 1] = temp;
						swapped = true;
					}
				}
			}
			
			// Med[2] will be the median
			
			double alpha = Med[2];
							
			L0[0] = alpha * Input[0] + (1 - alpha) * L0[1];
			L1[0] = - (1 - alpha) * L0[0] + L0[1] +(1 - alpha) * L1[1];
			L2[0] = - (1 - alpha) * L1[0] + L1[1] +(1 - alpha) * L2[1];
			L3[0] = - (1 - alpha) * L2[0] + L2[1] +(1 - alpha) * L3[1];
			
			Filt.Set((L0[0] + 2 * L1[0] + 2 * L2[0] + L3[0]) / 6);
        }

        #region Properties
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries Filt
        {
            get { return Values[0]; }
        }
		
		/// <summary>
		/// </summary>
		[Description("Numbers of bars used for calculations")]
		[GridCategory("Parameters")]		
		public int length
		{
			get { return Length; }
			set { Length = Math.Max(1, value); }
		}

        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private LaguerreAdaptiveFilter[] cacheLaguerreAdaptiveFilter = null;

        private static LaguerreAdaptiveFilter checkLaguerreAdaptiveFilter = new LaguerreAdaptiveFilter();

        /// <summary>
        /// Laguerre Adaptive Filter
        /// </summary>
        /// <returns></returns>
        public LaguerreAdaptiveFilter LaguerreAdaptiveFilter(int length)
        {
            return LaguerreAdaptiveFilter(Input, length);
        }

        /// <summary>
        /// Laguerre Adaptive Filter
        /// </summary>
        /// <returns></returns>
        public LaguerreAdaptiveFilter LaguerreAdaptiveFilter(Data.IDataSeries input, int length)
        {
            if (cacheLaguerreAdaptiveFilter != null)
                for (int idx = 0; idx < cacheLaguerreAdaptiveFilter.Length; idx++)
                    if (cacheLaguerreAdaptiveFilter[idx].length == length && cacheLaguerreAdaptiveFilter[idx].EqualsInput(input))
                        return cacheLaguerreAdaptiveFilter[idx];

            lock (checkLaguerreAdaptiveFilter)
            {
                checkLaguerreAdaptiveFilter.length = length;
                length = checkLaguerreAdaptiveFilter.length;

                if (cacheLaguerreAdaptiveFilter != null)
                    for (int idx = 0; idx < cacheLaguerreAdaptiveFilter.Length; idx++)
                        if (cacheLaguerreAdaptiveFilter[idx].length == length && cacheLaguerreAdaptiveFilter[idx].EqualsInput(input))
                            return cacheLaguerreAdaptiveFilter[idx];

                LaguerreAdaptiveFilter indicator = new LaguerreAdaptiveFilter();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.length = length;
                Indicators.Add(indicator);
                indicator.SetUp();

                LaguerreAdaptiveFilter[] tmp = new LaguerreAdaptiveFilter[cacheLaguerreAdaptiveFilter == null ? 1 : cacheLaguerreAdaptiveFilter.Length + 1];
                if (cacheLaguerreAdaptiveFilter != null)
                    cacheLaguerreAdaptiveFilter.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheLaguerreAdaptiveFilter = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Laguerre Adaptive Filter
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.LaguerreAdaptiveFilter LaguerreAdaptiveFilter(int length)
        {
            return _indicator.LaguerreAdaptiveFilter(Input, length);
        }

        /// <summary>
        /// Laguerre Adaptive Filter
        /// </summary>
        /// <returns></returns>
        public Indicator.LaguerreAdaptiveFilter LaguerreAdaptiveFilter(Data.IDataSeries input, int length)
        {
            return _indicator.LaguerreAdaptiveFilter(input, length);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Laguerre Adaptive Filter
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.LaguerreAdaptiveFilter LaguerreAdaptiveFilter(int length)
        {
            return _indicator.LaguerreAdaptiveFilter(Input, length);
        }

        /// <summary>
        /// Laguerre Adaptive Filter
        /// </summary>
        /// <returns></returns>
        public Indicator.LaguerreAdaptiveFilter LaguerreAdaptiveFilter(Data.IDataSeries input, int length)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.LaguerreAdaptiveFilter(input, length);
        }
    }
}
#endregion
