#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Draw a line at user-defined price level.
    /// </summary>
    [Description("Draw a line at user-defined price level.")]
    public class PriceLevel : Indicator
    {
        #region Variables
    
            private int fontSize = 10;
            private bool hidden = false; 
            private double priceValue = 0;
			private PriceTextPosition	textPosition	= PriceTextPosition.AtRightEdge;  
   			private int alertRearmSeconds = 10;
			private Color alertBackColor = Color.Blue;
			private Color alertForeColor = Color.Yellow;
			private bool alertBool = false; 
		
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {

			Add(new Plot(Color.DeepPink, "PriceLevel"));
            
			PaintPriceMarkers = false;
			CalculateOnBarClose = false;
			Overlay				= true;
			
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
						
			if (Hidden==false)
			{
			Price.Set(PriceValue);	
			
				switch (TextPosition)
				{	
					case PriceTextPosition.AtRightEdge:
						{
							DrawText("Price", true, "$" + PriceValue.ToString(), 0, PriceValue, FontSize, Plots[0].Pen.Color,new Font("Arial", FontSize),StringAlignment.Far ,Color.Empty, Color.Empty,0);
							break;
						}
						
					case PriceTextPosition.AtLeftEdge:
						{
							DrawText("Price", true, "$" + PriceValue.ToString(), CurrentBar - ChartControl.FirstBarPainted, PriceValue, FontSize, Plots[0].Pen.Color, new Font("Arial", FontSize),StringAlignment.Near ,Color.Empty, Color.Empty,0);
							break;
						}	
					
					case PriceTextPosition.Center:
						{
							DrawText("Price", true, "$" + PriceValue.ToString(), (CurrentBar - ChartControl.FirstBarPainted) / 2, PriceValue, FontSize, Plots[0].Pen.Color, new Font("Arial", FontSize),StringAlignment.Near ,Color.Empty, Color.Empty,0);
							break;
						}		
						
					case PriceTextPosition.DoNotShow:
						{
							DrawText("Price", false, "", (CurrentBar - ChartControl.FirstBarPainted) / 2, PriceValue, FontSize, Plots[0].Pen.Color, new Font("Arial", FontSize),StringAlignment.Near ,Color.Empty, Color.Empty,0);
							break;
						}		
				}
			}
			
			if (AlertBool)
			{
				if (CrossAbove(Close,PriceValue,1))
					Alert("CrossAbove", NinjaTrader.Cbi.Priority.Medium, Instrument.FullName + " crossed above $" + PriceValue.ToString(), "Alert4.wav", AlertRearmSeconds, AlertBackColor, AlertForeColor);
			
				if (CrossBelow(Close,PriceValue,1))
					Alert("CrossBelow", NinjaTrader.Cbi.Priority.Medium, Instrument.FullName + " crossed below $" + PriceValue.ToString(), "Alert4.wav", AlertRearmSeconds, AlertBackColor, AlertForeColor);
			}
        }

        #region Properties
		
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Price
		{
			get { return Values[0]; }
		}
		
        [Description("Text font size.")]
        [GridCategory("Parameters")]
        public int FontSize
        {
            get { return fontSize; }
            set { fontSize = Math.Max(1, value); }
        }

        [Gui.Design.DisplayNameAttribute("Hidden?")]
        [Description("Hide plot?")]
		[Category("Parameters")]
        public bool Hidden
        {
            get { return hidden; }
            set {hidden = value; }
        }

        [Description("Prive Level value.")]
        [GridCategory("Parameters")]
        public double PriceValue
        {
            get { return priceValue; }
            set { priceValue = Math.Max(0, value); }
        }
		
		[Description("Text position.")]
		[Category("Parameters")]
		public PriceTextPosition TextPosition
		{
			get { return textPosition; }
			set { textPosition = value; }
		}
		
		[Description("Seconds to wait until alert rearms")]
        [Category("Alert")]
        public int AlertRearmSeconds
        {
            get { return alertRearmSeconds; }
            set { alertRearmSeconds = Math.Max(1, value); }
        }
		
		[XmlIgnore()]
        [Description("Alert back color.")]
        [Category("Alert")]
        public Color AlertBackColor
        {
            get { return alertBackColor; }
            set { alertBackColor = value; }
        }
		
		// Serialize our Color object
		[Browsable(false)]
		public string BorderColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(alertBackColor); }
			set { alertBackColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}

		[XmlIgnore()]
		[Description("Alert fore color.")]
        [Category("Alert")]
        public Color AlertForeColor
        {
            get { return alertForeColor; }
            set {alertForeColor = value; }
        }
		
		[Browsable(false)]
		public string FillColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(alertForeColor); }
			set { alertForeColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		[Gui.Design.DisplayNameAttribute("Alert?")]
        [Description("Send an alert to Alerts window when a the current price crosses above or below the price value.")]
		[Category("Alert")]
        public bool AlertBool
        {
            get { return alertBool; }
            set { alertBool = value; }
        }
		
        #endregion
    }
}
public enum PriceTextPosition
{
		AtRightEdge,
		Center,
		AtLeftEdge,
		DoNotShow,
}
#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private PriceLevel[] cachePriceLevel = null;

        private static PriceLevel checkPriceLevel = new PriceLevel();

        /// <summary>
        /// Draw a line at user-defined price level.
        /// </summary>
        /// <returns></returns>
        public PriceLevel PriceLevel(int fontSize, bool hidden, double priceValue, PriceTextPosition textPosition)
        {
            return PriceLevel(Input, fontSize, hidden, priceValue, textPosition);
        }

        /// <summary>
        /// Draw a line at user-defined price level.
        /// </summary>
        /// <returns></returns>
        public PriceLevel PriceLevel(Data.IDataSeries input, int fontSize, bool hidden, double priceValue, PriceTextPosition textPosition)
        {
            if (cachePriceLevel != null)
                for (int idx = 0; idx < cachePriceLevel.Length; idx++)
                    if (cachePriceLevel[idx].FontSize == fontSize && cachePriceLevel[idx].Hidden == hidden && Math.Abs(cachePriceLevel[idx].PriceValue - priceValue) <= double.Epsilon && cachePriceLevel[idx].TextPosition == textPosition && cachePriceLevel[idx].EqualsInput(input))
                        return cachePriceLevel[idx];

            lock (checkPriceLevel)
            {
                checkPriceLevel.FontSize = fontSize;
                fontSize = checkPriceLevel.FontSize;
                checkPriceLevel.Hidden = hidden;
                hidden = checkPriceLevel.Hidden;
                checkPriceLevel.PriceValue = priceValue;
                priceValue = checkPriceLevel.PriceValue;
                checkPriceLevel.TextPosition = textPosition;
                textPosition = checkPriceLevel.TextPosition;

                if (cachePriceLevel != null)
                    for (int idx = 0; idx < cachePriceLevel.Length; idx++)
                        if (cachePriceLevel[idx].FontSize == fontSize && cachePriceLevel[idx].Hidden == hidden && Math.Abs(cachePriceLevel[idx].PriceValue - priceValue) <= double.Epsilon && cachePriceLevel[idx].TextPosition == textPosition && cachePriceLevel[idx].EqualsInput(input))
                            return cachePriceLevel[idx];

                PriceLevel indicator = new PriceLevel();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.FontSize = fontSize;
                indicator.Hidden = hidden;
                indicator.PriceValue = priceValue;
                indicator.TextPosition = textPosition;
                Indicators.Add(indicator);
                indicator.SetUp();

                PriceLevel[] tmp = new PriceLevel[cachePriceLevel == null ? 1 : cachePriceLevel.Length + 1];
                if (cachePriceLevel != null)
                    cachePriceLevel.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cachePriceLevel = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Draw a line at user-defined price level.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.PriceLevel PriceLevel(int fontSize, bool hidden, double priceValue, PriceTextPosition textPosition)
        {
            return _indicator.PriceLevel(Input, fontSize, hidden, priceValue, textPosition);
        }

        /// <summary>
        /// Draw a line at user-defined price level.
        /// </summary>
        /// <returns></returns>
        public Indicator.PriceLevel PriceLevel(Data.IDataSeries input, int fontSize, bool hidden, double priceValue, PriceTextPosition textPosition)
        {
            return _indicator.PriceLevel(input, fontSize, hidden, priceValue, textPosition);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Draw a line at user-defined price level.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.PriceLevel PriceLevel(int fontSize, bool hidden, double priceValue, PriceTextPosition textPosition)
        {
            return _indicator.PriceLevel(Input, fontSize, hidden, priceValue, textPosition);
        }

        /// <summary>
        /// Draw a line at user-defined price level.
        /// </summary>
        /// <returns></returns>
        public Indicator.PriceLevel PriceLevel(Data.IDataSeries input, int fontSize, bool hidden, double priceValue, PriceTextPosition textPosition)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.PriceLevel(input, fontSize, hidden, priceValue, textPosition);
        }
    }
}
#endregion
