#region Using declarations
using System;
using System.IO;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    // <summary>
  	//
	// Written by Ben L. at sbgtrading@yahoo.com
	// Modified by Laurent M. at ryker@wanadoo.fr
	//
	
	//	Description of the "ProfileType" parameter:
	//		I've given the option of creating the profile in 3 different ways:
	//			1)  TPO - This method disregards volume altogether, and gives a single hit to each price in the range of the bar.
	//						e.g.  A 5-minute bar has a range of 1.5 points with a High at 1534 and a Low at 1532.5, then
	//						1 contract (or "hit") is loaded to the seven prices in that range: 1532.50, 1532.75, 1533.0, 1533.25, 1533.50, 1533.75, and 1534
	//			2)  VPO - This method distribues the volume of a bar over the price range of the bar.
	//						e.g.  A 5-minute bar has a volume of 280 and a range of 1.5 points with a High at 1534 and a Low at 1532.5, then
	//						40 contracts (=280/7) are loaded to each of the seven prices in that range: 1532.50, 1532.75, 1533.0, 1533.25, 1533.50, 1533.75, and 1534
	//
	// </summary>
    [Description("MP Values as described in Min Over Markets")]
    public class MPValues : Indicator
    {
        #region Variables
        // Wizard generated variables
		private bool inclWeekendVol=false;
		private int openHour=14,openMinute=30;
		private double pctOfVolumeInVA = 0.7;
		private string profileType="TPO";
		private double sessionLengthInHours=6.50;

		// User defined variables (add any user defined variables below)
		private string path = Cbi.Core.UserDataDir.ToString()+"test.txt";
		private double[,] PriceHitsArray = new double[2,1000]; // Count Array
		private double dSessionHigh, dSessionLow;
		private string FormatString;
		private double dVAH=0.0,dVAL=0.0,dPOC=0.0;
		private DateTime StartTime,EndTime;
		private bool InitializeEndTime;
		private int LastBarOfSession;
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
            Add(new Plot(Color.Red, PlotStyle.Line, "VAH"));
            Add(new Plot(Color.Lime, PlotStyle.Line, "VAL"));
            Add(new Plot(Color.Blue, PlotStyle.Line, "POC"));
			AutoScale			= false;
            CalculateOnBarClose	= true;
            Overlay				= true;
            PriceTypeSupported	= false;

			InitializeEndTime = true;
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {			
			if(TickSize.ToString().Length<=2) FormatString="0";
			if(TickSize.ToString().Length==3) FormatString="0.0";
			if(TickSize.ToString().Length==4) FormatString="0.00";
			if(TickSize.ToString().Length==5) FormatString="0.000";
			if(TickSize.ToString().Length==6) FormatString="0.0000";
			int i;
			
			if(CurrentBar<10)	return;

			if(InitializeEndTime)
			{	if(Time[0].CompareTo(StartTime)>=0)
				{	InitializeEndTime=false;
					StartTime = new DateTime(Time[0].Year,Time[0].Month,Time[0].Day,OpenHour,OpenMinute,0,0,DateTimeKind.Utc);
					if(sessionLengthInHours>=24.0) EndTime = StartTime.AddHours(24.0-1.0/60.0);
					else EndTime = StartTime.AddHours(sessionLengthInHours);
				}
			}
			
			// On a new day
			if(!InitializeEndTime && Time[1].CompareTo(EndTime)<=0 && Time[0].CompareTo(EndTime)>0)
			{	DetermineHighLowOfSession(StartTime);
				if(LastBarOfSession < 0) return;

				StartTime = new DateTime(Time[0].Year,Time[0].Month,Time[0].Day,OpenHour,OpenMinute,0,0,DateTimeKind.Utc);
				if(sessionLengthInHours>=24) EndTime = StartTime.AddHours(24.0-1.0/60.0);
				else EndTime = StartTime.AddHours(sessionLengthInHours);

				// Initialize CountArray
				int iTicksInRange = (int) Math.Round((dSessionHigh-dSessionLow)/TickSize,0) + 1;
				if (iTicksInRange>=1000) Log("Potential data problem in MPValues at "+Time[0].ToString()+" Session H/L: "+dSessionHigh.ToString(FormatString)+" / "+dSessionLow.ToString(FormatString),LogLevel.Warning);
				if (iTicksInRange<0) Log("Potential data problem in MPValues at "+Time[0].ToString()+" Session H/L: "+dSessionHigh.ToString(FormatString)+" / "+dSessionLow.ToString(FormatString),LogLevel.Warning);
				for(i=0; i<1000; i++)
				{ 	PriceHitsArray[0,i]=(i*TickSize+dSessionLow); // Price
					PriceHitsArray[1,i]=0.0; // Volume
				}
				
				// Fill CountArray
				int index=0;
				i=1;
				while (i <= LastBarOfSession) //Accumulate the volume for each previous days bar into PriceVolume array
				{	if(!inclWeekendVol && (Time[i].DayOfWeek==DayOfWeek.Saturday || Time[i].DayOfWeek==DayOfWeek.Sunday))
						i++;
					else
					{
						if (profileType == "TPO") //Time Price Opportunity - disregards volume, only counts number of times prices are touched
						{	double BarH=High[i]; double BarP=Low[i];
							while(BarP<=BarH+TickSize/2.0)
							{	index = (int) Math.Round((BarP-dSessionLow)/TickSize,0);
								PriceHitsArray[1,index] = PriceHitsArray[1,index] + 1;
								BarP = BarP + TickSize;
							}
						}
						if (profileType == "VPO") //Volume Weighted Time Price Opportunity - Disperses the Volume of the bar over the range of the bar so each price touched is weighted with volume
						{	double BarH=High[i]; double BarP=Low[i];
							int TicksInBar = (int) Math.Round((BarH-Low[i])/TickSize+1,0);
							while(BarP<=BarH+TickSize/2.0)
							{	index = (int) Math.Round((BarP-dSessionLow)/TickSize,0);
								PriceHitsArray[1,index] = PriceHitsArray[1,index] + Volume[i]/TicksInBar;
								BarP = BarP + TickSize;
							}
						}
						i++;
					}
				}
				// Calculate MPValues
				CalculateMPValues();
			}
			// Assign values to the indicator
			VAH.Set(dVAH);
			VAL.Set(dVAL);
			POC.Set(dPOC);
			
			StartTime = new DateTime(Time[0].Year,Time[0].Month,Time[0].Day,OpenHour,OpenMinute,0,0,DateTimeKind.Utc);
			if(sessionLengthInHours>=24.0) EndTime = StartTime.AddHours(24.0-1.0/60.0);
			else EndTime = StartTime.AddHours(sessionLengthInHours);
		}

        #region Properties
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries VAH
        {
            get { return Values[0]; }
        }
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries VAL
        {
            get { return Values[1]; }
        }
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries POC
        {
            get { return Values[2]; }
        }

        [Description("Include Weekend Volume")]
        [Category("Parameters")]
        public bool InclWeekendVol
        {
            get { return inclWeekendVol; }
			set { inclWeekendVol = value;}
        }
        [Description("Type of profile TPO,VPO")]
        [Category("Parameters")]
        public string ProfileType
        {
            get { return profileType; }
			set { profileType = value.ToUpper();
				  if(profileType!="TPO" && profileType!="VPO") profileType="TPO";
				}
        }
        [Description("Percent of volume within Value Area")]
        [Category("Parameters")]
        public double PctOfVolumeInVA
        {
            get { return pctOfVolumeInVA; }
			set { pctOfVolumeInVA = Math.Max(0.01,Math.Min(1.0,value));}
        }
        [Description("Market open hour IN 24HR FORMAT")]
        [Category("Parameters")]
        public int OpenHour
        {
            get { return openHour; }
			set { openHour = Math.Max(0,value);}
        }
        [Description("Market open minute")]
        [Category("Parameters")]
        public int OpenMinute
        {
            get { return openMinute; }
			set { openMinute = Math.Max(0,value);}
        }
        [Description("Session length (in hours)")]
        [Category("Parameters")]
        public double SessionLengthInHours
        {
            get { return sessionLengthInHours; }
			set { sessionLengthInHours = value;}
        }
        #endregion
		
		//===================================================================================
		private void DetermineHighLowOfSession(DateTime SessionStartTime)
		{	int i=1; //first bar to check is the bar immediately prior to currentbar
			dSessionHigh = High[i];
			dSessionLow = Low[i];
			LastBarOfSession = -1;
			while(i<CurrentBar-1 && Time[i].CompareTo(SessionStartTime)>0)
			{	if(!inclWeekendVol && (Time[i].DayOfWeek==DayOfWeek.Saturday || Time[i].DayOfWeek==DayOfWeek.Sunday))
					i++;
				else
				{
					if(High[i] > dSessionHigh) dSessionHigh = High[i];
					if(Low[i] < dSessionLow)   dSessionLow  = Low[i];
					LastBarOfSession = i;
					i++;
				}
			}
		}
		//===================================================================================
		private void CalculateMPValues()
		{	int iRowPOC = 0;
			int iBarH = (int) Math.Round((dSessionHigh - dSessionLow) / TickSize,0);
			int iHalfLength = (int) Math.Floor((double)(iBarH + 1) / 2);
			int iMid = iHalfLength;
			int iCountBelow = iHalfLength - 1;
    		int iCountAbove = iHalfLength;
			double dMax = 0.0, dVA, dTPOCount = 0.0; 
			
			// Initialization if array has an odd number of values
    		if (iBarH % 2 == 1)
        	{	iRowPOC = iHalfLength;
				dMax = PriceHitsArray[1, iRowPOC]; // Grab the volume
        		iCountAbove = iCountAbove + 1;
        		iMid = iMid;
				dTPOCount = dMax; 
			}
			// Find the max and its place in the variant
			for(int i=0; i < iHalfLength; i++)
			{	double dAboveV = PriceHitsArray[1,iCountAbove + i], dBelowV = PriceHitsArray[1,iCountBelow - i];
				dTPOCount = dTPOCount + dAboveV + dBelowV;
				if (dAboveV > dMax)
				{	iRowPOC = iCountAbove + i;
					dMax = dAboveV;
				}
				if (dBelowV > dMax)
				{	iRowPOC = iCountBelow - i;
					dMax = dBelowV;
				}
			}
			// VA Calculation
			double dAbove1, dAbove2, dBelow1, dBelow2, dAbove, dBelow;
			int iVAHigh, iVALow;
			dVA = pctOfVolumeInVA * dTPOCount;
			iVAHigh = iRowPOC;
			iVALow = iRowPOC;
			iCountAbove = iRowPOC - 1;
    		iCountBelow = iRowPOC + 1;
			dVA = dVA - PriceHitsArray[1,iRowPOC]; // Substract POC
			while( (dVA > 0) || ( (iCountAbove < 1) && (iCountBelow > iBarH) )) 
			{	// Count the TPO above and below
        		if (iCountAbove < 1) 
				{	dAbove1 = 0;
            		dAbove2 = 0;
				} 
				else if (iCountAbove - 1 < 1)
				{	dAbove1 = PriceHitsArray[1,iCountAbove]; 
            		dAbove2 = 0;
        		} 
				else 
            	{	dAbove1 = PriceHitsArray[1,iCountAbove]; 
            		dAbove2 = PriceHitsArray[1,iCountAbove - 1];
				}
				if (iCountBelow > iBarH)
            	{	dBelow1 = 0;
            		dBelow2 = 0;
				} 
				else if (iCountBelow + 1 > iBarH)
       			{	dBelow1 = PriceHitsArray[1,iCountBelow];
            		dBelow2 = 0;
				} 
				else
				{	dBelow1 = PriceHitsArray[1,iCountBelow];
					dBelow2 = PriceHitsArray[1,iCountBelow + 1];
				}
				dAbove = dAbove1 + dAbove2;
        		dBelow = dBelow1 + dBelow2;
				// Find the greatest
				if ( (dAbove > dBelow) || ( (dAbove == dBelow) && (Math.Abs(iCountAbove - iMid) < Math.Abs(iCountBelow - iMid)) ) )
				{	dVA = dVA - dAbove1;
					iVAHigh = iCountAbove;
					if (dVA > 0)
					{	dVA = dVA - dAbove2;
						iVAHigh = iCountAbove - 1;
					}
					iCountAbove = iCountAbove - 2;
				} 
				else
				{	dVA = dVA - dBelow1;
					iVALow = iCountBelow;
					if (dVA > 0)
					{	dVA = dVA - dBelow2;
						iVALow = iCountBelow + 1;
					}
					iCountBelow = iCountBelow + 2;
				}
			}
			// Set MP Values
			if (iCountAbove < 0) iCountAbove = 0;
			if (iCountBelow > iBarH) iCountBelow = iBarH;
			dPOC = PriceHitsArray[0,iRowPOC];
			dVAH = PriceHitsArray[0,iCountBelow];
			dVAL = PriceHitsArray[0,iCountAbove];
		}
		//===================================================================================

	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private MPValues[] cacheMPValues = null;

        private static MPValues checkMPValues = new MPValues();

        /// <summary>
        /// MP Values as described in Min Over Markets
        /// </summary>
        /// <returns></returns>
        public MPValues MPValues(bool inclWeekendVol, int openHour, int openMinute, double pctOfVolumeInVA, string profileType, double sessionLengthInHours)
        {
            return MPValues(Input, inclWeekendVol, openHour, openMinute, pctOfVolumeInVA, profileType, sessionLengthInHours);
        }

        /// <summary>
        /// MP Values as described in Min Over Markets
        /// </summary>
        /// <returns></returns>
        public MPValues MPValues(Data.IDataSeries input, bool inclWeekendVol, int openHour, int openMinute, double pctOfVolumeInVA, string profileType, double sessionLengthInHours)
        {
            if (cacheMPValues != null)
                for (int idx = 0; idx < cacheMPValues.Length; idx++)
                    if (cacheMPValues[idx].InclWeekendVol == inclWeekendVol && cacheMPValues[idx].OpenHour == openHour && cacheMPValues[idx].OpenMinute == openMinute && Math.Abs(cacheMPValues[idx].PctOfVolumeInVA - pctOfVolumeInVA) <= double.Epsilon && cacheMPValues[idx].ProfileType == profileType && Math.Abs(cacheMPValues[idx].SessionLengthInHours - sessionLengthInHours) <= double.Epsilon && cacheMPValues[idx].EqualsInput(input))
                        return cacheMPValues[idx];

            lock (checkMPValues)
            {
                checkMPValues.InclWeekendVol = inclWeekendVol;
                inclWeekendVol = checkMPValues.InclWeekendVol;
                checkMPValues.OpenHour = openHour;
                openHour = checkMPValues.OpenHour;
                checkMPValues.OpenMinute = openMinute;
                openMinute = checkMPValues.OpenMinute;
                checkMPValues.PctOfVolumeInVA = pctOfVolumeInVA;
                pctOfVolumeInVA = checkMPValues.PctOfVolumeInVA;
                checkMPValues.ProfileType = profileType;
                profileType = checkMPValues.ProfileType;
                checkMPValues.SessionLengthInHours = sessionLengthInHours;
                sessionLengthInHours = checkMPValues.SessionLengthInHours;

                if (cacheMPValues != null)
                    for (int idx = 0; idx < cacheMPValues.Length; idx++)
                        if (cacheMPValues[idx].InclWeekendVol == inclWeekendVol && cacheMPValues[idx].OpenHour == openHour && cacheMPValues[idx].OpenMinute == openMinute && Math.Abs(cacheMPValues[idx].PctOfVolumeInVA - pctOfVolumeInVA) <= double.Epsilon && cacheMPValues[idx].ProfileType == profileType && Math.Abs(cacheMPValues[idx].SessionLengthInHours - sessionLengthInHours) <= double.Epsilon && cacheMPValues[idx].EqualsInput(input))
                            return cacheMPValues[idx];

                MPValues indicator = new MPValues();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.InclWeekendVol = inclWeekendVol;
                indicator.OpenHour = openHour;
                indicator.OpenMinute = openMinute;
                indicator.PctOfVolumeInVA = pctOfVolumeInVA;
                indicator.ProfileType = profileType;
                indicator.SessionLengthInHours = sessionLengthInHours;
                Indicators.Add(indicator);
                indicator.SetUp();

                MPValues[] tmp = new MPValues[cacheMPValues == null ? 1 : cacheMPValues.Length + 1];
                if (cacheMPValues != null)
                    cacheMPValues.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheMPValues = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// MP Values as described in Min Over Markets
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.MPValues MPValues(bool inclWeekendVol, int openHour, int openMinute, double pctOfVolumeInVA, string profileType, double sessionLengthInHours)
        {
            return _indicator.MPValues(Input, inclWeekendVol, openHour, openMinute, pctOfVolumeInVA, profileType, sessionLengthInHours);
        }

        /// <summary>
        /// MP Values as described in Min Over Markets
        /// </summary>
        /// <returns></returns>
        public Indicator.MPValues MPValues(Data.IDataSeries input, bool inclWeekendVol, int openHour, int openMinute, double pctOfVolumeInVA, string profileType, double sessionLengthInHours)
        {
            return _indicator.MPValues(input, inclWeekendVol, openHour, openMinute, pctOfVolumeInVA, profileType, sessionLengthInHours);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// MP Values as described in Min Over Markets
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.MPValues MPValues(bool inclWeekendVol, int openHour, int openMinute, double pctOfVolumeInVA, string profileType, double sessionLengthInHours)
        {
            return _indicator.MPValues(Input, inclWeekendVol, openHour, openMinute, pctOfVolumeInVA, profileType, sessionLengthInHours);
        }

        /// <summary>
        /// MP Values as described in Min Over Markets
        /// </summary>
        /// <returns></returns>
        public Indicator.MPValues MPValues(Data.IDataSeries input, bool inclWeekendVol, int openHour, int openMinute, double pctOfVolumeInVA, string profileType, double sessionLengthInHours)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.MPValues(input, inclWeekendVol, openHour, openMinute, pctOfVolumeInVA, profileType, sessionLengthInHours);
        }
    }
}
#endregion
