// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// This indicator is a real-time indicator and does not plot against historical data. Plots a histogram splitting volume between trades at the ask or higher and trades at the bid and lower, with the higher value always on top.
	/// </summary>
	[Description("This indicator is a real-time indicator and does not plot against historical data. Plots a histogram splitting volume between trades at the ask or higher and trades at the bid and lower, with the higher value always on top.")]
	public class BuySellVolumeTop : Indicator
	{
		#region Variables
		private int                             activeBar       = -1;
		private System.Collections.ArrayList    alBuys          = new System.Collections.ArrayList();
		private System.Collections.ArrayList    alSells         = new System.Collections.ArrayList();
        private double                          askPrice        = 0;
        private double                          bidPrice        = 0;
		private double                          buys            = 0;
		private bool                            firstPaint      = true;
		private double                          sells           = 0;
		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Plot(new Pen(Color.Red, 6), PlotStyle.Bar, "Sells"));	// Values[0]
			Add(new Plot(new Pen(Color.Green, 6), PlotStyle.Bar, "Buys"));	// Values[1]
			Add(new Plot(new Pen(Color.Green, 6), PlotStyle.Bar, "Buys"));	// Values[2]
			Add(new Plot(new Pen(Color.Red, 6), PlotStyle.Bar, "Sells"));	// Values[3]
			CalculateOnBarClose = false;
			DisplayInDataBox = false;
			PaintPriceMarkers = false;
			PlotsConfigurable = true;
		}

		/// <summary>
		/// Called on each bar update event (incoming tick)
		/// </summary>
		protected override void OnBarUpdate()
		{
			if (CurrentBar < activeBar)
			{
				if ((double)alSells[CurrentBar] < (double)alBuys[CurrentBar])
				{
					Values[0].Set((double)alSells[CurrentBar]);
					Values[1].Set((double)alSells[CurrentBar] + (double)alBuys[CurrentBar]);
				}
				else
				{
					Values[2].Set((double)alBuys[CurrentBar]);
					Values[3].Set((double)alBuys[CurrentBar] + (double)alSells[CurrentBar]);
				}
				return;
			}
			else if (CurrentBar != activeBar)
			{
				alBuys.Insert(Math.Max(activeBar, 0), Historical ? 0 : buys);
				alSells.Insert(Math.Max(activeBar, 0), Historical ? 0 : sells);
				buys = 0;
				sells = 0;
				activeBar = CurrentBar;
			}

			if (firstPaint)
				firstPaint = false;
			else if (!Historical)
			{
				if (sells < buys)
				{
					Values[0].Set(sells);
					Values[1].Set(buys + sells);
				}
				else
				{
					Values[2].Set(buys);
					Values[3].Set(buys + sells);
				}					
			}
		}

        /// <summary>
        /// Called on each incoming real time market data event
        /// </summary>
        protected override void OnMarketData(MarketDataEventArgs e)
        {
            if (e.MarketDataType == MarketDataType.Ask)
                askPrice = e.Price;
            else if (e.MarketDataType == MarketDataType.Bid)
                bidPrice = e.Price;
            else if (e.MarketDataType == MarketDataType.Last)
            {
                if (askPrice > 0 && e.Price >= askPrice)
                    buys += e.Volume;
                else if (bidPrice > 0 && e.Price <= bidPrice)
                    sells += e.Volume;
            }    
        }
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private BuySellVolumeTop[] cacheBuySellVolumeTop = null;

        private static BuySellVolumeTop checkBuySellVolumeTop = new BuySellVolumeTop();

        /// <summary>
        /// This indicator is a real-time indicator and does not plot against historical data. Plots a histogram splitting volume between trades at the ask or higher and trades at the bid and lower, with the higher value always on top.
        /// </summary>
        /// <returns></returns>
        public BuySellVolumeTop BuySellVolumeTop()
        {
            return BuySellVolumeTop(Input);
        }

        /// <summary>
        /// This indicator is a real-time indicator and does not plot against historical data. Plots a histogram splitting volume between trades at the ask or higher and trades at the bid and lower, with the higher value always on top.
        /// </summary>
        /// <returns></returns>
        public BuySellVolumeTop BuySellVolumeTop(Data.IDataSeries input)
        {

            if (cacheBuySellVolumeTop != null)
                for (int idx = 0; idx < cacheBuySellVolumeTop.Length; idx++)
                    if (cacheBuySellVolumeTop[idx].EqualsInput(input))
                        return cacheBuySellVolumeTop[idx];

            BuySellVolumeTop indicator = new BuySellVolumeTop();
            indicator.SetUp();
            indicator.BarsRequired = BarsRequired;
            indicator.CalculateOnBarClose = CalculateOnBarClose;
            indicator.Input = input;

            BuySellVolumeTop[] tmp = new BuySellVolumeTop[cacheBuySellVolumeTop == null ? 1 : cacheBuySellVolumeTop.Length + 1];
            if (cacheBuySellVolumeTop != null)
                cacheBuySellVolumeTop.CopyTo(tmp, 0);
            tmp[tmp.Length - 1] = indicator;
            cacheBuySellVolumeTop = tmp;
            Indicators.Add(indicator);

            return indicator;
        }

    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// This indicator is a real-time indicator and does not plot against historical data. Plots a histogram splitting volume between trades at the ask or higher and trades at the bid and lower, with the higher value always on top.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.BuySellVolumeTop BuySellVolumeTop()
        {
            return _indicator.BuySellVolumeTop(Input);
        }

        /// <summary>
        /// This indicator is a real-time indicator and does not plot against historical data. Plots a histogram splitting volume between trades at the ask or higher and trades at the bid and lower, with the higher value always on top.
        /// </summary>
        /// <returns></returns>
        public Indicator.BuySellVolumeTop BuySellVolumeTop(Data.IDataSeries input)
        {
            return _indicator.BuySellVolumeTop(input);
        }

    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// This indicator is a real-time indicator and does not plot against historical data. Plots a histogram splitting volume between trades at the ask or higher and trades at the bid and lower, with the higher value always on top.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.BuySellVolumeTop BuySellVolumeTop()
        {
            return _indicator.BuySellVolumeTop(Input);
        }

        /// <summary>
        /// This indicator is a real-time indicator and does not plot against historical data. Plots a histogram splitting volume between trades at the ask or higher and trades at the bid and lower, with the higher value always on top.
        /// </summary>
        /// <returns></returns>
        public Indicator.BuySellVolumeTop BuySellVolumeTop(Data.IDataSeries input)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.BuySellVolumeTop(input);
        }

    }
}
#endregion
