// 
// Copyright (C) 2007, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//
#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Triple Exponential Moving Average
    /// </summary>
    [Description("Triple Exponential Moving Average")]
	[Gui.Design.DisplayName("TEMASlope (Triple Exponential Moving Average)")]
    public class TEMASlope : Indicator
    {
        #region Variables
            private int period = 14;
			private DataSeries ema1;
			private DataSeries ema2;
			private DataSeries ema3;
		
			private double value = 0, value1 = 0;
		
			private DataSeries 	mASlope;   //used to store the value in degrees of the indicator line
			private double		radToDegrees = 180/Math.PI; // to convert Radians to Degrees for slope calc
			private bool showslope = false; // switch displaying slope on chart or not
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
			Add(new Plot(Color.Orange, "TEMASlope"));
			ema1 = new DataSeries(this);
			ema2 = new DataSeries(this);
			ema3 = new DataSeries(this);
			
			mASlope = new DataSeries( this );
			
            Overlay				= true;
            PriceTypeSupported	= true;
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
			ema1.Set(EMA(Input, Period)[0]);
			ema2.Set(EMA(ema1, Period)[0]);
			ema3.Set(EMA(ema2, Period)[0]);
            Value.Set(3 * ema1[0] - 3 * ema2[0] + ema3[0]);
			
			value1 = value;
			value = 3 * ema1[0] - 3 * ema2[0] + ema3[0];
			mASlope.Set((int)(radToDegrees*(Math.Atan((value-value1)/TickSize))));
			
			if(showslope)	DrawText("SlopeValue"+CurrentBar,mASlope[0].ToString(),0,High[0]+3*TickSize,Color.LimeGreen);
			
		
        }

        #region Properties

        [Description("Number of bars used for calculations")]
        [Category("Parameters")]
        public int Period
        {
            get { return period; }
            set { period = Math.Max(1, value); }
        }
		[Description("Show Slope value on chart")]
        [Category("Parameters")]
        public bool Showslope
        {
            get { return showslope; }
            set { showslope = value; }
        }
		
		
		public DataSeries MASlope
        {
            get 
			{
			Update();
			 return mASlope; 
			}
        }
        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private TEMASlope[] cacheTEMASlope = null;

        private static TEMASlope checkTEMASlope = new TEMASlope();

        /// <summary>
        /// Triple Exponential Moving Average
        /// </summary>
        /// <returns></returns>
        public TEMASlope TEMASlope(int period, bool showslope)
        {
            return TEMASlope(Input, period, showslope);
        }

        /// <summary>
        /// Triple Exponential Moving Average
        /// </summary>
        /// <returns></returns>
        public TEMASlope TEMASlope(Data.IDataSeries input, int period, bool showslope)
        {
            checkTEMASlope.Period = period;
            period = checkTEMASlope.Period;
            checkTEMASlope.Showslope = showslope;
            showslope = checkTEMASlope.Showslope;

            if (cacheTEMASlope != null)
                for (int idx = 0; idx < cacheTEMASlope.Length; idx++)
                    if (cacheTEMASlope[idx].Period == period && cacheTEMASlope[idx].Showslope == showslope && cacheTEMASlope[idx].EqualsInput(input))
                        return cacheTEMASlope[idx];

            TEMASlope indicator = new TEMASlope();
            indicator.BarsRequired = BarsRequired;
            indicator.CalculateOnBarClose = CalculateOnBarClose;
            indicator.Input = input;
            indicator.Period = period;
            indicator.Showslope = showslope;
            indicator.SetUp();

            TEMASlope[] tmp = new TEMASlope[cacheTEMASlope == null ? 1 : cacheTEMASlope.Length + 1];
            if (cacheTEMASlope != null)
                cacheTEMASlope.CopyTo(tmp, 0);
            tmp[tmp.Length - 1] = indicator;
            cacheTEMASlope = tmp;
            Indicators.Add(indicator);

            return indicator;
        }

    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Triple Exponential Moving Average
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.TEMASlope TEMASlope(int period, bool showslope)
        {
            return _indicator.TEMASlope(Input, period, showslope);
        }

        /// <summary>
        /// Triple Exponential Moving Average
        /// </summary>
        /// <returns></returns>
        public Indicator.TEMASlope TEMASlope(Data.IDataSeries input, int period, bool showslope)
        {
            return _indicator.TEMASlope(input, period, showslope);
        }

    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Triple Exponential Moving Average
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.TEMASlope TEMASlope(int period, bool showslope)
        {
            return _indicator.TEMASlope(Input, period, showslope);
        }

        /// <summary>
        /// Triple Exponential Moving Average
        /// </summary>
        /// <returns></returns>
        public Indicator.TEMASlope TEMASlope(Data.IDataSeries input, int period, bool showslope)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.TEMASlope(input, period, showslope);
        }

    }
}
#endregion
