// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// The Tick Counter displays the current tick count of a bar.
	/// </summary>
	[Description("Displays the range countdown of a bar, time between last 5 bars, and sound before range finish.")]
	public class RangeTimer : Indicator
	{
		#region Variables
		private bool			countDown			= true;
		private bool			isRangeDerivate		= false;
		private bool			rangeDerivateChecked= false;
		private StringFormat	stringFormat		= new StringFormat();
		private SolidBrush		textBrush			= new SolidBrush(Color.White);
		private Font			textFont			= new Font("Arial", 30);
		private float			textWidth			= 10;
		private float			textHeight			= 10;
		private float			textWidth2			= 10;
		private string			noRangeMessage		= "Range Counter only works on Range bars.";
		private float			noRangeTextWidth	= 10;
		private float			noRangeTextHeight	= 10;
		private string	alertFile			= "Alert2.wav";
		private int		alertInterval		= 10;
		private int		alertTicks			= 2;
		#endregion
		private System.Windows.Forms.Timer		timer;	// for minutes timer

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			ChartOnly			= true;
			Overlay				= true;
			CalculateOnBarClose = false;
			
		}

		/// <summary>
		/// </summary>
		protected override void OnBarUpdate()
		{
			if (rangeDerivateChecked)
				return;
			if (ChartControl == null || ChartControl.Bars == null || ChartControl.Bars.Length == 0)
				return;

			if (Data.BarsType.GetInstance(ChartControl.Bars[0].Period.Id).BuiltFrom == Data.PeriodType.Tick && ChartControl.Bars[0].Period.ToString().IndexOf("Range") >= 0)
				isRangeDerivate = true;
			rangeDerivateChecked = true;
		}

		public override void Plot(Graphics graphics, Rectangle bounds, double min, double max)
		{
		string	timeGo;
			
			if (Bars == null)
				return;

			// Recalculate the proper string size should the chart control object font and axis color change
			if (textBrush.Color != ChartControl.AxisColor || textFont != ChartControl.Font)
			{
				textBrush.Color = ChartControl.AxisColor;
				textFont = ChartControl.Font;

				SizeF size = graphics.MeasureString((CountDown ? "Range remaining = %" : "Range count = %") + Bars.Period.Value, textFont);
				textWidth		= size.Width + 20;
				textHeight		= size.Height + 20;

				SizeF noTickSize = graphics.MeasureString(noRangeMessage, textFont);
				noRangeTextWidth = noTickSize.Width + 5;
				noRangeTextHeight = noTickSize.Height + 5;
				
				size = graphics.MeasureString("00:00:00", textFont );
				textWidth2 = size.Width + 20;

			}

			// Calculate timer
			if (timer == null && DisplayTime())
           	{
               	timer = new System.Windows.Forms.Timer();
               	timer.Interval = 1000;
               	timer.Tick += new EventHandler(OnTimerTick);
               	timer.Enabled = true;
           	}

			
			// Plot the range count message to the lower right hand corner of the chart
			if (Bars.Period.Id == PeriodType.Range || isRangeDerivate)
			{
				int	actualRange	= (int) Math.Round(Math.Max(Close[0] - Low[0], High[0] - Close[0]) / Bars.Instrument.MasterInstrument.TickSize);
				int	rangeCount	= CountDown ? Bars.Period.Value - actualRange : actualRange;
				graphics.DrawString((CountDown ? "Range remaining = " + rangeCount : "Range count = " + rangeCount), ChartControl.Font, textBrush, bounds.X + bounds.Width - textWidth, bounds.Y + bounds.Height - textHeight, stringFormat);
				if (Bars.Period.Value - actualRange <= @AlertTicks )
				    Alert("myAlert", NinjaTrader.Cbi.Priority.High, "Reached threshold", alertFile, 10, Color.Black, Color.Yellow);
			// Draw timer
				if (DisplayTime())
				{
					if (timer != null && !timer.Enabled)
						timer.Enabled = true;
//					TimeSpan barTime = Bars.Get(Bars.Count - 1).Time.Subtract(Now);
					TimeSpan barTime0 = Time[0]-Time[1];
					TimeSpan barTime1 = Time[1] - Time[2];
					TimeSpan barTime2 = Time[2] - Time[3];
					TimeSpan barTime3 = Time[3] - Time[4];
					TimeSpan barTime4 = Time[4] - Time[5];
					
					timeGo = barTime0.Hours.ToString("00") + ":" + barTime0.Minutes.ToString("00") + ":" + barTime0.Seconds.ToString("00");
					graphics.DrawString(timeGo, ChartControl.Font, textBrush, bounds.X+bounds.Width-textWidth2, bounds.Y + bounds.Height - 2*textHeight, stringFormat );
					
					timeGo = barTime1.Hours.ToString("00") + ":" + barTime1.Minutes.ToString("00") + ":" + barTime1.Seconds.ToString("00");
					graphics.DrawString(timeGo, ChartControl.Font, textBrush, bounds.X+bounds.Width-textWidth2, bounds.Y + bounds.Height - 3*textHeight, stringFormat );
					
					timeGo = barTime2.Hours.ToString("00") + ":" + barTime2.Minutes.ToString("00") + ":" + barTime2.Seconds.ToString("00");
					graphics.DrawString(timeGo, ChartControl.Font, textBrush, bounds.X+bounds.Width-textWidth2, bounds.Y + bounds.Height - 4*textHeight, stringFormat );

					timeGo = barTime3.Hours.ToString("00") + ":" + barTime3.Minutes.ToString("00") + ":" + barTime3.Seconds.ToString("00");
					graphics.DrawString(timeGo, ChartControl.Font, textBrush, bounds.X+bounds.Width-textWidth2, bounds.Y + bounds.Height - 5*textHeight, stringFormat );

					timeGo = barTime4.Hours.ToString("00") + ":" + barTime4.Minutes.ToString("00") + ":" + barTime4.Seconds.ToString("00");
					graphics.DrawString(timeGo, ChartControl.Font, textBrush, bounds.X+bounds.Width-textWidth2, bounds.Y + bounds.Height - 6*textHeight, stringFormat );

				}

			}
			else
				graphics.DrawString(noRangeMessage, ChartControl.Font, textBrush, bounds.X + bounds.Width - noRangeTextWidth, bounds.Y + bounds.Height - noRangeTextHeight, stringFormat);
		}
		
		// Additional Minutes Timer Code
		private  bool DisplayTime()
        {
			if (ChartControl != null
				&& Bars != null
				&& Bars.Count > 0
				&& Bars.MarketData != null
//				&& Bars.MarketData.Connection.PriceStatus == Cbi.ConnectionStatus.Connected
				&& (Bars.MarketData.Connection.Options.Provider != Cbi.Provider.OpenTick || !(Bars.MarketData.Connection.Options as Cbi.OpenTickOptions).UseDelayedData))
				return true;

            return false;
        }

		private DateTime Now
		{
			get { return (Bars.MarketData.Connection.Options.Provider == Cbi.Provider.Replay ? Bars.MarketData.Connection.Now : DateTime.Now); }
		}

        private void OnTimerTick(object sender, EventArgs e)
        {
            if (DisplayTime())
                ChartControl.ChartPanel.Invalidate();
        }

		#region Properties
		/// <summary>
		/// </summary>
		[Gui.Design.VisualizationOnly()]
		[Description("Indicates if the indicator displays remaining ticks or current ticks of a bar.")]
		[Category("Parameters")]
		[Gui.Design.DisplayName("Count down")]
		public bool CountDown
		{
			get { return countDown; }
			set { countDown = value; }
		}

				[Description("How much ticks before Range finish?")]
				[Category("Sound and Display")]
				[Gui.Design.DisplayName("Ticks before end")]
				public int AlertTicks
				{
					get { return alertTicks; }
					set { alertTicks = value; }
				}

				[Description("How often do you want the alert to sound?")]
				[Category("Sound and Display")]
				[Gui.Design.DisplayName("Alert Interval (sec.)")]
				public int AlertInterval
				{
					get { return alertInterval; }
					set { alertInterval = value; }
				}
		
				[Description("Where is the alert sound file located?  File name of .wav file to play. Provide either the absolute file path or just the name if the file is located in NinjaTrader Installation Folder sounds folder")]
				[Category("Sound and Display")]
				[Gui.Design.DisplayName("Alert .wav File")]
				public string AlertFile
				{
					get { return alertFile; }
					set { alertFile = value; }
				}


        #endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private RangeTimer[] cacheRangeTimer = null;

        private static RangeTimer checkRangeTimer = new RangeTimer();

        /// <summary>
        /// Displays the range countdown of a bar, time between last 5 bars, and sound before range finish.
        /// </summary>
        /// <returns></returns>
        public RangeTimer RangeTimer(bool countDown)
        {
            return RangeTimer(Input, countDown);
        }

        /// <summary>
        /// Displays the range countdown of a bar, time between last 5 bars, and sound before range finish.
        /// </summary>
        /// <returns></returns>
        public RangeTimer RangeTimer(Data.IDataSeries input, bool countDown)
        {
            if (cacheRangeTimer != null)
                for (int idx = 0; idx < cacheRangeTimer.Length; idx++)
                    if (cacheRangeTimer[idx].CountDown == countDown && cacheRangeTimer[idx].EqualsInput(input))
                        return cacheRangeTimer[idx];

            lock (checkRangeTimer)
            {
                checkRangeTimer.CountDown = countDown;
                countDown = checkRangeTimer.CountDown;

                if (cacheRangeTimer != null)
                    for (int idx = 0; idx < cacheRangeTimer.Length; idx++)
                        if (cacheRangeTimer[idx].CountDown == countDown && cacheRangeTimer[idx].EqualsInput(input))
                            return cacheRangeTimer[idx];

                RangeTimer indicator = new RangeTimer();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.CountDown = countDown;
                Indicators.Add(indicator);
                indicator.SetUp();

                RangeTimer[] tmp = new RangeTimer[cacheRangeTimer == null ? 1 : cacheRangeTimer.Length + 1];
                if (cacheRangeTimer != null)
                    cacheRangeTimer.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheRangeTimer = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Displays the range countdown of a bar, time between last 5 bars, and sound before range finish.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.RangeTimer RangeTimer(bool countDown)
        {
            return _indicator.RangeTimer(Input, countDown);
        }

        /// <summary>
        /// Displays the range countdown of a bar, time between last 5 bars, and sound before range finish.
        /// </summary>
        /// <returns></returns>
        public Indicator.RangeTimer RangeTimer(Data.IDataSeries input, bool countDown)
        {
            return _indicator.RangeTimer(input, countDown);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Displays the range countdown of a bar, time between last 5 bars, and sound before range finish.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.RangeTimer RangeTimer(bool countDown)
        {
            return _indicator.RangeTimer(Input, countDown);
        }

        /// <summary>
        /// Displays the range countdown of a bar, time between last 5 bars, and sound before range finish.
        /// </summary>
        /// <returns></returns>
        public Indicator.RangeTimer RangeTimer(Data.IDataSeries input, bool countDown)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.RangeTimer(input, countDown);
        }
    }
}
#endregion
