// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// The MACD_Color (Moving Average Convergence/Divergence) is a trend following momentum indicator that shows the relationship between two moving averages of prices.
	/// </summary>
	[Description("The MACD_Color (Moving Average Convergence/Divergence) is a trend following momentum indicator that shows the relationship between two moving averages of prices.")]
	public class MACD_Color : Indicator
	{
		#region Variables
		private int					fast	= 12;
		private int					slow	= 26;
		private int					smooth	= 9;
		
		private Color macdColorUp = Color.DarkGreen;
		private Color macdColorDown = Color.Red;
		private Color avgColorUp = Color.DarkGreen;
		private Color avgColorDown = Color.Red;
		private Color histColorUp = Color.DarkGreen;
		private Color histColorDown = Color.Red;
		
		private bool		showEntries = false;//true;
		private bool		alerts = false;
		
		
		private	DataSeries		fastEma;
		private	DataSeries		slowEma;
		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Plot(Color.Green, "Macd"));
			Add(new Plot(Color.DarkViolet, "Avg"));
			Add(new Plot(Color.FromKnownColor(KnownColor.SkyBlue), PlotStyle.Bar, "Diff"));
			
			// Increase the default widths of the plots
			Plots[0].Pen.Width = 1;
			Plots[1].Pen.Width = 1;
			Plots[2].Pen.Width = 3;
			
			Add(new Line(Color.DarkGray, 0, "Zero line"));

			fastEma	= new DataSeries(this);
			slowEma	= new DataSeries(this);
			
			CalculateOnBarClose = false;
			DrawOnPricePanel = false;
		}

		/// <summary>
		/// Calculates the indicator value(s) at the current index.
		/// </summary>
		protected override void OnBarUpdate()
		{
			if (CurrentBar < fast + slow + smooth + 2)
			{
				fastEma.Set(Input[0]);
				slowEma.Set(Input[0]);
				Value.Set(0);
				Avg.Set(0);
				Diff.Set(0);
			}
			else
			{
				fastEma.Set((2.0 / (1 + Fast)) * Input[0] + (1 - (2.0 / (1 + Fast))) * fastEma[1]);
				slowEma.Set((2.0 / (1 + Slow)) * Input[0] + (1 - (2.0 / (1 + Slow))) * slowEma[1]);

				double macd		= fastEma[0] - slowEma[0];
				double macdAvg	= (2.0 / (1 + Smooth)) * macd + (1 - (2.0 / (1 + Smooth))) * Avg[1];
				
			Value.Set(macd);
			
			if (Value[0]>0)
			{
				// The indexers for PlotColors are PlotColors[plot index][bars back], so the below code would set the first plot (Upper) to black and the second plot (Lower) to green.
				PlotColors[0][0] = macdColorUp;
			}
			// If the average is falling, change the plot colors.
			else if (Value[0]<0)
			{
				PlotColors[0][0] = macdColorDown;
			}
			// If the average remains the same, set both plots to the same color.
			else
			{
				PlotColors[0][0] = Color.DarkGoldenrod;
			}
				
				
			Avg.Set(macdAvg);
			
			if (Avg[0]>0)
			{
				// The indexers for PlotColors are PlotColors[plot index][bars back], so the below code would set the first plot (Upper) to black and the second plot (Lower) to green.
				PlotColors[1][0] = avgColorUp;
			}
			// If the average is falling, change the plot colors.
			else if (Avg[0]<0)
			{
				PlotColors[1][0] = avgColorDown;
			}
			// If the average remains the same, set both plots to the same color.
			else
			{
				PlotColors[1][0] = Color.DarkGoldenrod;
			}
			
			
			Diff.Set(macd - macdAvg);
				
			// If the average of the two plots is rising, change the plot colors.
			if (Diff[0]>0)
			{
				// The indexers for PlotColors are PlotColors[plot index][bars back], so the below code would set the first plot (Upper) to black and the second plot (Lower) to green.
				PlotColors[2][0] = histColorUp;
			}
			// If the average is falling, change the plot colors.
			else if (Diff[0]<0)
			{
				PlotColors[2][0] = histColorDown;
			}
			// If the average remains the same, set both plots to the same color.
			else
			{
				PlotColors[2][0] = Color.DarkGoldenrod;
			}
				
				
			// Paint Bars	
				
				if (macd>0 && macd>macdAvg)
					BarColor = Color.LimeGreen;
				
				if (macd<0 && macd<macdAvg)
					BarColor = Color.Red;
				
				if (macd<0 && macd>macdAvg || macd>0 && macd<macdAvg)
					BarColor = Color.White;
				
			
				if(showEntries == true && CrossAbove(Value, 0, 1))	
				{DrawTriangleUp(CurrentBar.ToString() + "Up", true, 0, 0, Color.Green);}
				
				if (alerts == true && CrossAbove(Value, 0, 1))
				{Alert(CurrentBar.ToString() + "UpAlert",NinjaTrader.Cbi.Priority.Medium,"ZeroCrossup","Alert1.wav",6,Color.Black,Color.Yellow);}
		
				if(showEntries == true && CrossBelow(Value, 0, 1))	
				{DrawTriangleDown(CurrentBar.ToString() + "Down", true, 0, 0, Color.Red);}
				
				if (alerts == true && CrossBelow(Value, 0, 1))
				{Alert(CurrentBar.ToString() + "Downalert",NinjaTrader.Cbi.Priority.Medium,"ZeroCrossdown","Alert1.wav",6,Color.Black,Color.Yellow);}
			
			}
		
		}

		#region Properties
		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Avg
		{
			get { return Values[1]; }
		}

		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Default
		{
			get { return Values[0]; }
		}
		
        /// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Diff
		{
			get { return Values[2]; }
		}

		/// <summary>
		/// </summary>
		[Description("Number of bars for fast EMA")]
		[Category("Parameters")]
		public int Fast
		{
			get { return fast; }
			set { fast = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Description("Number of bars for slow EMA")]
		[Category("Parameters")]
		public int Slow
		{
			get { return slow; }
			set { slow = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Description("Number of bars for smoothing")]
		[Category("Parameters")]
		public int Smooth
		{
			get { return smooth; }
			set { smooth = Math.Max(1, value); }
		}
		
		
		
		[XmlIgnore()]
        [Description("01. Macd Color Over Zero")]
        [Category("Colors")]
        [Gui.Design.DisplayNameAttribute("01. Macd Color Over Zero")]
        public Color MacdColorUp
        {
            get { return macdColorUp; }
            set { macdColorUp = value; }
        }
		
        /// <summary>
        /// </summary>
        [Browsable(false)]
        public string macdColorUpSerialize
        {
            get { return NinjaTrader.Gui.Design.SerializableColor.ToString(macdColorUp); }
            set { macdColorUp = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
        }

		[XmlIgnore()]
        [Description("02. Macd Color Under Zero")]
        [Category("Colors")]
        [Gui.Design.DisplayNameAttribute("02. Macd Color under Zero")]
        public Color MacdColorDown
        {
            get { return macdColorDown; }
            set { macdColorDown = value; }
        }
		
        /// <summary>
        /// </summary>
        [Browsable(false)]
        public string macdColorDownSerialize
        {
            get { return NinjaTrader.Gui.Design.SerializableColor.ToString(macdColorDown); }
            set { macdColorDown = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
        }
		
		
		[XmlIgnore()]
        [Description("03. Avg Color Over Zero")]
        [Category("Colors")]
        [Gui.Design.DisplayNameAttribute("03. Avg Color Over Zero")]
        public Color AvgColorUp
        {
            get { return avgColorUp; }
            set { avgColorUp = value; }
        }
		
        /// <summary>
        /// </summary>
        [Browsable(false)]
        public string avgColorUpSerialize
        {
            get { return NinjaTrader.Gui.Design.SerializableColor.ToString(avgColorUp); }
            set { avgColorUp = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
        }

		[XmlIgnore()]
        [Description("04. Avg Color Under Zero")]
        [Category("Colors")]
        [Gui.Design.DisplayNameAttribute("04. Avg Color under Zero")]
        public Color AvgColorDown
        {
            get { return avgColorDown; }
            set { avgColorDown = value; }
        }
		
        /// <summary>
        /// </summary>
        [Browsable(false)]
        public string avgColorDownSerialize
        {
            get { return NinjaTrader.Gui.Design.SerializableColor.ToString(avgColorDown); }
            set { avgColorDown = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
        }
		
		[XmlIgnore()]
        [Description("05. Hist Color Over Zero")]
        [Category("Colors")]
        [Gui.Design.DisplayNameAttribute("05. Hist Color Over Zero")]
        public Color HistColorUp
        {
            get { return histColorUp; }
            set { histColorUp = value; }
        }
		
        /// <summary>
        /// </summary>
        [Browsable(false)]
        public string histColorUpSerialize
        {
            get { return NinjaTrader.Gui.Design.SerializableColor.ToString(histColorUp); }
            set { histColorUp = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
        }

		[XmlIgnore()]
        [Description("06. Hist Color Under Zero")]
        [Category("Colors")]
        [Gui.Design.DisplayNameAttribute("06. Hist Color under Zero")]
        public Color HistColorDown
        {
            get { return histColorDown; }
            set { histColorDown = value; }
        }
		
        /// <summary>
        /// </summary>
        [Browsable(false)]
        public string histColorDownSerialize
        {
            get { return NinjaTrader.Gui.Design.SerializableColor.ToString(histColorDown); }
            set { histColorDown = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
        }
		
		[Description("Show Entry Triangles")]
        [Category("Display Parameters")]
        public bool ShowEntries
        {
            get { return showEntries; }
            set { showEntries = value; }
        }
		
		[Description("Sound alert warning.")]
		[Gui.Design.DisplayName ("Turn sound alert on?")]
        [Category("Display Parameters")]
        public bool Alerts
        {
            get { return alerts; }
            set { alerts = value; }
        }
		
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private MACD_Color[] cacheMACD_Color = null;

        private static MACD_Color checkMACD_Color = new MACD_Color();

        /// <summary>
        /// The MACD_Color (Moving Average Convergence/Divergence) is a trend following momentum indicator that shows the relationship between two moving averages of prices.
        /// </summary>
        /// <returns></returns>
        public MACD_Color MACD_Color(int fast, int slow, int smooth)
        {
            return MACD_Color(Input, fast, slow, smooth);
        }

        /// <summary>
        /// The MACD_Color (Moving Average Convergence/Divergence) is a trend following momentum indicator that shows the relationship between two moving averages of prices.
        /// </summary>
        /// <returns></returns>
        public MACD_Color MACD_Color(Data.IDataSeries input, int fast, int slow, int smooth)
        {
            if (cacheMACD_Color != null)
                for (int idx = 0; idx < cacheMACD_Color.Length; idx++)
                    if (cacheMACD_Color[idx].Fast == fast && cacheMACD_Color[idx].Slow == slow && cacheMACD_Color[idx].Smooth == smooth && cacheMACD_Color[idx].EqualsInput(input))
                        return cacheMACD_Color[idx];

            lock (checkMACD_Color)
            {
                checkMACD_Color.Fast = fast;
                fast = checkMACD_Color.Fast;
                checkMACD_Color.Slow = slow;
                slow = checkMACD_Color.Slow;
                checkMACD_Color.Smooth = smooth;
                smooth = checkMACD_Color.Smooth;

                if (cacheMACD_Color != null)
                    for (int idx = 0; idx < cacheMACD_Color.Length; idx++)
                        if (cacheMACD_Color[idx].Fast == fast && cacheMACD_Color[idx].Slow == slow && cacheMACD_Color[idx].Smooth == smooth && cacheMACD_Color[idx].EqualsInput(input))
                            return cacheMACD_Color[idx];

                MACD_Color indicator = new MACD_Color();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.Fast = fast;
                indicator.Slow = slow;
                indicator.Smooth = smooth;
                Indicators.Add(indicator);
                indicator.SetUp();

                MACD_Color[] tmp = new MACD_Color[cacheMACD_Color == null ? 1 : cacheMACD_Color.Length + 1];
                if (cacheMACD_Color != null)
                    cacheMACD_Color.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheMACD_Color = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// The MACD_Color (Moving Average Convergence/Divergence) is a trend following momentum indicator that shows the relationship between two moving averages of prices.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.MACD_Color MACD_Color(int fast, int slow, int smooth)
        {
            return _indicator.MACD_Color(Input, fast, slow, smooth);
        }

        /// <summary>
        /// The MACD_Color (Moving Average Convergence/Divergence) is a trend following momentum indicator that shows the relationship between two moving averages of prices.
        /// </summary>
        /// <returns></returns>
        public Indicator.MACD_Color MACD_Color(Data.IDataSeries input, int fast, int slow, int smooth)
        {
            return _indicator.MACD_Color(input, fast, slow, smooth);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// The MACD_Color (Moving Average Convergence/Divergence) is a trend following momentum indicator that shows the relationship between two moving averages of prices.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.MACD_Color MACD_Color(int fast, int slow, int smooth)
        {
            return _indicator.MACD_Color(Input, fast, slow, smooth);
        }

        /// <summary>
        /// The MACD_Color (Moving Average Convergence/Divergence) is a trend following momentum indicator that shows the relationship between two moving averages of prices.
        /// </summary>
        /// <returns></returns>
        public Indicator.MACD_Color MACD_Color(Data.IDataSeries input, int fast, int slow, int smooth)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.MACD_Color(input, fast, slow, smooth);
        }
    }
}
#endregion
