// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
// Modified by Zondor for multi-color plotting and zero centering between +/-100

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// Multicolor plot zero centered stochastic for NT7, version 001   January 1, 2010
	/// The Stochastic Oscillator is made up of two lines that oscillate between a vertical scale of 0 to 100. The %K is the main line and it is drawn as a solid line. The second is the %D line and is a moving average of %K. The %D line is drawn as a dotted line. Use as a buy/sell signal generator, buying when fast moves above slow and selling when fast moves below slow.
	/// </summary>
	[Description("Osiclllates between 100 and -100. D is a trailing MA of K.")]
	public class StochNT7MultiColor : Indicator
	{
		#region Variables
		private int				periodD	= 4;	// SlowDperiod
		private int				periodK	= 14;	// Kperiod
		private int				smooth	= 3;	// SlowKperiod
		private DataSeries		den;
		private DataSeries		nom;
		private DataSeries      kay;
		private double test1;
		private double test2;
		private Color downColor = Color.Red;
		private Color upColor = Color.CornflowerBlue;
		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Plot(Color.Gray, "D"));  //Plot0
				Plots[0].Pen.DashStyle=DashStyle.Dot;
				Plots[0].Pen.Width =2;
			Add(new Plot(Color.Cyan, "K"));			//Plot1
				Plots[0].Pen.Width=2;

			Add(new Line(Color.YellowGreen, -60, "Lower"));
			Add(new Line(Color.YellowGreen, 60, "Upper"));
			Add(new Line(Color.DimGray, 0, "Center"));
			Lines[2].Pen.DashStyle = DashStyle.Dash;
			den			= new DataSeries(this);
			nom			= new DataSeries(this);
			
		}

		/// <summary>
		/// Calculates the indicator value(s) at the current index.
		/// </summary>
		protected override void OnBarUpdate()
		{
			nom.Set(Close[0] - MIN(Low, PeriodK)[0]);
			den.Set(MAX(High, PeriodK)[0] - MIN(Low, PeriodK)[0]);										// PeriodK = Kperiod
			
			K.Set((200 * SUM(nom, Smooth)[0] / (SUM(den, Smooth)[0] == 0 ? 1.0 : SUM(den, Smooth)[0])) -100   );	// Smooth  = SlowKperiod
			
			
			if(Rising(K))
				PlotColors[1][0] = ColorRising;
			if(Falling(K))
				PlotColors[1][0] = ColorFalling;
			
				D.Set(SMA(K, PeriodD)[0]);																	// PeriodD = SlowDperiod
		}

		#region Properties
		
		[XmlIgnore()]
        [Description("Color for Falling Plot")]
        [Category("Parameters")]
        public Color ColorFalling
        {
            get { return downColor; }
            set { downColor = value; }
        }
		
		// Serialize our Color object
		[Browsable(false)]
		public string ColorFallingSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(downColor); }
			set { downColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}

		[XmlIgnore()]
		[Description("Color for Rising Plot")]
        [Category("Parameters")]
        public Color ColorRising
        {
            get { return upColor; }
            set { upColor = value; }
        }
		
		[Browsable(false)]
		public string ColorRisingSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(upColor); }
			set { upColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		
		/// <summary>
		/// Gets the slow D value.
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries D
		{
			get { return Values[0]; }
		}

		/// <summary>
		/// Gets the slow K value.
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries K
		{
			get { return Values[1]; }
		}
		
		/// <summary>
		/// </summary>
		[Description("Numbers of bars used for moving average over K values")]
		[GridCategory("Parameters")]
		public int PeriodD
		{
			get { return periodD; }
			set { periodD = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Description("Numbers of bars used for calculating the K values")]
		[GridCategory("Parameters")]
		public int PeriodK
		{
			get { return periodK; }
			set { periodK = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Description("Number of bars for smoothing the slow K values")]
		[GridCategory("Parameters")]
		public int Smooth
		{
			get { return smooth; }
			set { smooth = Math.Max(1, value); }
		}
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private StochNT7MultiColor[] cacheStochNT7MultiColor = null;

        private static StochNT7MultiColor checkStochNT7MultiColor = new StochNT7MultiColor();

        /// <summary>
        /// Osiclllates between 100 and -100. D is a trailing MA of K.
        /// </summary>
        /// <returns></returns>
        public StochNT7MultiColor StochNT7MultiColor(Color colorFalling, Color colorRising, int periodD, int periodK, int smooth)
        {
            return StochNT7MultiColor(Input, colorFalling, colorRising, periodD, periodK, smooth);
        }

        /// <summary>
        /// Osiclllates between 100 and -100. D is a trailing MA of K.
        /// </summary>
        /// <returns></returns>
        public StochNT7MultiColor StochNT7MultiColor(Data.IDataSeries input, Color colorFalling, Color colorRising, int periodD, int periodK, int smooth)
        {
            if (cacheStochNT7MultiColor != null)
                for (int idx = 0; idx < cacheStochNT7MultiColor.Length; idx++)
                    if (cacheStochNT7MultiColor[idx].ColorFalling == colorFalling && cacheStochNT7MultiColor[idx].ColorRising == colorRising && cacheStochNT7MultiColor[idx].PeriodD == periodD && cacheStochNT7MultiColor[idx].PeriodK == periodK && cacheStochNT7MultiColor[idx].Smooth == smooth && cacheStochNT7MultiColor[idx].EqualsInput(input))
                        return cacheStochNT7MultiColor[idx];

            lock (checkStochNT7MultiColor)
            {
                checkStochNT7MultiColor.ColorFalling = colorFalling;
                colorFalling = checkStochNT7MultiColor.ColorFalling;
                checkStochNT7MultiColor.ColorRising = colorRising;
                colorRising = checkStochNT7MultiColor.ColorRising;
                checkStochNT7MultiColor.PeriodD = periodD;
                periodD = checkStochNT7MultiColor.PeriodD;
                checkStochNT7MultiColor.PeriodK = periodK;
                periodK = checkStochNT7MultiColor.PeriodK;
                checkStochNT7MultiColor.Smooth = smooth;
                smooth = checkStochNT7MultiColor.Smooth;

                if (cacheStochNT7MultiColor != null)
                    for (int idx = 0; idx < cacheStochNT7MultiColor.Length; idx++)
                        if (cacheStochNT7MultiColor[idx].ColorFalling == colorFalling && cacheStochNT7MultiColor[idx].ColorRising == colorRising && cacheStochNT7MultiColor[idx].PeriodD == periodD && cacheStochNT7MultiColor[idx].PeriodK == periodK && cacheStochNT7MultiColor[idx].Smooth == smooth && cacheStochNT7MultiColor[idx].EqualsInput(input))
                            return cacheStochNT7MultiColor[idx];

                StochNT7MultiColor indicator = new StochNT7MultiColor();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.ColorFalling = colorFalling;
                indicator.ColorRising = colorRising;
                indicator.PeriodD = periodD;
                indicator.PeriodK = periodK;
                indicator.Smooth = smooth;
                Indicators.Add(indicator);
                indicator.SetUp();

                StochNT7MultiColor[] tmp = new StochNT7MultiColor[cacheStochNT7MultiColor == null ? 1 : cacheStochNT7MultiColor.Length + 1];
                if (cacheStochNT7MultiColor != null)
                    cacheStochNT7MultiColor.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheStochNT7MultiColor = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Osiclllates between 100 and -100. D is a trailing MA of K.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.StochNT7MultiColor StochNT7MultiColor(Color colorFalling, Color colorRising, int periodD, int periodK, int smooth)
        {
            return _indicator.StochNT7MultiColor(Input, colorFalling, colorRising, periodD, periodK, smooth);
        }

        /// <summary>
        /// Osiclllates between 100 and -100. D is a trailing MA of K.
        /// </summary>
        /// <returns></returns>
        public Indicator.StochNT7MultiColor StochNT7MultiColor(Data.IDataSeries input, Color colorFalling, Color colorRising, int periodD, int periodK, int smooth)
        {
            return _indicator.StochNT7MultiColor(input, colorFalling, colorRising, periodD, periodK, smooth);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Osiclllates between 100 and -100. D is a trailing MA of K.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.StochNT7MultiColor StochNT7MultiColor(Color colorFalling, Color colorRising, int periodD, int periodK, int smooth)
        {
            return _indicator.StochNT7MultiColor(Input, colorFalling, colorRising, periodD, periodK, smooth);
        }

        /// <summary>
        /// Osiclllates between 100 and -100. D is a trailing MA of K.
        /// </summary>
        /// <returns></returns>
        public Indicator.StochNT7MultiColor StochNT7MultiColor(Data.IDataSeries input, Color colorFalling, Color colorRising, int periodD, int periodK, int smooth)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.StochNT7MultiColor(input, colorFalling, colorRising, periodD, periodK, smooth);
        }
    }
}
#endregion
