﻿/*
 * XPeriodChannel - multi-time frame capable indicator to reference different timeframe to plot channels on lower timeframe.
 * Works with NT 7 and above.
 * 
 * by J.D. Fagan - jd.fagan@gmail.com
 * Copyright (c) 2010
 */

using System;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;

namespace NinjaTrader.Indicator {
    /// <summary>
    /// Intraday indicator for showing the High/Low X Period Channel
    /// </summary>
    [Description("Intraday indicator for showing the High/Low X Period Channel")]
    public class XPeriodChannel : Indicator {
        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize() {
            //Add the other user defined PeriodType as secondary timeframe
            Add(PeriodType, PeriodValue);

            Add(new Plot(Color.FromKnownColor(KnownColor.Blue), PlotStyle.Line, "HighChannel"));
            Add(new Plot(Color.FromKnownColor(KnownColor.Blue), PlotStyle.Line, "MidChannel"));
            Add(new Plot(Color.FromKnownColor(KnownColor.Blue), PlotStyle.Line, "LowChannel"));

            CalculateOnBarClose = false;
            Overlay = true;
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate() {
            if (BarsInProgress == 0) {
                //In the primary timeframe - plot the channel
                HighChannel.Set(highLine);
                MidChannel.Set(midLine);
                LowChannel.Set(lowLine);
            } else if (BarsInProgress == 1) {
                //In the secondary timeframe - calculate channel
                hhv = High[HighestBar(High, Lookback)];
                llv = Low[LowestBar(Low, Lookback)];
                
                midLine = (hhv + llv + Close[0])/3;
                halfOfHighLowDelta = (hhv - llv)/2;

                highLine = halfOfHighLowDelta + midLine;
                lowLine = midLine - halfOfHighLowDelta;

                //Plot out channel prices on chart
                if (ShowPriceOnChart) {
                    string priceFormat = PriceFormat(TickSize);

                    DrawText("HighLine" + CurrentBar, String.Format(priceFormat, highLine), -10, highLine + 2*TickSize, Plots[0].Pen.Color);
                    DrawText("MidLine" + CurrentBar, String.Format(priceFormat, midLine), -10, midLine + 2*TickSize, Plots[1].Pen.Color);
                    DrawText("LowLine" + CurrentBar, String.Format(priceFormat, lowLine), -10, lowLine - 2*TickSize, Plots[2].Pen.Color);
                }
            }
        }

        public string PriceFormat(double tickSize) {
            short precision = 0;
            while (tickSize % 1 > 0.0) {
                tickSize *= 10;
                precision++;
            }
            return precision == 0 ? "{0:G}" : "{0:F" + precision + "}";
        }

        #region Properties
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries HighChannel {
            get { return Values[0]; }
        }

        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries MidChannel {
            get { return Values[1]; }
        }

        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries LowChannel {
            get { return Values[2]; }
        }

        private PeriodType _periodType = PeriodType.Day;
        [Description("Period type from which channel is built from")]
        [GridCategory("01 - Higher Timeframe")]
        [Gui.Design.DisplayName("01  Period Type")]
        public PeriodType PeriodType {
            get { return _periodType; }
            set { _periodType = value; }
        }

        private int _periodValue = 180;
        [Description("Period value to describe timeframe of period type")]
        [GridCategory("01 - Higher Timeframe")]
        [Gui.Design.DisplayName("02  Period Value")]
        public int PeriodValue {
            get { return _periodValue; }
            set { _periodValue = value; }
        }

        private int _lookback = 5; // Default setting for Lookback
        [Description("Lookback period")]
        [GridCategory("02 - Misc")]
        [Gui.Design.DisplayName("01  Lookback")]
        public int Lookback {
            get { return _lookback; }
            set { _lookback = Math.Max(1, value); }
        }

        private bool _showPriceOnChart;
        [Description("Toggle to show channel prices directly on chart")]
        [GridCategory("02 - Misc")]
        [Gui.Design.DisplayName("02  Show Channel Prices")]
        public bool ShowPriceOnChart {
            get { return _showPriceOnChart; }
            set { _showPriceOnChart = value; }
        }
        #endregion

        #region Variables

        protected double hhv = Double.NaN, llv = Double.NaN;
        protected double midLine = Double.NaN, halfOfHighLowDelta = Double.NaN, highLine = Double.NaN, lowLine = Double.NaN;

        #endregion

    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator {
    public partial class Indicator : IndicatorBase {
        private XPeriodChannel[] cacheXDayChannel = null;

        private static XPeriodChannel checkXDayChannel = new XPeriodChannel();

        /// <summary>
        /// Intraday indicator for showing the High/Low X Day Channel
        /// </summary>
        /// <returns></returns>
        public XPeriodChannel XDayChannel(int lookback) {
            return XDayChannel(Input, lookback);
        }

        /// <summary>
        /// Intraday indicator for showing the High/Low X Day Channel
        /// </summary>
        /// <returns></returns>
        public XPeriodChannel XDayChannel(Data.IDataSeries input, int lookback) {
            if (cacheXDayChannel != null)
                for (int idx = 0; idx < cacheXDayChannel.Length; idx++)
                    if (cacheXDayChannel[idx].Lookback == lookback && cacheXDayChannel[idx].EqualsInput(input))
                        return cacheXDayChannel[idx];

            lock (checkXDayChannel) {
                checkXDayChannel.Lookback = lookback;
                lookback = checkXDayChannel.Lookback;

                if (cacheXDayChannel != null)
                    for (int idx = 0; idx < cacheXDayChannel.Length; idx++)
                        if (cacheXDayChannel[idx].Lookback == lookback && cacheXDayChannel[idx].EqualsInput(input))
                            return cacheXDayChannel[idx];

                XPeriodChannel indicator = new XPeriodChannel();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.Lookback = lookback;
                Indicators.Add(indicator);
                indicator.SetUp();

                XPeriodChannel[] tmp = new XPeriodChannel[cacheXDayChannel == null ? 1 : cacheXDayChannel.Length + 1];
                if (cacheXDayChannel != null)
                    cacheXDayChannel.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheXDayChannel = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer {
    public partial class Column : ColumnBase {
        /// <summary>
        /// Intraday indicator for showing the High/Low X Day Channel
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.XPeriodChannel XDayChannel(int lookback) {
            return _indicator.XDayChannel(Input, lookback);
        }

        /// <summary>
        /// Intraday indicator for showing the High/Low X Day Channel
        /// </summary>
        /// <returns></returns>
        public Indicator.XPeriodChannel XDayChannel(Data.IDataSeries input, int lookback) {
            return _indicator.XDayChannel(input, lookback);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy {
    public partial class Strategy : StrategyBase {
        /// <summary>
        /// Intraday indicator for showing the High/Low X Day Channel
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.XPeriodChannel XDayChannel(int lookback) {
            return _indicator.XDayChannel(Input, lookback);
        }

        /// <summary>
        /// Intraday indicator for showing the High/Low X Day Channel
        /// </summary>
        /// <returns></returns>
        public Indicator.XPeriodChannel XDayChannel(Data.IDataSeries input, int lookback) {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.XDayChannel(input, lookback);
        }
    }
}
#endregion
